<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    image-defs.php                                          */
/* Author:      Paul Waite                                              */
/* Description: Definitions for managing and using IMAGES               */
/*                                                                      */
/* ******************************************************************** */
/** @package image */

/** Renderable classes */
include_once("renderable.php");

// ----------------------------------------------------------------------
/**
* Image class
* A class for managing a standard image.
* @package image
*/
class image extends RenderableObject {
  /** Name of the image element */
  var $name = "";
  /** Image ALT tag content */
  var $alt = "";
  /** Image file source URL/path */
  var $src = "";
  /** Width of image in pixels */
  var $width = 0;
  /** Height of image in pixels */
  var $height = 0;
  /** Padding at image sides in pixels */
  var $hspace = 0;
  /** Padding at image top & bottom in pixels */
  var $vspace = 0;
  /** Image alignment with respect to the page:
  * 'left' (default), 'right', 'top', 'bottom', 'middle',
  * 'absbottom', 'absmiddle', 'baseline', 'texttop'
  */
  var $align = "";
  /** Image border with in pixels */
  var $border = 0;
  /** Script to execute on mouse-over */
  var $onmouseover = "";
  /** Script to execute on mouse-out */
  var $onmouseout = "";
  /** Text to display in status area when mouse over image */
  var $linkover_text = "";
  /** Image map to use */
  var $map = "";
  //.....................................................................
  /**
  * Constructor
  * Creates the basic image object.
  * @deprecated
  * @param string  $name   Name of this image element
  * @param string  $src    URL or path to the actual image file
  * @param string  $alt    Image ALT tag content
  * @param integer $width  Width of the image in pixels
  * @param integer $height Height of the image in pixels
  * @param integer $border Size of border around the image in pixels
  * @param integer $hspace Padding to left & right of image in pixels
  * @param integer $vspace Padding top & bottom of image in pixels
  * @param string  $align  Alignment of image with respect to the page
  */
  function image($name, $src, $alt, $width=0, $height=0, $border=0, $hspace=0, $vspace=0, $align="") {
    $this->name = $name;
    $this->src = $src;
    $this->alt = $alt;
    $this->width = $width;
    $this->height = $height;
    $this->border = $border;
    $this->hspace = $hspace;
    $this->vspace = $vspace;
    $this->align = $align;
  } // image
  //.....................................................................
  /**
  * Set image details
  * Allow setting of src and metrics from outside.
  * @param string  $src    URL or path to the actual image file
  * @param integer $width  Width of the image in pixels
  * @param integer $height Height of the image in pixels
  * @param integer $border Size of border around the image in pixels
  * @param integer $hspace Padding to left & right of image in pixels
  * @param integer $vspace Padding top & bottom of image in pixels
  * @param string  $align  Alignment of image with respect to the page
  */
  function set_src($src, $width, $height, $border=0, $hspace=0, $vspace=0, $align="") {
    $this->src = $src;
    $this->width=$width;
    $this->height = $height;
    $this->border = $border;
    $this->hspace = $hspace;
    $this->vspace = $vspace;
    $this->align = $align;
    return $this;
  } // set_src
  //.....................................................................
  /**
  * Set linkover text
  * Defines the text to show when the mouse moves over the image.
  * @param string $txt Text to show in status area when mouse-over.
  */
  function set_linkover_text($txt) {
    $this->linkover_text = $txt;
  } // set_linkover_text
  //.....................................................................
  /**
  * Set image map
  * Defines the image map to use with this image.
  * @param string $map The name of the image map to associate with this image.
  */
  function use_map($map) {
    $this->map = $map;
  } // use_map
  //.....................................................................
  /**
  * Render image as javascript object
  * @return string Javascript code rendering of this image
  */
  function javascript() {
    $js = "";
    if (($this->height > 0) && ($this->width > 0)) {
      $js .= "var " . $this->name . "=new Image(" . $this->width . "," . $this->height . ");\n";
      $js .= "var " . $this->name . ".src=\"" . $this->src . "\";\n";
    }
    return $js;
  } // javascript
  // ....................................................................
  /**
  * This renders the field as WML.
  * @return string The field as WML.
  */
  function wml() {
    return $this->html();
  } // wml
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html() {
    global $RESPONSE;
    $html  = "<img src=\"$this->src\"";
    $html .= " name=\"$this->name\"";
    $html .= " alt=\"$this->alt\"";
    // IE will show alt as tooltip, but others use title..
    if (isset($RESPONSE) && $RESPONSE->browser != "IE" && $this->alt != "") {
      $html .= " title=\"$this->alt\"";
    }
    if ($this->width  > 0)  $html .= " width=\"$this->width\"";
    if ($this->height > 0)  $html .= " height=\"$this->height\"";
    if ($this->hspace > 0)  $html .= " hspace=\"$this->hspace\"";
    if ($this->vspace > 0)  $html .= " vspace=\"$this->vspace\"";
    if ($this->align != "") $html .= " align=\"$this->align\"";
    if ($this->map != "")   $html .= " usemap=\"$this->map\"";
    $html .= " border=\"$this->border\"";
    if ($this->linkover_text != "" && $this->onmouseover == "") {
      $this->onmouseover = "status='$this->linkover_text';return true;";
      $this->onmouseout  = "status='';return true;";
    }
    if ($this->onmouseover != "") $html .= " onmouseover=\"$this->onmouseover\"";
    if ($this->onmouseout != "")  $html .= " onmouseout=\"$this->onmouseout\"";
    $html .= ">";
    return $html;
  } // html
} // image class

// ----------------------------------------------------------------------
/**
* Clickable Image class
* A class for managing a clickable image. A clickable image is
* one which redirects to a given URL when clicked on.
* @package image
*/
class clickable_image extends RenderableObject {
  // Public
  /** URL to go to when image is clicked on */
  var $url = "";

  // Private
  /** The image object
      @access private */
  var $img;
  //.....................................................................
  /**
  * Constructor
  * Creates the basic clickable image object.
  * @param string  $name   Name of this image element
  * @param string  $url    URL to go to when image is clicked
  * @param string  $alt    Image ALT tag content
  * @param string  $src    URL or path to the actual image file
  * @param integer $width  Width of the image in pixels
  * @param integer $height Height of the image in pixels
  * @param integer $border Size of border around the image in pixels
  */
  function clickable_image($name, $url, $alt, $src, $width=0, $height=0, $border=0) {
    $this->img = new image($name, $src, $alt, $width, $height, $border);
    $this->url = $url;
  } // clickable_image
  //.....................................................................
  /**
  * Set linkover text
  * Defines the text to show when the mouse moves over the image.
  * @param string $txt Text to show in status area when mouse-over.
  */
  function set_linkover_text($txt) {
    if (isset($this->img)) {
      $this->img->set_linkover_text($txt);
    }
  } // set_linkover_text
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html() {
    $html = "<a href=\"" . $this->url . "\">";
    $html .= $this->img->html();
    $html .= "</a>";
    return $html;
  } // html
} // clickable_image class

// ----------------------------------------------------------------------
/**
* Hover class
* Provides and image which changes to a second image when the mouse
* is over the top of it, using onmouseover and onmouseout events.
* @package image
*/
class hover extends RenderableObject {
  // Public
  /** Name of this element */
  var $name = "";
  /** Name of the over version of this element */
  var $name_over = "";
  /** URL to go to when image is clicked */
  var $url = "";

  // Private
  /** Image object
      @access private */
  var $img;
  /** Image over object
      @access private */
  var $img_over;
  /** Flag, true if images have been defined
      @access private */
  var $images_defined = false;
  //.....................................................................
  /**
  * Constructor
  * Creates the hover object.
  * This creates two image objects, one for the normal image and
  * one for the image displayed when the mouse goes over it. You
  * must define these images:
  * @see set_image()
  * @see set_image_over()
  * @param string  $name   Name of this hover image element
  * @param string  $url    URL to go to when image is clicked
  * @param string  $alt    Image ALT tag content
  */
  function hover( $name, $url, $alt ) {
    $this->name = $name;
    $this->name_over = $name . "_over";
    $this->url = $url;
    $this->img = new image($this->name, $url, $alt);
    $this->img_over = new image($this->name_over, $url, $alt);
  } // hover
  //.....................................................................
  /**
  * Define normal image
  * This is the image which is displayed when the mouse is 'out'.
  * @param string  $src    URL or path to the actual image file
  * @param integer $width  Width of the image in pixels
  * @param integer $height Height of the image in pixels
  * @param integer $border Size of border around the image in pixels
  */
  function set_image($src, $width, $height, $border=0) {
    $this->img = $this->img->set_src($src, $width, $height, $border);
    return $this;
  } // set_image
  //.....................................................................
  /**
  * Define over image
  * This is the image which is displayed when the mouse is 'over'.
  * @param string  $src    URL or path to the actual image file
  * @param integer $width  Width of the image in pixels
  * @param integer $height Height of the image in pixels
  * @param integer $border Size of border around the image in pixels
  */
  function set_image_over($src, $width, $height, $border=0) {
    $this->img_over = $this->img_over->set_src($src, $width, $height, $border);
    return $this;
  } // set_image_over
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html() {
    $html = "";
    $html .= "<a href=\"" . $this->url . "\"";
    $html .= " onmouseover=\"$this->name.src='" . $this->img_over->src . "'; window.status='$this->name'; return true;\"\n";
    $html .= " onmouseout=\"$this->name.src='" . $this->img->src . "'; window.status=''; return true;\">\n";
    $html .= $this->img->html();
    $html .= "</a>\n";
    return $html;
  } // html
} // hover class

// ----------------------------------------------------------------------
?>