<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    json-xhr-defs.php                                       */
/* Author:      Paul Waite                                              */
/*                                                                      */
/* Description: Definitions for implementing XML HTTP Requests (XHR)    */
/*              using the Dojo Xhr & Json utilities and tools.          */
/*              These classes are for the basic instantiation, as       */
/*              Javascript in a webpage, of several types of item:      */
/*               - utility funcs:                                       */
/*                                                                      */
/*                                                                      */
/*                                                                      */
/* ******************************************************************** */
include_once("json-defs.php");

/**
 * Class which contains a basic XHR request properties in JSON format.
 * 
 * This is designed to be inherited by the jsonXHRGet and jsonXHRPost
 * classes.
 */
class jsonXHR {
  /**
   * Optional object. Contains properties with string values. These
   * properties will be serialized as name1=value2 and passed in
   * the request.
   */
  var $content = "";
  /** 
   * Optional client-side function to handle errors.
   */
  var $error;
  /**
   * Optional DOM node for a form. Used to extract the form values
   * and send to the server. Use dojoById() or dijitById().
   */
  var $form;
  /**
   * Optional string. Acceptable values are: text (default), json,
   * json-comment-optional, json-comment-filtered, javascript, xml
   */
  var $handleAs;
  /**
   * Optional object. Additional HTTP headers to send in the request.
   */
  var $headers;
  /**
   * Optional client-side function to handle the load of response data.
   */
  var $load;
  /**
   * Optional boolean. Default false. If true then a "dojo.preventCache"
   * parameter is sent in the request with a value that changes
   * with each request (timestamp). Useful only with GET-type requests
   */
  var $preventCache = false;
  /**
   * Optional boolean. Default false. Indicates whether the request
   * should be a synchronous (blocking) request.
   */
  var $sync = false;
  /**
   * Optional integer. Milliseconds to wait for the response. If this
   * time passes, the error callbacks are called.
   */
  var $timeout = 15000; // timeout (millisec)
  /**
   * Mandatory string. URL to server endpoint.
   */
  var $url = "";  
  // .....................................................................
  function jsonXHR($url) {
    $this->url = $url;
  } // jsonXHR
  // .....................................................................
  function addRPC($rpc_name, $rpc_parms, $rpc_callback_func="") {
    $this->rpcFuncs[$rpc_name] = $rpc_parms;
    if ($rpc_callback_func != "") {
      $this->rpcCallbackFuncs[$rpc_name] = $rpc_callback_func;
    }
  } // addRPC
  // .....................................................................
  /**
   * Render the RPC functionality for this page. This is wholly Javascript
   * generation, and it makes sure that the Dojo RPC system is initialised,
   * and the user-specified RPC functions and call-backs are defined.
   */
  function render() {
    global $RESPONSE;
    // If no definitions, then no service..
    if ($this->service_defs != "") {
      
      // Set up javascript for initialisation
      $RESPONSE->head->add_named_script(
            "var jsonRPC=null;\n"
          . "function jsonRPCinit() {\n"
          . " jsonRPC=new dojo.rpc.JsonService('$this->service_defs');\n"
          . "}\n"
          . "dojo.require('dojo.rpc.JsonService');\n"
          ,
            "dojo",
            "javascript",
            "append",
            ADD_ONCE_ONLY
            );
      // Add the user RPC functions
      foreach ($this->rpcFuncs as $funcname => $parmslist) {
        $js = "function " . $funcname . "(" . $parmslist . ") {\n"
            . " var rpc=jsonRPC." . $funcname . "(" . $parmslist . ");\n"
            ;
        if (isset($this->rpcCallbackFuncs[$funcname])) {
          $js .= " rpc.addCallback(" . $this->rpcCallbackFuncs[$funcname] . ");\n";
        }
        $js .= "}\n";
        $RESPONSE->head->add_named_script($js, "dojo", "javascript", "append", ADD_ONCE_ONLY);
      } // foreach

      // Make sure RPC platform gets initialised onload
      $RESPONSE->head->add_named_script(
            "dojo.addOnLoad(jsonRPCinit);\n",
            "dojo_onloads",
            "javascript",
            "append",
            ADD_ONCE_ONLY
            );
    }  
  } // render

} // jsonRPC class

// -----------------------------------------------------------------------
/**
 * Function which returns an RPC response as JSON. This is for use with
 * callback functions. The Javascript callback function should look like
 * this example:
 *   mycallback(response, ioargs) {
 *     result = response[0];
 *     error  = response[1];
 *     rpcid  = response[2];
 *     -- your processing --
 *   }
 * Ie. the response is passed as a three-element array of scalars.
 * @param string $result A scalar containing result data
 * @param string $error A scalar containing error info
 * @param integer The original RPC request ID
 */
function jsonRPCresponse($result, $error, $id) {

  $resp = array(
    "result" => $result,
    "error" => ($error != "" ? $error : "null"),
    "id" => $id
    );
  $conv = new JSONconverter();
  return $conv->toJSON($resp);
}

?>