<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    rss-defs.php                                            */
/* Author:      Paul Waite                                              */
/* Description: RSS support classes and functions.                      */
/*              Provides a single set of classes which will emit        */
/*              RSS 1.0 and RSS 2.0 feeds.                              */
/*                                                                      */
/* ******************************************************************** */
/** @package rss */

// Supported versions of RSS
define("RSS_1", "1");
define("RSS_2", "2");

// ----------------------------------------------------------------------
class RSSelement {
  var $title;         # title of this element
  var $link;          # link to access this element (full url)
  var $description;   # description of this element
  var $pubDate_ts;    # publish date, unix timestamp
  // ....................................................................
  /** Constructor 
   * @param string $title Title of this element
   * @param string $link Link to homepage for this element
   * @param string $desc Description of this element
   */
  function RSSelement($title, $link, $desc="") {
    $this->set_title($title);
    $this->set_link($link);
    $this->set_description($desc);
  }
  // ....................................................................
  /**
   * @param string $title Title of this element
   */
  function set_title($title) {
    $this->title = $title;
  }
  // ....................................................................
  /**
   * @param string $link Link to homepage for this element
   */
  function set_link($link) {
    $this->link = $link;
  }
  // ....................................................................
  /**
   * @param string $desc Description of this element
   */
  function set_description($desc) {
    $this->description = $desc;
  }
  // ....................................................................
  /**
   * Takes a plain string formatted how you like your dates and returns
   * a unix timestamp. Special case - with no parameters returns 'now'
   * timestamp.
   * @param string $date A formatted date-time string
   */
  function get_timestamp_of($date="now") {
    if ($date == "now") {
      $ts = time();
    }
    else {
      $ts = strtotime($date);
    }
    return $ts;
  } // get_timestamp_of
  // ....................................................................
  /**
   * Set the publication date for this element, defaults to 'now'. Note
   * that this is a plain string formatted how you like your dates. It will
   * be transformed into a unix timestamp.
   * @param string $date A formatted date-time string
   */
  function set_pubdate($date="now") {
    $this->pubDate = $this->get_timestamp_of($date);
  }
  // ....................................................................
  /**
   * Return the container tag for this element, including all the
   * common sub-tags, if they are defined.
   * @param string $element_type Either 'channel' or 'item'
   * @return object The container tag for this element
   */
  function get_xmltag($element_type) {
    $xmltag = new xmltag($element_type);
    if ($this->title) {
      $xmltag->childtag( new xmltag("title", $this->title) );
    }
    if ($this->link) {
      $xmltag->childtag( new xmltag("link", $this->link) );
    }
    if ($this->description) {
      $xmltag->childtag( new xmltag("description", $this->description) );
    }
    return $xmltag;
  } // gen_xmltag

} // RSSelement

// ----------------------------------------------------------------------
/**
 * Generic RSS Channel class
 */
class RSSchannel extends RSSelement {
  var $rss_version = RSS_2;
	var $lastBuildDate;
  var $language;
  var $copyright;
	var $managingEditor;
	var $webMaster;
  var $publisher;

  var $docs;
  var $ttl;
  var $generator;
	var $image_url;
  var $image_link;
  var $image_title;
	var $items = array();
  var $modules = array();
  // ....................................................................
  /** Constructor 
   * @param string $title Title of this
   * @param string $link Link to homepage for this channel
   * @param string $desc Description of this channel
   */
  function RSSchannel($title, $link, $desc="") {
    $this->RSSelement($title, $link, $desc);
    # always include the rdf module by default (rss 1.0 use only)
    $this->use_module("rdf", "http://www.w3.org/1999/02/22-rdf-syntax-ns#");
    $this->use_module("dc", "http://purl.org/dc/elements/1.1/");
    $this->use_module("", "http://purl.org/rss/1.0/");
  } // RSSchannel
  // ....................................................................
  /**
   * Sets the RSS version. Note that you can leave this until your call
   * the render() method, and pass it in that call as a parameter.
   * NB: The default is always RSS_2 if this is not explicitly set.
   * @param string $rssver The RSS version to render as
   */
  function set_rss_version($rssver) {
    if ($rssver == RSS_1 || $rssver == RSS_2) {
      $this->rss_version = $rssver;
    }
  } // set_rss_version
  // ....................................................................
  /**
   * Adds an RSSitem object to this channel.
   * @param object $itemobj The item object to add
   */
  function add_item($itemobj) {
    if (is_object($itemobj)) {
      $this->items[] = $itemobj;
    }
  } // add_item
  // ....................................................................
  /**
   * Create a new item object with the given title, link and optional
   * description, then add it to our channel.
   * @param string $title Title of the new item
   * @param string $link Link to content of the new item
   * @param string $desc Description of the new item
   */
  function create_item($title, $link, $desc="") {
    $item = new RSSitem($title, $link, $desc);
    $this->add_item($item);
  } // create_item
  //.....................................................................
  /**
   * Define metadata modules and URL to reference, eg. 'dc' is Dublin Core.
   * @param string $prefix The prefix for this metadata module (eg. 'dc')
   * @param string $uri The URL to the definitions of the module scheme
   */
  function use_module($prefix, $uri) {
    $this->modules[$prefix] = $uri;
  } // use_module
  // ....................................................................
  /**
   * Set an image for the channel
   * @param string $url A full URL to acquire the image
   * @param string $link Optional override image link, defaults to RSS feed link
   * @param string $title Optional override image title, defaults to RSS feed title
   */
  function set_image($src, $link="", $title="") {
    $this->image_url = $src;
    if ($link != "") {
      $this->image_link = $link;
    }
    if ($title != "") {
      $this->image_title = $title;
    }
  } // set_image
  // ....................................................................
  /**
   * Set the build date for this element
   * @param string $date A formatted date string
   */
  function set_lastbuilddate($date="now") {
    $this->lastBuildDate = $this->get_timestamp_of($date);
  } // set_lastbuilddate
  // ....................................................................
  /**
   * Optionally set one or more contact information fields for the channel.
   * These are: the Publisher, The Webmaster, and the Managing Editor.
   * The Publisher is typically an organisation name eg. 'Acme Web News'.
   * A webmaster is usually an email address to contact someone regarding
   * technical issues with the feed. The managing editor is the email
   * of the person with editorial control over feed content.
   * NOTE: if any of these are set to nullstring "", they won't be included
   * in the feed content.
   * @param string $publisher The publishing house or organisation name
   * @param string $editor The email of the editor of this feed
   * @param string $webmaster The email of the channel web master
   */
  function set_contactinfo($publisher, $editor="", $webmaster="") {
    $this->publisher = $publisher;
    $this->managingEditor = $editor;
    $this->webMaster = $webmaster;
  } // set_contactinfo
  // ....................................................................
	/**
	* Scrubs the channel content to make it acceptable RSS content. It
	* removes line-endings, and funny stuff which isn't so funny. This
	* is because feeds are generally 'news articles' and as such may
	* contain stuff authored on word-processors (that shall not be
	* named) and therefore contain chars of surpassing weirdness. If
	* you want to make use of this, invoke the method just before the
	* feed/channel is rendered.
	* @param string $s String to scrub
	* @return string Scrubbed version of given string
	*/
  function scrub() {
    $this->title = scrub_string($this->title);
    $this->description = scrub_string($this->description);
    $newitems = array();
    foreach ($this->items as $item) {
      $item->title = scrub_string($item->title);
      $item->description = scrub_string($item->description);
      if ($item->comments) {
        $item->comments = scrub_string($item->comments);
      }
      $newitems[] = $item;
    }
    $this->items = $newitems;

  } // scrub
  // ....................................................................
  /**
   * Returns the channel XML tag object
   */
  function get_xmltag() {
    $xmltag = RSSelement::get_xmltag("channel");
    return $xmltag;
  } // get_xmltag
  // ....................................................................
  /**
   * Returns the channel XML tag in the RSS 1.0 format.
   * This format is a 'stacked' bunch of tags: <channel>, <image> and
   * then multiple <item> tags, all contained by the <rdf:RDF> tag.
   */
  function render_rss_1() {
    $xml = "";

    if ($this->title != "" && $this->link != "" && $this->description != "") {
      # Create container and add tags to it..
      $rss = new xmltag("rdf:RDF");
      
      # Render the module namespaces
      foreach ($this->modules as $mod => $module_url) {
        $module = "xmlns" . (($mod != "") ? ":$mod" : "");
        $rss->setattribute($module, $module_url);
      }

      # Add the <channel> tag
      $channel = $this->get_xmltag($this->rss_version);
      $channel->setattribute("rdf:about", $this->link);

      if ($this->copyright != "") {
        $channel->childtag( new xmltag("dc:rights", $this->copyright) );
      }
      if ($this->language != "") {
        $channel->childtag( new xmltag("dc:language", $this->language) );
      }
      if ($this->publisher != "") {
        $channel->childtag( new xmltag("dc:publisher", $this->language) );
      }
      if ($this->managingEditor != "") {
        $channel->childtag( new xmltag("dc:creator", $this->managingEditor) );
      }

      if ($this->image_url) {
        $img = new xmltag("image");
        $img->setattribute("rdf:resource", $this->image_url);
        $channel->childtag( $img );
      }
  
      $items = new xmltag("items");
      $rdfseq = new xmltag("rdf:Seq");
      foreach ($this->items as $item) {
        $rdfli = new xmltag("rdf:li");
        $rdfli->setattribute("resource", $item->link);
        $rdfseq->childtag($rdfli);
      }
      $items->childtag($rdfseq);
      $channel->childtag($items);
      $rss->childtag($channel);

      # Add the <image> tag
      if ($this->image_url) {
        $image = new xmltag("image");
        $image->setattribute("rdf:about", $this->image_url);
        $image->childtag( new xmltag("link",  (isset($this->image_link)  ? $this->image_link  : $this->link))  );
        $image->childtag( new xmltag("title", (isset($this->image_title) ? $this->image_title : $this->title)) );
        $rss->childtag($image);
      }      

      # Add all the item tags
      foreach ($this->items as $item) {
        $rss->childtag( $item->get_xmltag($this->rss_version) );
      }
      
      // Now render the XML
      $xml = xmlheader("1.0", "iso-8859-1") . $rss->render();
    }
    return $xml;
  } // render_rss_1
  // ....................................................................
  /**
   * Returns the channel XML tag in the RSS 2.0 format.
   */
  function render_rss_2() {
    $xml = "";

    if ($this->title != "" && $this->link != "" && $this->description != "") {

      # Create RSS container and add tags to it..
      $rss = new xmltag("rss");
      $rss->setattribute("version", "2.0");

      $channel = $this->get_xmltag($this->rss_version);
      if ($this->docs == "") {
        $this->docs = "http://blogs.law.harvard.edu/tech/rss";
      }
      if ($this->generator == "") {
        $this->generator = "AxylRSS";
      }
      if ($this->language) {
        $channel->childtag( new xmltag("language", $this->language) );
      }
      if ($this->lastBuildDate) {
        $channel->childtag( new xmltag("lastBuildDate", timestamp_to_displaydate(RFC_822, $this->lastBuildDate)) );
      }
      if ($this->copyright) {
        $channel->childtag( new xmltag("copyright", $this->copyright) );
      }
      if ($this->managingEditor) {
        $channel->childtag( new xmltag("managingEditor", $this->managingEditor) );
      }
      if ($this->webMaster) {
        $channel->childtag( new xmltag("webMaster", $this->webMaster) );
      }
      if ($this->docs) {
        $channel->childtag( new xmltag("docs", $this->docs) );
      }
      if ($this->ttl) {
        $channel->childtag( new xmltag("ttl", $this->ttl) );
      }
      if ($this->generator) {
        $channel->childtag( new xmltag("generator", $this->generator) );
      }
      if ($this->image_url) {
        $image = new xmltag("image");
        $image->childtag( new xmltag("url", $this->image_url) );
        $image->childtag( new xmltag("link",  (isset($this->image_link)  ? $this->image_link  : $this->link))  );
        $image->childtag( new xmltag("title", (isset($this->image_title) ? $this->image_title : $this->title)) );
        $channel->childtag($image);
      }
  
      // Add the items to the channel
      foreach ($this->items as $item) {
        $channel->childtag( $item->get_xmltag($this->rss_version) );
      }
      
      # Finally add to container
      $rss->childtag( $channel );
      
      // Now render the XML
      $xml = xmlheader("1.0", "iso-8859-1") . $rss->render();
    }
    return $xml;

  } // render_rss_2
  // ....................................................................
  /**
   * Returns the channel XML tag in the RSS 2.0 format.
   */
  function render($rssver=false) {
    if ($rssver !== false) {
      $this->set_rss_version($rssver);
    }
    switch ($this->rss_version) {
      case RSS_1:
        return $this->render_rss_1();
        break;
        
      case RSS_2:
        return $this->render_rss_2();
        break;
      default:
        return "";
    } // switch
  }  // render
	
} // RSSchannel

// ----------------------------------------------------------------------
/**
 * Generic RSS Item class
 */
class RSSitem extends RSSelement {
  var $author;
  var $category;
  var $category_domain;
  var $comments;
  var $enclosure_url;
  var $enclosure_length;
  var $enclosure_mimetype;
  var $guid;
  var $source_url;
  var $source_name;
  // ....................................................................
  /* Constructor 
   * @param string $title Title of this item
   * @param string $link Link to homepage for this item
   * @param string $desc Description of this item
   */
  function RSSitem($title, $link, $desc="") {
    $this->RSSelement($title, $link, $desc);
  }
  // ....................................................................
  /**
   * Set the category for this item. This can comprise both a string
   * category name, and a URL or domain from which the categorisation
   * taxonomy can be acquired.
   * @param string $category Category string
   * @param string $domain Optional location of categorisation taxonomy
   */
  function set_category($category, $domain="") {
    $this->category = $category;
    $this->category_domain = $domain;
  }
  // ....................................................................
  /**
   * An enclosure is in essence a reference to content (usually media).
   * This is typically something like an mp3 (sound clip) or a movie clip.
   * @param string $url Location of the enclosure content
   * @param integer $length Length in bytes
   * @param string $mimtype The mime-type of the content (eg. 'audio/mpeg')
   */
  function set_enclosure($url, $length, $mimetype) {
    $this->enclosure_url = $url;
    $this->enclosure_length = $length;
    $this->enclosure_mimetype = $mimetype;
  }
  // ....................................................................
  /**
   * Set the source of this item. The URL is a link to the XMLisation of
   * the source for this item. Name is optional and is usually the name
   * of the source channel, if there is one. This is to enable the
   * propagation of credit for items.
   * @param string $url Location of the XMLisation of the source
   * @param string $name Optional name of the source/channel
   */
  function set_source($url, $name="") {
    $this->source_url = $url;
    $this->source_name = $name;
  }
  // ....................................................................
  /**
   * Set the unique ID for this item. This can be used by clients to
   * ascertain duplicated items.
   * @param string $id The unique ID string identifying this item
   */
  function set_guid($id) {
    $this->guid = $id;
  } // set_guid
  // ....................................................................
  /**
   * Returns the item XML tag object for a given RSS version.
   * @param string $rssver The version of RSS to return the tag as
   */
  function get_xmltag($rss_version) {
    
    // Acquire the basic item tag
    $xmltag = RSSelement::get_xmltag("item");

    // Do RSS version-specific stuff
    switch ($rss_version) {
      case RSS_1:
        $xmltag->setattribute("rdf:about", $this->link);
        if ($this->pubDate) {
          $xmltag->childtag( new xmltag("dc:date", timestamp_to_displaydate(ISO_8601, $this->pubDate)) );
        }
        if ($this->author) {
          $xmltag->childtag( new xmltag("dc:creator", $this->author) );
        }
        if ($this->category) {
          $xmltag->childtag( new xmltag("dc:subject", $this->category) );
        }
        break;        

      case RSS_2:
        if ($this->author) {
          $xmltag->childtag( new xmltag("author", $this->author) );
        }
        if ($this->category) {
          $cat = new xmltag("category", $this->category);
          if ($this->category_domain) {
            $cat->setattribute("domain", $this->category_domain);
          }
          $xmltag->childtag($cat);
        }
        if ($this->comments) {
          $xmltag->childtag( new xmltag("comments", $this->comments) );
        }
        if ($this->enclosure_url) {
          $enc = new xmltag("enclosure");
          $enc->setattribute("url", $this->enclosure_url);
          if ($this->enclosure_length) {
            $cat->setattribute("length", $this->enclosure_length);
          }
          if ($this->enclosure_type) {
            $cat->setattribute("type", $this->enclosure_type);
          }
          $xmltag->childtag($cat);
        }
        if ($this->guid) {
          $xmltag->childtag( new xmltag("guid", $this->guid) );
        }
        if ($this->source_url) {
          $src = new xmltag("source", $this->source_name);
          $src->setattribute("url", $this->source_url);
          $xmltag->childtag($src);
        }
        if ($this->pubDate) {
          $xmltag->childtag( new xmltag("pubDate", timestamp_to_displaydate(RFC_822, $this->pubDate)) );
        }
        break;
        
    } // switch
            
    return $xmltag;
  } // get_xmltag
  
  // ....................................................................
  /**
   * Returns the item XML tag in the correct format according to the
   * RSS version passed in.
   * @param string $rssver The version of RSS to return the tag as
   */
  function render($rss_version) {
    $xml = "";
    if ($this->title != "" || $this->description != "") {
      $xmltag = $this->get_xmltag($rss_version);
      if (is_object($xmltag)) {
        $xml = $xmltag->render();
      }
    }
    return $xml;
  } // render
  
} // RSSitem

// ----------------------------------------------------------------------
?>