/*
 *
 *   Bacula Director -- User Agent Database Retention Period Handling
 *
 *     Kern Sibbald, February MMII
 *
 *   Version $Id: ua_retention.c,v 1.15 2003/05/30 14:44:41 kerns Exp $
 */

/*
   Copyright (C) 2000-2003 Kern Sibbald and John Walker

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.

 */

#ifdef NEEDED

#include "bacula.h"
#include "dird.h"

#define MAX_DEL_LIST_LEN 1000000

/*
 * List of SQL commands terminated by NULL for deleting
 *  temporary tables and indicies 
 */
static char *drop_deltabs[] = {
   "DROP TABLE DelCandidates",
   "DROP TABLE DelTable",
   "DROP INDEX DelInx1", 
   "DROP INDEX DelInx2",
   "DROP INDEX DelInx3",
   NULL};

/*
 * List of SQL commands to create temp table and indicies
 */
static char *create_deltabs[] = {
   "CREATE TABLE DelCandidates ("
      "JobId INTEGER UNSIGNED NOT NULL, "
      "FileId INTEGER UNSIGNED, "
      "PathId INTEGER UNSIGNED, "
      "FilenameId INTEGER UNSIGNED, "
      "ClientId INTEGER UNSIGNED)", 
   "CREATE INDEX DelInx1 ON DelCandidates (FilenameId)", 
   "CREATE INDEX DelInx2 ON DelCandidates (PathId)", 
   "CREATE TABLE DelTable ("
      "JobId INTEGER UNSIGNED NOT NULL, "
      "FileId BIGINT UNSIGNED, "
      "PathId INTEGER UNSIGNED, "
      "FilenameId INTEGER UNSIGNED)",
   "CREATE INDEX DelInx3 ON DelTable (FileId)",
   NULL};


/* 
 * Select Jobs subject to being deleted -- for listing
 */
static char *jobs = 
   "SELECT JobId, Job, JobFiles FROM Job "
   "WHERE JobTDate < %d " 
   "AND PurgedFiles = 0";
/*
 * Fill retention table with all Files subject to being deleted
 */
static char *insert_delcand = 
   "INSERT INTO DelCandidates "
   "SELECT Job.JobId,File.FileId,File.PathId,File.FilenameId,Job.ClientId "
   "FROM Job,File "
   "WHERE Job.JobTDate < %d "
   "AND Job.PurgedFiles = 0 "
   "AND Job.JobId=File.JobId";

/*
 * Fill delete table with only Files that have a newer backup
 * This is the list of files to delete
 */
static char *insert_del =
   "INSERT INTO DelTable "
   "SELECT DelCandidates.JobId,DelCandidates.FileId,DelCandidates.PathId,DelCandidates.FilenameId "
   "FROM Job,DelCandidates,File "
   "WHERE Job.JobTDate >= %d "
   "AND Job.ClientId=DelCandidates.ClientId "
   "AND Job.JobId=File.JobId "
   "AND DelCandidates.FilenameId=File.FilenameId "
   "AND DelCandidates.PathId=File.PathId "
   "GROUP BY DelCandidates.FileId";

/*
 * List files to be deleted
 */
static char *list_del =
   "SELECT JobId,Path.Path,Filename.Name FROM "
   "DelTable,Path,Filename "
   "WHERE DelTable.PathId=Path.PathId "
   "AND DelTable.FilenameId=Filename.FilenameId";

/*
 * These are the JobIds for which we purged the
 * files, so mark them so.
 */
static char *update_purged = 
   "UPDATE Job Set PurgedFiles=1 "
   "WHERE JobTDate < %d " 
   "AND PurgedFiles = 0";


struct s_del_ctx {
   FileId_t *FileId;
   int num_ids; 		      /* ids stored */
   int max_ids; 		      /* size of array */
   int num_del; 		      /* number deleted */
   int tot_ids; 		      /* total to process */
};

/*
 * Called here to count entries to be deleted 
 */
static int count_handler(void *ctx, int num_fields, char **row)
{
   struct s_del_ctx *del = (struct s_del_ctx *)ctx;

   if (row[0]) {
      del->tot_ids = atoi(row[0]);
   } else {
      del->tot_ids = 0;
   }
   return 0;
}

/*
 * Called here to make in memory list of FileIds to be
 *  deleted. The in memory list will then be transversed
 *  to issue the SQL DELETE commands.  Note, the list
 *  is allowed to get to MAX_DEL_LIST_LEN to limit the
 *  maximum malloc'ed memory.
 */
static int delete_handler(void *ctx, int num_fields, char **row)
{
   struct s_del_ctx *del = (struct s_del_ctx *)ctx;

   if (del->num_ids == MAX_DEL_LIST_LEN) {  
      return 1;
   }
   if (del->num_ids == del->max_ids) {
      del->max_ids = (del->max_ids * 3) / 2;
      del->FileId = (FileId_t *)brealloc(del->FileId, sizeof(FileId_t) *
	 del->max_ids);
   }
   del->FileId[del->num_ids++] = (FileId_t)str_to_int64(row[0]);
   return 0;
}

static void drop_temp_tables(UAContext *ua) 
{
   int i;
   for (i=0; drop_deltabs[i]; i++) {
      db_sql_query(ua->db, drop_deltabs[i], NULL, (void *)NULL);
   }
}

/*
 * File Retention period handling
 */
int retentioncmd(UAContext *ua, char *cmd)
{
   int i, yy, mm, dd;
   struct s_del_ctx del;
   char *query = (char *)get_pool_memory(PM_MESSAGE);
   utime_t now, period;
   char ec1[20];
       
   del.FileId = NULL;
   now = (utime_t)time(NULL);

   if (!open_db(ua)) {
      goto bail_out;
   }

   bsendmsg(ua, _(
"This command will permit you to remove File records from the\n\
Catalog that are older than the Catalog File Retention period.\n\
Job and Media records will remain, but you will no longer be\n\
able to list the filenames that were saved for those Jobs.\n"));

   if (!get_cmd(ua, _("Enter Catalog File Retention period in days or yy-mm-dd: "))) {
      goto bail_out;
   }
   if (strchr(ua->cmd, '-')) {
      if (sscanf(ua->cmd, "%d-%d-%d", &yy, &mm, &dd) != 3 ||
	  yy < 2000 || mm < 1 || mm > 12 || dd < 1 || dd > 31) {
         bsendmsg(ua, _("Invalid date specification.\n"));
	 goto bail_out;
      }
      period = (int32_t)(date_encode(tm.tm_year+1900, tm.tm_mon+1, tm.tm_mday) -
	       date_encode(yy, mm, dd));
   } else {
      period = atoi(ua->cmd);
   }

   if (period <= 0) {
      bsendmsg(ua, _("%d is an invalid period.\n"), period);
      goto bail_out;
   }
   if (period < 30) {
      bsendmsg(ua, _("%d is rather small.\n"), period);
      if (!get_yesno(ua, _("Are you sure? (yes/no): ")) || !ua->pint32_val) {
	 goto bail_out;
      }
   }

   /* DROP any previous DelCandidates and DelTable tables */
   drop_temp_tables(ua);

   bsendmsg(ua, "The Files associated with the following Jobs will\n"
      "be considered for removal from the Catalog, providing they\n"
      "are backed up more recently.\n");

   Mmsg(&query, jobs, now - period);
/* bsendmsg(ua, "Query: %s\n", query); */
   db_list_sql_query(jcr, ua->db, query, prtit, ua, 1, HORZ_LIST);
   if (!get_yesno(ua, _("Continue? (yes/no): ")) || !ua->pint32_val) {
	 goto bail_out;
   }

   /* Create temp tables and indicies */
   for (i=0; create_deltabs[i]; i++) {
      if (!db_sql_query(ua->db, create_deltabs[i], NULL, (void *)NULL)) {
         bsendmsg(ua, "%s", db_strerror(ua->db));
	 goto bail_out;
      }
   }

   /* 
    * Select all files that are older than Catalog DelCandidates period
    *  and stuff them into the "DeletionCandidates" table.
    */
   Mmsg(&query, insert_delcand, now - period);
/* bsendmsg(ua, "Query: %s\n", query); */
   if (!db_sql_query(ua->db, query, NULL, (void *)NULL)) {
      bsendmsg(ua, "%s", db_strerror(ua->db));
      goto bail_out;
   }
   bsendmsg(ua, _("Deletion candidates table completed.\n"));
#ifdef xxx_debug_printout
   bsendmsg(ua, "DelCandidates table\n");
   db_list_sql_query(ua->db, "select * from DelCandidates", prtit, ua, 1, HORZ_LIST);
#endif

   /* 
    * Select all files that have a more recent backup from "DelCandidates"
    *  table and stuff them into "DelTable". These are the File items
    *  that will be deleted.
    */
   bsendmsg(ua, _("Building deletion table ...\n"));
   Mmsg(&query, insert_del, now - period);
/* bsendmsg(ua, "Query: %s\n", query); */
   if (!db_sql_query(ua->db, query, NULL, (void *)NULL)) {
      bsendmsg(ua, "%s", db_strerror(ua->db));
      goto bail_out;
   }
   bsendmsg(ua, _("Deletion table completed.\n"));

   del.num_ids = 0;
   del.tot_ids = 0;
   del.num_del = 0;
   del.max_ids = 0;
      
   /* Find out how many File rows to delete */
   if (!db_sql_query(ua->db, "SELECT count(*) FROM DelTable", count_handler, (void *)&del)) {
      bsendmsg(ua, "%s", db_strerror(ua->db));
      goto bail_out;
   }

   if (del.tot_ids == 0) {
      bsendmsg(ua, _("No files found to remove from the Catalog.\n"));
      goto bail_out;
   }

   if (del.tot_ids < MAX_DEL_LIST_LEN) {
      del.max_ids = del.tot_ids + 1;
   } else {
      del.max_ids = MAX_DEL_LIST_LEN; 
   }
   del.FileId = (FileId_t *)malloc(sizeof(FileId_t) * del.max_ids);

   bsendmsg(ua, _("There are %s files to be removed from the Catalog.\n"  
                "A rough time estimate is %d minutes.\n"),
	 edit_uint64_with_commas(del.tot_ids, ec1), del.tot_ids/25000);
   
   if (get_yesno(ua, _("Do you want to list the files? (yes/no): ")) && 
	  ua->pint32_val) {
      bsendmsg(ua, _("The following files will be removed from the Catalog.\n"));
      /* List Files to be deleted */
      db_list_sql_query(ua->db, list_del, prtit, ua, 1, HORZ_LIST);
   }

   if (!get_yesno(ua, _("Remove the catalog file records? (yes/no): ")) || 
	  !ua->pint32_val) {
	 goto bail_out;
   }

   /*
    * Here is where we actually delete the files.
    *   Using the "delete_handler", we make an in memory list
    *	of the FileIds to be deleted, then we wiffle through that
    *	list firing off DELETE FROM commands.
    *
    * Note, this algorithm runs MUCH (1000 times) slower if
    * we are doing multiple passes because we must delete
    * the entries from DelTable. The loop deleting from
    * DelTable is done after the DELETE FROM File so that
    * we access only one table at a time.
    */
   while (del.tot_ids > del.num_del) {
      del.num_ids = 0;
      /* Make in memory list up to MAX_DEL_LIST_LEN */
      db_sql_query(ua->db, "SELECT FileId FROM DelTable", delete_handler, (void *)&del);
      if (del.num_ids == 0) {
         bsendmsg(ua, "%s", db_strerror(ua->db));
	 goto bail_out;
      }
      bsendmsg(ua, _("Beginning deletion batch.\n"));
      for (i=0; i < del.num_ids; i++) {
         Mmsg(&query, "DELETE FROM File WHERE FileId=%" llu, (uint64_t)del.FileId[i]);
	 db_sql_query(ua->db, query, NULL, (void *)NULL);   
         Dmsg2(400, "Del %d FileId=%" llu "\n", del.num_del+i+1, (uint64_t)del.FileId[i]);
      }
      if (del.tot_ids > MAX_DEL_LIST_LEN) {
	 for (i=0; i < del.num_ids; i++) {
	    /*
	     * If we are making more than one pass, we must delete used 
	     * records from DelTable so that they are not seen the next
             * pass.  Gross, but I don't know any other way to do it.
	     */
            Mmsg(&query, "DELETE FROM DelTable WHERE FileId=%" llu, (uint64_t)del.FileId[i]);
	    db_sql_query(ua->db, query, NULL, (void *)NULL);
	 }
      }
      bsendmsg(ua, _("\nBatch of %d deleted.\n"), del.num_ids);
      del.num_del += del.num_ids;
      Dmsg3(400, "tot_ids=%d num_del=%d num_ids=%d\n", del.tot_ids, del.num_del, 
	 del.num_ids);
   }
   /* 
    * Now mark all appropriate JobIds as having their files purged */
   Mmsg(&query, update_purged, now - period);
   if (!db_sql_query(ua->db, query, NULL, (void *)NULL)) {
      bsendmsg(ua, "%s", db_strerror(ua->db));
      goto bail_out;
   }

    
	 
bail_out:

   if (del.FileId) {
      free(del.FileId);
   }
   drop_temp_tables(ua);

   free_pool_memory(query);
   return 1;
}
#endif
