#include "pitch_converter.hpp"

uint16_t PitchConverter::getPitchFM(Note note, int octave, int pitch)
{
	return centTableFM_[calculatePitchIndex(octave, note, pitch)];
}

uint16_t PitchConverter::getPitchSSGSquare(Note note, int octave, int pitch)
{
	return centTableSSGSquare_[calculatePitchIndex(octave, note, pitch)];
}

uint16_t PitchConverter::getPitchSSGSquare(int n)
{
	return centTableSSGSquare_[n];
}

uint16_t PitchConverter::getPitchSSGTriangle(Note note, int octave, int pitch)
{
	return centTableSSGTriangle_[calculatePitchIndex(octave, note, pitch)];
}

uint16_t PitchConverter::getPitchSSGSaw(Note note, int octave, int pitch)
{
	return centTableSSGSaw_[calculatePitchIndex(octave, note, pitch)];
}

int PitchConverter::calculatePitchIndex(int octave, Note note, int pitch)
{
	int idx = 384 * octave + static_cast<int>(note) + pitch;
	return clamp(idx, 0, 3071);
}

const uint16_t PitchConverter::centTableFM_[3072] = {
	0x026a, 0x026b, 0x026c, 0x026e, 0x026f, 0x0270, 0x0271, 0x0272, 0x0273, 0x0274, 0x0276, 0x0277,
	0x0278, 0x0279, 0x027a, 0x027b, 0x027c, 0x027e, 0x027f, 0x0280, 0x0281, 0x0282, 0x0283, 0x0284,
	0x0286, 0x0287, 0x0288, 0x0289, 0x028a, 0x028b, 0x028d, 0x028e, 0x028f, 0x0290, 0x0291, 0x0293,
	0x0294, 0x0295, 0x0296, 0x0297, 0x0299, 0x029a, 0x029b, 0x029c, 0x029d, 0x029f, 0x02a0, 0x02a1,
	0x02a2, 0x02a3, 0x02a5, 0x02a6, 0x02a7, 0x02a8, 0x02aa, 0x02ab, 0x02ac, 0x02ad, 0x02ae, 0x02b0,
	0x02b1, 0x02b2, 0x02b3, 0x02b5, 0x02b6, 0x02b7, 0x02b8, 0x02ba, 0x02bb, 0x02bc, 0x02be, 0x02bf,
	0x02c0, 0x02c1, 0x02c3, 0x02c4, 0x02c5, 0x02c6, 0x02c8, 0x02c9, 0x02ca, 0x02cc, 0x02cd, 0x02ce,
	0x02cf, 0x02d1, 0x02d2, 0x02d3, 0x02d5, 0x02d6, 0x02d7, 0x02d9, 0x02da, 0x02db, 0x02dd, 0x02de,
	0x02df, 0x02e1, 0x02e2, 0x02e3, 0x02e5, 0x02e6, 0x02e7, 0x02e9, 0x02ea, 0x02eb, 0x02ed, 0x02ee,
	0x02ef, 0x02f1, 0x02f2, 0x02f3, 0x02f5, 0x02f6, 0x02f7, 0x02f9, 0x02fa, 0x02fc, 0x02fd, 0x02fe,
	0x0300, 0x0301, 0x0303, 0x0304, 0x0305, 0x0307, 0x0308, 0x030a, 0x030b, 0x030c, 0x030e, 0x030f,
	0x0311, 0x0312, 0x0313, 0x0315, 0x0316, 0x0318, 0x0319, 0x031b, 0x031c, 0x031d, 0x031f, 0x0320,
	0x0322, 0x0323, 0x0325, 0x0326, 0x0328, 0x0329, 0x032a, 0x032c, 0x032d, 0x032f, 0x0330, 0x0332,
	0x0333, 0x0335, 0x0336, 0x0338, 0x0339, 0x033b, 0x033c, 0x033e, 0x033f, 0x0341, 0x0342, 0x0344,
	0x0345, 0x0347, 0x0348, 0x034a, 0x034b, 0x034d, 0x034e, 0x0350, 0x0351, 0x0353, 0x0355, 0x0356,
	0x0358, 0x0359, 0x035b, 0x035c, 0x035e, 0x035f, 0x0361, 0x0362, 0x0364, 0x0366, 0x0367, 0x0369,
	0x036a, 0x036c, 0x036d, 0x036f, 0x0371, 0x0372, 0x0374, 0x0375, 0x0377, 0x0379, 0x037a, 0x037c,
	0x037d, 0x037f, 0x0381, 0x0382, 0x0384, 0x0386, 0x0387, 0x0389, 0x038a, 0x038c, 0x038e, 0x038f,
	0x0391, 0x0393, 0x0394, 0x0396, 0x0398, 0x0399, 0x039b, 0x039d, 0x039e, 0x03a0, 0x03a2, 0x03a3,
	0x03a5, 0x03a7, 0x03a8, 0x03aa, 0x03ac, 0x03ad, 0x03af, 0x03b1, 0x03b3, 0x03b4, 0x03b6, 0x03b8,
	0x03b9, 0x03bb, 0x03bd, 0x03bf, 0x03c0, 0x03c2, 0x03c4, 0x03c6, 0x03c7, 0x03c9, 0x03cb, 0x03cd,
	0x03ce, 0x03d0, 0x03d2, 0x03d4, 0x03d5, 0x03d7, 0x03d9, 0x03db, 0x03dd, 0x03de, 0x03e0, 0x03e2,
	0x03e4, 0x03e5, 0x03e7, 0x03e9, 0x03eb, 0x03ed, 0x03ef, 0x03f0, 0x03f2, 0x03f4, 0x03f6, 0x03f8,
	0x03f9, 0x03fb, 0x03fd, 0x03ff, 0x0401, 0x0403, 0x0405, 0x0406, 0x0408, 0x040a, 0x040c, 0x040e,
	0x0410, 0x0412, 0x0414, 0x0415, 0x0417, 0x0419, 0x041b, 0x041d, 0x041f, 0x0421, 0x0423, 0x0425,
	0x0427, 0x0428, 0x042a, 0x042c, 0x042e, 0x0430, 0x0432, 0x0434, 0x0436, 0x0438, 0x043a, 0x043c,
	0x043e, 0x0440, 0x0442, 0x0444, 0x0446, 0x0448, 0x044a, 0x044c, 0x044e, 0x0450, 0x0452, 0x0454,
	0x0456, 0x0458, 0x045a, 0x045c, 0x045e, 0x0460, 0x0462, 0x0464, 0x0466, 0x0468, 0x046a, 0x046c,
	0x046e, 0x0470, 0x0472, 0x0474, 0x0476, 0x0478, 0x047a, 0x047c, 0x047e, 0x0480, 0x0483, 0x0485,
	0x0487, 0x0489, 0x048b, 0x048d, 0x048f, 0x0491, 0x0493, 0x0495, 0x0498, 0x049a, 0x049c, 0x049e,
	0x04a0, 0x04a2, 0x04a4, 0x04a6, 0x04a9, 0x04ab, 0x04ad, 0x04af, 0x04b1, 0x04b3, 0x04b6, 0x04b8,
	0x04ba, 0x04bc, 0x04be, 0x04c1, 0x04c3, 0x04c5, 0x04c7, 0x04c9, 0x04cc, 0x04ce, 0x04d0, 0x04d2,
	0x04d4, 0x04d7, 0x04d9, 0x04db, 0x04dd, 0x04e0, 0x04e2, 0x04e4, 0x04e6, 0x04e9, 0x04eb, 0x04ed,
	0x04f0, 0x04f2, 0x04f4, 0x04f6, 0x04f9, 0x04fb, 0x04fd, 0x0500, 0x0502, 0x0504, 0x0507, 0x0509,
	0x050b, 0x050e, 0x0510, 0x0512, 0x0515, 0x0517, 0x0519, 0x051c, 0x051e, 0x0520, 0x0523, 0x0525,
	0x0528, 0x052a, 0x052c, 0x052f, 0x0531, 0x0533, 0x0536, 0x0538, 0x053b, 0x053d, 0x0540, 0x0542,
	0x0544, 0x0547, 0x0549, 0x054c, 0x054e, 0x0551, 0x0553, 0x0556, 0x0558, 0x055a, 0x055d, 0x055f,
	0x0562, 0x0564, 0x0567, 0x0569, 0x056c, 0x056e, 0x0571, 0x0573, 0x0576, 0x0578, 0x057b, 0x057e,
	0x0580, 0x0583, 0x0585, 0x0588, 0x058a, 0x058d, 0x058f, 0x0592, 0x0595, 0x0597, 0x059a, 0x059c,
	0x059f, 0x05a2, 0x05a4, 0x05a7, 0x05a9, 0x05ac, 0x05af, 0x05b1, 0x05b4, 0x05b6, 0x05b9, 0x05bc,
	0x05be, 0x05c1, 0x05c4, 0x05c6, 0x05c9, 0x05cc, 0x05ce, 0x05d1, 0x05d4, 0x05d7, 0x05d9, 0x05dc,
	0x05df, 0x05e1, 0x05e4, 0x05e7, 0x05ea, 0x05ec, 0x05ef, 0x05f2, 0x05f4, 0x05f7, 0x05fa, 0x05fd,
	0x0600, 0x0602, 0x0605, 0x0608, 0x060b, 0x060d, 0x0610, 0x0613, 0x0616, 0x0619, 0x061c, 0x061e,
	0x0621, 0x0624, 0x0627, 0x062a, 0x062d, 0x062f, 0x0632, 0x0635, 0x0638, 0x063b, 0x063e, 0x0641,
	0x0644, 0x0646, 0x0649, 0x064c, 0x064f, 0x0652, 0x0655, 0x0658, 0x065b, 0x065e, 0x0661, 0x0664,
	0x0667, 0x066a, 0x066d, 0x0670, 0x0673, 0x0675, 0x0678, 0x067b, 0x067e, 0x0681, 0x0684, 0x0687,
	0x068b, 0x068e, 0x0691, 0x0694, 0x0697, 0x069a, 0x069d, 0x06a0, 0x06a3, 0x06a6, 0x06a9, 0x06ac,
	0x06af, 0x06b2, 0x06b5, 0x06b8, 0x06bc, 0x06bf, 0x06c2, 0x06c5, 0x06c8, 0x06cb, 0x06ce, 0x06d1,
	0x06d5, 0x06d8, 0x06db, 0x06de, 0x06e1, 0x06e5, 0x06e8, 0x06eb, 0x06ee, 0x06f1, 0x06f5, 0x06f8,
	0x06fb, 0x06fe, 0x0701, 0x0705, 0x0708, 0x070b, 0x070e, 0x0712, 0x0715, 0x0718, 0x071b, 0x071f,
	0x0722, 0x0725, 0x0729, 0x072c, 0x072f, 0x0733, 0x0736, 0x0739, 0x073d, 0x0740, 0x0743, 0x0747,
	0x074a, 0x074d, 0x0751, 0x0754, 0x0758, 0x075b, 0x075e, 0x0762, 0x0765, 0x0769, 0x076c, 0x076f,
	0x0773, 0x0776, 0x077a, 0x077d, 0x0781, 0x0784, 0x0788, 0x078b, 0x078f, 0x0792, 0x0796, 0x0799,
	0x079d, 0x07a0, 0x07a4, 0x07a7, 0x07ab, 0x07ae, 0x07b2, 0x07b5, 0x07b9, 0x07bd, 0x07c0, 0x07c4,
	0x07c7, 0x07cb, 0x07cf, 0x07d2, 0x07d6, 0x07d9, 0x07dd, 0x07e1, 0x07e4, 0x07e8, 0x07ec, 0x07ef,
	0x07f3, 0x07f7, 0x07fa, 0x07fe, 0x0c01, 0x0c03, 0x0c05, 0x0c06, 0x0c08, 0x0c0a, 0x0c0c, 0x0c0e,
	0x0c10, 0x0c12, 0x0c14, 0x0c15, 0x0c17, 0x0c19, 0x0c1b, 0x0c1d, 0x0c1f, 0x0c21, 0x0c23, 0x0c25,
	0x0c27, 0x0c28, 0x0c2a, 0x0c2c, 0x0c2e, 0x0c30, 0x0c32, 0x0c34, 0x0c36, 0x0c38, 0x0c3a, 0x0c3c,
	0x0c3e, 0x0c40, 0x0c42, 0x0c44, 0x0c46, 0x0c48, 0x0c4a, 0x0c4c, 0x0c4e, 0x0c50, 0x0c52, 0x0c54,
	0x0c56, 0x0c58, 0x0c5a, 0x0c5c, 0x0c5e, 0x0c60, 0x0c62, 0x0c64, 0x0c66, 0x0c68, 0x0c6a, 0x0c6c,
	0x0c6e, 0x0c70, 0x0c72, 0x0c74, 0x0c76, 0x0c78, 0x0c7a, 0x0c7c, 0x0c7e, 0x0c80, 0x0c83, 0x0c85,
	0x0c87, 0x0c89, 0x0c8b, 0x0c8d, 0x0c8f, 0x0c91, 0x0c93, 0x0c95, 0x0c98, 0x0c9a, 0x0c9c, 0x0c9e,
	0x0ca0, 0x0ca2, 0x0ca4, 0x0ca6, 0x0ca9, 0x0cab, 0x0cad, 0x0caf, 0x0cb1, 0x0cb3, 0x0cb6, 0x0cb8,
	0x0cba, 0x0cbc, 0x0cbe, 0x0cc1, 0x0cc3, 0x0cc5, 0x0cc7, 0x0cc9, 0x0ccc, 0x0cce, 0x0cd0, 0x0cd2,
	0x0cd4, 0x0cd7, 0x0cd9, 0x0cdb, 0x0cdd, 0x0ce0, 0x0ce2, 0x0ce4, 0x0ce6, 0x0ce9, 0x0ceb, 0x0ced,
	0x0cf0, 0x0cf2, 0x0cf4, 0x0cf6, 0x0cf9, 0x0cfb, 0x0cfd, 0x0d00, 0x0d02, 0x0d04, 0x0d07, 0x0d09,
	0x0d0b, 0x0d0e, 0x0d10, 0x0d12, 0x0d15, 0x0d17, 0x0d19, 0x0d1c, 0x0d1e, 0x0d20, 0x0d23, 0x0d25,
	0x0d28, 0x0d2a, 0x0d2c, 0x0d2f, 0x0d31, 0x0d33, 0x0d36, 0x0d38, 0x0d3b, 0x0d3d, 0x0d40, 0x0d42,
	0x0d44, 0x0d47, 0x0d49, 0x0d4c, 0x0d4e, 0x0d51, 0x0d53, 0x0d56, 0x0d58, 0x0d5a, 0x0d5d, 0x0d5f,
	0x0d62, 0x0d64, 0x0d67, 0x0d69, 0x0d6c, 0x0d6e, 0x0d71, 0x0d73, 0x0d76, 0x0d78, 0x0d7b, 0x0d7e,
	0x0d80, 0x0d83, 0x0d85, 0x0d88, 0x0d8a, 0x0d8d, 0x0d8f, 0x0d92, 0x0d95, 0x0d97, 0x0d9a, 0x0d9c,
	0x0d9f, 0x0da2, 0x0da4, 0x0da7, 0x0da9, 0x0dac, 0x0daf, 0x0db1, 0x0db4, 0x0db6, 0x0db9, 0x0dbc,
	0x0dbe, 0x0dc1, 0x0dc4, 0x0dc6, 0x0dc9, 0x0dcc, 0x0dce, 0x0dd1, 0x0dd4, 0x0dd7, 0x0dd9, 0x0ddc,
	0x0ddf, 0x0de1, 0x0de4, 0x0de7, 0x0dea, 0x0dec, 0x0def, 0x0df2, 0x0df4, 0x0df7, 0x0dfa, 0x0dfd,
	0x0e00, 0x0e02, 0x0e05, 0x0e08, 0x0e0b, 0x0e0d, 0x0e10, 0x0e13, 0x0e16, 0x0e19, 0x0e1c, 0x0e1e,
	0x0e21, 0x0e24, 0x0e27, 0x0e2a, 0x0e2d, 0x0e2f, 0x0e32, 0x0e35, 0x0e38, 0x0e3b, 0x0e3e, 0x0e41,
	0x0e44, 0x0e46, 0x0e49, 0x0e4c, 0x0e4f, 0x0e52, 0x0e55, 0x0e58, 0x0e5b, 0x0e5e, 0x0e61, 0x0e64,
	0x0e67, 0x0e6a, 0x0e6d, 0x0e70, 0x0e73, 0x0e75, 0x0e78, 0x0e7b, 0x0e7e, 0x0e81, 0x0e84, 0x0e87,
	0x0e8b, 0x0e8e, 0x0e91, 0x0e94, 0x0e97, 0x0e9a, 0x0e9d, 0x0ea0, 0x0ea3, 0x0ea6, 0x0ea9, 0x0eac,
	0x0eaf, 0x0eb2, 0x0eb5, 0x0eb8, 0x0ebc, 0x0ebf, 0x0ec2, 0x0ec5, 0x0ec8, 0x0ecb, 0x0ece, 0x0ed1,
	0x0ed5, 0x0ed8, 0x0edb, 0x0ede, 0x0ee1, 0x0ee5, 0x0ee8, 0x0eeb, 0x0eee, 0x0ef1, 0x0ef5, 0x0ef8,
	0x0efb, 0x0efe, 0x0f01, 0x0f05, 0x0f08, 0x0f0b, 0x0f0e, 0x0f12, 0x0f15, 0x0f18, 0x0f1b, 0x0f1f,
	0x0f22, 0x0f25, 0x0f29, 0x0f2c, 0x0f2f, 0x0f33, 0x0f36, 0x0f39, 0x0f3d, 0x0f40, 0x0f43, 0x0f47,
	0x0f4a, 0x0f4d, 0x0f51, 0x0f54, 0x0f58, 0x0f5b, 0x0f5e, 0x0f62, 0x0f65, 0x0f69, 0x0f6c, 0x0f6f,
	0x0f73, 0x0f76, 0x0f7a, 0x0f7d, 0x0f81, 0x0f84, 0x0f88, 0x0f8b, 0x0f8f, 0x0f92, 0x0f96, 0x0f99,
	0x0f9d, 0x0fa0, 0x0fa4, 0x0fa7, 0x0fab, 0x0fae, 0x0fb2, 0x0fb5, 0x0fb9, 0x0fbd, 0x0fc0, 0x0fc4,
	0x0fc7, 0x0fcb, 0x0fcf, 0x0fd2, 0x0fd6, 0x0fd9, 0x0fdd, 0x0fe1, 0x0fe4, 0x0fe8, 0x0fec, 0x0fef,
	0x0ff3, 0x0ff7, 0x0ffa, 0x0ffe, 0x1401, 0x1403, 0x1405, 0x1406, 0x1408, 0x140a, 0x140c, 0x140e,
	0x1410, 0x1412, 0x1414, 0x1415, 0x1417, 0x1419, 0x141b, 0x141d, 0x141f, 0x1421, 0x1423, 0x1425,
	0x1427, 0x1428, 0x142a, 0x142c, 0x142e, 0x1430, 0x1432, 0x1434, 0x1436, 0x1438, 0x143a, 0x143c,
	0x143e, 0x1440, 0x1442, 0x1444, 0x1446, 0x1448, 0x144a, 0x144c, 0x144e, 0x1450, 0x1452, 0x1454,
	0x1456, 0x1458, 0x145a, 0x145c, 0x145e, 0x1460, 0x1462, 0x1464, 0x1466, 0x1468, 0x146a, 0x146c,
	0x146e, 0x1470, 0x1472, 0x1474, 0x1476, 0x1478, 0x147a, 0x147c, 0x147e, 0x1480, 0x1483, 0x1485,
	0x1487, 0x1489, 0x148b, 0x148d, 0x148f, 0x1491, 0x1493, 0x1495, 0x1498, 0x149a, 0x149c, 0x149e,
	0x14a0, 0x14a2, 0x14a4, 0x14a6, 0x14a9, 0x14ab, 0x14ad, 0x14af, 0x14b1, 0x14b3, 0x14b6, 0x14b8,
	0x14ba, 0x14bc, 0x14be, 0x14c1, 0x14c3, 0x14c5, 0x14c7, 0x14c9, 0x14cc, 0x14ce, 0x14d0, 0x14d2,
	0x14d4, 0x14d7, 0x14d9, 0x14db, 0x14dd, 0x14e0, 0x14e2, 0x14e4, 0x14e6, 0x14e9, 0x14eb, 0x14ed,
	0x14f0, 0x14f2, 0x14f4, 0x14f6, 0x14f9, 0x14fb, 0x14fd, 0x1500, 0x1502, 0x1504, 0x1507, 0x1509,
	0x150b, 0x150e, 0x1510, 0x1512, 0x1515, 0x1517, 0x1519, 0x151c, 0x151e, 0x1520, 0x1523, 0x1525,
	0x1528, 0x152a, 0x152c, 0x152f, 0x1531, 0x1533, 0x1536, 0x1538, 0x153b, 0x153d, 0x1540, 0x1542,
	0x1544, 0x1547, 0x1549, 0x154c, 0x154e, 0x1551, 0x1553, 0x1556, 0x1558, 0x155a, 0x155d, 0x155f,
	0x1562, 0x1564, 0x1567, 0x1569, 0x156c, 0x156e, 0x1571, 0x1573, 0x1576, 0x1578, 0x157b, 0x157e,
	0x1580, 0x1583, 0x1585, 0x1588, 0x158a, 0x158d, 0x158f, 0x1592, 0x1595, 0x1597, 0x159a, 0x159c,
	0x159f, 0x15a2, 0x15a4, 0x15a7, 0x15a9, 0x15ac, 0x15af, 0x15b1, 0x15b4, 0x15b6, 0x15b9, 0x15bc,
	0x15be, 0x15c1, 0x15c4, 0x15c6, 0x15c9, 0x15cc, 0x15ce, 0x15d1, 0x15d4, 0x15d7, 0x15d9, 0x15dc,
	0x15df, 0x15e1, 0x15e4, 0x15e7, 0x15ea, 0x15ec, 0x15ef, 0x15f2, 0x15f4, 0x15f7, 0x15fa, 0x15fd,
	0x1600, 0x1602, 0x1605, 0x1608, 0x160b, 0x160d, 0x1610, 0x1613, 0x1616, 0x1619, 0x161c, 0x161e,
	0x1621, 0x1624, 0x1627, 0x162a, 0x162d, 0x162f, 0x1632, 0x1635, 0x1638, 0x163b, 0x163e, 0x1641,
	0x1644, 0x1646, 0x1649, 0x164c, 0x164f, 0x1652, 0x1655, 0x1658, 0x165b, 0x165e, 0x1661, 0x1664,
	0x1667, 0x166a, 0x166d, 0x1670, 0x1673, 0x1675, 0x1678, 0x167b, 0x167e, 0x1681, 0x1684, 0x1687,
	0x168b, 0x168e, 0x1691, 0x1694, 0x1697, 0x169a, 0x169d, 0x16a0, 0x16a3, 0x16a6, 0x16a9, 0x16ac,
	0x16af, 0x16b2, 0x16b5, 0x16b8, 0x16bc, 0x16bf, 0x16c2, 0x16c5, 0x16c8, 0x16cb, 0x16ce, 0x16d1,
	0x16d5, 0x16d8, 0x16db, 0x16de, 0x16e1, 0x16e5, 0x16e8, 0x16eb, 0x16ee, 0x16f1, 0x16f5, 0x16f8,
	0x16fb, 0x16fe, 0x1701, 0x1705, 0x1708, 0x170b, 0x170e, 0x1712, 0x1715, 0x1718, 0x171b, 0x171f,
	0x1722, 0x1725, 0x1729, 0x172c, 0x172f, 0x1733, 0x1736, 0x1739, 0x173d, 0x1740, 0x1743, 0x1747,
	0x174a, 0x174d, 0x1751, 0x1754, 0x1758, 0x175b, 0x175e, 0x1762, 0x1765, 0x1769, 0x176c, 0x176f,
	0x1773, 0x1776, 0x177a, 0x177d, 0x1781, 0x1784, 0x1788, 0x178b, 0x178f, 0x1792, 0x1796, 0x1799,
	0x179d, 0x17a0, 0x17a4, 0x17a7, 0x17ab, 0x17ae, 0x17b2, 0x17b5, 0x17b9, 0x17bd, 0x17c0, 0x17c4,
	0x17c7, 0x17cb, 0x17cf, 0x17d2, 0x17d6, 0x17d9, 0x17dd, 0x17e1, 0x17e4, 0x17e8, 0x17ec, 0x17ef,
	0x17f3, 0x17f7, 0x17fa, 0x17fe, 0x1c01, 0x1c03, 0x1c05, 0x1c06, 0x1c08, 0x1c0a, 0x1c0c, 0x1c0e,
	0x1c10, 0x1c12, 0x1c14, 0x1c15, 0x1c17, 0x1c19, 0x1c1b, 0x1c1d, 0x1c1f, 0x1c21, 0x1c23, 0x1c25,
	0x1c27, 0x1c28, 0x1c2a, 0x1c2c, 0x1c2e, 0x1c30, 0x1c32, 0x1c34, 0x1c36, 0x1c38, 0x1c3a, 0x1c3c,
	0x1c3e, 0x1c40, 0x1c42, 0x1c44, 0x1c46, 0x1c48, 0x1c4a, 0x1c4c, 0x1c4e, 0x1c50, 0x1c52, 0x1c54,
	0x1c56, 0x1c58, 0x1c5a, 0x1c5c, 0x1c5e, 0x1c60, 0x1c62, 0x1c64, 0x1c66, 0x1c68, 0x1c6a, 0x1c6c,
	0x1c6e, 0x1c70, 0x1c72, 0x1c74, 0x1c76, 0x1c78, 0x1c7a, 0x1c7c, 0x1c7e, 0x1c80, 0x1c83, 0x1c85,
	0x1c87, 0x1c89, 0x1c8b, 0x1c8d, 0x1c8f, 0x1c91, 0x1c93, 0x1c95, 0x1c98, 0x1c9a, 0x1c9c, 0x1c9e,
	0x1ca0, 0x1ca2, 0x1ca4, 0x1ca6, 0x1ca9, 0x1cab, 0x1cad, 0x1caf, 0x1cb1, 0x1cb3, 0x1cb6, 0x1cb8,
	0x1cba, 0x1cbc, 0x1cbe, 0x1cc1, 0x1cc3, 0x1cc5, 0x1cc7, 0x1cc9, 0x1ccc, 0x1cce, 0x1cd0, 0x1cd2,
	0x1cd4, 0x1cd7, 0x1cd9, 0x1cdb, 0x1cdd, 0x1ce0, 0x1ce2, 0x1ce4, 0x1ce6, 0x1ce9, 0x1ceb, 0x1ced,
	0x1cf0, 0x1cf2, 0x1cf4, 0x1cf6, 0x1cf9, 0x1cfb, 0x1cfd, 0x1d00, 0x1d02, 0x1d04, 0x1d07, 0x1d09,
	0x1d0b, 0x1d0e, 0x1d10, 0x1d12, 0x1d15, 0x1d17, 0x1d19, 0x1d1c, 0x1d1e, 0x1d20, 0x1d23, 0x1d25,
	0x1d28, 0x1d2a, 0x1d2c, 0x1d2f, 0x1d31, 0x1d33, 0x1d36, 0x1d38, 0x1d3b, 0x1d3d, 0x1d40, 0x1d42,
	0x1d44, 0x1d47, 0x1d49, 0x1d4c, 0x1d4e, 0x1d51, 0x1d53, 0x1d56, 0x1d58, 0x1d5a, 0x1d5d, 0x1d5f,
	0x1d62, 0x1d64, 0x1d67, 0x1d69, 0x1d6c, 0x1d6e, 0x1d71, 0x1d73, 0x1d76, 0x1d78, 0x1d7b, 0x1d7e,
	0x1d80, 0x1d83, 0x1d85, 0x1d88, 0x1d8a, 0x1d8d, 0x1d8f, 0x1d92, 0x1d95, 0x1d97, 0x1d9a, 0x1d9c,
	0x1d9f, 0x1da2, 0x1da4, 0x1da7, 0x1da9, 0x1dac, 0x1daf, 0x1db1, 0x1db4, 0x1db6, 0x1db9, 0x1dbc,
	0x1dbe, 0x1dc1, 0x1dc4, 0x1dc6, 0x1dc9, 0x1dcc, 0x1dce, 0x1dd1, 0x1dd4, 0x1dd7, 0x1dd9, 0x1ddc,
	0x1ddf, 0x1de1, 0x1de4, 0x1de7, 0x1dea, 0x1dec, 0x1def, 0x1df2, 0x1df4, 0x1df7, 0x1dfa, 0x1dfd,
	0x1e00, 0x1e02, 0x1e05, 0x1e08, 0x1e0b, 0x1e0d, 0x1e10, 0x1e13, 0x1e16, 0x1e19, 0x1e1c, 0x1e1e,
	0x1e21, 0x1e24, 0x1e27, 0x1e2a, 0x1e2d, 0x1e2f, 0x1e32, 0x1e35, 0x1e38, 0x1e3b, 0x1e3e, 0x1e41,
	0x1e44, 0x1e46, 0x1e49, 0x1e4c, 0x1e4f, 0x1e52, 0x1e55, 0x1e58, 0x1e5b, 0x1e5e, 0x1e61, 0x1e64,
	0x1e67, 0x1e6a, 0x1e6d, 0x1e70, 0x1e73, 0x1e75, 0x1e78, 0x1e7b, 0x1e7e, 0x1e81, 0x1e84, 0x1e87,
	0x1e8b, 0x1e8e, 0x1e91, 0x1e94, 0x1e97, 0x1e9a, 0x1e9d, 0x1ea0, 0x1ea3, 0x1ea6, 0x1ea9, 0x1eac,
	0x1eaf, 0x1eb2, 0x1eb5, 0x1eb8, 0x1ebc, 0x1ebf, 0x1ec2, 0x1ec5, 0x1ec8, 0x1ecb, 0x1ece, 0x1ed1,
	0x1ed5, 0x1ed8, 0x1edb, 0x1ede, 0x1ee1, 0x1ee5, 0x1ee8, 0x1eeb, 0x1eee, 0x1ef1, 0x1ef5, 0x1ef8,
	0x1efb, 0x1efe, 0x1f01, 0x1f05, 0x1f08, 0x1f0b, 0x1f0e, 0x1f12, 0x1f15, 0x1f18, 0x1f1b, 0x1f1f,
	0x1f22, 0x1f25, 0x1f29, 0x1f2c, 0x1f2f, 0x1f33, 0x1f36, 0x1f39, 0x1f3d, 0x1f40, 0x1f43, 0x1f47,
	0x1f4a, 0x1f4d, 0x1f51, 0x1f54, 0x1f58, 0x1f5b, 0x1f5e, 0x1f62, 0x1f65, 0x1f69, 0x1f6c, 0x1f6f,
	0x1f73, 0x1f76, 0x1f7a, 0x1f7d, 0x1f81, 0x1f84, 0x1f88, 0x1f8b, 0x1f8f, 0x1f92, 0x1f96, 0x1f99,
	0x1f9d, 0x1fa0, 0x1fa4, 0x1fa7, 0x1fab, 0x1fae, 0x1fb2, 0x1fb5, 0x1fb9, 0x1fbd, 0x1fc0, 0x1fc4,
	0x1fc7, 0x1fcb, 0x1fcf, 0x1fd2, 0x1fd6, 0x1fd9, 0x1fdd, 0x1fe1, 0x1fe4, 0x1fe8, 0x1fec, 0x1fef,
	0x1ff3, 0x1ff7, 0x1ffa, 0x1ffe, 0x2401, 0x2403, 0x2405, 0x2406, 0x2408, 0x240a, 0x240c, 0x240e,
	0x2410, 0x2412, 0x2414, 0x2415, 0x2417, 0x2419, 0x241b, 0x241d, 0x241f, 0x2421, 0x2423, 0x2425,
	0x2427, 0x2428, 0x242a, 0x242c, 0x242e, 0x2430, 0x2432, 0x2434, 0x2436, 0x2438, 0x243a, 0x243c,
	0x243e, 0x2440, 0x2442, 0x2444, 0x2446, 0x2448, 0x244a, 0x244c, 0x244e, 0x2450, 0x2452, 0x2454,
	0x2456, 0x2458, 0x245a, 0x245c, 0x245e, 0x2460, 0x2462, 0x2464, 0x2466, 0x2468, 0x246a, 0x246c,
	0x246e, 0x2470, 0x2472, 0x2474, 0x2476, 0x2478, 0x247a, 0x247c, 0x247e, 0x2480, 0x2483, 0x2485,
	0x2487, 0x2489, 0x248b, 0x248d, 0x248f, 0x2491, 0x2493, 0x2495, 0x2498, 0x249a, 0x249c, 0x249e,
	0x24a0, 0x24a2, 0x24a4, 0x24a6, 0x24a9, 0x24ab, 0x24ad, 0x24af, 0x24b1, 0x24b3, 0x24b6, 0x24b8,
	0x24ba, 0x24bc, 0x24be, 0x24c1, 0x24c3, 0x24c5, 0x24c7, 0x24c9, 0x24cc, 0x24ce, 0x24d0, 0x24d2,
	0x24d4, 0x24d7, 0x24d9, 0x24db, 0x24dd, 0x24e0, 0x24e2, 0x24e4, 0x24e6, 0x24e9, 0x24eb, 0x24ed,
	0x24f0, 0x24f2, 0x24f4, 0x24f6, 0x24f9, 0x24fb, 0x24fd, 0x2500, 0x2502, 0x2504, 0x2507, 0x2509,
	0x250b, 0x250e, 0x2510, 0x2512, 0x2515, 0x2517, 0x2519, 0x251c, 0x251e, 0x2520, 0x2523, 0x2525,
	0x2528, 0x252a, 0x252c, 0x252f, 0x2531, 0x2533, 0x2536, 0x2538, 0x253b, 0x253d, 0x2540, 0x2542,
	0x2544, 0x2547, 0x2549, 0x254c, 0x254e, 0x2551, 0x2553, 0x2556, 0x2558, 0x255a, 0x255d, 0x255f,
	0x2562, 0x2564, 0x2567, 0x2569, 0x256c, 0x256e, 0x2571, 0x2573, 0x2576, 0x2578, 0x257b, 0x257e,
	0x2580, 0x2583, 0x2585, 0x2588, 0x258a, 0x258d, 0x258f, 0x2592, 0x2595, 0x2597, 0x259a, 0x259c,
	0x259f, 0x25a2, 0x25a4, 0x25a7, 0x25a9, 0x25ac, 0x25af, 0x25b1, 0x25b4, 0x25b6, 0x25b9, 0x25bc,
	0x25be, 0x25c1, 0x25c4, 0x25c6, 0x25c9, 0x25cc, 0x25ce, 0x25d1, 0x25d4, 0x25d7, 0x25d9, 0x25dc,
	0x25df, 0x25e1, 0x25e4, 0x25e7, 0x25ea, 0x25ec, 0x25ef, 0x25f2, 0x25f4, 0x25f7, 0x25fa, 0x25fd,
	0x2600, 0x2602, 0x2605, 0x2608, 0x260b, 0x260d, 0x2610, 0x2613, 0x2616, 0x2619, 0x261c, 0x261e,
	0x2621, 0x2624, 0x2627, 0x262a, 0x262d, 0x262f, 0x2632, 0x2635, 0x2638, 0x263b, 0x263e, 0x2641,
	0x2644, 0x2646, 0x2649, 0x264c, 0x264f, 0x2652, 0x2655, 0x2658, 0x265b, 0x265e, 0x2661, 0x2664,
	0x2667, 0x266a, 0x266d, 0x2670, 0x2673, 0x2675, 0x2678, 0x267b, 0x267e, 0x2681, 0x2684, 0x2687,
	0x268b, 0x268e, 0x2691, 0x2694, 0x2697, 0x269a, 0x269d, 0x26a0, 0x26a3, 0x26a6, 0x26a9, 0x26ac,
	0x26af, 0x26b2, 0x26b5, 0x26b8, 0x26bc, 0x26bf, 0x26c2, 0x26c5, 0x26c8, 0x26cb, 0x26ce, 0x26d1,
	0x26d5, 0x26d8, 0x26db, 0x26de, 0x26e1, 0x26e5, 0x26e8, 0x26eb, 0x26ee, 0x26f1, 0x26f5, 0x26f8,
	0x26fb, 0x26fe, 0x2701, 0x2705, 0x2708, 0x270b, 0x270e, 0x2712, 0x2715, 0x2718, 0x271b, 0x271f,
	0x2722, 0x2725, 0x2729, 0x272c, 0x272f, 0x2733, 0x2736, 0x2739, 0x273d, 0x2740, 0x2743, 0x2747,
	0x274a, 0x274d, 0x2751, 0x2754, 0x2758, 0x275b, 0x275e, 0x2762, 0x2765, 0x2769, 0x276c, 0x276f,
	0x2773, 0x2776, 0x277a, 0x277d, 0x2781, 0x2784, 0x2788, 0x278b, 0x278f, 0x2792, 0x2796, 0x2799,
	0x279d, 0x27a0, 0x27a4, 0x27a7, 0x27ab, 0x27ae, 0x27b2, 0x27b5, 0x27b9, 0x27bd, 0x27c0, 0x27c4,
	0x27c7, 0x27cb, 0x27cf, 0x27d2, 0x27d6, 0x27d9, 0x27dd, 0x27e1, 0x27e4, 0x27e8, 0x27ec, 0x27ef,
	0x27f3, 0x27f7, 0x27fa, 0x27fe, 0x2c01, 0x2c03, 0x2c05, 0x2c06, 0x2c08, 0x2c0a, 0x2c0c, 0x2c0e,
	0x2c10, 0x2c12, 0x2c14, 0x2c15, 0x2c17, 0x2c19, 0x2c1b, 0x2c1d, 0x2c1f, 0x2c21, 0x2c23, 0x2c25,
	0x2c27, 0x2c28, 0x2c2a, 0x2c2c, 0x2c2e, 0x2c30, 0x2c32, 0x2c34, 0x2c36, 0x2c38, 0x2c3a, 0x2c3c,
	0x2c3e, 0x2c40, 0x2c42, 0x2c44, 0x2c46, 0x2c48, 0x2c4a, 0x2c4c, 0x2c4e, 0x2c50, 0x2c52, 0x2c54,
	0x2c56, 0x2c58, 0x2c5a, 0x2c5c, 0x2c5e, 0x2c60, 0x2c62, 0x2c64, 0x2c66, 0x2c68, 0x2c6a, 0x2c6c,
	0x2c6e, 0x2c70, 0x2c72, 0x2c74, 0x2c76, 0x2c78, 0x2c7a, 0x2c7c, 0x2c7e, 0x2c80, 0x2c83, 0x2c85,
	0x2c87, 0x2c89, 0x2c8b, 0x2c8d, 0x2c8f, 0x2c91, 0x2c93, 0x2c95, 0x2c98, 0x2c9a, 0x2c9c, 0x2c9e,
	0x2ca0, 0x2ca2, 0x2ca4, 0x2ca6, 0x2ca9, 0x2cab, 0x2cad, 0x2caf, 0x2cb1, 0x2cb3, 0x2cb6, 0x2cb8,
	0x2cba, 0x2cbc, 0x2cbe, 0x2cc1, 0x2cc3, 0x2cc5, 0x2cc7, 0x2cc9, 0x2ccc, 0x2cce, 0x2cd0, 0x2cd2,
	0x2cd4, 0x2cd7, 0x2cd9, 0x2cdb, 0x2cdd, 0x2ce0, 0x2ce2, 0x2ce4, 0x2ce6, 0x2ce9, 0x2ceb, 0x2ced,
	0x2cf0, 0x2cf2, 0x2cf4, 0x2cf6, 0x2cf9, 0x2cfb, 0x2cfd, 0x2d00, 0x2d02, 0x2d04, 0x2d07, 0x2d09,
	0x2d0b, 0x2d0e, 0x2d10, 0x2d12, 0x2d15, 0x2d17, 0x2d19, 0x2d1c, 0x2d1e, 0x2d20, 0x2d23, 0x2d25,
	0x2d28, 0x2d2a, 0x2d2c, 0x2d2f, 0x2d31, 0x2d33, 0x2d36, 0x2d38, 0x2d3b, 0x2d3d, 0x2d40, 0x2d42,
	0x2d44, 0x2d47, 0x2d49, 0x2d4c, 0x2d4e, 0x2d51, 0x2d53, 0x2d56, 0x2d58, 0x2d5a, 0x2d5d, 0x2d5f,
	0x2d62, 0x2d64, 0x2d67, 0x2d69, 0x2d6c, 0x2d6e, 0x2d71, 0x2d73, 0x2d76, 0x2d78, 0x2d7b, 0x2d7e,
	0x2d80, 0x2d83, 0x2d85, 0x2d88, 0x2d8a, 0x2d8d, 0x2d8f, 0x2d92, 0x2d95, 0x2d97, 0x2d9a, 0x2d9c,
	0x2d9f, 0x2da2, 0x2da4, 0x2da7, 0x2da9, 0x2dac, 0x2daf, 0x2db1, 0x2db4, 0x2db6, 0x2db9, 0x2dbc,
	0x2dbe, 0x2dc1, 0x2dc4, 0x2dc6, 0x2dc9, 0x2dcc, 0x2dce, 0x2dd1, 0x2dd4, 0x2dd7, 0x2dd9, 0x2ddc,
	0x2ddf, 0x2de1, 0x2de4, 0x2de7, 0x2dea, 0x2dec, 0x2def, 0x2df2, 0x2df4, 0x2df7, 0x2dfa, 0x2dfd,
	0x2e00, 0x2e02, 0x2e05, 0x2e08, 0x2e0b, 0x2e0d, 0x2e10, 0x2e13, 0x2e16, 0x2e19, 0x2e1c, 0x2e1e,
	0x2e21, 0x2e24, 0x2e27, 0x2e2a, 0x2e2d, 0x2e2f, 0x2e32, 0x2e35, 0x2e38, 0x2e3b, 0x2e3e, 0x2e41,
	0x2e44, 0x2e46, 0x2e49, 0x2e4c, 0x2e4f, 0x2e52, 0x2e55, 0x2e58, 0x2e5b, 0x2e5e, 0x2e61, 0x2e64,
	0x2e67, 0x2e6a, 0x2e6d, 0x2e70, 0x2e73, 0x2e75, 0x2e78, 0x2e7b, 0x2e7e, 0x2e81, 0x2e84, 0x2e87,
	0x2e8b, 0x2e8e, 0x2e91, 0x2e94, 0x2e97, 0x2e9a, 0x2e9d, 0x2ea0, 0x2ea3, 0x2ea6, 0x2ea9, 0x2eac,
	0x2eaf, 0x2eb2, 0x2eb5, 0x2eb8, 0x2ebc, 0x2ebf, 0x2ec2, 0x2ec5, 0x2ec8, 0x2ecb, 0x2ece, 0x2ed1,
	0x2ed5, 0x2ed8, 0x2edb, 0x2ede, 0x2ee1, 0x2ee5, 0x2ee8, 0x2eeb, 0x2eee, 0x2ef1, 0x2ef5, 0x2ef8,
	0x2efb, 0x2efe, 0x2f01, 0x2f05, 0x2f08, 0x2f0b, 0x2f0e, 0x2f12, 0x2f15, 0x2f18, 0x2f1b, 0x2f1f,
	0x2f22, 0x2f25, 0x2f29, 0x2f2c, 0x2f2f, 0x2f33, 0x2f36, 0x2f39, 0x2f3d, 0x2f40, 0x2f43, 0x2f47,
	0x2f4a, 0x2f4d, 0x2f51, 0x2f54, 0x2f58, 0x2f5b, 0x2f5e, 0x2f62, 0x2f65, 0x2f69, 0x2f6c, 0x2f6f,
	0x2f73, 0x2f76, 0x2f7a, 0x2f7d, 0x2f81, 0x2f84, 0x2f88, 0x2f8b, 0x2f8f, 0x2f92, 0x2f96, 0x2f99,
	0x2f9d, 0x2fa0, 0x2fa4, 0x2fa7, 0x2fab, 0x2fae, 0x2fb2, 0x2fb5, 0x2fb9, 0x2fbd, 0x2fc0, 0x2fc4,
	0x2fc7, 0x2fcb, 0x2fcf, 0x2fd2, 0x2fd6, 0x2fd9, 0x2fdd, 0x2fe1, 0x2fe4, 0x2fe8, 0x2fec, 0x2fef,
	0x2ff3, 0x2ff7, 0x2ffa, 0x2ffe, 0x3401, 0x3403, 0x3405, 0x3406, 0x3408, 0x340a, 0x340c, 0x340e,
	0x3410, 0x3412, 0x3414, 0x3415, 0x3417, 0x3419, 0x341b, 0x341d, 0x341f, 0x3421, 0x3423, 0x3425,
	0x3427, 0x3428, 0x342a, 0x342c, 0x342e, 0x3430, 0x3432, 0x3434, 0x3436, 0x3438, 0x343a, 0x343c,
	0x343e, 0x3440, 0x3442, 0x3444, 0x3446, 0x3448, 0x344a, 0x344c, 0x344e, 0x3450, 0x3452, 0x3454,
	0x3456, 0x3458, 0x345a, 0x345c, 0x345e, 0x3460, 0x3462, 0x3464, 0x3466, 0x3468, 0x346a, 0x346c,
	0x346e, 0x3470, 0x3472, 0x3474, 0x3476, 0x3478, 0x347a, 0x347c, 0x347e, 0x3480, 0x3483, 0x3485,
	0x3487, 0x3489, 0x348b, 0x348d, 0x348f, 0x3491, 0x3493, 0x3495, 0x3498, 0x349a, 0x349c, 0x349e,
	0x34a0, 0x34a2, 0x34a4, 0x34a6, 0x34a9, 0x34ab, 0x34ad, 0x34af, 0x34b1, 0x34b3, 0x34b6, 0x34b8,
	0x34ba, 0x34bc, 0x34be, 0x34c1, 0x34c3, 0x34c5, 0x34c7, 0x34c9, 0x34cc, 0x34ce, 0x34d0, 0x34d2,
	0x34d4, 0x34d7, 0x34d9, 0x34db, 0x34dd, 0x34e0, 0x34e2, 0x34e4, 0x34e6, 0x34e9, 0x34eb, 0x34ed,
	0x34f0, 0x34f2, 0x34f4, 0x34f6, 0x34f9, 0x34fb, 0x34fd, 0x3500, 0x3502, 0x3504, 0x3507, 0x3509,
	0x350b, 0x350e, 0x3510, 0x3512, 0x3515, 0x3517, 0x3519, 0x351c, 0x351e, 0x3520, 0x3523, 0x3525,
	0x3528, 0x352a, 0x352c, 0x352f, 0x3531, 0x3533, 0x3536, 0x3538, 0x353b, 0x353d, 0x3540, 0x3542,
	0x3544, 0x3547, 0x3549, 0x354c, 0x354e, 0x3551, 0x3553, 0x3556, 0x3558, 0x355a, 0x355d, 0x355f,
	0x3562, 0x3564, 0x3567, 0x3569, 0x356c, 0x356e, 0x3571, 0x3573, 0x3576, 0x3578, 0x357b, 0x357e,
	0x3580, 0x3583, 0x3585, 0x3588, 0x358a, 0x358d, 0x358f, 0x3592, 0x3595, 0x3597, 0x359a, 0x359c,
	0x359f, 0x35a2, 0x35a4, 0x35a7, 0x35a9, 0x35ac, 0x35af, 0x35b1, 0x35b4, 0x35b6, 0x35b9, 0x35bc,
	0x35be, 0x35c1, 0x35c4, 0x35c6, 0x35c9, 0x35cc, 0x35ce, 0x35d1, 0x35d4, 0x35d7, 0x35d9, 0x35dc,
	0x35df, 0x35e1, 0x35e4, 0x35e7, 0x35ea, 0x35ec, 0x35ef, 0x35f2, 0x35f4, 0x35f7, 0x35fa, 0x35fd,
	0x3600, 0x3602, 0x3605, 0x3608, 0x360b, 0x360d, 0x3610, 0x3613, 0x3616, 0x3619, 0x361c, 0x361e,
	0x3621, 0x3624, 0x3627, 0x362a, 0x362d, 0x362f, 0x3632, 0x3635, 0x3638, 0x363b, 0x363e, 0x3641,
	0x3644, 0x3646, 0x3649, 0x364c, 0x364f, 0x3652, 0x3655, 0x3658, 0x365b, 0x365e, 0x3661, 0x3664,
	0x3667, 0x366a, 0x366d, 0x3670, 0x3673, 0x3675, 0x3678, 0x367b, 0x367e, 0x3681, 0x3684, 0x3687,
	0x368b, 0x368e, 0x3691, 0x3694, 0x3697, 0x369a, 0x369d, 0x36a0, 0x36a3, 0x36a6, 0x36a9, 0x36ac,
	0x36af, 0x36b2, 0x36b5, 0x36b8, 0x36bc, 0x36bf, 0x36c2, 0x36c5, 0x36c8, 0x36cb, 0x36ce, 0x36d1,
	0x36d5, 0x36d8, 0x36db, 0x36de, 0x36e1, 0x36e5, 0x36e8, 0x36eb, 0x36ee, 0x36f1, 0x36f5, 0x36f8,
	0x36fb, 0x36fe, 0x3701, 0x3705, 0x3708, 0x370b, 0x370e, 0x3712, 0x3715, 0x3718, 0x371b, 0x371f,
	0x3722, 0x3725, 0x3729, 0x372c, 0x372f, 0x3733, 0x3736, 0x3739, 0x373d, 0x3740, 0x3743, 0x3747,
	0x374a, 0x374d, 0x3751, 0x3754, 0x3758, 0x375b, 0x375e, 0x3762, 0x3765, 0x3769, 0x376c, 0x376f,
	0x3773, 0x3776, 0x377a, 0x377d, 0x3781, 0x3784, 0x3788, 0x378b, 0x378f, 0x3792, 0x3796, 0x3799,
	0x379d, 0x37a0, 0x37a4, 0x37a7, 0x37ab, 0x37ae, 0x37b2, 0x37b5, 0x37b9, 0x37bd, 0x37c0, 0x37c4,
	0x37c7, 0x37cb, 0x37cf, 0x37d2, 0x37d6, 0x37d9, 0x37dd, 0x37e1, 0x37e4, 0x37e8, 0x37ec, 0x37ef,
	0x37f3, 0x37f7, 0x37fa, 0x37fe, 0x3c01, 0x3c03, 0x3c05, 0x3c06, 0x3c08, 0x3c0a, 0x3c0c, 0x3c0e,
	0x3c10, 0x3c12, 0x3c14, 0x3c15, 0x3c17, 0x3c19, 0x3c1b, 0x3c1d, 0x3c1f, 0x3c21, 0x3c23, 0x3c25,
	0x3c27, 0x3c28, 0x3c2a, 0x3c2c, 0x3c2e, 0x3c30, 0x3c32, 0x3c34, 0x3c36, 0x3c38, 0x3c3a, 0x3c3c,
	0x3c3e, 0x3c40, 0x3c42, 0x3c44, 0x3c46, 0x3c48, 0x3c4a, 0x3c4c, 0x3c4e, 0x3c50, 0x3c52, 0x3c54,
	0x3c56, 0x3c58, 0x3c5a, 0x3c5c, 0x3c5e, 0x3c60, 0x3c62, 0x3c64, 0x3c66, 0x3c68, 0x3c6a, 0x3c6c,
	0x3c6e, 0x3c70, 0x3c72, 0x3c74, 0x3c76, 0x3c78, 0x3c7a, 0x3c7c, 0x3c7e, 0x3c80, 0x3c83, 0x3c85,
	0x3c87, 0x3c89, 0x3c8b, 0x3c8d, 0x3c8f, 0x3c91, 0x3c93, 0x3c95, 0x3c98, 0x3c9a, 0x3c9c, 0x3c9e,
	0x3ca0, 0x3ca2, 0x3ca4, 0x3ca6, 0x3ca9, 0x3cab, 0x3cad, 0x3caf, 0x3cb1, 0x3cb3, 0x3cb6, 0x3cb8,
	0x3cba, 0x3cbc, 0x3cbe, 0x3cc1, 0x3cc3, 0x3cc5, 0x3cc7, 0x3cc9, 0x3ccc, 0x3cce, 0x3cd0, 0x3cd2
};

const uint16_t PitchConverter::centTableSSGSquare_[3072] = {
	0xee8, 0xee1, 0xeda, 0xed4, 0xecd, 0xec6, 0xebf, 0xeb8, 0xeb1, 0xeab, 0xea4, 0xe9d,
	0xe96, 0xe90, 0xe89, 0xe82, 0xe7c, 0xe75, 0xe6e, 0xe67, 0xe61, 0xe5a, 0xe54, 0xe4d,
	0xe46, 0xe40, 0xe39, 0xe33, 0xe2c, 0xe26, 0xe1f, 0xe18, 0xe12, 0xe0b, 0xe05, 0xdff,
	0xdf8, 0xdf2, 0xdeb, 0xde5, 0xdde, 0xdd8, 0xdd2, 0xdcb, 0xdc5, 0xdbe, 0xdb8, 0xdb2,
	0xdab, 0xda5, 0xd9f, 0xd99, 0xd92, 0xd8c, 0xd86, 0xd7f, 0xd79, 0xd73, 0xd6d, 0xd67,
	0xd60, 0xd5a, 0xd54, 0xd4e, 0xd48, 0xd42, 0xd3c, 0xd35, 0xd2f, 0xd29, 0xd23, 0xd1d,
	0xd17, 0xd11, 0xd0b, 0xd05, 0xcff, 0xcf9, 0xcf3, 0xced, 0xce7, 0xce1, 0xcdb, 0xcd5,
	0xccf, 0xcc9, 0xcc3, 0xcbe, 0xcb8, 0xcb2, 0xcac, 0xca6, 0xca0, 0xc9a, 0xc95, 0xc8f,
	0xc89, 0xc83, 0xc7d, 0xc78, 0xc72, 0xc6c, 0xc66, 0xc61, 0xc5b, 0xc55, 0xc50, 0xc4a,
	0xc44, 0xc3f, 0xc39, 0xc33, 0xc2e, 0xc28, 0xc22, 0xc1d, 0xc17, 0xc12, 0xc0c, 0xc06,
	0xc01, 0xbfb, 0xbf6, 0xbf0, 0xbeb, 0xbe5, 0xbe0, 0xbda, 0xbd5, 0xbcf, 0xbca, 0xbc5,
	0xbbf, 0xbba, 0xbb4, 0xbaf, 0xba9, 0xba4, 0xb9f, 0xb99, 0xb94, 0xb8f, 0xb89, 0xb84,
	0xb7f, 0xb79, 0xb74, 0xb6f, 0xb69, 0xb64, 0xb5f, 0xb5a, 0xb54, 0xb4f, 0xb4a, 0xb45,
	0xb40, 0xb3a, 0xb35, 0xb30, 0xb2b, 0xb26, 0xb21, 0xb1b, 0xb16, 0xb11, 0xb0c, 0xb07,
	0xb02, 0xafd, 0xaf8, 0xaf3, 0xaee, 0xae9, 0xae4, 0xadf, 0xad9, 0xad4, 0xacf, 0xaca,
	0xac6, 0xac1, 0xabc, 0xab7, 0xab2, 0xaad, 0xaa8, 0xaa3, 0xa9e, 0xa99, 0xa94, 0xa8f,
	0xa8a, 0xa86, 0xa81, 0xa7c, 0xa77, 0xa72, 0xa6d, 0xa69, 0xa64, 0xa5f, 0xa5a, 0xa55,
	0xa51, 0xa4c, 0xa47, 0xa42, 0xa3e, 0xa39, 0xa34, 0xa2f, 0xa2b, 0xa26, 0xa21, 0xa1d,
	0xa18, 0xa13, 0xa0f, 0xa0a, 0xa05, 0xa01, 0x9fc, 0x9f8, 0x9f3, 0x9ee, 0x9ea, 0x9e5,
	0x9e1, 0x9dc, 0x9d8, 0x9d3, 0x9ce, 0x9ca, 0x9c5, 0x9c1, 0x9bc, 0x9b8, 0x9b3, 0x9af,
	0x9aa, 0x9a6, 0x9a2, 0x99d, 0x999, 0x994, 0x990, 0x98b, 0x987, 0x983, 0x97e, 0x97a,
	0x975, 0x971, 0x96d, 0x968, 0x964, 0x960, 0x95b, 0x957, 0x953, 0x94e, 0x94a, 0x946,
	0x942, 0x93d, 0x939, 0x935, 0x931, 0x92c, 0x928, 0x924, 0x920, 0x91b, 0x917, 0x913,
	0x90f, 0x90b, 0x906, 0x902, 0x8fe, 0x8fa, 0x8f6, 0x8f2, 0x8ee, 0x8e9, 0x8e5, 0x8e1,
	0x8dd, 0x8d9, 0x8d5, 0x8d1, 0x8cd, 0x8c9, 0x8c5, 0x8c1, 0x8bd, 0x8b9, 0x8b4, 0x8b0,
	0x8ac, 0x8a8, 0x8a4, 0x8a0, 0x89c, 0x899, 0x895, 0x891, 0x88d, 0x889, 0x885, 0x881,
	0x87d, 0x879, 0x875, 0x871, 0x86d, 0x869, 0x865, 0x862, 0x85e, 0x85a, 0x856, 0x852,
	0x84e, 0x84a, 0x847, 0x843, 0x83f, 0x83b, 0x837, 0x834, 0x830, 0x82c, 0x828, 0x825,
	0x821, 0x81d, 0x819, 0x816, 0x812, 0x80e, 0x80a, 0x807, 0x803, 0x7ff, 0x7fc, 0x7f8,
	0x7f4, 0x7f1, 0x7ed, 0x7e9, 0x7e6, 0x7e2, 0x7de, 0x7db, 0x7d7, 0x7d3, 0x7d0, 0x7cc,
	0x7c9, 0x7c5, 0x7c1, 0x7be, 0x7ba, 0x7b7, 0x7b3, 0x7b0, 0x7ac, 0x7a8, 0x7a5, 0x7a1,
	0x79e, 0x79a, 0x797, 0x793, 0x790, 0x78c, 0x789, 0x785, 0x782, 0x77e, 0x77b, 0x778,
	0x774, 0x771, 0x76d, 0x76a, 0x766, 0x763, 0x760, 0x75c, 0x759, 0x755, 0x752, 0x74f,
	0x74b, 0x748, 0x744, 0x741, 0x73e, 0x73a, 0x737, 0x734, 0x730, 0x72d, 0x72a, 0x726,
	0x723, 0x720, 0x71d, 0x719, 0x716, 0x713, 0x70f, 0x70c, 0x709, 0x706, 0x702, 0x6ff,
	0x6fc, 0x6f9, 0x6f6, 0x6f2, 0x6ef, 0x6ec, 0x6e9, 0x6e6, 0x6e2, 0x6df, 0x6dc, 0x6d9,
	0x6d6, 0x6d3, 0x6cf, 0x6cc, 0x6c9, 0x6c6, 0x6c3, 0x6c0, 0x6bd, 0x6ba, 0x6b6, 0x6b3,
	0x6b0, 0x6ad, 0x6aa, 0x6a7, 0x6a4, 0x6a1, 0x69e, 0x69b, 0x698, 0x695, 0x692, 0x68f,
	0x68c, 0x689, 0x685, 0x682, 0x67f, 0x67c, 0x679, 0x676, 0x674, 0x671, 0x66e, 0x66b,
	0x668, 0x665, 0x662, 0x65f, 0x65c, 0x659, 0x656, 0x653, 0x650, 0x64d, 0x64a, 0x647,
	0x644, 0x642, 0x63f, 0x63c, 0x639, 0x636, 0x633, 0x630, 0x62d, 0x62b, 0x628, 0x625,
	0x622, 0x61f, 0x61c, 0x61a, 0x617, 0x614, 0x611, 0x60e, 0x60c, 0x609, 0x606, 0x603,
	0x600, 0x5fe, 0x5fb, 0x5f8, 0x5f5, 0x5f3, 0x5f0, 0x5ed, 0x5ea, 0x5e8, 0x5e5, 0x5e2,
	0x5e0, 0x5dd, 0x5da, 0x5d7, 0x5d5, 0x5d2, 0x5cf, 0x5cd, 0x5ca, 0x5c7, 0x5c5, 0x5c2,
	0x5bf, 0x5bd, 0x5ba, 0x5b7, 0x5b5, 0x5b2, 0x5af, 0x5ad, 0x5aa, 0x5a8, 0x5a5, 0x5a2,
	0x5a0, 0x59d, 0x59b, 0x598, 0x595, 0x593, 0x590, 0x58e, 0x58b, 0x589, 0x586, 0x583,
	0x581, 0x57e, 0x57c, 0x579, 0x577, 0x574, 0x572, 0x56f, 0x56d, 0x56a, 0x568, 0x565,
	0x563, 0x560, 0x55e, 0x55b, 0x559, 0x556, 0x554, 0x551, 0x54f, 0x54d, 0x54a, 0x548,
	0x545, 0x543, 0x540, 0x53e, 0x53c, 0x539, 0x537, 0x534, 0x532, 0x52f, 0x52d, 0x52b,
	0x528, 0x526, 0x524, 0x521, 0x51f, 0x51c, 0x51a, 0x518, 0x515, 0x513, 0x511, 0x50e,
	0x50c, 0x50a, 0x507, 0x505, 0x503, 0x500, 0x4fe, 0x4fc, 0x4f9, 0x4f7, 0x4f5, 0x4f3,
	0x4f0, 0x4ee, 0x4ec, 0x4e9, 0x4e7, 0x4e5, 0x4e3, 0x4e0, 0x4de, 0x4dc, 0x4da, 0x4d7,
	0x4d5, 0x4d3, 0x4d1, 0x4cf, 0x4cc, 0x4ca, 0x4c8, 0x4c6, 0x4c3, 0x4c1, 0x4bf, 0x4bd,
	0x4bb, 0x4b9, 0x4b6, 0x4b4, 0x4b2, 0x4b0, 0x4ae, 0x4ac, 0x4a9, 0x4a7, 0x4a5, 0x4a3,
	0x4a1, 0x49f, 0x49d, 0x49a, 0x498, 0x496, 0x494, 0x492, 0x490, 0x48e, 0x48c, 0x489,
	0x487, 0x485, 0x483, 0x481, 0x47f, 0x47d, 0x47b, 0x479, 0x477, 0x475, 0x473, 0x471,
	0x46f, 0x46c, 0x46a, 0x468, 0x466, 0x464, 0x462, 0x460, 0x45e, 0x45c, 0x45a, 0x458,
	0x456, 0x454, 0x452, 0x450, 0x44e, 0x44c, 0x44a, 0x448, 0x446, 0x444, 0x442, 0x440,
	0x43e, 0x43c, 0x43b, 0x439, 0x437, 0x435, 0x433, 0x431, 0x42f, 0x42d, 0x42b, 0x429,
	0x427, 0x425, 0x423, 0x421, 0x420, 0x41e, 0x41c, 0x41a, 0x418, 0x416, 0x414, 0x412,
	0x410, 0x40f, 0x40d, 0x40b, 0x409, 0x407, 0x405, 0x403, 0x401, 0x400, 0x3fe, 0x3fc,
	0x3fa, 0x3f8, 0x3f6, 0x3f5, 0x3f3, 0x3f1, 0x3ef, 0x3ed, 0x3eb, 0x3ea, 0x3e8, 0x3e6,
	0x3e4, 0x3e2, 0x3e1, 0x3df, 0x3dd, 0x3db, 0x3da, 0x3d8, 0x3d6, 0x3d4, 0x3d2, 0x3d1,
	0x3cf, 0x3cd, 0x3cb, 0x3ca, 0x3c8, 0x3c6, 0x3c4, 0x3c3, 0x3c1, 0x3bf, 0x3bd, 0x3bc,
	0x3ba, 0x3b8, 0x3b7, 0x3b5, 0x3b3, 0x3b1, 0x3b0, 0x3ae, 0x3ac, 0x3ab, 0x3a9, 0x3a7,
	0x3a6, 0x3a4, 0x3a2, 0x3a1, 0x39f, 0x39d, 0x39c, 0x39a, 0x398, 0x397, 0x395, 0x393,
	0x392, 0x390, 0x38e, 0x38d, 0x38b, 0x389, 0x388, 0x386, 0x384, 0x383, 0x381, 0x380,
	0x37e, 0x37c, 0x37b, 0x379, 0x378, 0x376, 0x374, 0x373, 0x371, 0x370, 0x36e, 0x36c,
	0x36b, 0x369, 0x368, 0x366, 0x365, 0x363, 0x361, 0x360, 0x35e, 0x35d, 0x35b, 0x35a,
	0x358, 0x357, 0x355, 0x353, 0x352, 0x350, 0x34f, 0x34d, 0x34c, 0x34a, 0x349, 0x347,
	0x346, 0x344, 0x343, 0x341, 0x340, 0x33e, 0x33d, 0x33b, 0x33a, 0x338, 0x337, 0x335,
	0x334, 0x332, 0x331, 0x32f, 0x32e, 0x32c, 0x32b, 0x32a, 0x328, 0x327, 0x325, 0x324,
	0x322, 0x321, 0x31f, 0x31e, 0x31c, 0x31b, 0x31a, 0x318, 0x317, 0x315, 0x314, 0x312,
	0x311, 0x310, 0x30e, 0x30d, 0x30b, 0x30a, 0x309, 0x307, 0x306, 0x304, 0x303, 0x302,
	0x300, 0x2ff, 0x2fd, 0x2fc, 0x2fb, 0x2f9, 0x2f8, 0x2f7, 0x2f5, 0x2f4, 0x2f2, 0x2f1,
	0x2f0, 0x2ee, 0x2ed, 0x2ec, 0x2ea, 0x2e9, 0x2e8, 0x2e6, 0x2e5, 0x2e4, 0x2e2, 0x2e1,
	0x2e0, 0x2de, 0x2dd, 0x2dc, 0x2da, 0x2d9, 0x2d8, 0x2d6, 0x2d5, 0x2d4, 0x2d3, 0x2d1,
	0x2d0, 0x2cf, 0x2cd, 0x2cc, 0x2cb, 0x2c9, 0x2c8, 0x2c7, 0x2c6, 0x2c4, 0x2c3, 0x2c2,
	0x2c0, 0x2bf, 0x2be, 0x2bd, 0x2bb, 0x2ba, 0x2b9, 0x2b8, 0x2b6, 0x2b5, 0x2b4, 0x2b3,
	0x2b1, 0x2b0, 0x2af, 0x2ae, 0x2ac, 0x2ab, 0x2aa, 0x2a9, 0x2a7, 0x2a6, 0x2a5, 0x2a4,
	0x2a3, 0x2a1, 0x2a0, 0x29f, 0x29e, 0x29d, 0x29b, 0x29a, 0x299, 0x298, 0x297, 0x295,
	0x294, 0x293, 0x292, 0x291, 0x28f, 0x28e, 0x28d, 0x28c, 0x28b, 0x28a, 0x288, 0x287,
	0x286, 0x285, 0x284, 0x283, 0x281, 0x280, 0x27f, 0x27e, 0x27d, 0x27c, 0x27a, 0x279,
	0x278, 0x277, 0x276, 0x275, 0x274, 0x272, 0x271, 0x270, 0x26f, 0x26e, 0x26d, 0x26c,
	0x26b, 0x26a, 0x268, 0x267, 0x266, 0x265, 0x264, 0x263, 0x262, 0x261, 0x260, 0x25e,
	0x25d, 0x25c, 0x25b, 0x25a, 0x259, 0x258, 0x257, 0x256, 0x255, 0x254, 0x253, 0x251,
	0x250, 0x24f, 0x24e, 0x24d, 0x24c, 0x24b, 0x24a, 0x249, 0x248, 0x247, 0x246, 0x245,
	0x244, 0x243, 0x242, 0x241, 0x240, 0x23e, 0x23d, 0x23c, 0x23b, 0x23a, 0x239, 0x238,
	0x237, 0x236, 0x235, 0x234, 0x233, 0x232, 0x231, 0x230, 0x22f, 0x22e, 0x22d, 0x22c,
	0x22b, 0x22a, 0x229, 0x228, 0x227, 0x226, 0x225, 0x224, 0x223, 0x222, 0x221, 0x220,
	0x21f, 0x21e, 0x21d, 0x21c, 0x21b, 0x21a, 0x219, 0x218, 0x217, 0x216, 0x216, 0x215,
	0x214, 0x213, 0x212, 0x211, 0x210, 0x20f, 0x20e, 0x20d, 0x20c, 0x20b, 0x20a, 0x209,
	0x208, 0x207, 0x206, 0x205, 0x204, 0x204, 0x203, 0x202, 0x201, 0x200, 0x1ff, 0x1fe,
	0x1fd, 0x1fc, 0x1fb, 0x1fa, 0x1f9, 0x1f8, 0x1f8, 0x1f7, 0x1f6, 0x1f5, 0x1f4, 0x1f3,
	0x1f2, 0x1f1, 0x1f0, 0x1ef, 0x1ef, 0x1ee, 0x1ed, 0x1ec, 0x1eb, 0x1ea, 0x1e9, 0x1e8,
	0x1e7, 0x1e7, 0x1e6, 0x1e5, 0x1e4, 0x1e3, 0x1e2, 0x1e1, 0x1e0, 0x1e0, 0x1df, 0x1de,
	0x1dd, 0x1dc, 0x1db, 0x1da, 0x1da, 0x1d9, 0x1d8, 0x1d7, 0x1d6, 0x1d5, 0x1d4, 0x1d4,
	0x1d3, 0x1d2, 0x1d1, 0x1d0, 0x1cf, 0x1cf, 0x1ce, 0x1cd, 0x1cc, 0x1cb, 0x1ca, 0x1ca,
	0x1c9, 0x1c8, 0x1c7, 0x1c6, 0x1c6, 0x1c5, 0x1c4, 0x1c3, 0x1c2, 0x1c1, 0x1c1, 0x1c0,
	0x1bf, 0x1be, 0x1bd, 0x1bd, 0x1bc, 0x1bb, 0x1ba, 0x1b9, 0x1b9, 0x1b8, 0x1b7, 0x1b6,
	0x1b5, 0x1b5, 0x1b4, 0x1b3, 0x1b2, 0x1b1, 0x1b1, 0x1b0, 0x1af, 0x1ae, 0x1ae, 0x1ad,
	0x1ac, 0x1ab, 0x1ab, 0x1aa, 0x1a9, 0x1a8, 0x1a7, 0x1a7, 0x1a6, 0x1a5, 0x1a4, 0x1a4,
	0x1a3, 0x1a2, 0x1a1, 0x1a1, 0x1a0, 0x19f, 0x19e, 0x19e, 0x19d, 0x19c, 0x19b, 0x19b,
	0x19a, 0x199, 0x198, 0x198, 0x197, 0x196, 0x195, 0x195, 0x194, 0x193, 0x193, 0x192,
	0x191, 0x190, 0x190, 0x18f, 0x18e, 0x18e, 0x18d, 0x18c, 0x18b, 0x18b, 0x18a, 0x189,
	0x189, 0x188, 0x187, 0x186, 0x186, 0x185, 0x184, 0x184, 0x183, 0x182, 0x182, 0x181,
	0x180, 0x17f, 0x17f, 0x17e, 0x17d, 0x17d, 0x17c, 0x17b, 0x17b, 0x17a, 0x179, 0x179,
	0x178, 0x177, 0x177, 0x176, 0x175, 0x175, 0x174, 0x173, 0x172, 0x172, 0x171, 0x170,
	0x170, 0x16f, 0x16f, 0x16e, 0x16d, 0x16d, 0x16c, 0x16b, 0x16b, 0x16a, 0x169, 0x169,
	0x168, 0x167, 0x167, 0x166, 0x165, 0x165, 0x164, 0x163, 0x163, 0x162, 0x162, 0x161,
	0x160, 0x160, 0x15f, 0x15e, 0x15e, 0x15d, 0x15c, 0x15c, 0x15b, 0x15b, 0x15a, 0x159,
	0x159, 0x158, 0x157, 0x157, 0x156, 0x156, 0x155, 0x154, 0x154, 0x153, 0x153, 0x152,
	0x151, 0x151, 0x150, 0x14f, 0x14f, 0x14e, 0x14e, 0x14d, 0x14c, 0x14c, 0x14b, 0x14b,
	0x14a, 0x149, 0x149, 0x148, 0x148, 0x147, 0x147, 0x146, 0x145, 0x145, 0x144, 0x144,
	0x143, 0x142, 0x142, 0x141, 0x141, 0x140, 0x140, 0x13f, 0x13e, 0x13e, 0x13d, 0x13d,
	0x13c, 0x13c, 0x13b, 0x13a, 0x13a, 0x139, 0x139, 0x138, 0x138, 0x137, 0x136, 0x136,
	0x135, 0x135, 0x134, 0x134, 0x133, 0x133, 0x132, 0x131, 0x131, 0x130, 0x130, 0x12f,
	0x12f, 0x12e, 0x12e, 0x12d, 0x12d, 0x12c, 0x12b, 0x12b, 0x12a, 0x12a, 0x129, 0x129,
	0x128, 0x128, 0x127, 0x127, 0x126, 0x126, 0x125, 0x124, 0x124, 0x123, 0x123, 0x122,
	0x122, 0x121, 0x121, 0x120, 0x120, 0x11f, 0x11f, 0x11e, 0x11e, 0x11d, 0x11d, 0x11c,
	0x11c, 0x11b, 0x11b, 0x11a, 0x11a, 0x119, 0x119, 0x118, 0x118, 0x117, 0x117, 0x116,
	0x116, 0x115, 0x115, 0x114, 0x114, 0x113, 0x113, 0x112, 0x112, 0x111, 0x111, 0x110,
	0x110, 0x10f, 0x10f, 0x10e, 0x10e, 0x10d, 0x10d, 0x10c, 0x10c, 0x10b, 0x10b, 0x10a,
	0x10a, 0x109, 0x109, 0x108, 0x108, 0x107, 0x107, 0x106, 0x106, 0x106, 0x105, 0x105,
	0x104, 0x104, 0x103, 0x103, 0x102, 0x102, 0x101, 0x101, 0x100, 0x100, 0x0ff, 0x0ff,
	0x0ff, 0x0fe, 0x0fe, 0x0fd, 0x0fd, 0x0fc, 0x0fc, 0x0fb, 0x0fb, 0x0fa, 0x0fa, 0x0fa,
	0x0f9, 0x0f9, 0x0f8, 0x0f8, 0x0f7, 0x0f7, 0x0f6, 0x0f6, 0x0f5, 0x0f5, 0x0f5, 0x0f4,
	0x0f4, 0x0f3, 0x0f3, 0x0f2, 0x0f2, 0x0f2, 0x0f1, 0x0f1, 0x0f0, 0x0f0, 0x0ef, 0x0ef,
	0x0ef, 0x0ee, 0x0ee, 0x0ed, 0x0ed, 0x0ec, 0x0ec, 0x0ec, 0x0eb, 0x0eb, 0x0ea, 0x0ea,
	0x0e9, 0x0e9, 0x0e9, 0x0e8, 0x0e8, 0x0e7, 0x0e7, 0x0e6, 0x0e6, 0x0e6, 0x0e5, 0x0e5,
	0x0e4, 0x0e4, 0x0e4, 0x0e3, 0x0e3, 0x0e2, 0x0e2, 0x0e2, 0x0e1, 0x0e1, 0x0e0, 0x0e0,
	0x0e0, 0x0df, 0x0df, 0x0de, 0x0de, 0x0dd, 0x0dd, 0x0dd, 0x0dc, 0x0dc, 0x0dc, 0x0db,
	0x0db, 0x0da, 0x0da, 0x0da, 0x0d9, 0x0d9, 0x0d8, 0x0d8, 0x0d8, 0x0d7, 0x0d7, 0x0d6,
	0x0d6, 0x0d6, 0x0d5, 0x0d5, 0x0d4, 0x0d4, 0x0d4, 0x0d3, 0x0d3, 0x0d3, 0x0d2, 0x0d2,
	0x0d1, 0x0d1, 0x0d1, 0x0d0, 0x0d0, 0x0d0, 0x0cf, 0x0cf, 0x0ce, 0x0ce, 0x0ce, 0x0cd,
	0x0cd, 0x0cd, 0x0cc, 0x0cc, 0x0cb, 0x0cb, 0x0cb, 0x0ca, 0x0ca, 0x0ca, 0x0c9, 0x0c9,
	0x0c9, 0x0c8, 0x0c8, 0x0c7, 0x0c7, 0x0c7, 0x0c6, 0x0c6, 0x0c6, 0x0c5, 0x0c5, 0x0c5,
	0x0c4, 0x0c4, 0x0c4, 0x0c3, 0x0c3, 0x0c3, 0x0c2, 0x0c2, 0x0c1, 0x0c1, 0x0c1, 0x0c0,
	0x0c0, 0x0c0, 0x0bf, 0x0bf, 0x0bf, 0x0be, 0x0be, 0x0be, 0x0bd, 0x0bd, 0x0bd, 0x0bc,
	0x0bc, 0x0bc, 0x0bb, 0x0bb, 0x0bb, 0x0ba, 0x0ba, 0x0ba, 0x0b9, 0x0b9, 0x0b9, 0x0b8,
	0x0b8, 0x0b8, 0x0b7, 0x0b7, 0x0b7, 0x0b6, 0x0b6, 0x0b6, 0x0b5, 0x0b5, 0x0b5, 0x0b4,
	0x0b4, 0x0b4, 0x0b3, 0x0b3, 0x0b3, 0x0b2, 0x0b2, 0x0b2, 0x0b1, 0x0b1, 0x0b1, 0x0b0,
	0x0b0, 0x0b0, 0x0af, 0x0af, 0x0af, 0x0af, 0x0ae, 0x0ae, 0x0ae, 0x0ad, 0x0ad, 0x0ad,
	0x0ac, 0x0ac, 0x0ac, 0x0ab, 0x0ab, 0x0ab, 0x0aa, 0x0aa, 0x0aa, 0x0aa, 0x0a9, 0x0a9,
	0x0a9, 0x0a8, 0x0a8, 0x0a8, 0x0a7, 0x0a7, 0x0a7, 0x0a7, 0x0a6, 0x0a6, 0x0a6, 0x0a5,
	0x0a5, 0x0a5, 0x0a4, 0x0a4, 0x0a4, 0x0a4, 0x0a3, 0x0a3, 0x0a3, 0x0a2, 0x0a2, 0x0a2,
	0x0a2, 0x0a1, 0x0a1, 0x0a1, 0x0a0, 0x0a0, 0x0a0, 0x09f, 0x09f, 0x09f, 0x09f, 0x09e,
	0x09e, 0x09e, 0x09d, 0x09d, 0x09d, 0x09d, 0x09c, 0x09c, 0x09c, 0x09b, 0x09b, 0x09b,
	0x09b, 0x09a, 0x09a, 0x09a, 0x09a, 0x099, 0x099, 0x099, 0x098, 0x098, 0x098, 0x098,
	0x097, 0x097, 0x097, 0x097, 0x096, 0x096, 0x096, 0x095, 0x095, 0x095, 0x095, 0x094,
	0x094, 0x094, 0x094, 0x093, 0x093, 0x093, 0x093, 0x092, 0x092, 0x092, 0x091, 0x091,
	0x091, 0x091, 0x090, 0x090, 0x090, 0x090, 0x08f, 0x08f, 0x08f, 0x08f, 0x08e, 0x08e,
	0x08e, 0x08e, 0x08d, 0x08d, 0x08d, 0x08d, 0x08c, 0x08c, 0x08c, 0x08c, 0x08b, 0x08b,
	0x08b, 0x08b, 0x08a, 0x08a, 0x08a, 0x08a, 0x089, 0x089, 0x089, 0x089, 0x088, 0x088,
	0x088, 0x088, 0x087, 0x087, 0x087, 0x087, 0x086, 0x086, 0x086, 0x086, 0x085, 0x085,
	0x085, 0x085, 0x084, 0x084, 0x084, 0x084, 0x083, 0x083, 0x083, 0x083, 0x083, 0x082,
	0x082, 0x082, 0x082, 0x081, 0x081, 0x081, 0x081, 0x080, 0x080, 0x080, 0x080, 0x07f,
	0x07f, 0x07f, 0x07f, 0x07f, 0x07e, 0x07e, 0x07e, 0x07e, 0x07d, 0x07d, 0x07d, 0x07d,
	0x07d, 0x07c, 0x07c, 0x07c, 0x07c, 0x07b, 0x07b, 0x07b, 0x07b, 0x07b, 0x07a, 0x07a,
	0x07a, 0x07a, 0x079, 0x079, 0x079, 0x079, 0x079, 0x078, 0x078, 0x078, 0x078, 0x077,
	0x077, 0x077, 0x077, 0x077, 0x076, 0x076, 0x076, 0x076, 0x076, 0x075, 0x075, 0x075,
	0x075, 0x074, 0x074, 0x074, 0x074, 0x074, 0x073, 0x073, 0x073, 0x073, 0x073, 0x072,
	0x072, 0x072, 0x072, 0x072, 0x071, 0x071, 0x071, 0x071, 0x071, 0x070, 0x070, 0x070,
	0x070, 0x070, 0x06f, 0x06f, 0x06f, 0x06f, 0x06f, 0x06e, 0x06e, 0x06e, 0x06e, 0x06e,
	0x06d, 0x06d, 0x06d, 0x06d, 0x06d, 0x06c, 0x06c, 0x06c, 0x06c, 0x06c, 0x06b, 0x06b,
	0x06b, 0x06b, 0x06b, 0x06a, 0x06a, 0x06a, 0x06a, 0x06a, 0x069, 0x069, 0x069, 0x069,
	0x069, 0x069, 0x068, 0x068, 0x068, 0x068, 0x068, 0x067, 0x067, 0x067, 0x067, 0x067,
	0x066, 0x066, 0x066, 0x066, 0x066, 0x066, 0x065, 0x065, 0x065, 0x065, 0x065, 0x064,
	0x064, 0x064, 0x064, 0x064, 0x064, 0x063, 0x063, 0x063, 0x063, 0x063, 0x062, 0x062,
	0x062, 0x062, 0x062, 0x062, 0x061, 0x061, 0x061, 0x061, 0x061, 0x061, 0x060, 0x060,
	0x060, 0x060, 0x060, 0x060, 0x05f, 0x05f, 0x05f, 0x05f, 0x05f, 0x05e, 0x05e, 0x05e,
	0x05e, 0x05e, 0x05e, 0x05d, 0x05d, 0x05d, 0x05d, 0x05d, 0x05d, 0x05c, 0x05c, 0x05c,
	0x05c, 0x05c, 0x05c, 0x05b, 0x05b, 0x05b, 0x05b, 0x05b, 0x05b, 0x05a, 0x05a, 0x05a,
	0x05a, 0x05a, 0x05a, 0x05a, 0x059, 0x059, 0x059, 0x059, 0x059, 0x059, 0x058, 0x058,
	0x058, 0x058, 0x058, 0x058, 0x057, 0x057, 0x057, 0x057, 0x057, 0x057, 0x056, 0x056,
	0x056, 0x056, 0x056, 0x056, 0x056, 0x055, 0x055, 0x055, 0x055, 0x055, 0x055, 0x054,
	0x054, 0x054, 0x054, 0x054, 0x054, 0x054, 0x053, 0x053, 0x053, 0x053, 0x053, 0x053,
	0x053, 0x052, 0x052, 0x052, 0x052, 0x052, 0x052, 0x051, 0x051, 0x051, 0x051, 0x051,
	0x051, 0x051, 0x050, 0x050, 0x050, 0x050, 0x050, 0x050, 0x050, 0x04f, 0x04f, 0x04f,
	0x04f, 0x04f, 0x04f, 0x04f, 0x04e, 0x04e, 0x04e, 0x04e, 0x04e, 0x04e, 0x04e, 0x04d,
	0x04d, 0x04d, 0x04d, 0x04d, 0x04d, 0x04d, 0x04c, 0x04c, 0x04c, 0x04c, 0x04c, 0x04c,
	0x04c, 0x04c, 0x04b, 0x04b, 0x04b, 0x04b, 0x04b, 0x04b, 0x04b, 0x04a, 0x04a, 0x04a,
	0x04a, 0x04a, 0x04a, 0x04a, 0x04a, 0x049, 0x049, 0x049, 0x049, 0x049, 0x049, 0x049,
	0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x047, 0x047, 0x047, 0x047,
	0x047, 0x047, 0x047, 0x047, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046,
	0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x044, 0x044, 0x044, 0x044,
	0x044, 0x044, 0x044, 0x044, 0x043, 0x043, 0x043, 0x043, 0x043, 0x043, 0x043, 0x043,
	0x042, 0x042, 0x042, 0x042, 0x042, 0x042, 0x042, 0x042, 0x041, 0x041, 0x041, 0x041,
	0x041, 0x041, 0x041, 0x041, 0x041, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040,
	0x040, 0x040, 0x03f, 0x03f, 0x03f, 0x03f, 0x03f, 0x03f, 0x03f, 0x03f, 0x03e, 0x03e,
	0x03e, 0x03e, 0x03e, 0x03e, 0x03e, 0x03e, 0x03e, 0x03d, 0x03d, 0x03d, 0x03d, 0x03d,
	0x03d, 0x03d, 0x03d, 0x03d, 0x03c, 0x03c, 0x03c, 0x03c, 0x03c, 0x03c, 0x03c, 0x03c,
	0x03c, 0x03c, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03a,
	0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x039, 0x039, 0x039,
	0x039, 0x039, 0x039, 0x039, 0x039, 0x039, 0x038, 0x038, 0x038, 0x038, 0x038, 0x038,
	0x038, 0x038, 0x038, 0x038, 0x037, 0x037, 0x037, 0x037, 0x037, 0x037, 0x037, 0x037,
	0x037, 0x037, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036,
	0x036, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x034,
	0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x033, 0x033,
	0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x032, 0x032, 0x032,
	0x032, 0x032, 0x032, 0x032, 0x032, 0x032, 0x032, 0x032, 0x031, 0x031, 0x031, 0x031,
	0x031, 0x031, 0x031, 0x031, 0x031, 0x031, 0x031, 0x030, 0x030, 0x030, 0x030, 0x030,
	0x030, 0x030, 0x030, 0x030, 0x030, 0x030, 0x030, 0x02f, 0x02f, 0x02f, 0x02f, 0x02f,
	0x02f, 0x02f, 0x02f, 0x02f, 0x02f, 0x02f, 0x02e, 0x02e, 0x02e, 0x02e, 0x02e, 0x02e,
	0x02e, 0x02e, 0x02e, 0x02e, 0x02e, 0x02e, 0x02d, 0x02d, 0x02d, 0x02d, 0x02d, 0x02d,
	0x02d, 0x02d, 0x02d, 0x02d, 0x02d, 0x02d, 0x02d, 0x02c, 0x02c, 0x02c, 0x02c, 0x02c,
	0x02c, 0x02c, 0x02c, 0x02c, 0x02c, 0x02c, 0x02c, 0x02b, 0x02b, 0x02b, 0x02b, 0x02b,
	0x02b, 0x02b, 0x02b, 0x02b, 0x02b, 0x02b, 0x02b, 0x02b, 0x02a, 0x02a, 0x02a, 0x02a,
	0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x029, 0x029, 0x029,
	0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x028,
	0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028,
	0x028, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027,
	0x027, 0x027, 0x027, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026,
	0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025,
	0x025, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025, 0x024, 0x024, 0x024,
	0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024,
	0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023,
	0x023, 0x023, 0x023, 0x023, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022,
	0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x021, 0x021, 0x021, 0x021,
	0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021,
	0x021, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020,
	0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f,
	0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f,
	0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e,
	0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d,
	0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d,
	0x01d, 0x01d, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c,
	0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01b, 0x01b, 0x01b,
	0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b,
	0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a,
	0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a,
	0x01a, 0x01a, 0x01a, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019,
	0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019,
	0x019, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018,
	0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018,
	0x018, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017,
	0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017,
	0x017, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016,
	0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016,
	0x016, 0x016, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015,
	0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015,
	0x015, 0x015, 0x015, 0x015, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014,
	0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014,
	0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x013, 0x013, 0x013, 0x013,
	0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013,
	0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013,
	0x013, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012,
	0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012,
	0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x011, 0x011, 0x011, 0x011,
	0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011,
	0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011,
	0x011, 0x011, 0x011, 0x011, 0x011, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010,
	0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010,
	0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010,
	0x010, 0x010, 0x010, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f,
	0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f
};

const uint16_t PitchConverter::centTableSSGTriangle_[3072] = {
	0x077, 0x077, 0x077, 0x077, 0x076, 0x076, 0x076, 0x076, 0x076, 0x075, 0x075, 0x075,
	0x075, 0x074, 0x074, 0x074, 0x074, 0x074, 0x073, 0x073, 0x073, 0x073, 0x073, 0x072,
	0x072, 0x072, 0x072, 0x072, 0x071, 0x071, 0x071, 0x071, 0x071, 0x070, 0x070, 0x070,
	0x070, 0x070, 0x06f, 0x06f, 0x06f, 0x06f, 0x06f, 0x06e, 0x06e, 0x06e, 0x06e, 0x06e,
	0x06d, 0x06d, 0x06d, 0x06d, 0x06d, 0x06c, 0x06c, 0x06c, 0x06c, 0x06c, 0x06b, 0x06b,
	0x06b, 0x06b, 0x06b, 0x06a, 0x06a, 0x06a, 0x06a, 0x06a, 0x069, 0x069, 0x069, 0x069,
	0x069, 0x069, 0x068, 0x068, 0x068, 0x068, 0x068, 0x067, 0x067, 0x067, 0x067, 0x067,
	0x066, 0x066, 0x066, 0x066, 0x066, 0x066, 0x065, 0x065, 0x065, 0x065, 0x065, 0x064,
	0x064, 0x064, 0x064, 0x064, 0x064, 0x063, 0x063, 0x063, 0x063, 0x063, 0x062, 0x062,
	0x062, 0x062, 0x062, 0x062, 0x061, 0x061, 0x061, 0x061, 0x061, 0x061, 0x060, 0x060,
	0x060, 0x060, 0x060, 0x060, 0x05f, 0x05f, 0x05f, 0x05f, 0x05f, 0x05e, 0x05e, 0x05e,
	0x05e, 0x05e, 0x05e, 0x05d, 0x05d, 0x05d, 0x05d, 0x05d, 0x05d, 0x05c, 0x05c, 0x05c,
	0x05c, 0x05c, 0x05c, 0x05b, 0x05b, 0x05b, 0x05b, 0x05b, 0x05b, 0x05a, 0x05a, 0x05a,
	0x05a, 0x05a, 0x05a, 0x05a, 0x059, 0x059, 0x059, 0x059, 0x059, 0x059, 0x058, 0x058,
	0x058, 0x058, 0x058, 0x058, 0x057, 0x057, 0x057, 0x057, 0x057, 0x057, 0x056, 0x056,
	0x056, 0x056, 0x056, 0x056, 0x056, 0x055, 0x055, 0x055, 0x055, 0x055, 0x055, 0x054,
	0x054, 0x054, 0x054, 0x054, 0x054, 0x054, 0x053, 0x053, 0x053, 0x053, 0x053, 0x053,
	0x053, 0x052, 0x052, 0x052, 0x052, 0x052, 0x052, 0x051, 0x051, 0x051, 0x051, 0x051,
	0x051, 0x051, 0x050, 0x050, 0x050, 0x050, 0x050, 0x050, 0x050, 0x04f, 0x04f, 0x04f,
	0x04f, 0x04f, 0x04f, 0x04f, 0x04e, 0x04e, 0x04e, 0x04e, 0x04e, 0x04e, 0x04e, 0x04d,
	0x04d, 0x04d, 0x04d, 0x04d, 0x04d, 0x04d, 0x04c, 0x04c, 0x04c, 0x04c, 0x04c, 0x04c,
	0x04c, 0x04c, 0x04b, 0x04b, 0x04b, 0x04b, 0x04b, 0x04b, 0x04b, 0x04a, 0x04a, 0x04a,
	0x04a, 0x04a, 0x04a, 0x04a, 0x04a, 0x049, 0x049, 0x049, 0x049, 0x049, 0x049, 0x049,
	0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x047, 0x047, 0x047, 0x047,
	0x047, 0x047, 0x047, 0x047, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046,
	0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x044, 0x044, 0x044, 0x044,
	0x044, 0x044, 0x044, 0x044, 0x043, 0x043, 0x043, 0x043, 0x043, 0x043, 0x043, 0x043,
	0x042, 0x042, 0x042, 0x042, 0x042, 0x042, 0x042, 0x042, 0x041, 0x041, 0x041, 0x041,
	0x041, 0x041, 0x041, 0x041, 0x041, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040,
	0x040, 0x040, 0x03f, 0x03f, 0x03f, 0x03f, 0x03f, 0x03f, 0x03f, 0x03f, 0x03e, 0x03e,
	0x03e, 0x03e, 0x03e, 0x03e, 0x03e, 0x03e, 0x03e, 0x03d, 0x03d, 0x03d, 0x03d, 0x03d,
	0x03d, 0x03d, 0x03d, 0x03d, 0x03c, 0x03c, 0x03c, 0x03c, 0x03c, 0x03c, 0x03c, 0x03c,
	0x03c, 0x03c, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03a,
	0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x039, 0x039, 0x039,
	0x039, 0x039, 0x039, 0x039, 0x039, 0x039, 0x038, 0x038, 0x038, 0x038, 0x038, 0x038,
	0x038, 0x038, 0x038, 0x038, 0x037, 0x037, 0x037, 0x037, 0x037, 0x037, 0x037, 0x037,
	0x037, 0x037, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036,
	0x036, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x034,
	0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x033, 0x033,
	0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x032, 0x032, 0x032,
	0x032, 0x032, 0x032, 0x032, 0x032, 0x032, 0x032, 0x032, 0x031, 0x031, 0x031, 0x031,
	0x031, 0x031, 0x031, 0x031, 0x031, 0x031, 0x031, 0x030, 0x030, 0x030, 0x030, 0x030,
	0x030, 0x030, 0x030, 0x030, 0x030, 0x030, 0x030, 0x02f, 0x02f, 0x02f, 0x02f, 0x02f,
	0x02f, 0x02f, 0x02f, 0x02f, 0x02f, 0x02f, 0x02e, 0x02e, 0x02e, 0x02e, 0x02e, 0x02e,
	0x02e, 0x02e, 0x02e, 0x02e, 0x02e, 0x02e, 0x02d, 0x02d, 0x02d, 0x02d, 0x02d, 0x02d,
	0x02d, 0x02d, 0x02d, 0x02d, 0x02d, 0x02d, 0x02d, 0x02c, 0x02c, 0x02c, 0x02c, 0x02c,
	0x02c, 0x02c, 0x02c, 0x02c, 0x02c, 0x02c, 0x02c, 0x02b, 0x02b, 0x02b, 0x02b, 0x02b,
	0x02b, 0x02b, 0x02b, 0x02b, 0x02b, 0x02b, 0x02b, 0x02b, 0x02a, 0x02a, 0x02a, 0x02a,
	0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x029, 0x029, 0x029,
	0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x028,
	0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028,
	0x028, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027,
	0x027, 0x027, 0x027, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026,
	0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025,
	0x025, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025, 0x024, 0x024, 0x024,
	0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024,
	0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023,
	0x023, 0x023, 0x023, 0x023, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022,
	0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x021, 0x021, 0x021, 0x021,
	0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021,
	0x021, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020,
	0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f,
	0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f,
	0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e,
	0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d,
	0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d,
	0x01d, 0x01d, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c,
	0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01b, 0x01b, 0x01b,
	0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b,
	0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a,
	0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a,
	0x01a, 0x01a, 0x01a, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019,
	0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019,
	0x019, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018,
	0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018,
	0x018, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017,
	0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017,
	0x017, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016,
	0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016,
	0x016, 0x016, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015,
	0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015,
	0x015, 0x015, 0x015, 0x015, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014,
	0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014,
	0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x013, 0x013, 0x013, 0x013,
	0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013,
	0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013,
	0x013, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012,
	0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012,
	0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x011, 0x011, 0x011, 0x011,
	0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011,
	0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011,
	0x011, 0x011, 0x011, 0x011, 0x011, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010,
	0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010,
	0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010,
	0x010, 0x010, 0x010, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f,
	0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f,
	0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f,
	0x00f, 0x00f, 0x00f, 0x00f, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e,
	0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e,
	0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e,
	0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00d, 0x00d, 0x00d, 0x00d,
	0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d,
	0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d,
	0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d,
	0x00d, 0x00d, 0x00d, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c,
	0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c,
	0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c,
	0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c,
	0x00c, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b,
	0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b,
	0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b,
	0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b,
	0x00b, 0x00b, 0x00b, 0x00b, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a,
	0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a,
	0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a,
	0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a,
	0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a,
	0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009,
	0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009,
	0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009,
	0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009,
	0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009,
	0x009, 0x009, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008,
	0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008,
	0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008,
	0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008,
	0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008,
	0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001
};

const uint16_t PitchConverter::centTableSSGSaw_[3072] = {
	0x0ef, 0x0ee, 0x0ee, 0x0ed, 0x0ed, 0x0ec, 0x0ec, 0x0ec, 0x0eb, 0x0eb, 0x0ea, 0x0ea,
	0x0e9, 0x0e9, 0x0e9, 0x0e8, 0x0e8, 0x0e7, 0x0e7, 0x0e6, 0x0e6, 0x0e6, 0x0e5, 0x0e5,
	0x0e4, 0x0e4, 0x0e4, 0x0e3, 0x0e3, 0x0e2, 0x0e2, 0x0e2, 0x0e1, 0x0e1, 0x0e0, 0x0e0,
	0x0e0, 0x0df, 0x0df, 0x0de, 0x0de, 0x0dd, 0x0dd, 0x0dd, 0x0dc, 0x0dc, 0x0dc, 0x0db,
	0x0db, 0x0da, 0x0da, 0x0da, 0x0d9, 0x0d9, 0x0d8, 0x0d8, 0x0d8, 0x0d7, 0x0d7, 0x0d6,
	0x0d6, 0x0d6, 0x0d5, 0x0d5, 0x0d4, 0x0d4, 0x0d4, 0x0d3, 0x0d3, 0x0d3, 0x0d2, 0x0d2,
	0x0d1, 0x0d1, 0x0d1, 0x0d0, 0x0d0, 0x0d0, 0x0cf, 0x0cf, 0x0ce, 0x0ce, 0x0ce, 0x0cd,
	0x0cd, 0x0cd, 0x0cc, 0x0cc, 0x0cb, 0x0cb, 0x0cb, 0x0ca, 0x0ca, 0x0ca, 0x0c9, 0x0c9,
	0x0c9, 0x0c8, 0x0c8, 0x0c7, 0x0c7, 0x0c7, 0x0c6, 0x0c6, 0x0c6, 0x0c5, 0x0c5, 0x0c5,
	0x0c4, 0x0c4, 0x0c4, 0x0c3, 0x0c3, 0x0c3, 0x0c2, 0x0c2, 0x0c1, 0x0c1, 0x0c1, 0x0c0,
	0x0c0, 0x0c0, 0x0bf, 0x0bf, 0x0bf, 0x0be, 0x0be, 0x0be, 0x0bd, 0x0bd, 0x0bd, 0x0bc,
	0x0bc, 0x0bc, 0x0bb, 0x0bb, 0x0bb, 0x0ba, 0x0ba, 0x0ba, 0x0b9, 0x0b9, 0x0b9, 0x0b8,
	0x0b8, 0x0b8, 0x0b7, 0x0b7, 0x0b7, 0x0b6, 0x0b6, 0x0b6, 0x0b5, 0x0b5, 0x0b5, 0x0b4,
	0x0b4, 0x0b4, 0x0b3, 0x0b3, 0x0b3, 0x0b2, 0x0b2, 0x0b2, 0x0b1, 0x0b1, 0x0b1, 0x0b0,
	0x0b0, 0x0b0, 0x0af, 0x0af, 0x0af, 0x0af, 0x0ae, 0x0ae, 0x0ae, 0x0ad, 0x0ad, 0x0ad,
	0x0ac, 0x0ac, 0x0ac, 0x0ab, 0x0ab, 0x0ab, 0x0aa, 0x0aa, 0x0aa, 0x0aa, 0x0a9, 0x0a9,
	0x0a9, 0x0a8, 0x0a8, 0x0a8, 0x0a7, 0x0a7, 0x0a7, 0x0a7, 0x0a6, 0x0a6, 0x0a6, 0x0a5,
	0x0a5, 0x0a5, 0x0a4, 0x0a4, 0x0a4, 0x0a4, 0x0a3, 0x0a3, 0x0a3, 0x0a2, 0x0a2, 0x0a2,
	0x0a2, 0x0a1, 0x0a1, 0x0a1, 0x0a0, 0x0a0, 0x0a0, 0x09f, 0x09f, 0x09f, 0x09f, 0x09e,
	0x09e, 0x09e, 0x09d, 0x09d, 0x09d, 0x09d, 0x09c, 0x09c, 0x09c, 0x09b, 0x09b, 0x09b,
	0x09b, 0x09a, 0x09a, 0x09a, 0x09a, 0x099, 0x099, 0x099, 0x098, 0x098, 0x098, 0x098,
	0x097, 0x097, 0x097, 0x097, 0x096, 0x096, 0x096, 0x095, 0x095, 0x095, 0x095, 0x094,
	0x094, 0x094, 0x094, 0x093, 0x093, 0x093, 0x093, 0x092, 0x092, 0x092, 0x091, 0x091,
	0x091, 0x091, 0x090, 0x090, 0x090, 0x090, 0x08f, 0x08f, 0x08f, 0x08f, 0x08e, 0x08e,
	0x08e, 0x08e, 0x08d, 0x08d, 0x08d, 0x08d, 0x08c, 0x08c, 0x08c, 0x08c, 0x08b, 0x08b,
	0x08b, 0x08b, 0x08a, 0x08a, 0x08a, 0x08a, 0x089, 0x089, 0x089, 0x089, 0x088, 0x088,
	0x088, 0x088, 0x087, 0x087, 0x087, 0x087, 0x086, 0x086, 0x086, 0x086, 0x085, 0x085,
	0x085, 0x085, 0x084, 0x084, 0x084, 0x084, 0x083, 0x083, 0x083, 0x083, 0x083, 0x082,
	0x082, 0x082, 0x082, 0x081, 0x081, 0x081, 0x081, 0x080, 0x080, 0x080, 0x080, 0x07f,
	0x07f, 0x07f, 0x07f, 0x07f, 0x07e, 0x07e, 0x07e, 0x07e, 0x07d, 0x07d, 0x07d, 0x07d,
	0x07d, 0x07c, 0x07c, 0x07c, 0x07c, 0x07b, 0x07b, 0x07b, 0x07b, 0x07b, 0x07a, 0x07a,
	0x07a, 0x07a, 0x079, 0x079, 0x079, 0x079, 0x079, 0x078, 0x078, 0x078, 0x078, 0x077,
	0x077, 0x077, 0x077, 0x077, 0x076, 0x076, 0x076, 0x076, 0x076, 0x075, 0x075, 0x075,
	0x075, 0x074, 0x074, 0x074, 0x074, 0x074, 0x073, 0x073, 0x073, 0x073, 0x073, 0x072,
	0x072, 0x072, 0x072, 0x072, 0x071, 0x071, 0x071, 0x071, 0x071, 0x070, 0x070, 0x070,
	0x070, 0x070, 0x06f, 0x06f, 0x06f, 0x06f, 0x06f, 0x06e, 0x06e, 0x06e, 0x06e, 0x06e,
	0x06d, 0x06d, 0x06d, 0x06d, 0x06d, 0x06c, 0x06c, 0x06c, 0x06c, 0x06c, 0x06b, 0x06b,
	0x06b, 0x06b, 0x06b, 0x06a, 0x06a, 0x06a, 0x06a, 0x06a, 0x069, 0x069, 0x069, 0x069,
	0x069, 0x069, 0x068, 0x068, 0x068, 0x068, 0x068, 0x067, 0x067, 0x067, 0x067, 0x067,
	0x066, 0x066, 0x066, 0x066, 0x066, 0x066, 0x065, 0x065, 0x065, 0x065, 0x065, 0x064,
	0x064, 0x064, 0x064, 0x064, 0x064, 0x063, 0x063, 0x063, 0x063, 0x063, 0x062, 0x062,
	0x062, 0x062, 0x062, 0x062, 0x061, 0x061, 0x061, 0x061, 0x061, 0x061, 0x060, 0x060,
	0x060, 0x060, 0x060, 0x060, 0x05f, 0x05f, 0x05f, 0x05f, 0x05f, 0x05e, 0x05e, 0x05e,
	0x05e, 0x05e, 0x05e, 0x05d, 0x05d, 0x05d, 0x05d, 0x05d, 0x05d, 0x05c, 0x05c, 0x05c,
	0x05c, 0x05c, 0x05c, 0x05b, 0x05b, 0x05b, 0x05b, 0x05b, 0x05b, 0x05a, 0x05a, 0x05a,
	0x05a, 0x05a, 0x05a, 0x05a, 0x059, 0x059, 0x059, 0x059, 0x059, 0x059, 0x058, 0x058,
	0x058, 0x058, 0x058, 0x058, 0x057, 0x057, 0x057, 0x057, 0x057, 0x057, 0x056, 0x056,
	0x056, 0x056, 0x056, 0x056, 0x056, 0x055, 0x055, 0x055, 0x055, 0x055, 0x055, 0x054,
	0x054, 0x054, 0x054, 0x054, 0x054, 0x054, 0x053, 0x053, 0x053, 0x053, 0x053, 0x053,
	0x053, 0x052, 0x052, 0x052, 0x052, 0x052, 0x052, 0x051, 0x051, 0x051, 0x051, 0x051,
	0x051, 0x051, 0x050, 0x050, 0x050, 0x050, 0x050, 0x050, 0x050, 0x04f, 0x04f, 0x04f,
	0x04f, 0x04f, 0x04f, 0x04f, 0x04e, 0x04e, 0x04e, 0x04e, 0x04e, 0x04e, 0x04e, 0x04d,
	0x04d, 0x04d, 0x04d, 0x04d, 0x04d, 0x04d, 0x04c, 0x04c, 0x04c, 0x04c, 0x04c, 0x04c,
	0x04c, 0x04c, 0x04b, 0x04b, 0x04b, 0x04b, 0x04b, 0x04b, 0x04b, 0x04a, 0x04a, 0x04a,
	0x04a, 0x04a, 0x04a, 0x04a, 0x04a, 0x049, 0x049, 0x049, 0x049, 0x049, 0x049, 0x049,
	0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x048, 0x047, 0x047, 0x047, 0x047,
	0x047, 0x047, 0x047, 0x047, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046, 0x046,
	0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x045, 0x044, 0x044, 0x044, 0x044,
	0x044, 0x044, 0x044, 0x044, 0x043, 0x043, 0x043, 0x043, 0x043, 0x043, 0x043, 0x043,
	0x042, 0x042, 0x042, 0x042, 0x042, 0x042, 0x042, 0x042, 0x041, 0x041, 0x041, 0x041,
	0x041, 0x041, 0x041, 0x041, 0x041, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040, 0x040,
	0x040, 0x040, 0x03f, 0x03f, 0x03f, 0x03f, 0x03f, 0x03f, 0x03f, 0x03f, 0x03e, 0x03e,
	0x03e, 0x03e, 0x03e, 0x03e, 0x03e, 0x03e, 0x03e, 0x03d, 0x03d, 0x03d, 0x03d, 0x03d,
	0x03d, 0x03d, 0x03d, 0x03d, 0x03c, 0x03c, 0x03c, 0x03c, 0x03c, 0x03c, 0x03c, 0x03c,
	0x03c, 0x03c, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03b, 0x03a,
	0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x03a, 0x039, 0x039, 0x039,
	0x039, 0x039, 0x039, 0x039, 0x039, 0x039, 0x038, 0x038, 0x038, 0x038, 0x038, 0x038,
	0x038, 0x038, 0x038, 0x038, 0x037, 0x037, 0x037, 0x037, 0x037, 0x037, 0x037, 0x037,
	0x037, 0x037, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036, 0x036,
	0x036, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x035, 0x034,
	0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x034, 0x033, 0x033,
	0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x033, 0x032, 0x032, 0x032,
	0x032, 0x032, 0x032, 0x032, 0x032, 0x032, 0x032, 0x032, 0x031, 0x031, 0x031, 0x031,
	0x031, 0x031, 0x031, 0x031, 0x031, 0x031, 0x031, 0x030, 0x030, 0x030, 0x030, 0x030,
	0x030, 0x030, 0x030, 0x030, 0x030, 0x030, 0x030, 0x02f, 0x02f, 0x02f, 0x02f, 0x02f,
	0x02f, 0x02f, 0x02f, 0x02f, 0x02f, 0x02f, 0x02e, 0x02e, 0x02e, 0x02e, 0x02e, 0x02e,
	0x02e, 0x02e, 0x02e, 0x02e, 0x02e, 0x02e, 0x02d, 0x02d, 0x02d, 0x02d, 0x02d, 0x02d,
	0x02d, 0x02d, 0x02d, 0x02d, 0x02d, 0x02d, 0x02d, 0x02c, 0x02c, 0x02c, 0x02c, 0x02c,
	0x02c, 0x02c, 0x02c, 0x02c, 0x02c, 0x02c, 0x02c, 0x02b, 0x02b, 0x02b, 0x02b, 0x02b,
	0x02b, 0x02b, 0x02b, 0x02b, 0x02b, 0x02b, 0x02b, 0x02b, 0x02a, 0x02a, 0x02a, 0x02a,
	0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x02a, 0x029, 0x029, 0x029,
	0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x029, 0x028,
	0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028, 0x028,
	0x028, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027, 0x027,
	0x027, 0x027, 0x027, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x026,
	0x026, 0x026, 0x026, 0x026, 0x026, 0x026, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025,
	0x025, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025, 0x025, 0x024, 0x024, 0x024,
	0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024, 0x024,
	0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023, 0x023,
	0x023, 0x023, 0x023, 0x023, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022,
	0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x022, 0x021, 0x021, 0x021, 0x021,
	0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021, 0x021,
	0x021, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x020,
	0x020, 0x020, 0x020, 0x020, 0x020, 0x020, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f,
	0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f, 0x01f,
	0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e,
	0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01e, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d,
	0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d, 0x01d,
	0x01d, 0x01d, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c,
	0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01c, 0x01b, 0x01b, 0x01b,
	0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b,
	0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01b, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a,
	0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a, 0x01a,
	0x01a, 0x01a, 0x01a, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019,
	0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019, 0x019,
	0x019, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018,
	0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018, 0x018,
	0x018, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017,
	0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017, 0x017,
	0x017, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016,
	0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016, 0x016,
	0x016, 0x016, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015,
	0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015, 0x015,
	0x015, 0x015, 0x015, 0x015, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014,
	0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014,
	0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x014, 0x013, 0x013, 0x013, 0x013,
	0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013,
	0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013, 0x013,
	0x013, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012,
	0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012,
	0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x012, 0x011, 0x011, 0x011, 0x011,
	0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011,
	0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011, 0x011,
	0x011, 0x011, 0x011, 0x011, 0x011, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010,
	0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010,
	0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010, 0x010,
	0x010, 0x010, 0x010, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f,
	0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f,
	0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f, 0x00f,
	0x00f, 0x00f, 0x00f, 0x00f, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e,
	0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e,
	0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e,
	0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00e, 0x00d, 0x00d, 0x00d, 0x00d,
	0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d,
	0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d,
	0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d, 0x00d,
	0x00d, 0x00d, 0x00d, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c,
	0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c,
	0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c,
	0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c, 0x00c,
	0x00c, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b,
	0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b,
	0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b,
	0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b, 0x00b,
	0x00b, 0x00b, 0x00b, 0x00b, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a,
	0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a,
	0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a,
	0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a,
	0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a, 0x00a,
	0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009,
	0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009,
	0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009,
	0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009,
	0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009, 0x009,
	0x009, 0x009, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008,
	0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008,
	0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008,
	0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008,
	0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008,
	0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008, 0x008,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007,
	0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x007, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006, 0x006,
	0x006, 0x006, 0x006, 0x006, 0x006, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005,
	0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x005, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004,
	0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x004, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003, 0x003,
	0x003, 0x003, 0x003, 0x003, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x002,
	0x002, 0x002, 0x002, 0x002, 0x002, 0x002, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001,
	0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001, 0x001
};

PitchConverter::PitchConverter()
{
}
