/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastient.laout@tuxfamily.org                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "additemdialog.h"
#include "basket.h"
#include "item.h"
#include "itemfactory.h"

#include <knuminput.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qcheckbox.h>
#include <qlayout.h>
#include <qimage.h>
#include <qpixmap.h>
#include <klocale.h>
#include <kurl.h>
#include <kopenwith.h>

#include <kicondialog.h>
#include <kiconloader.h>
//#include <qiconset.h>

/** ViewSizeDialog */

ViewSizeDialog::ViewSizeDialog(QWidget *parent, int w, int h)
 : QDialog(parent)
{
	setGeometry(x(), y(), w, h);
}

ViewSizeDialog::~ViewSizeDialog()
{
}

void ViewSizeDialog::resizeEvent(QResizeEvent *)
{
	setCaption( i18n("%1 by %2 pixels").arg(QString::number(width())).arg(QString::number(height())) );
}

/** AddItemWidget */

AddItemWidget::AddItemWidget(QWidget *parent, const char *name, WFlags f)
 : QWidget(parent, name, f)
{
	QLabel *label;
	QVBoxLayout *layout = new QVBoxLayout(this, 0, 5);
	QHBoxLayout *hLayout;
	QVBoxLayout *vLayout;
	QHBoxLayout *hLayoutRB;
	QSpacerItem *spacer;

	label = new QLabel(i18n("<b>Create an empty item of primitive type... :</b>"), this);
	layout->addWidget(label);

	hLayout = new QHBoxLayout(0, 0, 5);
	hLayout->addSpacing(20);

	vLayout = new QVBoxLayout(0, 0, 5);
	m_options.append( new QRadioButton(i18n("&Text"), this) );
	vLayout->addWidget( m_options.getLast() );
	m_options.append( new QRadioButton(i18n("&Rich text (HTML)"), this) );
	vLayout->addWidget( m_options.getLast() );
	m_options.append( new QRadioButton(i18n("L&ink (to a local file or a network URL)"), this) );
	vLayout->addWidget( m_options.getLast() );

	hLayoutRB = new QHBoxLayout(0, 0, 5);
	m_options.append( new QRadioButton(i18n("I&mage of"), this) );
	hLayoutRB->addWidget( m_options.getLast() );
	m_imgSizeX = new KIntNumInput(this);
	m_imgSizeX->setValue(200);
	m_imgSizeX->setMinValue(1);
	m_imgSizeX->setMaxValue(4096);
	m_imgSizeX->setReferencePoint(100);
	hLayoutRB->addWidget(m_imgSizeX);
	label = new QLabel(i18n("by"), this);
	hLayoutRB->addWidget(label);
	m_imgSizeY = new KIntNumInput(this);
	m_imgSizeY->setValue(100);
	m_imgSizeY->setMinValue(1);
	m_imgSizeY->setMaxValue(4096);
	m_imgSizeY->setReferencePoint(100);
	hLayoutRB->addWidget(m_imgSizeY);
	label = new QLabel(i18n("pixels"), this);
	hLayoutRB->addWidget(label);
	m_pushVisualize = new QPushButton(i18n("&Visualize..."), this);
	hLayoutRB->addWidget(m_pushVisualize);
	spacer = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum);
	hLayoutRB->addItem(spacer);
	vLayout->addLayout(hLayoutRB);

	m_options.append( new QRadioButton(i18n("Co&lor"), this) );
	vLayout->addWidget( m_options.getLast() );
	hLayout->addLayout(vLayout);
	layout->addLayout(hLayout);

	label = new QLabel(i18n("<b>... Or fill a new item with :</b>"), this);
	layout->addWidget(label);
	hLayout = new QHBoxLayout(0, 0, 5);
	hLayout->addSpacing(20);
	vLayout = new QVBoxLayout(0, 0, 5);

	hLayoutRB = new QHBoxLayout(0, 0, 5);
	m_options.append( new QRadioButton(i18n("&Application launcher"), this) );
	hLayoutRB->addWidget( m_options.getLast() );
	m_nameInAnnots = new QCheckBox(i18n("&Store app name in annotations"), this);
	hLayoutRB->addWidget(m_nameInAnnots);
	spacer = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum);
	hLayoutRB->addItem(spacer);
	vLayout->addLayout(hLayoutRB);

	hLayoutRB = new QHBoxLayout(0, 0, 5);
	m_options.append( new QRadioButton(i18n("Im&port an icon of"), this) );
	hLayoutRB->addWidget( m_options.getLast() );
	m_iconSize = new IconSizeCombo(false, this);
	hLayoutRB->addWidget(m_iconSize);
	QLabel *label2 = new QLabel(i18n("pixels"), this);
	hLayoutRB->addWidget(label2);
	spacer = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Minimum);
	hLayoutRB->addItem(spacer);
	vLayout->addLayout(hLayoutRB);

	m_options.append( new QRadioButton(i18n("Loa&d from a file"), this) );
	vLayout->addWidget( m_options.getLast() );

	m_options.append( new QRadioButton(i18n("Mirror a &file"), this) );
	vLayout->addWidget( m_options.getLast() );

	hLayout->addLayout(vLayout);
	layout->addLayout(hLayout);

	label = new QLabel(i18n("<i>Tip : You also can drag and drop objects to baskets (or system tray icon), "
		"paste them or paste selection (with middle mouse click).</i>"), this);
	layout->addWidget(label);
	//spacer = new QSpacerItem(0, 0, QSizePolicy::Minimum, QSizePolicy::Expanding);
	//layout->addItem(spacer);

	m_options.first()->toggle();
	for (QRadioButton *option = m_options.first(); option; option = m_options.next())
		connect( option, SIGNAL(toggled(bool)), this, SLOT(toggled(bool)) );

	connect( m_imgSizeX,      SIGNAL(valueChanged(int)), this, SLOT(activateImageOption(int))     );
	connect( m_imgSizeY,      SIGNAL(valueChanged(int)), this, SLOT(activateImageOption(int))     );
	connect( m_pushVisualize, SIGNAL(clicked()),         this, SLOT(activateImageOption())        );
	connect( m_pushVisualize, SIGNAL(clicked()),         this, SLOT(visualize())                  );
	connect( m_nameInAnnots,  SIGNAL(toggled(bool)),     this, SLOT(activateLauncherOption(bool)) );
	connect( m_iconSize,      SIGNAL(activated(int)),    this, SLOT(activateImportOption(int))    );

	//setMinimumSize( QSize(geometry().x(), geometry().y()) );
	layout->setResizeMode(QLayout::Minimum);

	m_options.at(7)->setEnabled(false);
	m_options.at(8)->setEnabled(false);
}

AddItemWidget::~AddItemWidget()
{
}

void AddItemWidget::toggled(bool on)
{
	if (on)
		for (QRadioButton *option = m_options.first(); option; option = m_options.next())
			if (option != sender() && option->isOn())
				option->toggle();

	bool oneChecked = false;
	for (QRadioButton *option = m_options.first(); option; option = m_options.next())
		if (option->isOn())
			oneChecked = true;
	if ( ! oneChecked )
		((QRadioButton*) sender())->toggle();
}

int AddItemWidget::selected()
{
	for (uint i = 0; i < m_options.count(); ++i)
		if ( m_options.at(i) && m_options.at(i)->isOn() )
			return i;
	return 0;
}

void AddItemWidget::visualize()
{
	ViewSizeDialog *size = new ViewSizeDialog(this, m_imgSizeX->value(), m_imgSizeY->value());
	size->exec();
	m_imgSizeX->setValue(size->width());
	m_imgSizeY->setValue(size->height());
}

void AddItemWidget::activateImageOption(int)
{
	m_options.at(3)->toggle();
}

void AddItemWidget::activateLauncherOption(bool)
{
	m_options.at(5)->toggle();
}

void AddItemWidget::activateImportOption(int)
{
	m_options.at(6)->toggle();
}

QSize AddItemWidget::imageSize()
{
	return QSize(m_imgSizeX->value(), m_imgSizeY->value());
}

int AddItemWidget::iconSize()
{
	return m_iconSize->iconSize();
}

/** AddItemDialog */

AddItemDialog::AddItemDialog(Basket *parent, const QString &caption)
 : KDialogBase(parent, "addItemDialog", true, caption, KDialogBase::Ok | KDialogBase::Cancel, KDialogBase::Ok, false)
{
	m_basket = parent;
	m_widget = new AddItemWidget(this);
	setMainWidget(m_widget);
 }

AddItemDialog::~AddItemDialog()
{
}

void AddItemDialog::slotOk()
{
	Item    *item = 0;
	QPixmap *pixmap;
	switch (m_widget->selected()) {
		case 0:	// Text
			item = ItemFactory::createItemText("", m_basket);
			break;
		case 1: // Rich text
			item = ItemFactory::createItemHtml("", m_basket);
			break;
		case 2: // Link
			item = ItemFactory::createItemUrl(KURL(""), m_basket);
			break;
		case 3: // Image
			pixmap = new QPixmap(m_widget->imageSize());
			pixmap->fill();
			item = ItemFactory::createItemImage(*pixmap, m_basket);
			break;
		case 4: // Color
			item = ItemFactory::createItemColor(QColor(), m_basket);
			break;
		case 5: // App launcher
			item = appLauncher(m_widget->m_nameInAnnots->isChecked());
			return;
			break;
		case 6: // Icon
			item = importIcon();
			return;
			break;
		case 7: // Load
			break;
		case 8: // Mirror
			;
	}
	delayedDestruct();
	if (item)
		item->slotEdit();
}

Item* AddItemDialog::appLauncher(bool nameInAnnots)
{
	hide();
	KOpenWithDlg *dlg = new KOpenWithDlg(this);
	dlg->exec();

	if (dlg->text().isEmpty()) {
		delayedDestruct();
		return (Item*)0;
	}

	delayedDestruct();
	return ItemFactory::createItemLauncher(dlg->text(), dlg->service(), nameInAnnots, m_basket);
}

Item* AddItemDialog::importIcon()
{
	QString iconName;

	hide();
	iconName = KIconDialog::getIcon(KIcon::Desktop, KIcon::Application, false, m_widget->iconSize());
	if (iconName.isEmpty()) {
		delayedDestruct();
		return (Item*)0;
	}
	KIconLoader	icol;
	QPixmap		pix;
	pix = icol.loadIcon(iconName, KIcon::Desktop, m_widget->iconSize());

	delayedDestruct();
	return ItemFactory::createItemImage(pix, m_basket);
}

#include "additemdialog.moc"
