/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastient.laout@tuxfamily.org                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qlayout.h>
#include <qstring.h>
#include <qpixmap.h>
#include <qcolor.h>
#include <kpopupmenu.h>
#include <qpalette.h>
#include <qcursor.h>
#include <qaction.h>
#include <kstdaccel.h>

#include <kapplication.h>
#include <qinputdialog.h>
#include <qdragobject.h>
#include <kurldrag.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmimetype.h>

#include <qstringlist.h>
#include <qdir.h>
#include <qdom.h>
#include <kurl.h>
#include <qmessagebox.h>
//#include <iostream>

#include "basket.h"
#include "item.h"
#include "itemfactory.h"
#include "additemdialog.h"
#include "appsettings.h"
#include "debugwindow.h"

// FIXME:
//  QLayout::ResizeMode
//   QLayout::FreeResize - The widget is not constrained.

// FIXME:
//  Use QBoxLayout with properties :
//   LeftToRight, Down, RightToLeft or Up.

Basket::Basket(QWidget *parent, const char *name, WFlags fl)
 : QWidget(parent, name, fl)
{
	initActions(name);

	m_layout = new QVBoxLayout(this);

	m_checkable = false;
	setAcceptDrops(true);

	m_contentOnClickAction = 0;
	m_fileOnClickAction    = 2;
	m_runOverrideClick     = true;

	m_showAnnotations = true;
	m_hAlign          = 0;
	m_vAlign          = 1;
	m_isLocked        = false;

	m_newItemPlacement       = -1;
	m_allowFreeDrop          = true;
	m_pasteSelAtCursorPos    = true;
	m_systrayStackFrom       = 2;
	m_deleteAfterSystrayDrag = false;

	resetInsertTo();
	m_frameInsertTo = new QFrame(this);
	m_frameInsertTo->setFrameShape(QFrame::HLine);
	m_frameInsertTo->setFrameShadow(QFrame::Plain);
	m_frameInsertTo->setLineWidth(2);
	m_frameInsertTo->setPaletteForegroundColor(KApplication::palette().active().dark());
	m_frameInsertTo->hide();

	m_useColor    = false;
	m_color       = parent->paletteBackgroundColor();
	m_useAltColor = true;
	m_altColor    = QColor(KApplication::palette().active().mid()); // lightGray
}

Basket::~Basket()
{
}

void Basket::initActions(const char *name)
{
	KIconLoader icol;
	QPixmap     ico;

	ico = icol.loadIcon( "filenew",   KIcon::Toolbar, KIcon::SizeSmall );
	actAddItem = new QAction( i18n("Add item..."),        ico, i18n("Add &item..."),
	                          KStdAccel::shortcut(KStdAccel::New),   this );
	connect( actAddItem, SIGNAL(activated()) , this, SLOT(addItemDialog()) );

	ico = icol.loadIcon( "editpaste", KIcon::Toolbar, KIcon::SizeSmall );
	actPaste   = new QAction( i18n("Paste"),              ico, i18n("&Paste"),
	                          KStdAccel::shortcut(KStdAccel::Paste), this );
	connect( actPaste, SIGNAL(activated()) , this, SLOT(pasteItem())       );

	// Dynamic actions (the name of the basket can change) :
	ico = icol.loadIcon( "misc",      KIcon::Toolbar, KIcon::SizeSmall );
	actProp    = new QAction( i18n("Basket properties"),  ico, i18n("&Properties of \"%1\"...").arg(name),
	                          i18n("Basket properties", "F2"/*"Ctrl+P"*/),   this );
	connect( actProp,  SIGNAL(activated()) , AppSettings::mainContainer, SLOT(propBasket()) );

	// Dynamic actions (the name of the basket can change) :
	actLock    = new QAction( i18n("Lock/Unlock basket"),      i18n("&Lock \"%1\"").arg(name),
	                          i18n("(Un)Lock basket", "Ctrl+L"),     this );
	actLock->setToggleAction(true);
	connect( actLock,  SIGNAL(activated()) , AppSettings::mainContainer, SLOT(slotLockClicked()) ); // Yes : AppSettings::mainContainer is uggly too !
}

void Basket::setName(const char *name)
{
	QObject::setName(name);

	actProp->setMenuText( i18n("&Properties of \"%1\"...").arg(name) );
	actLock->setMenuText( i18n("&Lock \"%1\"").arg(name) );

	emit nameChanged(this, QString(name));
}

void Basket::setIcon(const QString &icon)
{
	m_icon = icon;
	emit iconChanged(this, icon);
}

void Basket::setCheckable(bool checkable)
{
	if (checkable == m_checkable)
		return;

	m_checkable = checkable;
	for (Item *cur = m_itemsList.first(); cur; cur = m_itemsList.next())
		cur->checkableChanged(checkable);
}

void Basket::setShowAnnotations(bool show)
{
	if (show == m_showAnnotations)
		return;

	m_showAnnotations = show;
	for (Item *cur = m_itemsList.first(); cur; cur = m_itemsList.next())
		cur->showAnnotationsChanged(show);
}

void Basket::setAlign(int hAlign, int vAlign)
{
	if (hAlign == m_hAlign && vAlign == m_vAlign)
		return;

	m_hAlign = hAlign;
	m_vAlign = vAlign;
	for (Item *cur = m_itemsList.first(); cur; cur = m_itemsList.next())
		cur->alignChanged(hAlign, vAlign);
}

void Basket::setLocked(bool lock)
{
	if (lock == m_isLocked)
		return;

	m_isLocked = lock;
	for (Item *cur = m_itemsList.first(); cur; cur = m_itemsList.next())
		cur->lockedChanged(lock);

	actLock->setOn(lock);

	actProp->setEnabled(!lock);

	actAddItem->setEnabled(!lock);
	actPaste->setEnabled(!lock);
}

void Basket::addItemDialog()
{
	AddItemDialog *dlg = new AddItemDialog(this, i18n("Add item"));
	dlg->exec();
}

void Basket::delItem(Item *item)
{
	if (item->useFile())
		QFile::remove( m_savePath + item->fileName() ); // FIXME: Directly in Basket ?

	m_layout->remove(item);
	m_itemsList.remove(m_itemsList.find(item));
	save();

	item->hide();  // We don't delete item because app crash time to time when move items by drag and drop
	//delete item; //  FIXME: I know it is memory unefficient but if anyone find why it crash, it will be
	               //  a very good programer (for that, comment first line and uncomment the secund one) !

	recolorizeItems();

	if (AppSettings::tray)
		AppSettings::tray->updateToolTip();
}

void Basket::contextMenuEvent(QContextMenuEvent *e)
{
	/* If no Item contained : show default Basket contextual menu */
	KPopupMenu *menu = new KPopupMenu(this);

	actAddItem->addTo(menu);
	actPaste->addTo(menu);

	menu->exec(e->globalPos());
	e->accept();
}


void Basket::dragEnterEvent(QDragEnterEvent* event)
{
	if (isLocked())
		return;

	event->accept( // FIXME: Alwayse accept ?
		true /*
		QImageDrag::canDecode(event) ||
		KURLDrag::canDecode(event)   ||
		QTextDrag::canDecode(event)  ||
		QColorDrag::canDecode(event)   */ );
}

void Basket::dragNotice(QWidget *item, int offset)
{
	if ( isLocked() || ! m_allowFreeDrop )
		return;

	m_insertTo = m_itemsList.find((Item*)item) + offset;
	m_layout->remove(m_frameInsertTo);
	m_frameInsertTo->show();
	m_layout->insertWidget(m_insertTo, m_frameInsertTo);
}

Item* Basket::currentStackItem()
{
	switch (m_systrayStackFrom) {
		case 1:  return m_itemsList.first();
		case 2:  return m_itemsList.last();
		default: return (Item*) 0;
	}
}

void Basket::dragStackItem()
{
	currentStackItem()->dragItem();
}

void Basket::dragLeaveEvent(QDragLeaveEvent*)
{
	// This event can be redirected from an item the most times)
	//  So, we only cancel drag if cursor is really outside of the basket
	if ( ! rect().contains(mapFromGlobal(QCursor::pos())) ) {
		m_frameInsertTo->hide();
		resetInsertTo();
	}
}

void Basket::dragMoveEvent(QDragMoveEvent*)
{
	if ( ! m_allowFreeDrop )
		m_frameInsertTo->show();
}

void Basket::dropEvent(QDropEvent *event)
{
	m_frameInsertTo->hide();

	ItemFactory::dropItem(event, this);

	resetInsertTo();
}

void Basket::mousePressEvent(QMouseEvent *qme)
{
	// if MidButton    BUT NOT ALT PRESSED to allow Alt+middleClick to launch actions
	//                 Because KWin handle Alt+click.
	//                 So it's a temporary hack to provide an alternative to alt+click !
	if ( (qme->button() & Qt::MidButton) && (qme->state() == 0) ) {
		if (m_pasteSelAtCursorPos) { // Calculate where to paste the item
			for (Item *cur = m_itemsList.first(); cur; cur = m_itemsList.next()) {
				QRect globalRect = cur->rect();
				globalRect.moveTopLeft(cur->pos());
				if ( globalRect.contains(qme->pos()) ) {
					int index = m_itemsList.find(cur);
					int y = globalRect.center().y();
					m_insertTo = index + (qme->pos().y() > y);
					break;
				}
			}
		}
		pasteItem(QClipboard::Selection);
		qme->accept();
	}
}

void Basket::pasteItem(QClipboard::Mode mode) // TODO: Return a Item* ??? More pratical
{
	if (m_isLocked)
		return;

	ItemFactory::dropItem( KApplication::clipboard()->data(mode), this );

	resetInsertTo();
}

void Basket::insertItem(Item *item)
{
	if (m_insertTo == -1)
		m_itemsList.append(item);
	else
		m_itemsList.insert((unsigned int)m_insertTo, item);
	m_layout->insertWidget(m_insertTo, item);
	m_lastInsertedItem = item;
	item->show();

	connect( item, SIGNAL(wantDelete(Item*)),           this, SLOT(delItem(Item*))              );
	connect( item, SIGNAL(wantPaste(QClipboard::Mode)), this, SLOT(pasteItem(QClipboard::Mode)) );
	connect( item, SIGNAL(wantNew()),                   this, SLOT(addItemDialog())             );

	m_layout->remove(m_frameInsertTo);
	resetInsertTo();
	recolorizeItems();

	if (AppSettings::tray)
		AppSettings::tray->updateToolTip();
}

Item* Basket::lastInsertedItem()
{
	return m_lastInsertedItem;
}

void Basket::recolorizeItems()
{
	QColor color;
	QColor altColor;

	if (m_useColor)
		color = m_color;
	else
		color = KApplication::palette().active().background();
	if (m_useAltColor)
		altColor = m_altColor; // FIXME: QColor QColor::light ( int factor = 150 ) const ?
	else
		altColor = color;

	m_layout->remove(m_frameInsertTo); // Do not falsifie results

	int i = 1;
	for (Item *cur = m_itemsList.first(); cur; cur = m_itemsList.next(), ++i) {
		if ( ! (i % 2) )
			cur->setPaletteBackgroundColor(altColor);  // FIXME: If background picture, it overwrite it :-/
		else  // void QWidget::unsetPalette () ??????????? : no because colors will be personalizbles
			cur->setPaletteBackgroundColor(color);
	}
}

void Basket::setColor(bool useColor, const QColor &color, bool useAltColor, const QColor &altColor)
{
	m_useColor    = useColor;
	m_color       = color;
	m_useAltColor = useAltColor;
	m_altColor    = altColor;

	recolorizeItems();
	// save(); // Saved with dialog !
}

void Basket::linkLookChanged()
{
	for (Item *cur = m_itemsList.first(); cur; cur = m_itemsList.next())
		cur->linkLookChanged();
}

/** Save work */

void Basket::setSavePath(const QString &path)
{
	QDir	dir;

	m_savePath = path;
	if ( path.isEmpty() )
		chooseNewSavePath();

	dir = QDir(m_savePath);
	if ( dir.exists() ) {
		load();
	} else {		// First save
		dir.mkdir(m_savePath);
		save();
		saveItemsContent();
	}
}

/*
 * Return a string like basketAppPath + "basket00001/"
 */
void Basket::chooseNewSavePath()
{
	QString	path;
	QDir	dir;

	for (int i = 1; ; ++i) {
		path = QString::number(i);
		//path = path.rightJustify(5, '0');
		path = AppSettings::appPath + "basket" + path + "/";
		dir  = QDir(path);
		if ( ! dir.exists() )
			break;
	}

	m_savePath = path;
}

void Basket::save()
{
	*AppSettings::debugWindow << "Basket::save() Start";

	QDomDocument doc("basket");
	QDomElement root = doc.createElement("basket");
	doc.appendChild(root);

	QDomElement properties = doc.createElement("properties");
	root.appendChild(properties);

	// TODO: addElement(name, content, attributesList);;
	QDomElement tag1 = doc.createElement("name");
	properties.appendChild(tag1);
	QDomText t1 = doc.createTextNode(name());
	tag1.appendChild(t1);

	QDomElement tag2 = doc.createElement("icon");
	properties.appendChild(tag2);
	QDomText t2 = doc.createTextNode(icon());
	tag2.appendChild(t2);

	QDomElement background = doc.createElement("background");
	properties.appendChild(background);
	background.setAttribute( "usecolor",    trueOrFalse(m_useColor)    );
	background.setAttribute( "color",       m_color.name()             );
	background.setAttribute( "usealtcolor", trueOrFalse(m_useAltColor) );
	background.setAttribute( "altcolor",    m_altColor.name()          );

	QDomElement tag3 = doc.createElement("showcheckboxes");
	properties.appendChild(tag3);
	QDomText t3 = doc.createTextNode( trueOrFalse(isCheckable()) );
	tag3.appendChild(t3);

	QDomElement tag4 = doc.createElement("showtooltips");
	properties.appendChild(tag4);
	QDomText t4 = doc.createTextNode( trueOrFalse(isShowAnnotations()) );
	tag4.appendChild(t4);

	QDomElement align = doc.createElement("alignment");
	properties.appendChild(align);
	align.setAttribute( "hor", QString::number(hAlign()) );
	align.setAttribute( "ver", QString::number(vAlign()) );

	QDomElement tag7 = doc.createElement("locked");
	properties.appendChild(tag7);
	QDomText t7 = doc.createTextNode( trueOrFalse(isLocked()) );
	tag7.appendChild(t7);

	QDomElement additempolicy = doc.createElement("additempolicy");
	properties.appendChild(additempolicy);
	additempolicy.setAttribute( "newitemplacement",       QString::number(m_newItemPlacement)   );
	additempolicy.setAttribute( "dropatcursorpos",        trueOrFalse(m_allowFreeDrop)          );
	additempolicy.setAttribute( "pasteselatcursorpos",    trueOrFalse(m_pasteSelAtCursorPos)    );
	additempolicy.setAttribute( "systraystackfrom",       QString::number(m_systrayStackFrom)   );
	additempolicy.setAttribute( "deleteaftersystraydrag", trueOrFalse(m_deleteAfterSystrayDrag) );

	QDomElement onClick = doc.createElement("onclick");
	properties.appendChild(onClick);
	onClick.setAttribute( "content",          m_contentOnClickAction          );
	onClick.setAttribute( "file",             m_fileOnClickAction             );
	onClick.setAttribute( "runoverrideclick", trueOrFalse(m_runOverrideClick) );

	QDomElement items = doc.createElement("items");

	for (Item *cur = m_itemsList.first(); cur; cur = m_itemsList.next()) {
		QString type;
		QDomElement content = doc.createElement("content");
		QDomText cont;
		switch (cur->type()) { // FIXME: bi.lowerTypeName();;
			case Item::Text:
				type = "text";
				content.setAttribute( "font", QString::number(cur->textFontType()) );
				content.setAttribute( "color", cur->paletteForegroundColor().name() );
				cont = doc.createTextNode( cur->fileName() );
				content.appendChild(cont);
				break;
			case Item::Html:
				type = "html";
				content.setAttribute( "showsource", trueOrFalse(cur->isShowSource()) );
				cont = doc.createTextNode( cur->fileName() );
				content.appendChild(cont);
				break;
			case Item::Url:
				type = "link";
				content.setAttribute( "title", cur->title() );
				content.setAttribute( "icon",  cur->icon()  );
				content.setAttribute( "autotitle", trueOrFalse(cur->autoTitle()) );
				content.setAttribute( "autoicon", trueOrFalse(cur->autoIcon()) );
				cont = doc.createTextNode( cur->url().prettyURL() );
				content.appendChild(cont);
				break;
			case Item::Image:
				type = "image";
				cont = doc.createTextNode( cur->fileName() );
				content.appendChild(cont);
				break;
			case Item::Color:
				type = "color";
				cont = doc.createTextNode( cur->color().name() );
				content.appendChild(cont);
				break;
		}
		QDomElement item = doc.createElement("item");
		items.appendChild(item);
		item.setAttribute( "type", type );
		item.setAttribute( "checked", trueOrFalse(cur->isChecked()) );
		if ( ! cur->runCommand().isEmpty() )
			item.setAttribute( "runcommand", cur->runCommand() );

		item.appendChild(content);

		if ( ! cur->annotations().isEmpty() ) {
			QDomElement annotations = doc.createElement("annotations");
			item.appendChild(annotations);
			QDomText tannot = doc.createTextNode( cur->annotations() );
			annotations.appendChild(tannot);
		}
	}

	root.appendChild(items);

	QFile file(m_savePath + ".basket");
	if ( file.open(IO_WriteOnly) ) {
		QTextStream stream(&file);
		QString xml = doc.toString();
		stream << "<?xml version=\"1.0\" encoding=\"ISO-8859-15\"?>\n";	// FIXME: Else : accentued characters in links are not keeped
		stream << xml;
		file.close();
	}

	*AppSettings::debugWindow << "Basket::save() End";
}

void Basket::saveItemsContent()
{
	for (Item *cur = m_itemsList.first(); cur; cur = m_itemsList.next())
		cur->saveContent();
}

void Basket::load()
{
	*AppSettings::debugWindow << "Basket::load() Start";

	// xmlOpenDoc
	QDomDocument doc("basket");
	QFile file(m_savePath + ".basket");
	if ( !file.open(IO_ReadOnly) )
		return;
	if ( ! doc.setContent(&file) ) {
//		QMessageBox::information(this, "Load", "Error : malformed content");
		file.close();
		return;
	}
	file.close();

	QDomElement docElem = doc.documentElement();
	QDomElement properties = getXmlElement(docElem, "properties");
	setName(                        getXmlElementText(properties, "name",           "")       );
	setIcon(                        getXmlElementText(properties, "icon",           "")       );
	setCheckable(       trueOrFalse(getXmlElementText(properties, "showcheckboxes", "false")) );
	// Keep compatibility with 0.3.x versions :
	setCheckable(       trueOrFalse(getXmlElementText(properties, "checkable",      trueOrFalse(isCheckable()))) );
	setShowAnnotations( trueOrFalse(getXmlElementText(properties, "showtooltips",   "true"))  );
	// Keep compatibility with 0.3.x versions :
	setShowAnnotations( trueOrFalse(getXmlElementText(properties, "annotations",    trueOrFalse(isShowAnnotations())))  );
	setLocked(          trueOrFalse(getXmlElementText(properties, "locked",         "false")) );

	QDomElement align = getXmlElement(properties, "alignment");
	setAlign( align.attribute("hor", "0" ).toInt(), align.attribute("ver", "1" ).toInt() );
	// Keep compatibility with 0.3.x versions :
	setAlign( getXmlElementText( properties, "halign", QString::number(hAlign()) ).toInt(),
	          getXmlElementText( properties, "valign", QString::number(vAlign()) ).toInt() );

	QDomElement background = getXmlElement(properties, "background");
	setColor(   trueOrFalse(background.attribute("usecolor",    "false"    )),
	            QColor(     background.attribute("color",       ((QWidget*)parent())->paletteBackgroundColor().name() )),
	            trueOrFalse(background.attribute("usealtcolor", "true"     )),
	            QColor(     background.attribute("altcolor",    "lightGray")) );

	QDomElement onClick = getXmlElement(properties, "onclick");
	m_contentOnClickAction =             onClick.attribute("content",          "0"   ).toInt();
	m_fileOnClickAction    =             onClick.attribute("file",             "2"   ).toInt();
	m_runOverrideClick     = trueOrFalse(onClick.attribute("runoverrideclick", "true"));

	loadItems( getXmlElement(docElem, "items") );

	/* Load add item policy AFTER items because else items will be loaded
	   in reverse order if user selected add new items on top ! */
	QDomElement additempolicy = getXmlElement(properties, "additempolicy");
	setNewItemPlacement(                   additempolicy.attribute("newitemplacement",       "-1"    ).toInt()  );
	setAllowFreeDrop(          trueOrFalse(additempolicy.attribute("dropatcursorpos",        "true"  ))         );
	setPasteSelAtCursorPos(    trueOrFalse(additempolicy.attribute("pasteselatcursorpos",    "true"  ))         );
	setSystrayStackFrom(                   additempolicy.attribute("systraystackfrom",       "2"     ).toInt()  );
	setDeleteAfterSystrayDrag( trueOrFalse(additempolicy.attribute("deleteaftersystraydrag", "false" ))         );

	*AppSettings::debugWindow << "Basket::load() End";
}

void Basket::loadItems(const QDomElement &items)
{
	*AppSettings::debugWindow << "Basket::loadItems() Start";

	QDomNode n = items.firstChild();
	Item *item = 0;
	while ( ! n.isNull() ) {
		QDomElement e = n.toElement();
		if ( (!e.isNull()) && e.tagName() == "item" ) {
			QString type = e.attribute("type");
			QDomElement content = getXmlElement(e, "content");
			QString runCommand = e.attribute("runcommand");
			runCommand = getXmlElementText(e, "action", runCommand); // Keep compatibility with 0.3.x versions
			QString annotations = getXmlElementText(e, "annotations");
			bool checked = trueOrFalse(e.attribute("checked"), false);
			if (type == "text") {
				item = new Item( QString(), content.attribute("font").toInt(), QColor(content.attribute("color")),
				                 annotations, checked, runCommand,
				                 this, content.text() );
				item->loadContent();
			}
			if (type == "html") {
				item = new Item( QString(), trueOrFalse(content.attribute("showsource"), false),
				                 annotations, checked, runCommand,
				                 this, content.text() );
				item->loadContent();
			}
			if (type == "link") {
				bool defAutoTitle = content.attribute("title") == content.text();
				bool defAutoIcon  = content.attribute("icon")  == ItemFactory::iconForURL(KURL(content.text()));
				item = new Item( KURL(content.text()), content.attribute("title"), content.attribute("icon"),
				                 trueOrFalse( content.attribute("autotitle"), defAutoTitle),
				                 trueOrFalse( content.attribute("autoicon"),  defAutoIcon),
				                 annotations, checked, runCommand,
				                 this );
			}
			if (type == "image") {
				item = new Item( QPixmap(), //(content.text(), "PNG"),
				                 annotations, checked, runCommand,
				                 this, content.text() );
				item->loadContent();
			}
			if (type == "color") {
				item = new Item( QColor(content.text()),
				                 annotations, checked, runCommand,
				                 this );
			}
			insertItem(item); // No save needed (because we load :-) )
		}
		n = n.nextSibling();
	}

	*AppSettings::debugWindow << "Basket::loadItems() End";
}

void Basket::deleteFiles()
{
	QDir dir(m_savePath, QString::null, QDir::Name | QDir::IgnoreCase, QDir::All | QDir::Hidden);
	QStringList list = dir.entryList();

	for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it )
		if ( *it != "." && *it != ".." ) {
			QFile::remove(m_savePath + *it);}

	dir.rmdir(m_savePath);
}

QDomElement Basket::getXmlElement(const QDomElement &startElement, const QString &elementPath)
{
	QStringList elements = QStringList::split("/", elementPath, false);
	QDomNode n = startElement.firstChild();
	for (unsigned int i = 0; i < elements.count(); ++i) {               // For each elements
		while ( ! n.isNull() ) {                                        // Browse theire sub elements
			QDomElement e = n.toElement();                              //  and search the good one
/*DEBUG*/	/*QMessageBox::information( this,
			"Tag search", "Start : " + startElement.tagName() +
			"\nSearch : " + elementPath +
			"\nElement : " + e.tagName() );*/
/*DEBUG2*/	/*std::cout << "Start : " + startElement.tagName() +
			"\t\tSearch : " + elementPath +
			"\t\tElement : " + e.tagName() + "\n";*/
			if ( (!e.isNull()) && e.tagName() == *elements.at(i) ) {    // If found
				if ( i + 1 == elements.count() )                        // And if it is the asked element
					return e;                                           // Return the first corresponding
				else {                                                  // Or if it is an intermediate element
					n = e.firstChild();                                 // Continue with the next sub element
					break;
				}
			}
			n = n.nextSibling();
		}
	}
	return QDomElement();                                               // Not found !
}

QString Basket::getXmlElementText(const QDomElement &startElement, const QString &elementPath, const QString &defaultTxt)
{
	QDomElement e = getXmlElement(startElement, elementPath);
	if (e.isNull())
		return defaultTxt;
	else
		return e.text();
}

bool Basket::trueOrFalse(const QString &value, bool defaultValue)
{
	if ( value == "true"  || value == "1" || value == "on"  )
		return true;
	if ( value == "false" || value == "0" || value == "off" )
		return false;
	return defaultValue;
}

QString Basket::trueOrFalse(bool value)
{
	return value ? "true" : "false";
}

#include "basket.moc"
