/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastient.laout@tuxfamily.org                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef ITEM_H
#define ITEM_H

#include <qcheckbox.h>
#include <qframe.h>
#include <qclipboard.h>
#include <kprocess.h> // class KProcess;

#include <qstring.h>
#include <qpixmap.h>
#include <qcolor.h>
#include <kurl.h>

class QWidget;
class QHBoxLayout;
class QVBoxLayout;
class QSpacerItem;
class QLabel;
class QColor;
class KURL;
class QDragObject;

class Basket;
class LinkLabel;

/**Provide an vertically alignable QCheckBox.
  *@author Sbastien Laot
  */
class AlignableCheckBox : public QWidget
{
  Q_OBJECT
  public:
	AlignableCheckBox(const QString &text, QWidget *parent, const char *name = 0);
	~AlignableCheckBox();
	inline QCheckBox* checkBox() { return &m_check; }
	void setAlignment(int align);
  private:
	QVBoxLayout *m_layout;
	QSpacerItem *m_spacer;
	QCheckBox    m_check;
};

/**Implement all droped objects.
  *@author Sbastien Laot
  */
class Item : public QFrame
{
  Q_OBJECT
  public:
	/** Construtors, initializer and destructor */
	Item(const QString &text,        int fontType, const QColor &fontColor,
	     const QString &annotations, bool checked, const QString &runCommand,
	     Basket *parent,             const QString &fileName);
	Item(const QString &html,        bool showSource,
	     const QString &annotations, bool checked, const QString &runCommand,
	     Basket *parent,             const QString &fileName);
	Item(const KURL    &url,         const QString &title, const QString &icon,
	     bool autoTitle, bool autoIcon,
	     const QString &annotations, bool checked, const QString &runCommand,
	     Basket *parent);
	Item(const QPixmap &image,
	     const QString &annotations, bool checked, const QString &runCommand,
	     Basket *parent,             const QString &fileName);
	Item(const QColor  &color,
	     const QString &annotations, bool checked, const QString &runCommand,
	     Basket *parent);
	~Item();
  private:
	void initItem(bool checked); // Create and initialize a new item
  public:
	/** Return the type of the stored contain */
	enum ContentType { Text = 1, Html, Url, Image/*, File, Movie*/, Color };
	inline ContentType type() { return m_type; } // TODO: getTypeName( lowerCase, pretty...
	QString contentTypeName();
	QString toString();
	/*enum ColorFormat { Hex = 1, Rvb, Hsv, Name };*/
	/** Various functions to change contents (according to the type of the item) */
	// Text
	QString text();
	int textFontType();
	void setText(const QString &text);
	void setTextFontType(int type);
	void setText(const QString &text, int type, QColor fontColor);
	// Html
	QString html();
	bool isShowSource();
	void setShowSource(bool show);
	void setHtml(const QString &html);
	void setHtml(const QString &html, bool showSource);
	// Url
	KURL url();
	QString title();
	QString icon();
	bool autoTitle() { return m_autoTitle; }
	bool autoIcon()  { return m_autoIcon;  }
	void setUrl(const KURL &url, const QString &title, const QString &icon);
	void setAuto(bool autoTitle, bool autoIcon) { m_autoTitle = autoTitle; m_autoIcon = autoIcon; }
	// Image
	QPixmap* pixmap();
	void setPixmap(const QPixmap &pixmap);
	// Color
	QColor color();
	void setColor(QColor color);
	/** Common configs */
	QString annotations();
	void setAnnotations(const QString &annotations);
	bool isChecked();
	void setChecked(bool check);
	QString fileName()   { return m_fileName;      }
	bool setFileName(const QString &fileName);
	QString runCommand() { return m_runCommand;    }
	void setRunCommand(const QString &runCommand);
	bool useFile()       { return (m_type == Text || m_type == Html || m_type == Image); }
	bool useLinkLabel()  { return (m_type == Url); }
  signals:
	/** Pass signals to parent basket */
	void wantDelete(Item *item);
	void wantNew();
	void wantPaste(QClipboard::Mode mode);
  protected:
	virtual void mousePressEvent(QMouseEvent *qme);
	virtual void mouseReleaseEvent(QMouseEvent *qme);
	virtual void mouseMoveEvent(QMouseEvent *qme);
	virtual void dragMoveEvent(QDragMoveEvent *e);
	virtual void dragLeaveEvent(QDragLeaveEvent *event);
	virtual void dragEnterEvent(QDragEnterEvent *event);
	virtual void dropEvent(QDropEvent *event);
	virtual void contextMenuEvent(QContextMenuEvent *e);
  public:
	void dragItem();
	// Propage event because m_title QLabel's LinkLabel doesn't care of color changes
	void setPaletteBackgroundColor(const QColor &color);
  public slots:
	void editImage();
	/** Update display according of new basket properties */	// private or friend basket
	void checkableChanged(bool checkable);
	void showAnnotationsChanged(bool show);
	void alignChanged(int hAlign, int vAlign);
	void linkLookChanged();
	void lockedChanged(bool lock);
	/** Save & load work */
	void saveProperties();
	void saveContent();
	void loadContent();
	void saveAs(const KURL &dest);
	void loadFrom(const KURL &source);
	/** User interaction */
	void slotEdit(bool editAnnotations = false);
	void slotEditAnnotations(); // Provided to item popupMenu ! Please use slotEdit(true)
	void slotDelete();
	void slotNew();
	void slotCut();
	void slotCopy(bool toSelection = false);
	void slotCopySelection();   // Provided to item popupMenu ! Please use slotCopy(true)
	void slotPaste();
	void slotOpen();
	void slotOpenWith();
	void slotSaveAs();
	void slotEndEdit(KProcess*);
	void slotRunCommand();
	void slotChecked();
//  private: // Disallow standard QWidget reparenting...
//	virtual void reparent(QWidget *parent, WFlags, const QPoint &, bool showIt = false);
//	void reparent(QWidget *parent, const QPoint &, bool showIt = false);
//  public:  // ... but allow reparent to a Basket (to memorize the new parent and adopt its properties) !!
//	void reparent(Basket *parent, const QPoint &point, bool showIt = false);	// Too complicated for the moment

	int onClickActionPolicy();
  private:
	/* Widgets for the UI */
	QHBoxLayout       *m_layout;
	AlignableCheckBox *m_check;
	QLabel            *m_item;
	LinkLabel         *m_linkLabel;

	/* Type and properties */
	ContentType  m_type;
	Basket      *m_parentBasket;
	QString      m_annotations;
	QString      m_runCommand;
	QString      m_fileName;

	/* Special variables to handle contents by type */
	int      m_textFontType;
	bool     m_showSource;
	QColor  *m_color;
	KURL    *m_url;
	QString *m_title;
	QString *m_icon;
	bool     m_autoTitle;
	bool     m_autoIcon;

	/* Other variables and flags */
	bool     m_canDrag;
	QPoint   m_pressPos;
	KProcess m_editProcess;
  public:
	static QString urlWithoutProtocol(const KURL &url) {
		return url.prettyURL().right(url.prettyURL().length() - url.protocol().length() - 1);
	}
};

#endif // ITEM_H
