/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastient.laout@tuxfamily.org                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qbuffer.h>
#include <qdragobject.h>
#include <kurldrag.h>
#include <kcolordrag.h>
#include <kapplication.h>
#include <kiconloader.h>
#include <qimage.h>

#include "itemdrag.h"
#include "itemfactory.h"
#include "basket.h"

const char * const ItemDrag::item_mime_string  = "application/x-basket-item";
const char * const ItemDrag::text_mime_string  = "text/plain";
const char * const ItemDrag::color_mime_string = "application/x-color";

ItemDrag::ItemDrag(/*const */Item &item, QWidget *parent, const char *name)
 : QDragObject(parent, name)
{
	setItem(item);
}

ItemDrag::~ItemDrag()
{
	// FIXME: delete m_object; ???
}

void ItemDrag::setItem(/*const */Item &item)
{
	QPixmap      icon;
	QImage       image; // For temporary scale purpose
	const int    iconSize = 22;
	const QPoint iconHotPoint = QPoint(-5, -7); //iconSize / 2;

	/* Init item */
	m_type        = item.type();
	m_isChecked   = item.isChecked();
	m_annotations = item.annotations();
	m_runCommand  = item.runCommand();

	/* Init item content */
	switch (m_type) {
		case Item::Text:
			m_text         = item.text();
			m_textFontType = item.textFontType();
			m_color        = item.paletteForegroundColor();
			break;
		case Item::Html:
			m_text         = item.html();
			m_showSource   = item.isShowSource();
			break;
		case Item::Image:
			m_pixmap       = QPixmap(*(item.pixmap()));
			break;
		case Item::Url:
			m_url          = item.url();
			m_title        = item.title();
			m_icon         = item.icon();
			m_autoTitle    = item.autoTitle();
			m_autoIcon     = item.autoIcon();
			break;
		case Item::Color:
			m_color        = item.color();
			break;
	}

	/* Init alternative formats for item content */
	m_object = contentDrag();

	/* Init icon during drag */
	if (1/*TODO*/)
		switch (m_type) {
			case Item::Text:
				icon = kapp->iconLoader()->loadIcon("text", KIcon::NoGroup, iconSize);
				setPixmap(icon, iconHotPoint);
				break;
			case Item::Html:
				icon = kapp->iconLoader()->loadIcon("html", KIcon::NoGroup, iconSize);
				setPixmap(icon, iconHotPoint);
				break;
			case Item::Image:
				image = m_pixmap.convertToImage();
				image = image.smoothScale(40, 30, QImage::ScaleMin);
				icon = QPixmap(image);
				setPixmap(icon, iconHotPoint);
				break;
			case Item::Url:
				icon = kapp->iconLoader()->loadIcon(m_icon, KIcon::NoGroup, iconSize);
				setPixmap(icon, iconHotPoint);
				break;
			case Item::Color:
				setPixmap(m_object->pixmap(), m_object->pixmapHotSpot());
				break;
		}
}

QDragObject* ItemDrag::contentDrag() const
{
	switch (m_type) {
		case Item::Text:  return new QTextDrag(  m_text,                    0     );
		case Item::Html:  return new QTextDrag(  m_text,                    0     );
		case Item::Image: return new QImageDrag( m_pixmap.convertToImage(), 0     );
		case Item::Url:   return new KURLDrag(   KURL::List(m_url),         0, "" );
		case Item::Color: return new KColorDrag( m_color,                   0     );
	}
	return (QDragObject*) 0;
}

const char* ItemDrag::format(int i) const
{
	// First supported MIME is the item format (to drop into application)
	if (i == 0)
		return item_mime_string;

	// ItemDrag also provide other formats that represent the item content
	const char *format;

	if (m_type == Item::Color) {
		if (i == 1)                    // QColorDrag just support "application/x-color"
			format = color_mime_string;
		else if (i == 2)               // And we add a "text/plain" support (must be in KColorDrag of KDE 3.2,
			format = text_mime_string; //  but we must keep compatibility with older versions)
		else
			format = 0;
	} else
		format = m_object->format(i-1);

	// TODO: Add "text/uri-list" for all file based items !
	// TODO: Add "text/html" (with HTML tags) for HTML items and "text/plain" (converted to a text plain) !

	return format;
}

QByteArray ItemDrag::encodedItem() const
{
	QBuffer buffer;

	if (buffer.open(IO_WriteOnly)) {
		QDataStream stream(&buffer);
		stream << (Q_INT32)m_type
		       << (Q_INT32)m_isChecked // Basket::trueOrFalse() // Store a QString "true" or "false" because I don't know how to store bools
		       << m_annotations                    //  and (Q_INT32)
		       << m_runCommand;
		switch (m_type) {
			case Item::Text:
				stream << m_text
				       << m_textFontType
				       << m_color;
				break;
			case Item::Html:
				stream << m_text
				       << (Q_INT32)m_showSource;
				break;
			case Item::Image:
				stream << m_pixmap;
				break;
			case Item::Url:
				stream << m_url
				       << m_title
				       << m_icon
				       << (Q_INT32)m_autoTitle
				       << (Q_INT32)m_autoIcon;
				break;
			case Item::Color:
				stream << m_color;
				break;
		}
		buffer.close();
	}

	return buffer.buffer();
}

QByteArray ItemDrag::encodedData(const char *mimeType) const
{
	QString mime = QString(mimeType); // To be able to do comparisons between strings

	// Item MIME type : return encoded
	if (mime == item_mime_string)
		return encodedItem();

	// Color MIME type : return encoded color in "text/plain" MIME format
	if (m_type == Item::Color && mime == text_mime_string)
		// FIXME: Is it good to create all those additionnal QDragObject ?? (It doesn't work with a QBuffer!)
		return (new QTextDrag(m_color.name(), 0))->encodedData(text_mime_string);
	else
		// Return encoded content
		return m_object->encodedData(mime);
}

bool ItemDrag::canDecode(QMimeSource *source)
{
	if (source != 0)
		return source->provides(item_mime_string);
	else
		return false;
}

//bool ItemDrag::decode(QMimeSource *source, Item &item)
Item* ItemDrag::decode(QMimeSource *source, Basket *parent)
{
	/* Variables to handle item type and global properties */
	Item::ContentType type;
	Q_INT32           tmp_isChecked;
	bool              isChecked;
	QString           annotations;
	QString           runCommand;

	/* Special variables to handle contents by type */
	QString text;
	int     textFontType;
	Q_INT32 showSource;
	QPixmap pixmap;
	QColor  color;
	KURL    url;
	QString title;
	QString icon;
	Q_INT32 autoTitle;
	Q_INT32 autoIcon;

	QBuffer buffer(source->encodedData(item_mime_string));
	Item *item = 0;

	if (buffer.open(IO_ReadOnly)) {
		QDataStream stream(&buffer);
		stream >> (Q_INT32&)type // FIXME: Alwayse Q_INT32 ?
		       >> tmp_isChecked
		       >> annotations
		       >> runCommand;
		isChecked = tmp_isChecked != 0;
		switch (type) {
			case Item::Text:
				stream >> text
				       >> textFontType
				       >> color;
				item = new Item( text, textFontType, color,
				                 annotations, isChecked, runCommand,
				                 parent, ItemFactory::fileNameForNewItem(parent) );
				break;
			case Item::Html:
				stream >> text
				       >> showSource;
				item = new Item( text, showSource != 0,
				                 annotations, isChecked, runCommand,
				                 parent, ItemFactory::fileNameForNewItem(parent) );
				break;
			case Item::Image:
				stream >> pixmap;
				item = new Item( pixmap,
				                 annotations, isChecked, runCommand,
				                 parent, ItemFactory::fileNameForNewItem(parent) );
				break;
			case Item::Url:
				stream >> url
				       >> title
				       >> icon
				       >> autoTitle
				       >> autoIcon;
				item = new Item( url, title, icon,
				                 autoTitle != 0, autoIcon != 0,
				                 annotations, isChecked, runCommand,
				                 parent );
				break;
			case Item::Color:
				stream >> color;
				item = new Item( color,
				                 annotations, isChecked, runCommand,
				                 parent );
				break;
		}
		buffer.close();
	}

	return item;
}

#include "itemdrag.moc"
