/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastient.laout@tuxfamily.org                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qstring.h>
#include <kurl.h>
#include <qpixmap.h>
#include <qcolor.h>
#include <qregexp.h>
#include <kcolordrag.h>
#include <kurldrag.h>
#include <qstylesheet.h>
#include <qdir.h>
#include <kmimetype.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <kdesktopfile.h>
#include <kapplication.h>
#include <kiconloader.h>

#include "basket.h"
#include "item.h"
#include "itemfactory.h"
#include "itemdrag.h"
#include "linklabel.h"
#include "appsettings.h"

#include "debugwindow.h"

/** Create items from scratch (just a content) */

Item* ItemFactory::createItemText(const QString &text, Basket *parent)
{
	Item *item = new Item(text, 0, QColor(), QString(), false, QString(),
	                      parent, fileNameForNewItem(parent));
	parent->insertItem(item);
	item->saveContent();
	parent->save();

	return item;
}

Item* ItemFactory::createItemHtml(const QString &html, Basket *parent)
{
	Item *item = new Item(html, false, QString(), false, QString(),
	                      parent, fileNameForNewItem(parent));
	parent->insertItem(item);
	item->saveContent();
	parent->save();

	return item;
}

Item* ItemFactory::createItemUrl(const KURL &url, Basket *parent)
{
	Item *item = new Item(url, url.prettyURL(), iconForURL(url), true, true,
	                      QString(), false, QString(), parent);
	parent->insertItem(item);
	parent->save();

	return item;
}

Item* ItemFactory::createItemImage(const QPixmap &image, Basket *parent)
{
	Item *item = new Item(image, QString(), false, QString(),
	                      parent, fileNameForNewItem(parent));
	parent->insertItem(item);
	item->saveContent();
	parent->save();

	return item;
}

Item* ItemFactory::createItemColor(const QColor &color, Basket *parent)
{
	Item *item = new Item(color, QString(), false, QString(),
	                      parent);
	parent->insertItem(item);
	parent->save();

	return item;
}

Item* ItemFactory::createItem(const QString &text, Basket *parent)
{
	QString newText = text.stripWhiteSpace();	// The text for a new item, without useless spaces
	QString ltext   = newText.lower();			// The same, in lower cases

	/* Search for mail adress ("*@*.*" ; "*" can contain '_', '-', or '.') and add protocol to it */
	QRegExp mailExp("^[\\w-\\.]+@[\\w-\\.]+\\.[\\w]+$");
	if (mailExp.search(ltext) != -1) {
		ltext.insert(0, "mailto:");
		newText.insert(0, "mailto:");
	}

	/* Search for an url and create an URL item */
	if ( ltext.startsWith("/") && ltext[1] != '/' && ltext[1] != '*' || // Take files but not C/C++/... comments !
		 ltext.startsWith("file:")   ||
		 ltext.startsWith("http://") ||
		 ltext.startsWith("www.")    ||
		 ltext.startsWith("ftp")     ||
		 ltext.startsWith("mailto:")    ) {

		/* First, correct the text to use the good format for the url */
		if (ltext.startsWith( "/"))
			newText.insert(0, "file:");
		if (ltext.startsWith("www."))
			newText.insert(0, "http://");
		if (ltext.startsWith("ftp."))
			newText.insert(0, "ftp://");

		/* And create the Url item (or launcher if URL point a .desktop file) */
		return createItemUrlOrLauncher(KURL(newText), parent);
	}

	/* Search for a color (#RGB , #RRGGBB , #RRRGGGBBB , #RRRRGGGGBBBB) and create a color item */
	QRegExp exp("^#(?:[a-fA-F\\d]{3}){1,4}$");
	if ( exp.search(text) != -1 )
		return createItemColor(QColor(text), parent);

	/* Else, it's a text or an HTML item, so, create it */
	if (QStyleSheet::mightBeRichText(newText))   // ISITUSEFULL: Verifie all lines ?
		return createItemHtml(newText, parent);
	else
		return createItemText(newText, parent);
}

Item* ItemFactory::createItemLauncher(const QString &prog, KService::Ptr service, bool nameInAnnots, Basket *parent)
{
	QString name;
	QString icon;
	QString runC;
	QString comment;
	QString annot;

	/* Debug work */
	*AppSettings::debugWindow << "<b>Import desktop service to link item :</b>";
	if (service == 0)
		*AppSettings::debugWindow << "Personnalized command that isn't in K menu";
	else
		*AppSettings::debugWindow
			<< "&nbsp;&nbsp;&nbsp;<b>type :</b> "               + service->type()
			<< "&nbsp;&nbsp;&nbsp;<b><u>name :</u></b> "        + service->name()
			<< "&nbsp;&nbsp;&nbsp;<b><u>exec :</u></b> "        + service->exec()
			<< "&nbsp;&nbsp;&nbsp;<b>library :</b> "            + service->library()
			<< "&nbsp;&nbsp;&nbsp;<b>init :</b> "               + service->init()
			<< "&nbsp;&nbsp;&nbsp;<b><u>icon :</u></b> "        + service->icon()
			<< "&nbsp;&nbsp;&nbsp;<b>terminal :</b> "           + QString(service->terminal()?"true":"false")
			<< "&nbsp;&nbsp;&nbsp;<b>terminalOptions :</b> "    + service->terminalOptions()
			<< "&nbsp;&nbsp;&nbsp;<b>substituteUid :</b> "      + QString(service->substituteUid()?"true":"false")
			<< "&nbsp;&nbsp;&nbsp;<b>username :</b> "           + service->username()
			<< "&nbsp;&nbsp;&nbsp;<b>desktopEntryPath :</b> "   + service->desktopEntryPath()
			<< "&nbsp;&nbsp;&nbsp;<b>desktopEntryName :</b> "   + service->desktopEntryName()
			//DCOPServiceType_t  DCOPServiceType () const
			<< "&nbsp;&nbsp;&nbsp;<b>path :</b> "               + service->path()
			<< "&nbsp;&nbsp;&nbsp;<b><u>comment :</u></b> "     + service->comment()
			<< "&nbsp;&nbsp;&nbsp;<b>genericName :</b> "        + service->genericName()
			<< "&nbsp;&nbsp;&nbsp;<b>keywords :</b> "           + service->keywords().join(",")
			<< "&nbsp;&nbsp;&nbsp;<b>categories :</b> "         + service->categories().join(",")
			<< "&nbsp;&nbsp;&nbsp;<b>serviceTypes :</b> "       + service->serviceTypes().join(",")
			//bool  hasServiceType ( const QString& _service ) const
			<< "&nbsp;&nbsp;&nbsp;<b>allowAsDefault :</b> "     + QString(service->allowAsDefault()?"true":"false")
			<< "&nbsp;&nbsp;&nbsp;<b>allowMultipleFiles :</b> " + QString(service->allowMultipleFiles()?"true":"false")
			<< "&nbsp;&nbsp;&nbsp;<b>initialPreference :</b> "  + QString::number(service->initialPreference())
			<< "&nbsp;&nbsp;&nbsp;<b>noDisplay :</b> "          + QString(service->noDisplay()?"true":"false")
			<< "&nbsp;&nbsp;&nbsp;<b>parentApp :</b> "          + service->parentApp()
			//virtual QVariant  property ( const QString& _name ) const
			//virtual QStringList  propertyNames () const
			<< "&nbsp;&nbsp;&nbsp;<b><i>isValid :</i></b> "     + QString(service->isValid()?"true":"false");
	*AppSettings::debugWindow << "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<b><u>Real exec :</u></b> " +
		prog + " " + (nameInAnnots?"(Place name in annotations)":"");

	/* Data guess */
	if (service && service->isValid()) { // First from service, if exist
		name    = service->name();
		icon    = service->icon();  // This function return icon with extension but KIconLoader doesn't care of that
		comment = service->genericName();
		if (name.isEmpty())
			name = service->exec(); // without terminal call (more beautiful)
		if (icon.isEmpty())         // Try to find : use first word as icon : typically the program without argument
			icon = QStringList::split(' ', service->exec()).first();
		if (comment.isEmpty())
			comment = service->comment();
	}
	runC = prog;        // Use full run command : with terminal call, if asked
	if (name.isEmpty())
		name = runC;
	if (comment.lower() == name.lower().remove(' ')) // Avoid annotations to be same as name, even with case
		comment = "";                                //  or spaces differences (such as "K Write" and "KWrite")
	// exeName = exeName.mid( exeName.findRev( '/' ) + 1 ); // strip path if given [But it doesn't care of such "myprogram /my/path/argument" -> return "argument". Would must first strip first word and then strip path... Useful ??
	if (icon.isEmpty()) // Last luck : use first word as icon : typically the program without argument
		icon = QStringList::split(' ', runC).first();
	if (kapp->iconLoader()->loadIcon(icon, KIcon::NoGroup, 16, KIcon::DefaultState, 0L, true).isNull()) // If icon isn't found
		icon = QStringList::split('-', runC).first(); // e.g. This allow to use "gimp" icon if run command is "gimp-1.3"
	if (kapp->iconLoader()->loadIcon(icon, KIcon::NoGroup, 16, KIcon::DefaultState, 0L, true).isNull()) // If icon isn't found
		icon = "exec"; // Very last ressort : a generic icon

	/* Move program name in annotations, if asked */
	annot = comment;
	if (nameInAnnots && !name.isEmpty())
		annot = name + "\n" + comment;

	/* And finaly create the item */
	Item *item = new Item( "", (nameInAnnots ? QString() : name), icon,
	                       false, false,/*true, true,*/ // TODO: titleForProg() and iconForProg() !!
	                       annot, false, runC, parent );
	parent->insertItem(item);
	parent->save();

	return item;

	// FIXME: Always use .desktop files to store links ??
	//        Can I add "autoTitle" and "autoIcon" entries to it, or just store them in basket, as now...
}

Item* ItemFactory::createItemUrlOrLauncher(const KURL &url, Basket *parent)
{
	KService::Ptr service = new KService(Item::urlWithoutProtocol(url));

	if (service->isValid()) // First try if link point to a .desktop file to add a launcher
		return createItemLauncher(service->exec(), service, false, parent); // FIXME: No care of "run in terminal"
	else                                      //  or other options ! Later: Use KService to store apps launchers !
		return createItemUrl(url, parent);
}

Item* ItemFactory::dropItem(QMimeSource *source, Basket *parent)
{
	Item *item = 0;

	/* DEBUG */
	*AppSettings::debugWindow << "<b>Drop :</b>";
	for(int i = 0; source->format(i); ++i)
		if ( *(source->format(i)) )
			*AppSettings::debugWindow << "\t[" + QString::number(i) + "] " + QString(source->format(i));

	/* Copy or move an Item */
	if (ItemDrag::canDecode(source)) {
		item = ItemDrag::decode(source, parent);
		parent->insertItem(item);
		item->saveContent();
		parent->save();
		return item;
	}

	/*switch ( e->action() ) {
		case QDropEvent::Copy:
			str = "Copy";
			break;
		case QDropEvent::Move:
			str = "Move";
			e->acceptAction();
			break;
		case QDropEvent::Link:
			str = "Link";
			e->acceptAction();
			break;
		default:
			str = "Unknown";
	}*/

	/* Else : Drop object to item */

	QPixmap pixmap;
	if ( QImageDrag::decode(source, pixmap) )
		return createItemImage(pixmap, parent);

	QColor color;
	if ( KColorDrag::decode(source, color) )
		return createItemColor(color, parent);

//	if ( source->provides("text/uri-list") ) { // Because when we copy text or html from BasKet, URL lists are pasted (very strange because when drag the same, it's work)
	KURL::List urls;
	if ( KURLDrag::decode(source, urls) ) {
		int insertTo = parent->m_insertTo; // To past all URLS at the same place (and not one in the right place
		                                   //  and the others at start or end)
		int delta = insertTo > 0;          // Increment insertTo, unless we must drop at start (0) or end (-1)
		for ( KURL::List::iterator it = urls.begin(); it != urls.end(); ++it ) {
			item = createItemUrlOrLauncher(*it, parent);
			parent->m_insertTo = (insertTo += delta); // After last item was inserted, the counter is reseted : re-set it
		}
		return item;
	}
//	}

	if ( source->provides("text/html") ) { // TODO: Create a QHtmlDrag object (also xHTML ? )...
		QByteArray html = source->encodedData("text/html");
		return createItemHtml(html, parent);
	}

	QString text;
	if ( QTextDrag::decode(source, text) )
		return createItem(text, parent);

	/* Unsucceful drop */
	QString mimes = "";
	for(int i = 0; source->format(i); ++i)
		if ( *(source->format(i)) )
			mimes += "&nbsp;&nbsp;&nbsp;&nbsp;" + QString(source->format(i)) + "<br>";
	QString message = i18n("<qt>BasKet not currently support the data you've dropped."
	                       "<p>Please contact developer to tell him this lack, and join the following MIME type(s) :<br>"
	                       "%1"
	                       "<br>Thanks.").arg(mimes);
	KMessageBox::error(parent, message, i18n("Unsupported MIME type(s)"));
	return (Item*) 0;
}

QString ItemFactory::fileNameForNewItem(Basket *parent)
{
	QString fileName;
	QString itemNumber;
	QDir    dir;

	for (int i = 1; ; ++i) {
		itemNumber = QString::number(i);
		//itemNumber = itemNumber.rightJustify(5, '0');
		fileName = parent->savePath() + "item" + itemNumber;
		dir = QDir(fileName);
		if ( ! dir.exists(fileName) )
			break;
	}

	return "item" + itemNumber;
}

QFont ItemFactory::fontForFontType(int type)
{
	QFont font;

	switch (type) {
		case 1:  font.setFamily("SansSerif"); break;
		case 2:  font.setFamily("Serif");     break;
		case 3:  font.setFamily("Courier");   break;
		default: font.setFamily("System");           // case O or another illegal type...
	}

	return font;
}

QString ItemFactory::titleForURL(const KURL &url)
{
	return url.prettyURL(); // TODO: Use only the filename if isLocal, etc...
}

QString ItemFactory::iconForURL(const KURL &url)
{
	QString icon = KMimeType::iconForURL(url.url());
	if ( url.protocol() == "mailto" )
		icon = "message";
	return icon;
}
