/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastient.laout@tuxfamily.org                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qwidget.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <klocale.h>
#include <kmessagebox.h>

#include "basket.h"
#include "appsettings.h"
#include "onclickaction.h"

/** OnClickAction **/

// return : 0 : Copy to clipboard
//          1 : Copy to selection
//          2 : Copy to both
int OnClickAction::whichCopy(bool shiftPressed)
{
	int a = 1;

	switch (AppSettings::whereToCopy()) {
		case 0: a = (shiftPressed ? 1 : 0); break;
		case 1: a = (shiftPressed ? 2 : 0); break;
		case 2: a = (shiftPressed ? 0 : 1); break;
		case 3: a = (shiftPressed ? 2 : 1); break;
		case 4: a = (shiftPressed ? 0 : 2); break;
		case 5: a = (shiftPressed ? 1 : 2); break;
	}

	return a;
}

QString OnClickAction::whichCopyName(bool shiftPressed)
{
	int copy = whichCopy(shiftPressed);

	switch (copy) {
		case 0: return i18n("Copy to clipboard");
		case 1: return i18n("Copy to selection");
		case 2: return i18n("Copy to both");
	}

	return 0;
}

QString OnClickAction::hotkeyForAction(Action action, int policy, bool isRCOnClick, bool hasRC)
{
	QString s;

	// For copys :
	switch (policy) {
		case 0: s = i18n("click");      break;
		case 1: s = i18n("Ctrl+click"); break;
		case 2: s = i18n("Alt+click");  break;
	}

	switch (action) {
		case Copy:
			switch (AppSettings::whereToCopy()) {
				case 0: case 1:                              break;
				case 2: case 4: s = i18n("Shift+%1").arg(s); break;
				case 3: case 5: s = "";                      break;
			}
			break;
		case CopyToSel:
			switch (AppSettings::whereToCopy()) {
				case 2: case 3:                              break;
				case 0: case 5: s = i18n("Shift+%1").arg(s); break;
				case 1: case 4: s = "";                      break;
			}
			break;
		case CopyToBoth:
			switch (AppSettings::whereToCopy()) {
				case 1: case 3:                              break;
				case 4: case 5: s = i18n("Shift+%1").arg(s); break;
				case 0: case 2: s = "";                      break;
			}
			break;
		case Edit:
		case EditAnnot:
			switch (policy) {
				case 0: s = i18n("Ctrl+click"); break;
				case 1: s = i18n("click");      break;
				case 2: s = i18n("Ctrl+click"); break;
			}
			if (action == EditAnnot)
				s = i18n("Shift+%1").arg(s);
			break;
		case Open:
		case OpenWith:
			switch (policy) {
				case 0: s = i18n("Alt+click"); break;
				case 1: s = i18n("Alt+click"); break;
				case 2: s = i18n("click");     break;
			}
			if (action == OpenWith)
				s = i18n("Shift+%1").arg(s);
			break;
		case RunCommand:
			s = QString("\t%1").arg(isRCOnClick && hasRC ? i18n("click") : i18n("Ctrl+Alt+Click"));
			return s; // Because if s == "click" it will be set to "" in few lines of code =>
		case NoAction:
			break;    // No action to execute
	}

	if (s == "click" && isRCOnClick && hasRC)
		s = ""; // Because in this case RunCommand override click config

	if ( ! s.isEmpty() )
		return QString("\t%1").arg(s); // s == "\t"+s
	else
		return s;                      // s == ""
}

OnClickAction::Action OnClickAction::whichCopyAction(bool shiftPressed)
{
	if (shiftPressed) {
		switch (AppSettings::whereToCopy()) {
			case 2: case 4: return Copy;       break;
			case 0: case 5: return CopyToSel;  break;
			case 1: case 3: return CopyToBoth; break;
		}
	} else {
		switch (AppSettings::whereToCopy()) {
			case 0: case 1: return Copy;       break;
			case 2: case 3: return CopyToSel;  break;
			case 4: case 5: return CopyToBoth; break;
		}
	}

	return NoAction;
}

OnClickAction::Action OnClickAction::actionForButton(Qt::ButtonState state, int policy, bool isRCOnClick, bool hasRC)
{
	if ((state & Qt::ControlButton) && (state & Qt::AltButton)) {
		if (hasRC)
			return RunCommand;
		else
			return NoAction;
	} else if (state & Qt::ControlButton) {
		if (state & Qt::ShiftButton) {
			switch (policy) {
				case 0: return EditAnnot;              break;
				case 1: return whichCopyAction(true);  break;
				case 2: return EditAnnot;              break;
			}
		} else {
			switch (policy) {
				case 0: return Edit;                   break;
				case 1: return whichCopyAction(false); break;
				case 2: return Edit;                   break;
			}
		}
	} else if (state & Qt::AltButton) {
		if (state & Qt::ShiftButton) {
			switch (policy) {
				case 0: return OpenWith;               break;
				case 1: return OpenWith;               break;
				case 2: return whichCopyAction(true);  break;
			}
		} else {
			switch (policy) {
				case 0: return Open;                   break;
				case 1: return Open;                   break;
				case 2: return whichCopyAction(false); break;
			}
		}
	} else {                                  // Simple click without hotkey
		if (state & Qt::ShiftButton) {
			switch (policy) {
				case 0: return whichCopyAction(true);  break;
				case 1: return EditAnnot;              break;
				case 2: return OpenWith;               break;
			}
		} else if (isRCOnClick && hasRC)
			return RunCommand;
		else {
			switch (policy) {
				case 0: return whichCopyAction(false); break;
				case 1: return Edit;                   break;
				case 2: return Open;                   break;
			}
		}
	}

	return NoAction;
}

void OnClickAction::showHelp(QWidget *parent)
{
	KMessageBox::information( parent,
		// The table to display actions
		i18n("<table>"
		     "<tr><td></td>"                        "<td><b>Config 1</b></td>"  "<td><b>Config 2</b></td>"  "<td><b>Config 3</b></td></tr>"
		     "<tr><td><b>click</b></td>"            "<td>%1</td>"               "<td>Edit</td>"             "<td>Open</td></tr>"
		     "<tr><td><b>Shift+click</b></td>"      "<td>%2</td>"               "<td>Edit annotations</td>" "<td>Open with...</td></tr>"
		     "<tr><td><b>Ctrl+click</b></td>"       "<td>Edit</td>"             "<td>%3</td>"               "<td>Edit</td>"
		     "<tr><td><b>Shift+Ctrl+click</b></td>" "<td>Edit annotations</td>" "<td>%4</td>"               "<td>Edit annotations</td></tr>"
		     "<tr><td><b>Alt+click</b></td>"        "<td>Open</td>"             "<td>Open</td>"             "<td>%5</td></tr>"
		     "<tr><td><b>Shift+Alt+click</b></td>"  "<td>Open with...</td>"     "<td>Open with...</td>"     "<td>%6</td></tr>"
		     "<tr><td><b>Ctrl+Alt+click</b></td>"   "<td>Run command</td>"      "<td>Run command</td>"      "<td>Run command</td></tr>"
		     "</table>")
		.arg(whichCopyName(false)) .arg(whichCopyName(true)) // %1 %2
		.arg(whichCopyName(false)) .arg(whichCopyName(true)) // %3 %4
		.arg(whichCopyName(false)) .arg(whichCopyName(true)) // %5 %6
		// Because else, some long texts (such as "Edit annotations") will be displayed on two lines
		//  (too cluttered (not clear for the user) and not beautiful)
		.replace(' ', "&nbsp;")
		// Add some additional help and tips
		+ "<br>"
		+ i18n("<b>Config 1</b> is typically used for clip baskets to easy copy items with a simple click.<br>"
		       "<b>Config 2</b> is typically used for notes baskets to quickly edit notes with a simple click.<br>"
		       "<b>Config 3</b> is typically used for links for a more intuitive work : links are openned with a simple click.<br>"
		       "<br>"
		       "You can exchange copy to clipboard and to selection (or to both) by configuring Global on click copy."),
		// The message box title
		i18n("Available mouse click actions and configurations") );
}

/** OnClickActionCombo **/

OnClickActionCombo::OnClickActionCombo(int action, QWidget *parent, const char *name)
 : QComboBox(parent, name)
{
	insertItem(i18n("Click to copy, Ctrl+click to edit, Alt+click to open (Config 1)"));
	insertItem(i18n("Click to edit, Ctrl+click to copy, Alt+click to open (Config 2)"));
	insertItem(i18n("Click to open, Ctrl+click to edit, Alt+click to copy (Config 3)"));

	setCurrentItem(action);
}

OnClickActionCombo::~OnClickActionCombo()
{
}

/** OnClickActionCopyDialog **/

OnClickActionCopyDialog::OnClickActionCopyDialog(QWidget *parent)
 : KDialogBase(KDialogBase::Plain, i18n("Global on click copy configuration"),
               KDialogBase::Ok | KDialogBase::Apply | KDialogBase::Cancel, KDialogBase::Ok, parent)
{
	QFrame *frame = plainPage();

	QButtonGroup *grp = new QButtonGroup(6, Qt::Vertical, frame);
	grp->setFrameShape(QFrame::NoFrame);
	grp->setExclusive(true);
	new QRadioButton(i18n("Copy to &clipboard on click, and to selection if Shift is pressed during click"), grp);
	new QRadioButton(i18n("Copy to c&lipboard on click, and to both (clipboard and selection) if Shift is pressed during click"), grp);
	new QRadioButton(i18n("Copy to &selection on click, and to clipboard if Shift is pressed during click"), grp);
	new QRadioButton(i18n("Copy to s&election on click, and to both if Shift is pressed during click"), grp);
	new QRadioButton(i18n("Copy to &both on click, and to clipboard if Shift is pressed during click"), grp);
	new QRadioButton(i18n("Copy to bo&th on click, and to selection if Shift is pressed during click"), grp);

	m_choosen = AppSettings::whereToCopy();
	grp->setButton(m_choosen);

	connect( grp, SIGNAL(clicked(int)), this, SLOT(selectOption(int)) );

	QLabel *lbl = new QLabel(i18n(
		"Note that click can means \"click\", \"Ctrl+click\" or \"Alt+click\".\n"
		"Idem for Shift pressed : can means \"Shift+click\", \"Shift+Ctrl+click\" or \"Shift+Alt+click\".\n"
		"Those modifier keys can be configured for each baskets."), frame);

	QVBoxLayout *vLay = new QVBoxLayout(frame, 0, 6);
	vLay->addWidget(grp);
	vLay->addWidget(lbl);
}

OnClickActionCopyDialog::~OnClickActionCopyDialog()
{
}

void OnClickActionCopyDialog::selectOption(int id)
{
	m_choosen = id;
}

void OnClickActionCopyDialog::slotApply()
{
	KDialogBase::slotApply();
}

void OnClickActionCopyDialog::slotOk()
{
	AppSettings::setWhereToCopy(m_choosen);

	slotApply();
	KDialogBase::slotOk();
}

#include "onclickaction.moc"
