# Copyright 2001-2003 Hewlett-Packard Corporation
# This file contains all of the HP-unique subroutines
# $Id: HP_API.pm,v 1.111 2003/04/01 23:02:54 tyler_e Exp $

####################################################################
#
#  This module makes up the HP-UX specific API routines.
#  
####################################################################
#
#  Subroutine Listing:
#     &HP_ConfigureForDistro: adds all used file names to global
#                             hashes and generates a global IPD
#                             hash for SD modification lookup.
#
#     &getGlobalSwlist($):    Takes a fully qualified file name
#                             and returns product:filset info
#                             for that file.  returns undef if
#                             the file is not present in the IPD
#
#     &B_check_system:        Runs a series of system queries to
#                             determine if Bastille can be safely
#                             ran on the current system.
#
#     &B_swmodify($):         Takes a file name and runs the
#                             swmodify command on it so that the
#                             IPD is updated after changes
#
#     &B_System($$):          Takes a system command and the system
#                             command that should be used to revert
#                             whatever was done. Returns 1 on 
#                             success and 0 on failure
#
#     &B_load_ipf_rules($):   Loads a set of ipfrules into ipf, storing
#                             current rules for later reversion.
#
#     &B_Schedule($$):        Takes a pattern and a crontab line.
#                             Adds or replaces the crontab line to
#                             the crontab file, depending on if a 
#                             line matches the pattern
#
#     &B_ch_rc($$):           Takes a the rc.config.d flag name and
#                             new value as well as the init script
#                             location. This will stop a services
#                             and set the service so that it will
#                             not be restarted.
#
#     &B_set_value($$$):      Takes a param, value, and a filename
#                             and sets the given value in the file.
#                             Uses ch_rc, but could be rewritten using
#                             Bastille API calls to make it work on Linux
#
#     &B_TODO($):             Appends the give string to the TODO.txt
#                             file.
#
#     &B_chperm($$$$):        Takes new perm owner and group of given
#                             file.  TO BE DEPRECATED!!!
#
#     &B_install_jail($$):    Takes the jail name and the jail config
#                             script location for a give jail...
#                             These scripts can be found in the main
#                             directory e.g. jail.bind.hpux
#
#####################################################################

##############################################################################
#
#                     HP-UX Bastille directory structure
#
##############################################################################
#
#  /opt/sec_mgmt/bastille/bin/   -- location of Bastille binaries
#  /opt/sec_mgmt/bastille/lib/   -- location of Bastille modules
#  /opt/sec_mgmt/bastille/doc/   -- location of Bastille doc files
#
#  /etc/opt/sec_mgmt/bastille/   -- location of Bastille config files
#
#  /var/opt/sec_mgmt/bastille/log         -- location of Bastille log files
#  /var/opt/sec_mgmt/bastille/revert        -- directory holding all Bastille-
#                                            created revert scripts
#  /var/opt/sec_mgmt/bastille/revert/backup -- directory holding the original
#                                            files that Bastille modifies,
#                                            with permissions intact
#
##############################################################################


#####################################################################
# HP_ConfigureForDistro:  This subroutine is run when Bastille's
#    API is initiated.  HP-UX specific and configures Bastille's
#    GLOBAL_* Hashes for the rest of the implementation.
#####################################################################

sub HP_ConfigureForDistro {
    my $returnval=1;
    

    if (&getActualDistro =~ "^HP-UX11"){

	# First, let's make sure that we do not create any files or 
        # directories with more permissive permissions than we 
        # intend via setting the Perl umask
	umask(077);
	# Original "Prepare to Run" Global Variables

	$GLOBAL_BDIR{"log"} = "/var/opt/sec_mgmt/bastille/log";
	$GLOBAL_BFILE{"action-log"} = &getGlobal('BDIR', "log") . "/action-log";	
	$GLOBAL_BFILE{"error-log"} = &getGlobal('BDIR', "log") . "/error-log";
	$GLOBAL_BFILE{"debug-log"} = &getGlobal('BDIR', "log") . "/debug-log";

	$GLOBAL_BDIR{"revert"}= "/var/opt/sec_mgmt/bastille/revert";
	$GLOBAL_BDIR{"backup"}= &getGlobal('BDIR', "revert") . "/backup";
	$GLOBAL_BDIR{"config"} = "/etc/opt/sec_mgmt/bastille";
	$GLOBAL_BDIR{"home"}= "/etc/opt/sec_mgmt/bastille";
	$GLOBAL_BDIR{"share"}= "/etc/opt/sec_mgmt/bastille";
	$GLOBAL_BDIR{"jail"} = "/var/jail";
	$GLOBAL_BDIR{"apachejail"} = &getGlobal('BDIR',"jail") . "/apache";
	$GLOBAL_BDIR{"apachejail2"} = &getGlobal('BDIR',"jail") . "/apache2";
	$GLOBAL_BDIR{"hpws_apachejail"} = &getGlobal('BDIR',"jail") . "/hpws_apache";
	$GLOBAL_BDIR{"oldconfig"}= "/var/opt/sec_mgmt/bastille/oldconfig";

	$GLOBAL_BFILE{"config"} = &getGlobal('BDIR', "config") . "/config";
	$GLOBAL_BFILE{"credits"}="/opt/sec_mgmt/bastille/docs/Credits";
	$GLOBAL_BFILE{"sum.csv"}= &getGlobal('BDIR', "revert") . "/sum.csv";
	$GLOBAL_BFILE{"required_security_patches"}= "/var/opt/sec_mgmt/bastille/required_security_patches.txt";
	$GLOBAL_BFILE{"revert-actions"}= &getGlobal('BDIR', "revert") . "/revert-actions";
	$GLOBAL_BFILE{"revert-directory-perms.sh"}=&getGlobal('BDIR', "revert") . "/revert-directory-perms.sh";
	$GLOBAL_BFILE{"credits"}="/opt/sec_mgmt/bastille/docs/Credits";
	$GLOBAL_BFILE{"TODO"}= "/var/opt/sec_mgmt/bastille/TODO.txt";
	$GLOBAL_BFILE{"TOREVERT"}= "/var/opt/sec_mgmt/bastille/TOREVERT.txt";
	$GLOBAL_BFILE{"directory-perms.sh"} = "/var/opt/sec_mgmt/bastille/directory-perms.sh";
	$GLOBAL_BFILE{"suid-perms.sh"} = "/var/opt/sec_mgmt/bastille/suid-perms.sh";
	$GLOBAL_BFILE{"finish-named-chroot.sh"}="/var/opt/sec_mgmt/bastille/finish-named-chroot.sh";
	$GLOBAL_BFILE{"Xaccess.bastille"}="/etc/dt/config/Xaccess.bastille";
	$GLOBAL_BFILE{"Xconfig"}="/etc/dt/config/Xconfig";
	$GLOBAL_BFILE{"jail.generic.hpux"}= &getGlobal('BDIR','config') . "/jail.generic.hpux";
	$GLOBAL_BFILE{"jail.bind.hpux"}= &getGlobal('BDIR','config') . "/jail.bind.hpux";
	$GLOBAL_BFILE{"jail.bind9.hpux"}= &getGlobal('BDIR','config') . "/jail.bind9.hpux";
	$GLOBAL_BFILE{"nodisclaimer"}=    &getGlobal('BDIR', "home") . "/.nodisclaimer";
	$GLOBAL_BFILE{"BastilleBackEnd"} = "/opt/sec_mgmt/bastille/bin/bastille -b";
	$GLOBAL_BFILE{'incomplete.xbm'} = "/opt/sec_mgmt/bastille/lib/incomplete.xbm";
	$GLOBAL_BFILE{'complete.xbm'} = "/opt/sec_mgmt/bastille/lib/complete.xbm";
	$GLOBAL_BFILE{'lockfile'} = "/var/opt/sec_mgmt/bastille/bastille-lock";
	$GLOBAL_BFILE{'last.config'} = "/var/opt/sec_mgmt/bastille/last.config";
    }

    if (&GetDistro =~ "^HP-UX11"){

	# Set necessary binaries
	$GLOBAL_BIN{"at"}="/usr/bin/at";
	$GLOBAL_BIN{"auth.adm"}="/usr/sbin/auth.adm";
	$GLOBAL_BIN{"awk"}="/usr/bin/awk";
	$GLOBAL_BIN{"bdf"}="/usr/bin/bdf";	
	$GLOBAL_BIN{"cat"}="/usr/bin/cat";
	$GLOBAL_BIN{"chsh"}="/usr/bin/chsh";
	$GLOBAL_BIN{"ch_rc"}="/usr/sbin/ch_rc";
	$GLOBAL_BIN{"cksum"}="/usr/bin/cksum";
	$GLOBAL_BIN{"clear"}="/usr/bin/clear";
	$GLOBAL_BIN{"crontab"}="/usr/bin/crontab";
	$GLOBAL_BIN{"cut"}="/usr/bin/cut";
	$GLOBAL_BIN{"dtlogin.rc"}="/sbin/init.d/dtlogin.rc";
	$GLOBAL_BIN{"dump"}="/usr/sbin/dump";
	$GLOBAL_BIN{"echo"}="/usr/bin/echo";
	$GLOBAL_BIN{"false"}="/usr/bin/false";
	$GLOBAL_BIN{"getprdef"}="/usr/lbin/getprdef";
	$GLOBAL_BIN{"getprpw"}="/usr/lbin/getprpw";
	$GLOBAL_BIN{"groups"}="/usr/bin/groups";
	$GLOBAL_BIN{"head"}="/usr/bin/head";
	$GLOBAL_BIN{"inetd"}="/usr/sbin/inetd";
	$GLOBAL_BIN{"ipf"}="/sbin/ipf";
	$GLOBAL_BIN{"ipfstat"}="/sbin/ipfstat";
	$GLOBAL_BIN{"lpalt"}="/usr/bin/lpalt";
	$GLOBAL_BIN{"mail"}="/usr/bin/mailx";
	$GLOBAL_BIN{"more"}="/usr/bin/more";
	$GLOBAL_BIN{"passwd"}="/usr/bin/passwd";
	$GLOBAL_BIN{"ping"}="/usr/sbin/ping";
	$GLOBAL_BIN{"rcp"}="/usr/bin/rcp";
	$GLOBAL_BIN{"rdist"}="/usr/bin/rdist";
	$GLOBAL_BIN{"restore"}="/sbin/restore";
	$GLOBAL_BIN{"rexec"}="/usr/bin/rexec";
	$GLOBAL_BIN{"rexecd"}="/usr/lbin/rexecd";
	$GLOBAL_BIN{"rlogin"}="/usr/bin/rlogin";
	$GLOBAL_BIN{"rlogind"}="/usr/lbin/rlogind";
	$GLOBAL_BIN{"rsh"}="/usr/bin/remsh";
	$GLOBAL_BIN{"rshd"}="/usr/lbin/remshd";
	$GLOBAL_BIN{"sh"}="/usr/bin/sh";
	$GLOBAL_BIN{"tail"}="/usr/bin/tail";
	$GLOBAL_BIN{"traceroute"}="/usr/contrib/bin/traceroute";
	$GLOBAL_BIN{"uname"}="/usr/bin/uname";
        $GLOBAL_BIN{"chmod"}="/usr/bin/chmod";
        $GLOBAL_BIN{"chown"}="/usr/bin/chown";
        $GLOBAL_BIN{"chgrp"}="/usr/bin/chgrp";
        $GLOBAL_BIN{"cp"}="/usr/bin/cp";
	$GLOBAL_BIN{"hpws_httpd"} = "/opt/hpws/apache/bin/httpd";
	$GLOBAL_BIN{"diff"}="/usr/bin/diff";
	$GLOBAL_BIN{"grep"}="/usr/bin/grep";
        $GLOBAL_BIN{"groupadd"}="/usr/sbin/groupadd";
        $GLOBAL_BIN{"groupdel"}="/usr/sbin/groupdel";
        $GLOBAL_BIN{"killall"}="/usr/sbin/killall";
        $GLOBAL_BIN{"kctune"}="/usr/sbin/kctune";
        $GLOBAL_BIN{"kmtune"}="/usr/sbin/kmtune";
	$GLOBAL_BIN{"ln"}="/usr/bin/ln";
        $GLOBAL_BIN{"lpq"}="/usr/bin/lpstat";
        $GLOBAL_BIN{"lpr"}="/usr/bin/lp";
        $GLOBAL_BIN{"lprm"}="/usr/bin/cancel";
	$GLOBAL_BIN{"mk_kernel"}="/usr/sbin/mk_kernel";
        $GLOBAL_BIN{"mknod"}="/usr/sbin/mknod";
        $GLOBAL_BIN{"mkfifo"}="/usr/bin/mkfifo";
        $GLOBAL_BIN{"modprdef"}="/usr/lbin/modprdef";
        $GLOBAL_BIN{"modprpw"}="/usr/lbin/modprpw";
        $GLOBAL_BIN{"mount"}="/usr/sbin/mount";
        $GLOBAL_BIN{"mv"}="/usr/bin/mv";	
        $GLOBAL_BIN{"named-xfer"}="/usr/sbin/named-xfer";
        $GLOBAL_BIN{"ndd"}="/usr/bin/ndd";
        $GLOBAL_BIN{"ps"}="/usr/bin/ps";
        $GLOBAL_BIN{"pwconv"}="/usr/sbin/pwconv";
        $GLOBAL_BIN{"pwget"}="/usr/bin/pwget";
        $GLOBAL_BIN{"pwunconv"}="/usr/sbin/pwunconv";
        $GLOBAL_BIN{"swacl"}="/usr/sbin/swacl";
        $GLOBAL_BIN{"swagentd"}="/usr/sbin/swagentd";
	$GLOBAL_BIN{"swlist"}="/usr/sbin/swlist";
	$GLOBAL_BIN{"swmodify"}="/usr/sbin/swmodify";
	$GLOBAL_BIN{"syslogd"}="/usr/sbin/syslogd";	
        $GLOBAL_BIN{"tsconvert"}="/usr/lbin/tsconvert";
        $GLOBAL_BIN{"umask"}="umask";
        $GLOBAL_BIN{"umount"}="/usr/sbin/umount";
        $GLOBAL_BIN{"useradd"}="/usr/sbin/useradd";
        $GLOBAL_BIN{"userdel"}="/usr/sbin/userdel";
	$GLOBAL_BIN{"rm"}="/usr/bin/rm";
	$GLOBAL_BIN{"rmdir"}="/usr/bin/rmdir";
	$GLOBAL_BIN{"sendmail"}="/usr/sbin/sendmail";
	$GLOBAL_BIN{"sleep"}="/usr/bin/sleep";
        $GLOBAL_BIN{"touch"}="/usr/bin/touch";
        $GLOBAL_BIN{"what"}="/usr/bin/what";
	$GLOBAL_BIN{"who"}="/usr/bin/who";
 	
	# Directories
	$GLOBAL_DIR{"home"}="/home";
	$GLOBAL_DIR{"initd"}="/sbin/init.d";
	$GLOBAL_DIR{"dlkm_kernel_build"}="/stand/build/dlkm.vmunix_test";
        $GLOBAL_DIR{"rcd"}="/sbin";
	$GLOBAL_DIR{"sbin"}="/sbin";
	$GLOBAL_DIR{"share"}= "/etc/opt/sec_mgmt/bastille";
        $GLOBAL_DIR{"xinetd.d"}="";
	$GLOBAL_DIR{"rc.config.d"}="/etc/rc.config.d";

	#
	# Config/misc files
	#
	$GLOBAL_FILE{"apachectl"}="/opt/apache/bin/apachectl";
	$GLOBAL_FILE{"apachectl2"}="/opt/hpapache2/bin/apachectl";
	$GLOBAL_FILE{"auto_master"}="/etc/auto_master";
	$GLOBAL_FILE{"chkconfig_acct"}="/sbin/init.d/acct";
	$GLOBAL_FILE{"chkconfig_auditing"}="/sbin/init.d/auditing";
	$GLOBAL_FILE{"chkconfig_comsec"}="/sbin/init.d/comsec";
	$GLOBAL_FILE{"chkconfig_lpd"}="/sbin/init.d/lp";
	$GLOBAL_FILE{"chkconfig_named"}="/sbin/init.d/named";
 	$GLOBAL_FILE{"chkconfig_nfs"}="/sbin/init.d/nfs.server";
	$GLOBAL_FILE{"chkconfig_nfs_client"}="/sbin/init.d/nfs.client";
 	$GLOBAL_FILE{"chkconfig_nfs_server"}="/sbin/init.d/nfs.server";
  	$GLOBAL_FILE{"chkconfig_nfs_core"}="/sbin/init.d/nfs.server";
 	$GLOBAL_FILE{"chkconfig_nisp_client"}="/sbin/init.d/nisplus.client";
	$GLOBAL_FILE{"chkconfig_nisp_server"}="/sbin/init.d/nisplus.server";
	$GLOBAL_FILE{"chkconfig_nis_client"}="/sbin/init.d/nis.client";
	$GLOBAL_FILE{"chkconfig_nis_server"}="/sbin/init.d/nis.server";
	$GLOBAL_FILE{"chkconfig_pty"}="/sbin/init.d/ptydaemon";
	$GLOBAL_FILE{"chkconfig_pwgrd"}="/sbin/init.d/pwgr";
	$GLOBAL_FILE{"chkconfig_rbootd"}="/sbin/init.d/rbootd";
	$GLOBAL_FILE{"chkconfig_snmpd"}="/sbin/init.d/SnmpMaster";
	$GLOBAL_FILE{"chkconfig_snmpd_HP"}="/sbin/init.d/SnmpHpunix";
	$GLOBAL_FILE{"chkconfig_snmpd_Master"}="/sbin/init.d/SnmpMaster";
	$GLOBAL_FILE{"chkconfig_snmpd_Mib2"}="/sbin/init.d/SnmpMib2";
	$GLOBAL_FILE{"chkconfig_snmpd_Trp"}="/sbin/init.d/SnmpTrpDst";
	$GLOBAL_FILE{"chkconfig_vt"}="/sbin/init.d/vt";
	$GLOBAL_FILE{"chkconfig_ypbind"}="/sbin/init.d/nis.client";
	$GLOBAL_FILE{"csh.login"}="/etc/csh.login";
	$GLOBAL_FILE{"chroot_os_cp.sh"}="/opt/apache/bin/chroot_os_cp.sh";
	$GLOBAL_FILE{"chroot_os_cp.sh2"}="/opt/hpapache2/util/chroot_os_cp.sh";
	$GLOBAL_FILE{"customipfrules"}="/etc/opt/sec_mgmt/bastille/ipf.customrules";
	$GLOBAL_FILE{"devassign"}="/tcb/files/devassign";
	$GLOBAL_FILE{"ftpaccess"}="/etc/ftpd/ftpaccess";
	$GLOBAL_FILE{"ftpusers"}="/etc/ftpd/ftpusers";
	$GLOBAL_FILE{"group"}="/etc/group";
	$GLOBAL_FILE{"hpws_chroot_os_cp.sh"}="/opt/hpws/apache/util/chroot_os_cp.sh";
	$GLOBAL_FILE{"hpws_httpd.conf"}="/opt/hpws/apache/conf/httpd.conf";
	$GLOBAL_FILE{"hpws_apachectl"}="/opt/hpws/apache/bin/apachectl";
	$GLOBAL_FILE{"idsagent"}="/opt/ids/bin/idsagent";
	$GLOBAL_FILE{"inetd.conf"}="/etc/inetd.conf";
	$GLOBAL_FILE{"inittab"}="/etc/inittab";
	$GLOBAL_FILE{"ipf.conf"}="/etc/opt/ipf/ipf.conf";
	$GLOBAL_FILE{"issue"}="/etc/issue";
	$GLOBAL_FILE{"kernel_params"}="/stand/system";
	$GLOBAL_FILE{"lpr"}="/usr/bin/lpr";
	$GLOBAL_FILE{"motd"}="/etc/motd";
        $GLOBAL_FILE{"mtab"}="/etc/mnttab";
	$GLOBAL_FILE{"named"}="/usr/sbin/named";
	$GLOBAL_FILE{"named.conf"}="/etc/named.conf";
	$GLOBAL_FILE{"named.pid"}="/var/run/named.pid";
	$GLOBAL_FILE{"nddconf"}="/etc/rc.config.d/nddconf";
	$GLOBAL_FILE{"namesvrs"}="/etc/rc.config.d/namesvrs";
	$GLOBAL_FILE{"nsswitch.conf"}="/etc/nsswitch.conf";
	$GLOBAL_FILE{"pam_access.conf"}="/etc/pam.conf";
	$GLOBAL_FILE{"profile"}="/etc/profile";
	$GLOBAL_FILE{"Questions"} = &getGlobal('BDIR', "share") . "/Questions.txt";
	$GLOBAL_FILE{"root_cron"}="/var/spool/cron/crontabs/root";
	$GLOBAL_FILE{"root_cronaids"}="/var/spool/cron/.cronaids/root";
	$GLOBAL_FILE{"rootprofile"} = "/.profile";
	$GLOBAL_FILE{"rsh"}="/usr/bin/remsh";
	$GLOBAL_FILE{"securetty"}="/etc/securetty";
        $GLOBAL_FILE{"security"}="/etc/default/security";
	$GLOBAL_FILE{"sendmail.cf"}="/etc/mail/sendmail.cf";
	$GLOBAL_FILE{"shadow"}="/etc/shadow";
	$GLOBAL_FILE{"shells"}="/etc/shells";
	$GLOBAL_FILE{"spc"}="/opt/sec_mgmt/spc/bin/security_patch_check";
	$GLOBAL_FILE{"sshd"}="/opt/ssh/sbin/sshd";
	$GLOBAL_FILE{"swinstall_lock"}="/var/adm/sw/products/swlock";
	$GLOBAL_FILE{"swmodify_lock"}="/var/spool/sw/catalog/swlock";
	$GLOBAL_FILE{"sysconfig_sendmail"}="/sbin/init.d/sendmail";
	$GLOBAL_FILE{"sysconfig_cron"}="/sbin/init.d/cron";
	$GLOBAL_FILE{"cimserverd"}="/opt/wbem/sbin/cimserverd";
	$GLOBAL_FILE{"webadmlogin"}="/opt/hpwebadmin/htdocs/common/login.cgi";
	$GLOBAL_FILE{'Xconfig'}="/usr/dt/config/Xconfig";
	$GLOBAL_FILE{"ypserv"}="/usr/lib/netsvc/yp/ypserv";
	$GLOBAL_FILE{"zprofile"}="/etc/zprofile";

	$GLOBAL_FILE{"httpd.conf"}="/opt/apache/conf/httpd.conf";
	$GLOBAL_FILE{"httpd"}="/opt/apache/bin/httpd";
	$GLOBAL_FILE{"httpd_access.conf"}="/opt/apache/conf/access.conf";

	$GLOBAL_FILE{"httpd.conf2"}="/opt/hpapache2/conf/httpd.conf";
	$GLOBAL_FILE{"httpd2"}="/opt/hpapache2/bin/httpd";
	$GLOBAL_FILE{"httpd_access.conf2"}="/opt/hpapache2/conf/access.conf";

        #####################################################################
        # SERVICES
        #####################################################################

        $GLOBAL_SERVICE{'apache'}=['APACHE2_START'];          # Default: 1
        $GLOBAL_SERVTYPE{'apache'}="rc";
        $GLOBAL_PROCESS{'apache'}=['/opt/hpapache2/bin/httpd'];
	
        $GLOBAL_SERVICE{'hpws_apache'}=['HPWS_APACHE_START'];  # Default: 0
        $GLOBAL_SERVTYPE{'hpws_apache'}="rc";
        $GLOBAL_PROCESS{'hpws_apache'}=['/opt/hpws/apache/bin/httpd'];
	

        $GLOBAL_SERVICE{'asian'}=['RUN_EUSRV'];               # Default: 0
        $GLOBAL_SERVTYPE{'asian'}="rc";
        $GLOBAL_PROCESS{'asian'}=['eusrv'];

        $GLOBAL_SERVICE{'audio'}=['AUDIO_SERVER'];            # Default: 0
        $GLOBAL_SERVTYPE{'audio'}="rc";
        $GLOBAL_PROCESS{'audio'}=['Aserver'];

        $GLOBAL_SERVICE{'auditing'}=['AUDITING'];            # Default: 0
        $GLOBAL_SERVTYPE{'auditing'}="rc";
        $GLOBAL_PROCESS{'auditing'}=['audomon','audevent'];

        $GLOBAL_SERVICE{'bootps'}=['bootps'];
        $GLOBAL_SERVTYPE{'bootps'}="inet";
        $GLOBAL_PROCESS{'bootps'}=['bootpd'];

        $GLOBAL_SERVICE{'builtin'}=['daytime', 'chargen', 'echo', 'discard'];
        $GLOBAL_SERVTYPE{'builtin'}="inet";
        $GLOBAL_PROCESS{'builtin'}=[];

        $GLOBAL_SERVICE{'cron'}=['CRON'];                     # Default: 1
        $GLOBAL_SERVTYPE{'cron'}="rc";
        $GLOBAL_PROCESS{'cron'}=['cron'];

#        $GLOBAL_SERVICE{'diagnostic'}=['DIAGNOSTICS'];        # Default: 1
#        $GLOBAL_SERVTYPE{'diagnostic'}="rc";
#        $GLOBAL_PROCESS{'diagnostic'}=['diagmond','diaglogd'];

        $GLOBAL_SERVICE{'dmi'}=['START_DMI'];                 # Default: 1
        $GLOBAL_SERVTYPE{'dmi'}="rc";
        $GLOBAL_PROCESS{'dmi'}=['dmisp', 'hpuxci', 'swci', 'sdci'];

        $GLOBAL_SERVICE{'dttools'}=['dtspc','[^#].*ttdbserver','[^#].*cmsd'];
        $GLOBAL_SERVTYPE{'dttools'}="inet";
        $GLOBAL_PROCESS{'dttools'}=['dtspcd','rpc.ttdbserver','rpc.cmsd'];

        # The "EG Convert daemon"
        $GLOBAL_SERVICE{'egcd'}=['EGCD'];                     # Default: 1
        $GLOBAL_SERVTYPE{'egcd'}="rc";
        $GLOBAL_PROCESS{'egcd'}=['egcd'];

        $GLOBAL_SERVICE{'ems'}= ['AUTOSTART_EMSAGT'];         # Default: 1
        $GLOBAL_SERVTYPE{'ems' }="rc";
        $GLOBAL_PROCESS{'ems' }=['emsagent'];

        $GLOBAL_SERVICE{'finger'}=['finger'];
        $GLOBAL_SERVTYPE{'finger'}="inet";
        $GLOBAL_PROCESS{'finger'}=['fingerd'];

        $GLOBAL_SERVICE{'ftp'}=['ftp'];
        $GLOBAL_SERVTYPE{'ftp'}="inet";
        $GLOBAL_PROCESS{'ftp'}=['ftpd'];

        $GLOBAL_SERVICE{'ident'}=['ident','auth'];
        $GLOBAL_SERVTYPE{'ident'}="inet";
        $GLOBAL_PROCESS{'ident'}=['identd'];

        # kerberized rtools
        $GLOBAL_SERVICE{'ktools'}=['kshell', 'klogin'];
        $GLOBAL_SERVTYPE{'ktools'}="inet";
        $GLOBAL_PROCESS{'ktools'}=['remshd -K','rlogind -K'];

        $GLOBAL_SERVICE{'licenseserver'}=['START_I4LMD'];     # Default: 0
        $GLOBAL_SERVTYPE{'licenseserver'}="rc";

        $GLOBAL_SERVICE{'lp'}=['LP'];                         # Default: 1
        $GLOBAL_SERVTYPE{'lp'}="rc";
        $GLOBAL_PROCESS{'lp'}=['lpsched'];

        $GLOBAL_SERVICE{'measureware'}=['MWA_START'];         # Default: 1
        $GLOBAL_SERVTYPE{'measureware'}="rc";
        $GLOBAL_PROCESS{'measureware'}=['perflbd', 'alarmgen'];

        $GLOBAL_SERVICE{'mountd'}=['START_MOUNTD'];           # Default: 1
        $GLOBAL_SERVTYPE{'mountd'}="rc";
        $GLOBAL_PROCESS{'mountd'}=[];

        $GLOBAL_SERVICE{'named'}=['NAMED'];                   # Default: 0
        $GLOBAL_SERVTYPE{'named'}="rc";
        $GLOBAL_PROCESS{'named'}=['named'];

        $GLOBAL_SERVICE{'nfs_automount'}=['AUTOMOUNT'];       # Default: 1
        $GLOBAL_SERVTYPE{'nfs_automount'}="rc";
        $GLOBAL_PROCESS{'nfs_automount'}=[];

        $GLOBAL_SERVICE{'nfs_client'}=['NFS_CLIENT'];         # Default: 1
        $GLOBAL_SERVTYPE{'nfs_client'}="rc";
        $GLOBAL_PROCESS{'nfs_client'}=['biod'];

        $GLOBAL_SERVICE{'nfs_server'}=['NFS_SERVER'];         # Default: 1
        $GLOBAL_SERVTYPE{'nfs_server'}="rc";
        $GLOBAL_PROCESS{'nfs_server'}=['nfsd'];

        $GLOBAL_SERVICE{'nis_client'}=['NIS_CLIENT'];         # Default: 1
        $GLOBAL_SERVTYPE{'nis_client'}="rc";
        $GLOBAL_PROCESS{'nis_client'}=['ypbind'];

        $GLOBAL_SERVICE{'nis_server'}=['NIS_MASTER_SERVER', 'NIS_SLAVE_SERVER'];   # Default: 1
        $GLOBAL_SERVTYPE{'nis_server'}="rc";
        $GLOBAL_PROCESS{'nis_server'}=['ypserv', 'ypserv'];

        $GLOBAL_SERVICE{'nisp_client'}=['NISPLUS_CLIENT'];    # Default: 0
        $GLOBAL_SERVTYPE{'nisp_client'}=['NISPLUS_CLIENT'];
        $GLOBAL_PROCESS{'nisp_client'}=['NISPLUS_CLIENT'];

        $GLOBAL_SERVICE{'nisp_server'}=['NISPLUS_SERVER'];    # Default: 0
        $GLOBAL_SERVTYPE{'nisp_server'}=['NISPLUS_SERVER'];
        $GLOBAL_PROCESS{'nisp_server'}=['NISPLUS_SERVER'];

        $GLOBAL_SERVICE{'pcnfs_server'}=['PCNFS_SERVER'];     # Default: 0
        $GLOBAL_SERVTYPE{'pcnfs_server'}="rc";
        $GLOBAL_PROCESS{'pcnfs_server'}=[];

        $GLOBAL_SERVTYPE{'printer'}="inet";
        $GLOBAL_SERVICE{'printer'}=['printer'];
        $GLOBAL_PROCESS{'printer'}=['rlpdaemon -i'];

        $GLOBAL_SERVICE{'ptydaemon'}=['PTYDAEMON_START'];     # Default: 1
        $GLOBAL_SERVTYPE{'ptydaemon'}="rc";
        $GLOBAL_PROCESS{'ptydaemon'}=['ptydaemon'];

        $GLOBAL_SERVICE{'pwgr'}=['PWGR'];                     # Default: 1
        $GLOBAL_SERVTYPE{'pwgr'}="rc";
        $GLOBAL_PROCESS{'pwgr'}=['pwgrd'];

        $GLOBAL_SERVICE{'rbootd'}=['START_RBOOTD'];           # Default: 1
        $GLOBAL_SERVTYPE{'rbootd'}="rc";
        $GLOBAL_PROCESS{'rbootd'}=['rbootd'];

        $GLOBAL_SERVICE{'recserv'}=['recserv'];
        $GLOBAL_SERVTYPE{'recserv'}="inet";
        $GLOBAL_PROCESS{'recserv'}=['recserv'];

        $GLOBAL_SERVICE{'registrar'}=['registrar'];
        $GLOBAL_SERVTYPE{'registrar'}="inet";
        $GLOBAL_PROCESS{'registrar'}=['registrar'];

        #$GLOBAL_SERVICE{'reversearp'}=['RARP'];               # Default: 0

        #$GLOBAL_SERVICE{'routerdiscover'}=['RDPD'];           # Default: 0

        $GLOBAL_SERVICE{'rpcd'}=['START_RPCD'];               # Default: 1
        $GLOBAL_SERVTYPE{'rpcd'}="rc";
        $GLOBAL_PROCESS{'rpcd'}=['rpcd'];

        $GLOBAL_SERVICE{'rtools'}=['login', 'shell', 'exec'];
        $GLOBAL_SERVTYPE{'rtools'}="inet";
        $GLOBAL_PROCESS{'rtools'}=['rlogind','remshd','rexecd'];

        $GLOBAL_SERVICE{'samba'}=['RUN_SAMBA'];               # Default: 0
        $GLOBAL_SERVTYPE{'samba'}="rc";
        $GLOBAL_PROCESS{'samba'}=['smbd','nmbd'];

        $GLOBAL_SERVICE{'samd'}=['samd'];
        $GLOBAL_SERVTYPE{'samd'}="inittab";
        $GLOBAL_PROCESS{'samd'}=['samd'];

        $GLOBAL_SERVICE{'scrdaemon'}=['SCR_DAEMON'];          # Default: 1
        $GLOBAL_SERVTYPE{'scrdaemon'}="rc";
        $GLOBAL_PROCESS{'scrdaemon'}=['scrdaemon'];

        $GLOBAL_SERVICE{'screen'}=['SLSD_DAEMON'];            # Default: 0

        $GLOBAL_SERVICE{'sendmail'}=['SENDMAIL_SERVER'];      # Default: 1
        $GLOBAL_SERVTYPE{'sendmail'}="rc";
        $GLOBAL_PROCESS{'sendmail'}=['sendmail'];

        $GLOBAL_SERVICE{'snmpd'}=['SNMP_HPUNIX_START', 
                                  'SNMP_MASTER_START', 
                                  'SNMP_MIB2_START', 
                                  'SNMP_TRAPDEST_START'];      # Defaults: 1 
        $GLOBAL_SERVTYPE{'snmpd'}="rc";
        $GLOBAL_PROCESS{'snmpd'}=['snmpdm'];

        $GLOBAL_SERVICE{'sshd'}=['SSHD_START'];               # Default: 1
        $GLOBAL_SERVTYPE{'sshd'}="rc";
        $GLOBAL_PROCESS{'sshd'}=['sshd'];

        $GLOBAL_SERVICE{'swat'}=['swat'];
        $GLOBAL_SERVTYPE{'swat'}="inet";
        $GLOBAL_PROCESS{'swat'}=['swat'];

        $GLOBAL_SERVICE{'syncer'}=['SYNCER'];                 # Default: 1
        $GLOBAL_SERVTYPE{'syncer'}="rc";
        $GLOBAL_PROCESS{'syncer'}=['syncer'];

        $GLOBAL_SERVICE{'talk'}=['ntalk'];
        $GLOBAL_SERVTYPE{'talk'}="inet";
        $GLOBAL_PROCESS{'talk'}=['ntalkd'];

        $GLOBAL_SERVICE{'telnet'}=['telnet'];
        $GLOBAL_SERVTYPE{'telnet'}="inet";
        $GLOBAL_PROCESS{'telnet'}=['telnetd'];

        $GLOBAL_SERVICE{'tftp'}=['tftp'];
        $GLOBAL_SERVTYPE{'tftp'}="inet";
        $GLOBAL_PROCESS{'tftp'}=['tftpd'];

        $GLOBAL_SERVICE{'time'}=['time'];
        $GLOBAL_SERVTYPE{'time'}="inet";
        $GLOBAL_PROCESS{'time'}=[];

        #$GLOBAL_SERVICE{'tomcat'}=['TOMCAT2_START'];          # Default: 0

        $GLOBAL_SERVICE{'uucp'}=['uucp'];
        $GLOBAL_SERVTYPE{'uucp'}="inet";
        $GLOBAL_PROCESS{'uucp'}=['uucpd'];

        $GLOBAL_SERVICE{'vjed'}=['VJED'];                     # Default: 1
        $GLOBAL_SERVTYPE{'vjed'}="rc";
        $GLOBAL_PROCESS{'vjed'}=['vjed'];

        $GLOBAL_SERVICE{'vt'}=['VTDAEMON_START'];             # Default: 1
        $GLOBAL_SERVTYPE{'vt'}="rc";
        $GLOBAL_PROCESS{'vt'}=['vtdaemon'];

        $GLOBAL_SERVICE{'webadmin'}=['WEBADMIN'];             # Default: 0
        $GLOBAL_SERVTYPE{'webadmin'}="rc";
        # can't distinguish this from a normal web server, so don't try
        $GLOBAL_PROCESS{'webadmin'}=[];

        $GLOBAL_SERVICE{'webmin'}=['WEBMIN2_START'];          # Default: 0
        $GLOBAL_SERVTYPE{'webmin'}="rc";
        $GLOBAL_PROCESS{'webmin'}=['miniserv.pl'];

        $GLOBAL_SERVICE{'xfs'}=['RUN_X_FONT_SERVER'];         # Default: 0
        $GLOBAL_SERVTYPE{'xfs'}="rc";
        $GLOBAL_PROCESS{'xfs'}=['xfs'];

    } # End if GLOBAL_DISTRO starts with "HP-UX11"
    else {
       $returnval=0;
    }

    return $returnval;
} # End Subroutine

####################################################################    
# &getGlobalSwlist ($file);
#   This function returns the product and fileset information for 
#   a given file or directory if it exists in the IPD otherwise
#   it returns undefined "undef"
#
#   uses $GLOBAL_SWLIST{"$FILE"}
####################################################################
sub getGlobalSwlist($){

    my $file = $_[0];


    if(! defined %GLOBAL_SWLIST) {
	# Generating swlist database for swmodify changes that will be required
	# The database will be a hash of fully qualified file names that reference
	# the files product name and fileset.  These values are required to use
	# swmodify...
	# temp variable to keep swlist command /usr/sbin/swlist
	my $swlist = &getGlobal('BIN',"swlist");
	# listing of each directory and file that was installed by SD on the target machine
	my @fileList = `${swlist} -l file`;
	# listing of each patch and the patches that supersede each.
	# hash which is indexed by patch.fileset on the system
	my %patchSuperseded;
	   
	my @patchList = `${swlist} -l fileset -a superseded_by *.*,c=patch 2>&1`;
	# check to see if any patches are present on the system
	if(($? >> 8) == 0) {

	    # determining patch suppression for swmodify.
	    foreach my $patchState (@patchList) {
		# removing empty lines and commented lines.
		if($patchState !~ /^\s*\#/ && $patchState !~ /^\s*$/) {
		    
		    # removing leading white space
		    $patchState =~ s/^\s+//;
		    my @patches = split /\s+/, $patchState;
		    if($#patches == 0){
			# patch is not superseded
			$patchSuperseded{$patches[0]} = 0;
		    }
		    else {
			# patch is superseded
			$patchSuperseded{$patches[0]} = 1;
		    }
		}
	    }
	}
	else {
	    &ActionLog("No patches found on the system.\n");
	}

	if($#fileList >= 0){
	    # foreach line of swlist output
	    foreach my $fileEntry ( @fileList ){
		#filter out commented portions
		if( $fileEntry !~ /^\s*\#/ ){
		    chomp $fileEntry;
		    # split the output into two fields filename and product.fileset
		    my ($productInfo,$file) = split /: /, $fileEntry;
		    $productInfo =~ s/\s+//;
		    $file =~ s/\s+//;
		    # if the product is a patch
		    if($productInfo =~ /PH(CO|KL|NE|SS)/){
			# if the patch is not superseded by another patch
			if($patchSuperseded{$productInfo} == 0){
			    # add the patch to the list of owner for this file
			    push @{$GLOBAL_SWLIST{"$file"}}, $productInfo;
			}
		    }
		    # not a patch.
		    else {
			# add the product to the list of owners for this file
			push @{$GLOBAL_SWLIST{"$file"}}, $productInfo;
		    }

		}
	    }
	}
	else{
	    # defining GLOBAL_SWLIST in error state.
	    $GLOBAL_SWLIST{"ERROR"} = "ERROR";
	    &ErrorLog("ERROR:   Could not execute swlist.  Swmodifys will not be attempted");
	}
    }

    if(exists $GLOBAL_SWLIST{"$file"}){
	return $GLOBAL_SWLIST{"$file"};
    }
    else {
	return undef;
    }
}

###################################################################
#  &B_check_system;
#    This subroutine is called to validate that bastille may be
#    safely run on the current system.  It will check to insure
#    that there is enough file system space, mounts are rw, nfs
#    mounts are not mounted noroot, and swinstall, swremove and 
#    swmodify are not running
#
#    uses ErrorLog
#
##################################################################
sub B_check_system {
    # exitFlag is one if a conflict with the successful execution
    # of bastille is found.
    my $exitFlag = 0;

    # first check for swinstall, swmodify, or swremove processes
    my $ps = &getGlobal('BIN',"ps") . " -el";
    my @processTable = `$ps`;
    foreach my $process (@processTable) {
	if($process =~ /swinstall/ ) {
	    &ErrorLog("Bastille cannot run while a swinstall is in progress.\n" . 
		      "Complete the swinstall operation and then run Bastille.\n\n");
	    $exitFlag = 1;
	}

	if($process =~ /swremove/ ) {
	    &ErrorLog("Bastille cannot run while a swremove is in progress.\n" .
		      "Complete the swremove operation and then run Bastille.\n\n");
	    $exitFlag = 1;
	}
	
	if($process =~ /swmodify/ ) {
	    &ErrorLog("Bastille cannot run while a swmodify is in progress.\n" .
		      "Complete the swmodify operation and then run Bastille.\n\n");
	    $exitFlag = 1;
	}
	
    }

    # check for root read only mounts for /var /usr /opt /etc /tmp /stand /
    # Bastille is required to make changes to these file systems.
    my $mount = &getGlobal('BIN',"mount");
    my $rm = &getGlobal('BIN',"rm");
    my $touch = &getGlobal('BIN',"touch");

    my @mnttab = `$mount`;

    foreach my $record (@mnttab) {
	my @fields = split /\s+/, $record;

	my $mountPoint = $fields[0];
	my $mountType =  $fields[2];
	my $mountOptions = $fields[3];
	if($mountPoint =~ /^\/$|^\/etc\/$|^\/tmp$|^\/opt\/$|^\/stand\/$|^\/var\/$/) {
	    
	    if($mountOptions =~ /^ro,|,ro,|,ro$/) {
		&ErrorLog("$mountPoint is mounted read-only.  Bastille needs to make\n" . 
			  "modifications to this file system.  Please remount\n" . 
			  "$mountPoint read-write and then run Bastille again.\n\n");
		$exitFlag = 1;
	    }
	    # looking for an nfs mounted file system
	    if($mountType =~/.+:\//){
		`$touch $mountPoint/.bastille 1>/dev/null 2>&1`;
		if(! -e "$mountPoint/.bastille") {
		    &ErrorLog("$mountPoint is an nfs mounted file system that does\n" .
			      "not allow root to write to.  Bastille needs to make\n" . 
			      "modifications to this file system.  Please remount\n" . 
			      "$mountPoint giving root access and then run Bastille\n" . 
			      "again.\n\n");
		    $exitFlag = 1;
		}
		`$rm -f $mountPoint/.bastille 1>/dev/null 2>&1`;
	    }
	    
	}
	
    }

    # checks for enough disk space in directories that bastille writes to.
    my $bdf = &getGlobal('BIN',"bdf");
    #directories that bastille writes to => required space in kilobytes.
    my %bastilleDirs = ( "/etc/opt/sec_mgmt/bastille" => "4", "/var/opt/sec_mgmt/bastille"=> "1000");
    for my $directory (sort keys %bastilleDirs) {
	my @diskUsage = `$bdf $directory`;

	# removing bdf header line from usage information.
	shift @diskUsage;

	$diskUsage[0] =~ s/^\s+//;

	my @fields = split /\s+/, $diskUsage[0];
	my $mountPoint = $fields[5];
	my $diskAvail = $fields[3];

	if($diskAvail <= $bastilleDirs{"$directory"}) {
	    &ErrorLog("$mountPoint does not contain enough available space\n" . 
		      "for bastille to run properly.  $directory needs\n" . 
		      "at least $bastilleDirs{$directory} kilobytes of space.  Please clear at least\n" . 
		      "that amount of space from $mountPoint and run Bastille again.\n" . 
		      "Current Free Space available = ${diskAvail} k\n\n");
	    $exitFlag = 1;
	}
    }

    
    # check to make sure that we are in at least run level 2 before we attempt to run
    $who = &getGlobal('BIN', "who") . " -r";
    $levelInfo = `$who`;
    chomp $levelInfo;
    @runlevel = split /\s+/, $levelInfo;
    if ($runlevel[3] < 2 ) {
	&ErrorLog("Bastille requires a run-level of 2 or more to run properly.\n" . 
		  "Please move your system to a higher run level and then\n" .
		  "run 'bastille -b'.\n" .
		  "Current run-level is '$runlevel[3]'.\n");
	$exitFlag=1;
    }
    else {
	&ActionLog("System run-level is $runlevel[3]\n");
    }


    if($exitFlag) { 
	exit(1);
    }

}

###################################################################
#  &B_swmodify($file);
#    This subroutine is called after a file is modified.  It will
#    redefine the file in the IPD with it's new properties.  If 
#    the file is not in the IPD it does nothing.
#
#    uses B_System to make the swmodifications.
##################################################################
sub B_swmodify($){
    my $file = $_[0];
    if(defined &getGlobalSwlist($file)){
	my $swmodify = &getGlobal('BIN',"swmodify");
	my @productsInfo = @{&getGlobalSwlist($file)};
	# running swmodify on files that were altered by this function but
	# were created and maintained by SD
	foreach my $productInfo (@productsInfo) {
	    &B_System("$swmodify -x files='$file' $productInfo",
		      "$swmodify -x files='$file' $productInfo");
	}
    }
}

####################################################################
#  &B_System($command,$revertcommand);
#    This function executes a command, then places the associated 
#    revert command in revert file. It takes two parameters, the
#    command and the command that reverts that command.
#
#   uses ActionLog and ErrorLog for logging purposes.
###################################################################
sub B_System ($$) {
    local ($command,$revertcmd)=@_;  # changed scoping so eval below can read it
# Log only flag is set.
#    if($GLOBAL_LOG_ONLY) {
# in a log only situation we will send intended commands
# to the action-log
#	&ActionLog("LOG ONLY (no action taken):\n" .
#		   " Would Execute:\n" .
#		   "   $command\n" . 
#		   " Would Add Revert Action:\n" .
#		   "   $revertcmd\n");
	# returning success as if the command executed normally
#	return 1;
#    }
    
    local $SIG{'ALRM'} = sub {  die "timeout" }; # This subroutine exits the "eval" below.  The program 
    # can then move on to the next operation.  Used "local"
    # to avoid name space collision with disclaim alarm.
    local $WAIT_TIME=120; # Wait X seconds for system commands
    eval{
	alarm($WAIT_TIME); # start a time-out for command to complete.  Some commands hang, and we want to 
	                   # fail gracefully.  When we call "die" it exits this eval statement 
	                   # with a value we use below
	my $commandOutput = `$command 2>&1`; # run the command and gather its output
	my $commandRetVal = ($? >> 8);  # find the commands return value
	if ($commandRetVal == 0) { 
	    &ActionLog("Executed Command: " . $command . "\n");
	    &ActionLog("Command Output: " . $commandOutput . "\n");	
	    &B_revert_log ("$revertcmd \n");
	    die "success";
	} else { 
	    &ActionLog("WARNING: Failed to Execute Command: " . $command . "\n");
	    &ActionLog("         Command Output: " . $commandOutput . "\n");	
	    &ErrorLog("WARNING: Failed to Execute Command: " . $command . "\n");
	    &ErrorLog("         Command Output: " . $commandOutput . "\n");
	    die "failure";
	};  
    };  

    my $exitcode=$@;

    alarm(0);  # End of the timed operation
    
    if ($exitcode) {  # The eval command above will exit with one of the 3 values below
	if ($exitcode =~ /timeout/) {
	    &ActionLog("WARNING: No response received from $command after $WAIT_TIME seconds.\n");
	    &ActionLog("         Command Output: " . $commandOutput . "\n");	
	    &ErrorLog("WARNING: No response received from $command after $WAIT_TIME seconds.\n");
	    &ErrorLog("         Command Output: " . $commandOutput . "\n");	
	    return undef;
	} elsif ($exitcode =~ /success/) {
	    return 1;
	} elsif ($exitcode =~ /failure/) {
	    return 0;
	} else {
	    &ErrorLog("ERROR:   Unexpected return state from command execution: $command\n");
	    &ErrorLog("Command Output: " . $commandOutput . "\n");	
	    exit 1;
	}
    }
}
####################################################################
#  &B_load_ipf_rules($ipfruleset);
#    This function enables an ipfruleset.  It's a little more
#    specific than most API functions, but necessary because
#    ipf doesn't return correct exit codes (syntax error results
#    in a 0 exit code)
#
#   uses ActionLog and ErrorLog to log
#   calls crontab directly (to list and to read in new jobs)
###################################################################
sub B_load_ipf_rules ($) {
   my $ipfruleset=$_[0];

   &ActionLog("# sub B_load_ipf_rules");

   # TODO: grab ipf.conf dynamically from the rc.config.d files
   my $ipfconf = &getGlobal('FILE','ipf.conf');

   # file system changes - these are straightforward, and the API
   # will take care of the revert
   &B_create_file($ipfconf);
   &B_blank_file($ipfconf, 'a$b');
   &B_append_line($ipfconf, 'a$b', $ipfruleset);

   # runtime changes

   # define binaries
   my $ipf = &getGlobal('BIN', 'ipf');
   my $ipfstat = &getGlobal('BIN', 'ipfstat');
   my $grep = &getGlobal('BIN', 'grep');

   # create backup rules
   # This will exit with a non-zero exit code because of the grep
   my @oldrules = `$ipfstat -io 2>&1 | $grep -v empty`;

   my @errors=`$ipf -I -Fa -f $ipfconf 2>&1`;

   if(($? >> 8) == 0) {

      &B_set_rc("IPF_START","1");
      &B_set_rc("IPF_CONF","$ipfconf");

      # swap the rules in
      &B_System("$ipf -s","$ipf -s");

      # now create a "here" document with the previous version of
      # the rules and put it into the revert-actions script
      &B_revert_log("$ipf -I -Fa -f - <<EOF\n@{oldrules}EOF");

      if (@errors) {
        &ErrorLog("ERROR:  ipfilter produced the following errors when\n" .
                  "        loading $ipfconf.  You probably had an invalid\n" .
                  "        rule in ". &getGlobal('FILE','customipfrules') ."\n".
                  "@errors\n");
      }

   } else {
     &ErrorLog("ERROR:  Unable to run $ipf\n");
   }

}



####################################################################
#  &B_Schedule($pattern,$cronjob);
#    This function schedules a cronjob.  If $pattern exists in the
#    crontab file, that job will be replaced.  Otherwise, the job
#    will be appended.
#
#   uses ActionLog and ErrorLog to log
#   calls crontab directly (to list and to read in new jobs)
###################################################################
sub B_Schedule ($$) {
   my ($pattern,$cronjob)=@_;
   $cronjob .= "\n";

   &ActionLog("# sub B_Schedule");
   my $crontab = &getGlobal('BIN','crontab');

   my @oldjobs = `$crontab -l 2>/dev/null`;
   my @newjobs;
   my $patternfound=0;

   foreach my $oldjob (@oldjobs) {
      if ($oldjob =~ m/$pattern/ ) {
        push @newjobs, $cronjob;
        $patternfound=1;
        &ActionLog("changing existing cron job which matches $pattern with\n" .
                   "$cronjob");
      } else {
        &ActionLog("keeping existing cron job $oldjob");
        push @newjobs, $oldjob;
      }
   }

   unless ($patternfound) {
     &ActionLog("adding cron job\n$cronjob\n");
     push @newjobs, $cronjob;
   }

   if(open(CRONTAB, "|$crontab - 2> /dev/null")) {
     print CRONTAB "@newjobs";

     # now create a "here" document with the previous version of
     # the crontab file and put it into the revert-actions script
     &B_revert_log("$crontab <<EOF\n@{oldjobs}EOF");
     close CRONTAB;
   } 

   # Now check to make sure it happened, since cron will exit happily 
   # (retval 0) with no changes if there are any syntax errors
   my @editedjobs = `$crontab -l 2>/dev/null`;

   if (@editedjobs ne @newjobs) {
     &ErrorLog("ERROR:   failed to add cron job:\n$cronjob\n" .
               "         You probably had an invalid crontab file to start with.");
   }

}


#This function turns off a service.  The first parameter is the parameter that 
#controls the operation of the service, the second parameter is the script that
#turns on and off the service at boot-time.
sub B_ch_rc($$) { 

    my ($ch_rc_parameter, $startup_script)=@_;

    if (&GetDistro != "^HP-UX") {
       &ErrorLog("ERROR:   Tried to call ch_rc $ch_rc_parameter on a non-HP-UX\n".
                 "         system!  Internal Bastille error.");
       return undef;
    }
    my $configfile="";
    my $command = &getGlobal('BIN', 'ch_rc');
    my $orig_value = &B_get_rc($ch_rc_parameter);

    if ( $orig_value !~ "1" ) { #If param is not already 1, the "stop" script won't work
	&B_System (&getGlobal('BIN',"ch_rc") . " -a -p $ch_rc_parameter=1"," ");
    }elsif ($orig_value eq "" ) { #If param is not initialized in a file, this section looks for file(s) 
	                          #that mentions that parameter (like a comment), it then explicitly tells
	                          #ch_rc to use those files by setting the configfile variable
	                          #We could have just grabbed the first one, but that could lead to a false
	                          #sense of security if we got the wrong one.  Other files will ignore addition
	my $filecommand=&getGlobal('BIN','grep')." -l $ch_rc_parameter ".&getGlobal('DIR','rc.config.d')."/*";
	$configfile=`$filecommand`;
 	chomp $configfile;
	$configfile =~ s/\n/ /g; #grep returns \n's, but ch_rc expects files to be separated with spaces
    }	
    &B_System ($startup_script  . " stop", #stop service, then restart if the user runs bastille -r
	       $startup_script . " start");  

    # set parameter, so that service will stay off after reboots
    &B_System (&getGlobal('BIN',"ch_rc") . " -a -p $ch_rc_parameter=0 $configfile" , 
	       &getGlobal('BIN',"ch_rc") . " -a -p $ch_rc_parameter=$orig_value");
}


# This routine sets a value in a given file
sub B_set_value($$$) { 
    my ($param, $value, $file)=@_;
    
    if (! -e $file ) {
	&B_create_file("$file");
    }

    # If a value is already set to something other than $value then reset it.
    &B_replace_line($file,"^$param\\s*=\\s*","$param=$value\n");
    # If the value is not already set to something then set it.
    &B_append_line($file,"^$param\\s*=\\s*$value","$param=$value\n");

}


# This function adds something to the To Do List.  It takes one argument, the string you want
# to add to the To Do List.
sub B_TODO ($) {
    my $text = $_[0];
    my $multilineString = "";

    # trim off any leading and trailing new lines.
    $text =~ s/^\n+(.*)/$1/;
    $text =~ s/(.*)\n+$/$1/;

    if ( ! -e &getGlobal('BFILE',"TODO") ) {
	# Make the TODO list file for HP-UX Distro
	&B_create_file(&getGlobal('BFILE', "TODO"));
	&B_append_line(&getGlobal('BFILE', "TODO"),'a$b',
          "Please take the steps below to make your system more secure,\n".
          "then delete the item from this file and record what you did along\n".
          "with the date and time in your system administration log.  You\n".
          "will need that information in case you ever need to revert your\n".
          "changes.\n\n");
    }


    if (open(TODO,"<" . &getGlobal('BFILE', "TODO"))) {
	while (my $line = <TODO>) {
	    # getting rid of all meta characters.
	    $line =~ s/(\\|\||\(|\)|\[|\]|\{|\}|\^|\$|\*|\+|\?|\.)//g;
	    $multilineString .= $line;
	}
	chomp $multilineString;
        $multilineString .= "\n";
	
	close(TODO);
    }
    else {
	&ErrorLog("Unable to read TODO.txt file.\n" . 
		  "The following text could not be appended to the TODO list:\n" . 
		  $text .
		  "End of TODO text\n");
    }

    my $textPattern = $text;

    # getting rid of all meta characters.
    $textPattern =~ s/(\\|\||\(|\)|\[|\]|\{|\}|\^|\$|\*|\+|\?|\.)//g;

    if( $multilineString !~  "$textPattern") {
	my $datestamp = "{" . localtime() . "}";
	unless ( &B_append_line(&getGlobal('BFILE', "TODO"), "", $datestamp . "\n" . $text . "\n\n\n") ) {
	    &ActionLog("TODO Failed for text: " . $text );
	    &ErrorLog("TODO Failed for text: " . $text );
	}
    }

}

##################################################################################
# &B_chperm($owner,$group,$mode,$filename(s))
#   This function changes ownership and mode of a list of files. Takes four
#   arguments first the owner next the group and third the new mode in oct and
#   last a list of files that the permissions changes should take affect on.
#
#   uses: &swmodify and &B_revert_log
##################################################################################
sub B_chperm($$$$) {
    my ($newown, $newgrp, $newmode, $file_expr) = @_;
    my @files = glob($file_expr);
    
    my $return = 1;
    
    foreach my $file (@files){
	my @filestat = stat $file;
	my $oldmode = (($filestat[2]/512) % 8) .
	    (($filestat[2]/64) % 8) .
		(($filestat[2]/8) % 8) .
		    (($filestat[2]) % 8);

	if((chown $newown, $newgrp, $file) != 1 ){
	    &ErrorLog("ERROR:   Could not change ownership of $file to $newown:$newgrp\n");
	    $return = 0;
	}
	else{
	    &ActionLog("Changed ownership of $file to $newown:$newgrp\n");
	    # swmodifying file if possible...
	    &B_swmodify($file);
	    &B_revert_log(&getGlobal('BIN',"chown") . " $filestat[4]:$filestat[5] $file\n");
	}
	
        $newmode_formatted=sprintf "%5lo",$newmode;

	if((chmod $newmode, $file) != 1){
	    &ErrorLog("ERROR:   Could not change mode of $file to $newmode_formatted\n");
	    $return = 0;
	}
	else{
	    &ActionLog("Changed mode of $file to $newmode_formatted\n");
	    &B_revert_log(&getGlobal('BIN',"chmod") . " $oldmode $file\n");
	}   

		      
    }
    return $return;
}

############################################################################
# &B_install_jail($jailname, $jailconfigfile);
# This function takes two arguments ( jail_name, jail_config )
# It's purpose is to take read in config files that define a 
# chroot jail and then generate it bases on that specification
############################################################################
sub B_install_jail($$) {

    my $jailName = $_[0];  # Name of the jail e.g bind
    my $jailConfig = $_[1]; # Name of the jails configuration file
    # create the root directory of the jail if it does not exist
    &B_create_dir( &getGlobal('BDIR','jail'));
    &B_chperm(0,0,0555,&getGlobal('BDIR','jail'));

    # create the Jail dir if it does not exist
    &B_create_dir( &getGlobal('BDIR','jail') . "/" . $jailName);
    &B_chperm(0,0,0555,&getGlobal('BDIR','jail') . "/". $jailName);
    
    
    my $jailPath = &getGlobal('BDIR','jail') . "/" . $jailName;
    my @lines; # used to store no commented no empty config file lines
    # open configuration file for desired jail and parse in commands
    if(open(JAILCONFIG,"< $jailConfig")) {
	while(my $line=<JAILCONFIG>){
	    if($line !~ /^\s*\#|^\s*$/){
		chomp $line;
		push(@lines,$line);
	    }
	}
        close JAILCONFIG;
    }
    else{
	&ErrorLog("Open Failed on filename: $jailConfig\n");
	return 0;
    }
    # read through commands and execute
    foreach my $line (@lines){
        &ActionLog("Install jail: $line\n");
	my @confCmd = split /\s+/,$line;
	if($confCmd[0] =~ /dir/){ # if the command say to add a directory
	    if($#confCmd == 4) { # checking dir Cmd form
		if(! (-d  $jailPath . "/" . $confCmd[1])){
		    #add a directory and change its permissions according 
                    #to the conf file
		    &B_create_dir( $jailPath . "/" . $confCmd[1]);
                    &B_chperm((getpwnam($confCmd[3]))[2],
                              (getgrnam($confCmd[4]))[2],
                               oct($confCmd[2]),
                               $jailPath . "/" . $confCmd[1]);
		}
	    }
	    else {
		&ErrorLog("Badly Formed Configuration Line\n");
		&ErrorLog("\tERROR: $line\n");
	    }
	}       
	elsif($confCmd[0] =~ /file/) {
	    if($#confCmd == 5) { # checking file cmd form
		if(&B_cp($confCmd[1],$jailPath . "/" . $confCmd[2])){ 
		    # for copy command cp file and change perms
		    &B_chperm($confCmd[4],$confCmd[5],oct($confCmd[3]),$jailPath . "/" . $confCmd[2]);
		}
		else {
		    &ErrorLog("Could not complete copy on specified files\n");
		    &ErrorLog("\tERROR: $line\n");
		}
	    }
	    else {
		&ErrorLog("Badly Formed Configuration Line\n");
		&ErrorLog("\tERROR: $line\n");
	    }
	}           
	elsif($confCmd[0] =~ /slink/) {
	    if($#confCmd == 2) { # checking file cmd form
		if(!(-e $jailPath . "/" . $confCmd[2])){
		    #for symlink command create the symlink
		    &B_symlink($jailPath . "/" . $confCmd[1], $confCmd[2]);
		}
	    }
	    else {
		&ErrorLog("Badly Formed Configuration Line\n");
		&ErrorLog("\tERROR: $line\n");
	    }
	}	    
	else {
	    &ErrorLog("Unrecognized Configuration Line\n");
	    &ErrorLog("\tERROR: $line\n");
	}
    }
    return 1;
}



###########################################################################
#  &B_list_processes($service)                                            #
#                                                                         #
#  This subroutine uses the GLOBAL_PROCESS hash to determine if a         # 
#  service's corresponding processes are running on the system.           #    
#  If any of the processes are found to be running then the process       #
#  name(s) is/are returned by this subroutine in the form of an list      #
#  If none of the processes that correspond to the service are running    #
#  then an empty list is returned.                                        #
###########################################################################
sub B_list_processes($) {

    # service name
    my $service = $_[0];
    # list of processes related to the service
    my @processes=@{ &getGlobal('PROCESS',$service)};

    # current systems process information
    my $ps = &getGlobal('BIN',"ps");
    my $psTable = `$ps -elf`;
    
    # the list to be returned from the function
    my @running_processes;

    # for every process associated with the service
    foreach my $process (@processes) {
	# if the process is in the process table then 
	if($psTable =~ m/$process/) {
	    # add the process to the list, which will be returned
	    push @running_processes, $process;
	}

    }
    
    # return the list of running processes
    return @running_processes;

}


################################################################################
#  &B_deactivate_inetd_service($service);                                      #
#                                                                              #
#  This subroutine will disable all inetd services associated with the input   #
#  service name.  Service name must be a reference to the following hashes     #
#  GLOBAL_SERVICE GLOBAL_SERVTYPE and GLOBAL_PROCESSES.  If processes are left #
#  running it will note these services in the TODO list as well as instruct the#
#  user in how they remaining processes can be disabled.                       #
################################################################################
sub B_deactivate_inetd_service($) {
    my $service = $_[0];
    my $servtype = &getGlobal('SERVTYPE',"$service");
    my $inetd_conf = &getGlobal('FILE',"inetd.conf");

    # check the service type to ensure that it can be configured by this subroutine.
    if($servtype ne 'inet') {
	&ActionLog("The service \"$service\" is not an inet service so it cannot be\n" .
		   "configured by this subroutine\n");
	return 0;
    }

    # check for the inetd configuration files existence so it may be configured by
    # this subroutine.
    if(! -e $inetd_conf ) {
	&ActionLog("The file \"$inetd_conf\" cannot be located.\n" .
		   "Unable to configure inetd\n");
	return 0;
    }

    # list of service identifiers present in inetd.conf file.
    my @inetd_entries = @{ &getGlobal('SERVICE',"$service") };
    
    foreach my $inetd_entry (@inetd_entries) {
	&B_hash_comment_line($inetd_conf, "^\\s*$inetd_entry");
    }

    # list of processes associated with this service which are still running
    # on the system
    my @running_processes = &B_list_processes($service);

    if($#running_processes >= 0) {
        my $todoString = "\n" . 
	                 "---------------------------------------\n" . 
	                 "Deactivating Inetd Service: $service\n" . 
			 "---------------------------------------\n" . 
			 "The following process(es) are associated with the inetd service \"$service\".\n" .
			 "They are most likely associated with a session which was initiated prior to\n" . 
			 "running Bastille.  To disable a process see \"kill(1)\" man pages or reboot\n" .
			 "the system\n" .
			 "Active Processes:\n" .
			 "###################################\n";
	foreach my $running_process (@running_processes) {
	    $todoString .= "\t$running_process\n";
	}
	$todoString .= 	 "###################################\n";
	
	&B_TODO($todoString);
    }

}


################################################################################
# B_get_rc($key);                                                              #
#                                                                              #
#  This subroutine will use the ch_rc binary to get rc.config.d variables      #
#  values properly escaped and quoted.                                         #
################################################################################
sub B_get_rc($) {
    my $key=$_[0];
    my $ch_rc = &getGlobal('BIN',"ch_rc");

    # get the current value of the given parameter.
    my $currentValue=`$ch_rc -l -p $key`;
    chomp $currentValue;
    
    if(($? >> 8) == 0 ) {
	# escape all meta characters.
	$currentValue =~ s/([\"\`\$\\])/\\$1/g;
	$currentValue = '"' . $currentValue . '"';

    }
    else {
	return undef;
    }

    return $currentValue;
}



################################################################################
# B_set_rc($key,$value);                                                       #
#                                                                              #
#  This subroutine will use the ch_rc binary to set rc.config.d variables.  As #
#  well as setting the variable this subroutine will set revert strings.       #
#                                                                              #
################################################################################
sub B_set_rc($$) {
    
    my ($key,$value)=@_;
    my $ch_rc = &getGlobal('BIN',"ch_rc");

    # get the current value of the given parameter.
    my $currentValue=&B_get_rc($key);

    if(defined $currentValue ) {

	if ( &B_System("$ch_rc -a -p $key=$value",
		       "$ch_rc -a -p $key=$currentValue") ) {
	    #ch_rc success
	    return 1;
	} 
	else {
	    #ch_rc failure.
	    return 0;
	}
    }
    else {
	&ErrorLog("ERROR:   ch_rc was unable to lookup $key\n");
	return 0;
    }

}

1;
