// Copyright (C) 2000 Open Source Telecom Corporation.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <server.h>
#include "bayonne_rpc.h"
#include <cstring>

#ifdef	CCXX_NAMESPACES
using namespace ost;
using namespace std;
#endif

inline char* safe_strdup( const char* pcSource ) { return strdup( pcSource != NULL ? pcSource : "" ); }

bool_t bayonne_command_1_svc(bayonne_command *cmd, bayonne_error *result, struct svc_req *svc)
{
	Module *mod = getModule(MODULE_FIFO, cmd->mod_name);
	if(!mod)
	{
		*result = BAYONNE_INVALID_MODULE;
		return TRUE;
	}

	if(!mod->command(cmd->argv.argv_val, NULL))
		*result = BAYONNE_FAILURE;
	
	*result = BAYONNE_SUCCESS;
	return TRUE;
}

bool_t bayonne_module_1_svc(char **name, bayonne_error *result, struct svc_req *svc)
{

	*result = BAYONNE_SUCCESS;
	if(!getModule(MODULE_ANY, *name))
		*result = BAYONNE_INVALID_MODULE;

	return TRUE;
}

bool_t bayonne_reserve_1_svc(bayonne_reserve *res, bayonne_error *error, struct svc_req *svc)
{
	Mixer *mix = driver->getMixer(res->conf_mixer);
	Conference *cf;
	int group;

	*error = BAYONNE_SUCCESS;
	if(!mix)
	{
		*error = BAYONNE_INVALID_MIXER;
		return TRUE;
	}

	if(!mix->setMixer(res->conf_groups, res->conf_alloc))
	{
		*error = BAYONNE_FAILURE;
		return TRUE;
	}		
	if(!res->conf_groups)
		return TRUE;

	for(group = 0; group < mix->getGroups(); ++group)
	{
		cf = mix->getConference(group);
		if(!cf)
		{
			*error = BAYONNE_INVALID_CONFERENCE;
			return TRUE;
		}
		if(group < res->conf_groups)
			cf->setConference(res->conf_limits.conf_limits_val[group]);
		else
			cf->setConference(0);
	}
	return TRUE;
}

bool_t bayonne_conference_1_svc(int *id, bayonne_conf *conf, struct svc_req *svc)
{
	Conference *cf = driver->getConference(*id);
	memset(conf, 0, sizeof(bayonne_conf));
	if(cf)
	{
		conf->count = cf->getMembers();
		conf->limit = cf->getLimit();
		conf->members.members_len = conf->count;
		conf->members.members_val = cf->getMembership();
	}
	return TRUE;
}

bool_t bayonne_mixer_1_svc(int *id, bayonne_mixer *mixer, struct svc_req *svc)
{
	bayonne_conf* pbcArg;
	Mixer *mix = driver->getMixer(*id);
	Conference *cf;
	int groups;

	memset(mixer, 0, sizeof(mixer));
	if(mix)
	{
		groups = mix->getGroups();
		mixer->mixer_avail = mix->getAvail();
		mixer->mixer_used = mix->getMembers();
		mixer->mixer_groups = groups;
		mixer->mixer_conf.mixer_conf_len = groups;
		pbcArg = new bayonne_conf[ groups ];
		mixer->mixer_conf.mixer_conf_val = pbcArg;
		for ( int grp = 0; grp < groups; grp++ )
		{
			cf = mix->getConference(grp);
			if(!cf)
			{
				++grp;
				continue;
			}
			pbcArg[grp].count = cf->getMembers();
			pbcArg[grp].limit = cf->getLimit();
			pbcArg[grp].members.members_len = pbcArg[grp].count;
			pbcArg[grp].members.members_val = cf->getMembership();
		} // End of for
	} // End of if
	return TRUE;
}

bool_t bayonne_query_1_svc( void*, bayonne_info* info, struct svc_req* reqsvc )
{
	TrunkGroup *grp = getGroup("*");

	memset( reinterpret_cast<void*>( info ), 0, sizeof ( bayonne_info ) );
	if ( grp )
	 info->tgi_policy = safe_strdup( grp->getLast( "groups" ) );
	info->tgi_node = safe_strdup( keyserver.getNode( ) );
	info->tgi_user = safe_strdup( keyserver.getLast( "user" ) );
	info->tgi_token = safe_strdup( keyserver.getToken( ) );
	info->tgi_driver = safe_strdup( plugins.getDriverName( ) );
	info->tgi_version = safe_strdup( getenv( "SERVER_VERSION" ) );
	info->ports = driver->getTrunkCount();
	info->used = driver->getTrunkUsed();
	info->mixers = driver->getMixers();
	info->conferences = driver->getGroups();
	info->uid = keyserver.getUid();
	info->gid = keyserver.getGid();
	info->nodes = keyserver.getNodeCount();
	
	return TRUE;
}

bool_t bayonne_status_1_svc( void*, char** status, struct svc_req* req )
{
	driver->getStatus( *status);
	return TRUE;
}

bool_t bayonne_down_1_svc( void*, bayonne_error* pbeResult, struct svc_req* req )
{
	*pbeResult = BAYONNE_SUCCESS;

	raise(SIGINT);
	return TRUE;
}

bool_t bayonne_compile_1_svc( void*, bayonne_error* pbeResult, struct svc_req* req)
{
	*pbeResult = BAYONNE_SUCCESS;

	driver->getImage();
	return TRUE;
}

bool_t bayonne_schedule_1_svc( char** sched, bayonne_error* pbeResult, struct svc_req* req )
{
	*pbeResult = BAYONNE_SUCCESS;

	if(!stricmp(*sched, "*"))
		scheduler.altSchedule(NULL);
	else
		scheduler.altSchedule(*sched);
	return TRUE;
}	

bool_t bayonne_reload_1_svc( char** name, bayonne_error* pbeResult, struct svc_req* req )
{
	Module *mod;

	*pbeResult = BAYONNE_SUCCESS;
	
	mod = getModule(MODULE_FIFO, *name);
	if(mod)
		mod->reload();
	else
		*pbeResult = BAYONNE_INVALID_MODULE;

	return TRUE;
}

bool_t bayonne_network_1_svc( char** name, bayonne_nodes* nodes, struct svc_req* req )
{
	nodes->bayonne_nodes_len = 0;
	nodes->bayonne_nodes_val = (bayonne_node *)getNodes(*name);

	if(nodes->bayonne_nodes_val)
	{
		if(*name)
			nodes->bayonne_nodes_len = 1;
		else
			nodes->bayonne_nodes_len = keyserver.getNodeCount();
	}
	return TRUE;
}

bool_t bayonne_disconnect_1_svc( int* port, bayonne_error* pbeResult, struct svc_req* req )
{
	Trunk *trunk = driver->getTrunkPort(*port);
	TrunkEvent event;

	*pbeResult = BAYONNE_SUCCESS;
	if(!trunk)
	{
		*pbeResult = BAYONNE_INVALID_PORT;
		return TRUE;
	}

	event.id = TRUNK_STOP_DISCONNECT;
	if(!trunk->postEvent(&event))
		*pbeResult = BAYONNE_FAILURE;

	return TRUE;
}

bool_t bayonne_busy_1_svc( int* port, bayonne_error* pbeResult, struct svc_req* req )
{
	Trunk *trunk = driver->getTrunkPort(*port);
	TrunkEvent event;

	*pbeResult = BAYONNE_SUCCESS;
	if(!trunk)
	{
		*pbeResult = BAYONNE_INVALID_PORT;
		return TRUE;
	}

	event.id = TRUNK_MAKE_BUSY;
	if(!trunk->postEvent(&event))
		*pbeResult = BAYONNE_FAILURE;

	return TRUE;
}

bool_t bayonne_idle_1_svc( int* port, bayonne_error* pbeResult, struct svc_req* req )
{
	Trunk *trunk = driver->getTrunkPort(*port);
	TrunkEvent event;

	*pbeResult = BAYONNE_SUCCESS;
	if(!trunk)
	{
		*pbeResult = BAYONNE_INVALID_PORT;
		return TRUE;
	}

	event.id = TRUNK_MAKE_IDLE;
	if(!trunk->postEvent(&event))
		*pbeResult = BAYONNE_FAILURE;

	return TRUE;
}

bool_t bayonne_start_1_svc( bayonne_start* start, bayonne_error* pbeResult, struct svc_req* req )
{
	Trunk *trunk;
	TrunkEvent event;
	int port = driver->getTrunkCount();
	TrunkGroup *pol, *my = getGroup(start->policy);

	*pbeResult = BAYONNE_SUCCESS;
	if(start->port > -1)
	{
		if(!trunk)
		{	
			*pbeResult = BAYONNE_INVALID_PORT;
			return TRUE;
		}
		event.parm.argv = start->argv.argv_val;
		event.id = TRUNK_START_SCRIPT;
		if(!trunk->postEvent(&event))
			*pbeResult = BAYONNE_FAILURE;
		return TRUE;
	}	
	while(port)
	{
		--port;
		pol = driver->getTrunkGroup(port);
		if(!pol)
			continue;
		if(pol != my)
			continue;
		event.id = TRUNK_START_SCRIPT;
		event.parm.argv = start->argv.argv_val;
		if(trunk->postEvent(&event))
			return TRUE;
	}
	*pbeResult = BAYONNE_FAILURE;
	return TRUE;
}			

bool_t bayonne_ring_1_svc( int* port, bayonne_error* pbeResult, struct svc_req* req )
{
	const char *argv[] = {NULL};
	Trunk *trunk = driver->getTrunkPort(*port);
	TrunkEvent event;

	*pbeResult = BAYONNE_SUCCESS;
	if(!trunk)
	{
		*pbeResult = BAYONNE_INVALID_PORT;
		return TRUE;
	}

	// Doesn't matter as the argument is never changed anyway
	event.parm.argv = const_cast<char**>( argv );
	event.id = TRUNK_RING_START;
	if(!trunk->postEvent(&event))
		*pbeResult = BAYONNE_FAILURE;

	return TRUE;
}

bool_t bayonne_request_1_svc( bayonne_request* req, bayonne_error* pbeResult, struct svc_req* svc )
{
	TrunkGroup *pol = getGroup(req->req_policy);

	if(!pol)
	{
		*pbeResult = BAYONNE_INVALID_POLICY;
		return TRUE;
	}
	*pbeResult = BAYONNE_SUCCESS;
	request(pol, req->argv.argv_val, req->req_timeout);
	return TRUE;
}

bool_t bayonne_policy_1_svc( char** name, bayonne_policy* policy, struct svc_req* req )
{
	TrunkGroup*	pol = getGroup(*name);
	int count	= driver->getTrunkCount();

	memset( reinterpret_cast<void*>( policy ), 0, sizeof ( bayonne_policy ) );

	pol = getGroup(*name);
	if(!pol)
	 return TRUE;

	// Lets hope this is enough
	policy->pol_sched = new char[ BAYONNE_POLICY_SCHED_SZ ];
	pol->getSchedule( policy->pol_sched );

	policy->pol_name = safe_strdup( pol->getLast( "name" ) );
	policy->pol_number = safe_strdup( pol->getLast( "number" ) );

	policy->pol_members = 0;
	policy->pol_ports.pol_ports_len = 0;
	policy->pol_ports.pol_ports_val = new int[ count ];
	policy->pol_active = 0;
	policy->max_incoming = policy->max_outgoing = 0;
	policy->tot_incoming = policy->tot_outgoing = 0;


	for(int id = 0; id < count; ++id)
	{
		if(driver->getTrunkGroup(id) != pol)
			continue;

		policy->pol_ports.pol_ports_val[ policy->pol_members++ ] = id;
	} // End of for
	policy->pol_active = pol->getStat(STAT_ACTIVE_CALLS);
	policy->max_incoming = pol->getStat(STAT_MAX_INCOMING);
	policy->max_outgoing = pol->getStat(STAT_MAX_OUTGOING);
	policy->tot_incoming = pol->getStat(STAT_TOT_INCOMING);
	policy->tot_outgoing = pol->getStat(STAT_TOT_OUTGOING);
	policy->pol_ports.pol_ports_len = policy->pol_members;			
	return TRUE;
}

bool_t bayonne_port_1_svc( int* id, bayonne_port* port, struct svc_req* req )
{
	Trunk *trunk = driver->getTrunkPort(*id);
	
	memset( ( void *) port, 0, sizeof ( bayonne_port ) );
	if ( !trunk )
	 return TRUE;

	port->port_caller = safe_strdup( trunk->getSymbol( SYM_CALLER ) );
	port->port_dialed = safe_strdup( trunk->getSymbol( SYM_DIALED ) );
	port->port_name = safe_strdup( trunk->getSymbol( SYM_NAME ) );
	port->port_user = safe_strdup( trunk->getSymbol( SYM_USER ) );
	port->port_info = safe_strdup( trunk->getSymbol( SYM_INFODIGITS ) );
	port->port_lang = safe_strdup( trunk->getSymbol( SYM_LANGUAGE ) );
	port->port_gid = safe_strdup( trunk->getSymbol( SYM_GID ) );
	port->port_policy = safe_strdup( trunk->getSymbol( SYM_POLICY ) );
	port->port_caps = trunk->getCapabilities();
	port->port_duration = 0;
	if ( trunk->getSymbol( SYM_DURATION ) != NULL )
	{
		port->port_duration = atol( trunk->getSymbol( SYM_DURATION ) );
	} // End of if

	return TRUE;
}

int bayonne_program_1_freeresult( SVCXPRT* psxHandle, xdrproc_t xptTransform, caddr_t ctArg )
{ // Start of bayonne_program_1_freeresult( SVCXPRT*, xdrproc_t, caddr_t )
	bayonne_info*	pbiArg;
	bayonne_mixer*	pbmArg;
	bayonne_policy*	pbplArg;
	bayonne_port*	pbpoArg;

	if ( xptTransform == reinterpret_cast<xdrproc_t>( xdr_bayonne_mixer ) )
	{
		// Clean up results
		pbmArg = reinterpret_cast<bayonne_mixer*>( ctArg );
		delete [] pbmArg->mixer_conf.mixer_conf_val;

		// Paranoia
		pbmArg->mixer_conf.mixer_conf_len = 0;
	}
	else if ( xptTransform == reinterpret_cast<xdrproc_t>( xdr_bayonne_info ) )
	{
		// Unfortunately strdup( ) requires free
		pbiArg = reinterpret_cast<bayonne_info*>( ctArg );
		free( pbiArg->tgi_policy );
		free( pbiArg->tgi_node );
		free( pbiArg->tgi_user );
		free( pbiArg->tgi_token );
		free( pbiArg->tgi_driver );
		free( pbiArg->tgi_version );
	}
	else if ( xptTransform == reinterpret_cast<xdrproc_t>( xdr_bayonne_policy ) )
	{
		pbplArg = reinterpret_cast<bayonne_policy*>( ctArg );
		delete [] pbplArg->pol_sched;
		delete [] pbplArg->pol_ports.pol_ports_val;
		free( pbplArg->pol_name );
		free( pbplArg->pol_number );
	}
	else if ( xptTransform == reinterpret_cast<xdrproc_t>( xdr_bayonne_port ) )
	{
		pbpoArg = reinterpret_cast<bayonne_port*>( ctArg );
		free( pbpoArg->port_caller );
		free( pbpoArg->port_dialed );
		free( pbpoArg->port_name );
		free( pbpoArg->port_user );
		free( pbpoArg->port_info );
		free( pbpoArg->port_lang );
		free( pbpoArg->port_gid );
		free( pbpoArg->port_policy );
	} // End of if

	return TRUE;
} // End of bayonne_program_1_freeresult( SVCXPRT*, xdrproc_t, caddr_t )
