/***************************************************************************
                         -- plugin.cc --
                         -------------------
    begin                : 2001-09-01
    copyright            : (C) 2001 - 2002
    authors              : Jason Kasper (vanRijn)
                         : Ben Jansens (xOr)
    URL                  : http://bbconf.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 // E_O_H_VR
#include <qfile.h>
#include <qdir.h>
#include <qtextstream.h>

#include <stdlib.h>
#include <iostream>
#include <fstream>

#include "plugin.h"

QString expandTilde(const QString& s) {
  if (!s.startsWith("~")) return s;

  const char* const home = getenv("HOME");
  if (home == NULL) return s;

  return QString(home + s.mid(s.find('/'),s.length()));
}


Plugin::Plugin(QObject *p, const char *n)
		: QObject(p, n) {
	tree = new TreeNode<QWidget>;
	m_modified = false;
	m_configfile = "";
	moddb = new QList<QString>;
	moddb->setAutoDelete(true);
}

Plugin::~Plugin() {
	if (tree)
		delete tree;
	if (moddb)
			delete moddb;
}

void Plugin::setIsModified() {
	setIsModified(true);
}

void Plugin::setIsModified(bool b) {
	m_modified = b;
}

bool Plugin::loaddb(QString file)
{
	clearSaveResources();
	if (!m_configfile.isEmpty())
		if ((db = XrmGetFileDatabase(m_configfile)) != NULL) {
			return true;	// m_configfile is still good
		}
	if ((db = XrmGetFileDatabase(file)) == NULL) {
		fprintf(stderr,"Could not open config file: %s\n",
			file.latin1() );
		fprintf(stderr,"Using internal defaults for plugin.\n");
   	return false;
	}
	m_configfile = file;
	return true;
}

bool Plugin::loaddb(QString global_bb, QString local_bb, QString global_nobb,
		QString local_nobb)
{
	clearSaveResources();
	QString global, local;

	if (!m_configfile.isEmpty())
		if ((db = XrmGetFileDatabase(m_configfile)) != NULL) {
			return true;	// m_configfile is still good
		}
	// pick the appropriate set of files to try, based on the existance of a
	// ~/.blackboxrc file
	QString bboxfile = getenv("HOME");
	bboxfile.append("/.blackboxrc");
	QFile f(bboxfile);
	if (f.open(IO_ReadOnly)) {
		global = global_bb;	// use the "with bb" configs
		local = local_bb;
	} else {
		global = global_nobb;	// use the "without bb" configs
		local = local_nobb;
	}
	f.close();
			
	m_configfile = getenv("HOME");
	m_configfile.append("/");
	m_configfile.append(local);	// try the local config
	if ((db = XrmGetFileDatabase(m_configfile)) == NULL) {
		m_configfile = global;		// try the global config
		if ((db = XrmGetFileDatabase(m_configfile)) == NULL) {
			fprintf(stderr,"Could not open default config file: %s\n",
					m_configfile.latin1() );
			fprintf(stderr,"Using internal defaults for plugin.\n");
			m_configfile = "";
   		return false;
		}
	}
	return true;
}

bool Plugin::loadResource(const QString rname, const QString rclass,
		const bool defval)
{
	char *val_type;
	XrmValue val;
	if (XrmGetResource(db, rname.latin1(), rclass.latin1(), &val_type,
			&val)) {
		if (!strncasecmp("true", val.addr, val.size))
			return true;
		else
			return false;
	} else
		return defval;
}

int Plugin::loadResource(const QString rname, const QString rclass,
		const int defval)
{
	char *val_type;
	XrmValue val;
	int ret;
	if (XrmGetResource(db,  rname.latin1(), rclass.latin1(), &val_type,
			&val)) {
		if (sscanf(val.addr, "%i", &ret) != 1)
			return defval;
		else
			return ret;
	} else
		return defval;
}

QString *Plugin::loadResource(const QString rname, const QString rclass,
		const char *defval)
{
	char *val_type;
	XrmValue val;
	if (XrmGetResource(db, rname.latin1(), rclass.latin1(), &val_type,
			&val)) {
		return new QString(val.addr);
	} else {
		return new QString(defval);
	}
}

QColor *Plugin::loadResource(const QString rname, const QString rclass,
		QColor &defval)
{
	char *val_type;
	XrmValue val;

	if (XrmGetResource(db, rname.latin1(), rclass.latin1(), &val_type,
			&val)) {
		QString strcolor = val.addr;
		if (strcolor.contains("rgb:")) {
			strcolor.remove(0, 4);	// remove the "rgb:"
			QString rstr = strcolor;
			rstr.remove(rstr.findRev("/"), rstr.length());
			rstr.remove(rstr.findRev("/"), rstr.length());
printf("r: %s\n", rstr.latin1());
			QString gstr = strcolor;
			gstr.remove(0, gstr.find("/")+1);
			gstr.remove(gstr.findRev("/"), gstr.length());
printf("g: %s\n",gstr.latin1());
			QString bstr = strcolor;
			bstr.remove(0, bstr.find("/")+1);
			bstr.remove(0, bstr.find("/")+1);
printf("b: %s\n",bstr.latin1());
			bool ok;
			int r = rstr.toInt(&ok, 16);
			int g = gstr.toInt(&ok, 16);
			int b = bstr.toInt(&ok, 16);
			return new QColor(r,g,b);
		} else
			return new QColor(val.addr);	// it's a color name
	} else {
		return new QColor(defval);
	}
}

void Plugin::saveResource(const QString name, const bool val) {
	QString strbool;
	if (val)
		strbool = "true";
	else
		strbool = "false";
	QString *strval = new QString;
	QTextOStream(strval) << name << ":\t" << strbool;
	moddb->append(strval);
}

void Plugin::saveResource(const QString name, const int val) {
	QString *strval = new QString;
	QTextOStream(strval) << name << ":\t" << val;
	moddb->append(strval);
}

void Plugin::saveResource(const QString name, const char *val) {
	QString *strval = new QString;
	QTextOStream(strval) << name << ":\t" << val;
	moddb->append(strval);
}

void Plugin::saveResource(const QString name, const QColor &val) {
	QString *strval = new QString;
	int r, g, b;
	val.rgb(&r, &g, &b);
	QString rstr, gstr, bstr;
	rstr.setNum(r, 16);
	gstr.setNum(g, 16);
	bstr.setNum(b, 16);
	QTextOStream(strval) << name << ":\trgb:" << rstr << "/" << gstr << "/" << bstr;
	moddb->append(strval);
}

void Plugin::saveResourceBlank() {
	QString *strval = new QString("");
	moddb->append(strval);
}

void Plugin::saveResourceComment(QString comment) {
	QString *strval = new QString("! ");
	strval->append(comment);
	moddb->append(strval);
}

void Plugin::saveResourceText(QString text) {
	QString *strval = new QString(text);
	moddb->append(strval);
}

bool Plugin::savedb(const char *file)
{
	bool saved = writedbfile(file);
	// reset our resource list so we don't keep appending to it
	clearSaveResources();
	return saved;
}

bool Plugin::savedb(const char *bb, const char *nobb)
{
	QString dbfile = getenv("HOME");
	dbfile.append("/");

	QString bboxfile = getenv("HOME");
	bboxfile.append("/.blackboxrc");
	QFile f(bboxfile);
	if (f.open(IO_ReadOnly)) {
		dbfile.append(bb);
	} else {
		dbfile.append(nobb);
	}
	f.close();

	bool saved = writedbfile(dbfile);
	// reset our resource list so we don't keep appending to it
	clearSaveResources();
	return saved;
}

void Plugin::clearSaveResources() {
	moddb->clear();
}

bool Plugin::writedbfile(const char *file) {
	// check to make sure the directory exists for the file that we're
	// asking to write to. If it doesn't exist, create it if possible,
	// then proceed....

	QFileInfo fi( (QString)file );
	QString dir = fi.dirPath(true);
	QDir d(dir);
	QFileInfo fid( dir );

	// qWarning( "QDir path is ->%s<-", (const char*)d.path() );
	// qWarning( "fid path is ->%s<-", (const char*)fid.absFilePath() );

	if ( ! fid.isDir() ) {
		qWarning( "directory ->%s<- doesn't exist. creating....", 
			(const char*)dir.latin1() );
		if (!d.mkdir(dir)) {
			qWarning("could not create directory ->%s<-",
				(const char *)d.absFilePath(dir) );

		}
	}

	

	std::ofstream out(file, std::ios::out);
	if (!out)
		return false;
	QListIterator<QString> it(*moddb);
	for (; it.current(); ++it)
		out << it.current()->latin1() << std::endl;
	out.close();
	return true;
}
