    // ObjectManager.cpp: implementation of the CObjectManager class.  
    //  
    //////////////////////////////////////////////////////////////////////  
  
#include "../../Include/Base/ObjectManager.h"  
#include "../../Include/Base/LongToObjectCollection.h"  
#include <stddef.h>  
#include <stdio.h>  
#include <string.h>  
 
    //////////////////////////////////////////////////////////////////////  
    // Construction/Destruction  
    //////////////////////////////////////////////////////////////////////  
      
    CObjectManager *CObjectManager::m_instance = 0;  
      
    CObjectManager::CObjectManager()  
    {  
		PASS_INITLOCK (&lock);
    	firstObject = NULL;  
    	lastObject = NULL;  
    	currentObjectID = 1; // When an object have an ID = 0, this object is no more used.  
    }  
      
    CObjectManager::~CObjectManager()  
    { 
        RemoveAll();
		PASS_DESTROYLOCK (&lock); 
    }  
      
    CObjectManager *CObjectManager::Instance (void)  
    {  
    	if (m_instance == 0)  
    	{  
    		m_instance = new CObjectManager;  
      
    	}  
    	return m_instance;  
    }  
      
    CObject *CObjectManager::CreateByName (char *objectName)  
    {  	
	    PASS_LOCK (&lock);
    	__ObjectList *tempPtr;  
      
    	if (firstObject == NULL)  
    	{  
		    PASS_UNLOCK (&lock);
    		return NULL;  
    	}  
      
    	tempPtr = firstObject;  
      
    	while (tempPtr)  
    	{  
    		if (tempPtr->objectName != NULL)  
    		{  
    			if (strcmp (tempPtr->objectName,objectName) == 0)  
    			{  
#ifdef PASS_EXCEPTION  
					try			  
#endif // PASS_EXCEPTION  
    				{  
    					CObject *classObject = tempPtr->classObject;  
  					CObject *tempObject = 0;  
					if (classObject != NULL)  
					{  
						tempObject = classObject->Clone();  
					}  
  
    					if (tempObject != NULL)  
    					{  
    						// Assign new ID  
    						tempObject->SetID(currentObjectID++);  
      
    					} 
					    PASS_UNLOCK (&lock); 
    					return tempObject;  
    				}  
#ifdef PASS_EXCEPTION  
    				catch (...)  
    				{  
    					// Can't close object return NULL;  
					    PASS_UNLOCK (&lock);
  				    	return NULL;  
    				}  
#endif // PASS_EXCEPTION  
    			}  
    		}  
    		tempPtr = tempPtr->nextObject;  
    	}  
      
	PASS_UNLOCK (&lock);
  	return NULL;  
    }  
      
    CObject *CObjectManager::CreateByNameAndType (char *objectName, char *objectType)  
    {  
	    PASS_LOCK (&lock);
    	__ObjectList *tempPtr;  
    	BOOL nameOK = FALSE;  
    	BOOL typeOK = FALSE;  
      
    	if (firstObject == NULL)  
    	{  
		    PASS_UNLOCK (&lock);
  		    return NULL;  
    	}  
      
    	// Must have name AND type to create new instance  
    	if (objectName == NULL || objectType == NULL)  
    	{  
		    PASS_UNLOCK (&lock);
  		    return NULL;  
    	}  
    	tempPtr = firstObject;  
      
    	while (tempPtr)  
    	{  
    		if (tempPtr->objectName != NULL && tempPtr->objectType != NULL)  
    		{  
    			if (strcmp (tempPtr->objectName,objectName) == 0)  
    			{  
    				nameOK = TRUE;  
    			}  
      
    			if (strcmp (tempPtr->objectType,objectType) == 0)  
    			{  
    				typeOK = TRUE;  
    			}  
      
    			if (nameOK == TRUE && typeOK == TRUE)  
    			{  
#ifdef PASS_EXCEPTION  
    				try  
#endif // PASS_EXCEPTION  
    				{  
    					CObject *classObject = tempPtr->classObject;  
  					CObject *tempObject = 0;  
					if (classObject != NULL)  
					{  
						tempObject = classObject->Clone();  
					}  
    					if (tempObject != NULL)  
    					{  
    						// Assign new ID  
    						tempObject->SetID(currentObjectID++);  
      
    					} 
					    PASS_UNLOCK (&lock); 
    					return tempObject;  
    				}  
#ifdef PASS_EXCEPTION  
    				catch (...)  
    				{  
    					// Can't close object return NULL;  
					    PASS_UNLOCK (&lock);
  					    return NULL;  
    				}  
#endif // PASS_EXCEPTION  
    			}  
    		}  
    		nameOK = FALSE;  
    		typeOK = FALSE;  
    		tempPtr = tempPtr->nextObject;  
    	}  
      
	    PASS_UNLOCK (&lock);
  	    return NULL;  
    }  
      
    CObject *CObjectManager::CreateBySerializedObject (struct SerializedObject *objectSerialized)  
    {  
	    PASS_LOCK (&lock);
    	__ObjectList *tempPtr;  
      
    	if (firstObject == NULL)  
    	{  
		    PASS_UNLOCK (&lock);
  		    return NULL;  
    	}  
      
    	// Get the object name  
      
    	char *objectName = new char [objectSerialized->nameSize + 1];  
    	strncpy (objectName, objectSerialized->name, objectSerialized->nameSize);  
    	memset (objectName + objectSerialized->nameSize, '\0', 1);   
      
    	tempPtr = firstObject;  
      
    	while (tempPtr)  
    	{  
    		if (strcmp (tempPtr->objectName,objectName) == 0)  
    		{  
    			// Create a new instance  
  			    CObject *tempObject = tempPtr->classObject->Clone ();  
      
    			// Unserialized the new instance  
      
    			tempObject->SetSerialMultiPass ();  
    			tempObject->UnSerialize (objectSerialized);  
      
    			if (tempObject != NULL)  
    			{  
    				// Assign new ID  
    				tempObject->SetID(currentObjectID++);  
    			}  
    			delete objectSerialized->name;  
    			delete objectSerialized->type;  
    			delete objectSerialized->serializedObject;  
    			delete objectSerialized;  
    			delete objectName; 
 
			    PASS_UNLOCK (&lock);
    			return tempObject;  
    		}  
    		tempPtr = tempPtr->nextObject;  
    	}  
      
    	delete objectSerialized->name;  
    	delete objectSerialized->type;  
    	delete objectSerialized->serializedObject;  
    	delete objectSerialized;  
    	delete objectName;  
	    PASS_UNLOCK (&lock);
  	    return NULL;  
    }  
      
    BOOL CObjectManager::RegisterObjectByNameAndType (  
    	CObject *anObject,  
    	char *name,  
    	char *type)  
    {  
	    PASS_LOCK (&lock);
    	__ObjectList *tempPtr;  
    	  
    	// No object to register, exit  
    	if (anObject == NULL)  
    	{  
		    PASS_UNLOCK (&lock);
    		return FALSE;  
    	}  
      
    	// No name and type, exit  
    	if (name == NULL && type == NULL)  
    	{  
		    PASS_UNLOCK (&lock);
    		return FALSE;  
    	}  
      
    	if (firstObject == NULL && lastObject == NULL)  
    	{  
    		firstObject = new __ObjectList;  
    		lastObject = firstObject;  
  		    firstObject->nextObject = NULL;  
  		    lastObject->nextObject = NULL;  
    		tempPtr = firstObject;  
    	}  
    	else  
    	{  
    		tempPtr = new __ObjectList;  
    		lastObject->nextObject = tempPtr;  
    	}  
      
  	    tempPtr->nextObject = NULL;  
    	tempPtr->classObject = anObject;  
      
    	if (name == NULL)  
    	{  
  		    tempPtr->objectName = NULL;  
    	}  
    	else  
    	{  
    		int nameLength = strlen (name);  
    		  
    		char *tempName = new char [nameLength + 1];  
    		strcpy (tempName,name);  
      
    		tempPtr->objectName = tempName;  
    	}  
      
    	if (type == NULL)  
    	{  
  		    tempPtr->objectType = NULL;  
    	}  
    	else  
    	{  
    		int typeLength = strlen (type);  
    		  
    		char *tempType = new char [typeLength + 1];  
    		strcpy (tempType,type);  
      
    		tempPtr->objectType = tempType;  
    	}  
      
    	lastObject = tempPtr;  
      
	    PASS_UNLOCK (&lock);
 	    return TRUE;  
	}  
    
    BOOL CObjectManager::RegisterObject (CObject *anObject)  
    {  
    	return RegisterObjectByNameAndType (anObject,anObject->GetName(),anObject->GetType());  
    }  
      
    BOOL CObjectManager::RegisterObjectByName (CObject *anObject, char *name)  
    {  
    	return RegisterObjectByNameAndType (anObject,name,anObject->GetType());  
    }  
      
    CLongToObjectCollection *CObjectManager::GetObjectsForType (char *type)  
    {  
	    PASS_LOCK (&lock);
    	__ObjectList *tempPtr;  
    	long objectCount = 0;  
    	CLongToObjectCollection *tempCollection = new CLongToObjectCollection;  
      
    	if (firstObject == NULL)  
    	{  
		    PASS_UNLOCK (&lock);
  		    return NULL;  
    	}  
      
    	// Must have name AND type to create new instance  
    	if (type == NULL)  
    	{  
		    PASS_UNLOCK (&lock);
  		    return NULL;  
    	}  
      
    	tempPtr = firstObject;  
      
    	while (tempPtr)  
    	{  
    		if (tempPtr->objectType != NULL)  
    		{  
    			if (strcmp (tempPtr->objectType,type) == 0)  
    			{  
#ifdef PASS_EXCEPTION 
    				try  
#endif // PASS_EXCEPTION 
    				{  
    					CObject *classObject = tempPtr->classObject;  
    					tempCollection->Add (objectCount++,classObject);  
    				}  
#ifdef PASS_EXCEPTION 
    				catch (...)  
    				{  
    					// Can't close object return NULL;  
					    PASS_UNLOCK (&lock);
  					    return NULL;  
    				}  
#endif // PASS_EXCEPTION 
    			}  
    		}  
    		tempPtr = tempPtr->nextObject;  
    	}  
      
	    PASS_UNLOCK (&lock);
    	return tempCollection;  
    }  

    void CObjectManager::RemoveAll()
    {
	    PASS_LOCK (&lock);
    	__ObjectList *tempPtr;  
    	__ObjectList *tempPtr2;  

    	tempPtr = firstObject;  
    	tempPtr2 = NULL;  
      
    	while (tempPtr)  
    	{  
    		if (tempPtr->objectName != NULL)  
    		{ 
                delete tempPtr->objectName;
                tempPtr->objectName = NULL;
            }
    		if (tempPtr->objectType != NULL)  
    		{ 
                delete tempPtr->objectType;
                tempPtr->objectType = NULL;
            }
    		tempPtr2 = tempPtr->nextObject;
            delete tempPtr;
            tempPtr = tempPtr2;
    	}
	    PASS_UNLOCK (&lock);
        m_instance = NULL;
    }
