// Connection_Local.cpp: implementation of the CConnection_Local class.
//
//////////////////////////////////////////////////////////////////////

#include "../../Include/Comm/Connection_Local.h"

#include "../../Include/Base/ObjectManager.h"
#include "../../Include/Base/StringContainer.h"
#include "../../Include/Base/StringToObjectCollection.h"
#include "../../Include/Base/LoggerManager.h"

#include "../../Include/Comm/MessageStack.h"
#include "../../Include/Comm/Message.h"
#include "../../Include/Comm/ServerSession.h"

#include <string.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CConnection_Local::CConnection_Local()
{
	m_SendStack = new CMessageStack; 
	m_ReceiveStack = new CMessageStack; 

	m_ulMessageIDCounter = 1; 
	m_bConnectionOpened = FALSE;
	memset (m_szSessionObjectName,0,sizeof(m_szSessionObjectName));
	m_ulWaitMessageTimeOut = 60000; // Max 60 Sec default

	PASS_INITLOCK (&lock);
}

CConnection_Local::~CConnection_Local()
{
	// Clean stacks 
	CMessage *message; 
 
	while ((message = m_SendStack->GetMessage()) != NULL) 
	{ 
		delete message; 
	} 
 
	while ((message = m_ReceiveStack->GetMessage()) != NULL) 
	{ 
		delete message; 
	} 
 
	delete m_SendStack; 
	delete m_ReceiveStack; 

	// Now stop and delete server session 
	if (m_Session != NULL)
	{ 
		unsigned long timeout = 0;

		// Stop session
		m_Session->Stop();

		while (m_Session->CanBeDeleted() == FALSE && timeout < 600)
		{
			PASS_MILLISLEEP (100);
			timeout++;
		}

		if (timeout == 600)
		{
			m_Session->Kill();
			Trace (__FILE__,__LINE__,LOGMASK_COMM,"Server session killed, this application can be unstable, please consider to restart it");
		}

		delete m_Session; 
		m_Session = NULL;
	} 

	PASS_DESTROYLOCK (&lock);
}

BOOL CConnection_Local::Open (void)
{
	// Verify session name 

	if (m_szSessionObjectName == NULL)
	{
		Trace (__FILE__,__LINE__,LOGMASK_COMM,"Can't create new connection, no session object name defined");

		return FALSE;
	}

    // Try to instanciate the session 
    CObjectManager *manager = CObjectManager::Instance(); 
    CServerSession *localSession = (CServerSession *)manager->CreateByNameAndType (m_szSessionObjectName,"ServerSession"); 

    if (localSession == NULL) 
    { 
		Trace (__FILE__,__LINE__,LOGMASK_COMM,"Can't create new connection, can't instantiate session object, please register a session object");

		return FALSE;
	}

#ifdef PASS_EXCEPTION 
	try 
#endif // PASS_EXCEPTION 
	{
		localSession->SetReceiveStack (m_SendStack); 
		localSession->SetSendStack (m_ReceiveStack); 
 		localSession->SetConnection (this); 
		localSession->SetTickTime (2); 
		localSession->Start (); 
	}
#ifdef PASS_EXCEPTION 
    catch (...) 
    { 
		// Error seems to be the wrong object; 
		unsigned long timeout = 0;

		// Stop session
		localSession->Stop();

		while (localSession->CanBeDeleted() == FALSE && timeout < 600)
		{
			PASS_MILLISLEEP (100);
			timeout++;
		}

		if (timeout == 600)
		{
			localSession->Kill();
			Trace (__FILE__,__LINE__,LOGMASK_COMM,"Server session killed, this application can be unstable, please consider to restart it");
		}

		delete localSession; 
		localSession = NULL;

		return FALSE;
	}
#endif // PASS_EXCEPTION

	m_Session = localSession;
	m_bConnectionOpened = TRUE;

	return TRUE;
}

BOOL CConnection_Local::Close (void)
{
	unsigned long timeout = 0;

	// Stop session
	m_Session->Stop();

	while (m_Session->CanBeDeleted() == FALSE && timeout < 600)
	{
		PASS_MILLISLEEP (100);
		timeout++;
	}

	if (timeout == 600)
	{
		m_Session->Kill();
		Trace (__FILE__,__LINE__,LOGMASK_COMM,"Server session killed, this application can be unstable, please consider to restart it");
	}

	delete m_Session; 
	m_Session = NULL;

	// Now clean stacks
	// Clean stacks 
	CMessage *message; 
 
	while ((message = m_SendStack->GetMessage()) != NULL) 
	{ 
		delete message; 
	} 
 
	while ((message = m_ReceiveStack->GetMessage()) != NULL) 
	{ 
		delete message; 
	}

	m_bConnectionOpened = FALSE;

	return TRUE;
}

CMessage *CConnection_Local::_GetMessageForID (unsigned long messageID) 
{ 
#ifdef PASS_EXCEPTION 
	try 
#endif // PASS_EXCEPTION 
	{ 
		return m_ReceiveStack->GetMessageForID (messageID); 
	} 
#ifdef PASS_EXCEPTION 
	catch (...) 
	{ 
		// Exception, try more 
		return NULL; 
	} 
#endif // PASS_EXCEPTION
} 

CMessage *CConnection_Local::GetMessageForID (unsigned long messageID)
{
	PASS_LOCK (&lock); 
 
	CMessage *localMessage = NULL; 
	localMessage = _GetMessageForID (messageID); 
 
	PASS_UNLOCK (&lock); 
	return localMessage; 
}

CMessage *CConnection_Local::WaitMessageForID (unsigned long messageID)
{
	PASS_LOCK (&lock); 
 
	if (messageID == 0) 
	{ 
		PASS_UNLOCK (&lock); 
		return NULL; 
	} 
 
	CMessage *returnMessage = NULL; 
	unsigned long countTimeOut = 0; 
 
#ifdef PASS_EXCEPTION 
	try  
#endif //PASS_EXCEPTION 
	{ 
		while ((returnMessage = _GetMessageForID(messageID)) == NULL) 
		{ 
			if (countTimeOut++ > m_ulWaitMessageTimeOut) 
			{ 
				returnMessage = NULL; 
				break; 
			} 
 
			PASS_MILLISLEEP (1); 
		} 
	} 
#ifdef PASS_EXCEPTION 
	catch (...) 
	{ 
		PASS_UNLOCK (&lock); 
		return NULL; 
	} 
#endif //PASS_EXCEPTION 
 
	PASS_UNLOCK (&lock); 
	return returnMessage; 
}

BOOL CConnection_Local::SetConnectionParam (CStringToObjectCollection *param)
{
	// Needed params : StringContainer - 'ServerAddress' and ULongContainer - 'ServerPort' 
	// No optional params 
 
	if (param == NULL) 
	{ 
		// Set error 
		return FALSE; 
	} 
 
	if (param->GetName() == NULL || param->GetType() == NULL) 
	{ 
		// Set Error 
		return FALSE; 
	} 
 
	if (strcmp (param->GetName(),"StringToObjectCollection") != 0 || strcmp (param->GetType(),"Collection") != 0) 
	{ 
		// Set Error 
		return FALSE; 
	} 

	// Now get session object name
 
	// session object need to be a stringContainer 
	CStringContainer *sessionObject = NULL; 
 
	sessionObject = (CStringContainer *)param->Lookup ("SessionObject"); 
 
	if (sessionObject == NULL) 
	{ 
		// Can't find session object 
		// Set error 
		return FALSE; 
	} 
 
	if (sessionObject->GetName() == NULL || sessionObject->GetType() == NULL) 
	{ 
		// no name or type 
		//  Set errror 
		return FALSE; 
	} 

	if (strcmp(sessionObject->GetName(),"StringContainer") != 0 || strcmp(sessionObject->GetType(),"Container") != 0) 
	{ 
		// Not a string container 
		// Set error 
		return FALSE; 
	} 

	strncpy (m_szSessionObjectName,sessionObject->GetString(),sizeof (m_szSessionObjectName)); 

	return TRUE;
}

unsigned long CConnection_Local::SendMessage (CMessage *aMessage)
{
	// Is the message valid ? 
	if (aMessage == NULL) 
	{ 
		// Invalid message return 0; 
		// Set Error 
		return 0; 
	} 
 
	// Is the connection open ? 
	if (m_bConnectionOpened == FALSE) 
	{ 
		// Connection not yet opened, return 0 and set error 
		// Set Error 
		return 0; 
	} 
        		 
    if (aMessage->GetMessageID() == 0) 
    { 
		// Message ID not yet set 
        aMessage->SetMessageID (m_ulMessageIDCounter++); 
		m_SendStack->PutMessage (aMessage); 
        return m_ulMessageIDCounter - 1; 
    } 
 
	m_SendStack->PutMessage (aMessage); 
 
 	return aMessage->GetMessageID(); 
}
