// Config.cpp: implementation of the CConfig class.
//
//////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "Config.h"
#include <openssl/conf.h>
#include <algorithm>
#include <wx/dynload.h>

#define CONFFILE_BEID "BEID.conf"

#define CONFFILE_BEID_SERVICE "BEID_service"
#define CONFFILE_BEID_SERVICE_ADDRESS "serverAddress"
#define CONFFILE_BEID_SERVICE_PORT "serverPort"
#define CONFFILE_BEID_SERVICE_LOGMODE "logmode"
#define CONFFILE_BEID_SERVICE_LOGFILE "logfile"
#define CONFFILE_BEID_SERVICE_HTTPCRLSTORE "httpcrlstore"
#define CONFFILE_BEID_SERVICE_SCHEDULE "crlschedule"
#define CONFFILE_BEID_SERVICE_LOCKTIMEOUT "locktimeout"
#define CONFFILE_BEID_SERVICE_IPADDRESS "IPAddr_allowed"

#define CONFFILE_BEID_DEFAULT "BEID_default"
#define CONFFILE_BEID_DEFAULT_MAINDIR "maindir"
#define CONFFILE_BEID_DEFAULT_CRL "crl"
#define CONFFILE_BEID_DEFAULT_PROXYHOST "proxyHost"
#define CONFFILE_BEID_DEFAULT_PROXYPORT "proxyPort"

class CFindKey
{
private:
    std::string m_strKey;
public:
    explicit CFindKey(std::string strKey)
    {
        m_strKey = strKey;
    }

    bool operator() (std::pair<std::string, std::string> oValue)
    {
#ifdef _WIN32
        return 0 == stricmp(oValue.first.c_str(), m_strKey.c_str());
#else
        return 0 == strcasecmp(oValue.first.c_str(), m_strKey.c_str());
#endif
    }
};

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

std::map<std::string, std::vector<std::pair<std::string, std::string> > > CConfig::m_oMapConfig;

CConfig::CConfig()
{

}

CConfig::~CConfig()
{

}

wxString CConfig::GetGlobalDir()
{
  wxString strDir;

#ifdef __VMS__ // Note if __VMS is defined __UNIX is also defined
    strDir = wxT("sys$manager:");
#elif defined(__WXMAC__)
    strDir = wxMacFindFolder(  (short) kOnSystemDisk, kPreferencesFolderType, kDontCreateFolder ) ;
#elif defined( __UNIX__ )
    strDir = wxT("/etc/");
#elif defined(__WXPM__)
    ULONG aulSysInfo[QSV_MAX] = {0};
    UINT drive;
    APIRET rc;

    rc = DosQuerySysInfo( 1L, QSV_MAX, (PVOID)aulSysInfo, sizeof(ULONG)*QSV_MAX);
    if (rc == 0)
    {
        drive = aulSysInfo[QSV_BOOT_DRIVE - 1];
        strDir.Printf(wxT("%c:\\OS2\\"), 'A'+drive-1);
    }
#elif defined(__WXSTUBS__)
    wxASSERT_MSG( FALSE, wxT("TODO") ) ;
#elif defined(__DOS__)
    // There's no such thing as global cfg dir in MS-DOS, let's return
    // current directory (FIXME_MGL?)
    return wxT(".\\");
#else // Windows
    wxChar szWinDir[MAX_PATH];
    ::GetWindowsDirectory(szWinDir, MAX_PATH);
    wxDynamicLibrary pathLoader("Kernel32");
    if(pathLoader.IsLoaded())
    {
        typedef UINT (WINAPI *TGetSystemWindowsDirectoryA)(LPSTR,UINT);
        TGetSystemWindowsDirectoryA pFunc  = (TGetSystemWindowsDirectoryA) ::GetProcAddress( pathLoader.GetLibHandle(), "GetSystemWindowsDirectoryA" );
        if(pFunc != NULL)
        {
            memset(szWinDir, 0, sizeof(szWinDir));
            pFunc(szWinDir, MAX_PATH);
        }
    }
    strDir = szWinDir;
    strDir << wxT('\\');
#endif // Unix/Windows

    return strDir;
}

void CConfig::Load()
{
    int i = 0;
    long lErr = 0;
    STACK_OF(CONF_VALUE) *pSec = NULL;
    CONF_VALUE *pItem = NULL;
    CONF *pConf = NULL;
    wxString strConfPath = GetGlobalDir() + CONFFILE_BEID;

    m_oMapConfig.clear();

    pConf = NCONF_new(NCONF_default());
    if (NCONF_load(pConf, strConfPath.GetData(), &lErr) > 0)
    {
        std::vector<std::pair<std::string, std::string> > oTempVec(8);
        if (NULL != (pSec = NCONF_get_section(pConf, CONFFILE_BEID_SERVICE)))
        {
            for (i = 0; i < sk_CONF_VALUE_num(pSec); i++)
            {
                pItem = sk_CONF_VALUE_value(pSec, i);
                oTempVec.push_back(std::pair<std::string, std::string>(pItem->name, pItem->value));
            }
            m_oMapConfig[CONFFILE_BEID_SERVICE] = oTempVec;
            pSec = NULL;
        }
        oTempVec.clear();

        if (NULL != (pSec = NCONF_get_section(pConf, CONFFILE_BEID_DEFAULT)))
        {
            for (i = 0; i < sk_CONF_VALUE_num(pSec); i++)
            {
                pItem = sk_CONF_VALUE_value(pSec, i);
                oTempVec.push_back(std::pair<std::string, std::string>(pItem->name, pItem->value));
            }
            m_oMapConfig[CONFFILE_BEID_DEFAULT] = oTempVec;
            pSec = NULL;
        }
        oTempVec.clear();
    }
    NCONF_free(pConf);
}

void CConfig::Unload()
{
    m_oMapConfig.clear();
}

const char *CConfig::FindValue(char *pszSection, char *pszKey)
{
    ConfigMapIT itmap;
    itmap = m_oMapConfig.find(pszSection);
    if(itmap != m_oMapConfig.end())
    {
        ConfigVecIT itvec;
        itvec = std::find_if((*itmap).second.begin(), (*itmap).second.end(), CFindKey(pszKey));
        if(itvec != (*itmap).second.end())
        {
            return (*itvec).second.c_str();
        }
    }
    return NULL;
}

const char *CConfig::GetServerAddress()
{
    return FindValue(CONFFILE_BEID_SERVICE, CONFFILE_BEID_SERVICE_ADDRESS);
}

const unsigned long CConfig::GetServerPort()
{
    const char *pszPort = FindValue(CONFFILE_BEID_SERVICE, CONFFILE_BEID_SERVICE_PORT);
    if(pszPort != NULL)
    {
        return atol(pszPort);
    }
    return 0;
}

const char *CConfig::GetLogFile()
{
    return FindValue(CONFFILE_BEID_SERVICE, CONFFILE_BEID_SERVICE_LOGFILE);
}

BOOL CConfig::GetLogMode()
{
    BOOL bRet = FALSE;
    const char *pMode = FindValue(CONFFILE_BEID_SERVICE, CONFFILE_BEID_SERVICE_LOGMODE);
    if(pMode != NULL && 0 == strcmp(pMode, "1"))
    {
        bRet = TRUE;
    }
    return bRet;
}

BOOL CConfig::GetCRLSchedule()
{
    BOOL bRet = FALSE;
    const char *pMode = FindValue(CONFFILE_BEID_SERVICE, CONFFILE_BEID_SERVICE_SCHEDULE);
    if(pMode != NULL && 0 == strcmp(pMode, "1"))
    {
        bRet = TRUE;
    }
    return bRet;
}

const char *CConfig::GetHttpCRLStorePath()
{
    return FindValue(CONFFILE_BEID_SERVICE, CONFFILE_BEID_SERVICE_HTTPCRLSTORE);
}

const char *CConfig::GetCRLStorePath()
{
    return FindValue(CONFFILE_BEID_DEFAULT, CONFFILE_BEID_DEFAULT_CRL);
}

long CConfig::GetLockTimeout()
{
    long lRet = 60;
    const char *pTimeOut = FindValue(CONFFILE_BEID_SERVICE, CONFFILE_BEID_SERVICE_LOCKTIMEOUT);
    if(pTimeOut != NULL)
    {
        lRet = atol(pTimeOut);
        if(lRet < 60)
        {
            lRet = 60;
        }
    }
    return lRet;
}

char **CConfig::GetAllowedIPAddress()
{
    const char *pValue = FindValue(CONFFILE_BEID_SERVICE, CONFFILE_BEID_SERVICE_IPADDRESS);
    if(NULL != pValue)
    {
        return SplitString(pValue, strlen(pValue), ',');
    }
    return NULL;
}

const char *CConfig::GetProxyHost()
{
    return FindValue(CONFFILE_BEID_DEFAULT, CONFFILE_BEID_DEFAULT_PROXYHOST);
}

const char *CConfig::GetProxyPort()
{
    return FindValue(CONFFILE_BEID_DEFAULT, CONFFILE_BEID_DEFAULT_PROXYPORT);
}

char **CConfig::SplitString(const char *psStr, int iLength, char cSep)
{
     const char * psSource;
     char * psTemp;
     char * psDest;
     char ** ppsList;
     int i;
     int iFields;
 
     psTemp = (char *)malloc(iLength + 1);
 
     iFields = 1;
     for (psSource = psStr, psDest = psTemp, i = 0; i < iLength; i++, psSource++, psDest++) 
     {
         *psDest = *psSource;
         if (*psDest == cSep) iFields++;
     }
 
     *psDest = '\0';
 
     ppsList = (char **)malloc(sizeof(char *) * (iFields + 1));
 
     psDest = psTemp;
     ppsList[0] = psDest;
     i = 1;
     while (i < iFields) 
     {
         if (*psDest == cSep) 
         {
             ppsList[i++] = psDest + 1;
             *psDest = 0;
         }
         psDest++;
     }
 
     ppsList[i] = NULL;
 
     return ppsList;
}
