// TransactionStack.cpp: implementation of the CTransactionStack class.
//
//////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "TransactionStack.h"
#include "Config.h"

#include <sys/timeb.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CTransactionStack::CTransactionStack()
{
	m_StackTop = NULL;
	m_StackBottom = NULL;
	m_ulAppCount = 0;
    m_lMaxTime = CConfig::GetLockTimeout(); //Default 60 sec in stack
	SetTickTime (5000);
	PASS_INITLOCK(&lock);
}

CTransactionStack::~CTransactionStack()
{
	// Clean up the stack
	__TransactionStack *temp = NULL;
    __TransactionStack *temp2 = NULL;
	temp = m_StackTop;

	while (temp != NULL)
	{
		temp2 = temp;
		temp = temp->next;
		delete temp2;
	}

	PASS_DESTROYLOCK (&lock);
}

void CTransactionStack::SetMaxTime (long lSec)
{
	m_lMaxTime = lSec;
}

void CTransactionStack::AddHandle(SCARDHANDLE hCard)
{
	PASS_LOCK (&lock);

    //Find existing application
    BOOL bFound = FALSE;
	__TransactionStack *temp = NULL;
	temp = m_StackTop;

	while (temp != NULL)
	{
        if(temp->hCard == hCard)
        {
            // Found One
            temp->lLockCount++;
	        // Initialize the time stamp
#ifdef _WIN32
	        struct _timeb timebuffer;
	        _ftime( &timebuffer );
#else
	        struct timeb timebuffer;
	        ftime( &timebuffer );
#endif
	        temp->time = timebuffer.time;
            bFound = TRUE;
            break;
        }
		temp = temp->next;
	}

    if(!bFound)
    {
	    // New message to stack
	    __TransactionStack *tempStack = new __TransactionStack;
	    tempStack->hCard = hCard;
	    tempStack->lLockCount = 1;
	    tempStack->time = 0;

	    if (m_StackTop == NULL)
	    {
	        // Initialize the time stamp
#ifdef _WIN32
	        struct _timeb timebuffer;
	        _ftime( &timebuffer );
#else
	        struct timeb timebuffer;
	        ftime( &timebuffer );
#endif
	        tempStack->time = timebuffer.time;

		    tempStack->next = NULL;
		    m_StackBottom = tempStack;
	    }
	    else
	    {
		    tempStack->next = m_StackTop;
	    }

	    // New head
	    m_StackTop = tempStack;

	    m_ulAppCount++;
    }
	PASS_UNLOCK (&lock);
}

void CTransactionStack::DeleteHandle(SCARDHANDLE hCard)
{
	PASS_LOCK (&lock);

    //Find existing application
    BOOL bFound = FALSE;
	__TransactionStack *temp = NULL;
	__TransactionStack *previous = NULL;
	temp = m_StackTop;

	while (temp != NULL && bFound == FALSE)
	{
        if(temp->hCard == hCard)
        {
            bFound = TRUE;
            temp->lLockCount--;
        }
        else
        {
			previous = temp;
			temp = temp->next;
        }
    }

    if(bFound && 0 == temp->lLockCount)
    {
        m_ulAppCount--;
		try
		{
			// Found the event, now remove it from the list;
			if (previous == NULL && temp->next == NULL)
			{
				// This was the only one in the list
				m_StackTop = NULL;
				m_StackBottom = NULL;
				delete temp;
			} 
			else if (previous == NULL && temp->next != NULL)
			{
				// This is the first of the list
				m_StackTop = temp->next;
				delete temp;
			}
			else if (previous != NULL && temp->next == NULL)
			{
				// This is the last one.
				previous->next = NULL;
				m_StackBottom = previous;
				delete temp;
			}
			else if (previous != NULL && temp->next != NULL)
			{
				// Somewhere in the list
				previous->next = temp->next;
				delete temp;
			}
		}
		catch (...)
		{
			// Error deleting
		}
	}

	PASS_UNLOCK (&lock);
}

void CTransactionStack::DeleteAllHandle(SCARDHANDLE hCard)
{
	PASS_LOCK (&lock);

    //Find existing application
    BOOL bFound = FALSE;
	__TransactionStack *temp = NULL;
	__TransactionStack *previous = NULL;
	temp = m_StackTop;

	while (temp != NULL && bFound == FALSE)
	{
        if(temp->hCard == hCard)
        {
            bFound = TRUE;
            temp->lLockCount = 0;
        }
        else
        {
			previous = temp;
			temp = temp->next;
        }
    }

    if(bFound)
    {
        m_ulAppCount--;
		try
		{
			// Found the event, now remove it from the list;
			if (previous == NULL && temp->next == NULL)
			{
				// This was the only one in the list
				m_StackTop = NULL;
				m_StackBottom = NULL;
				delete temp;
			} 
			else if (previous == NULL && temp->next != NULL)
			{
				// This is the first of the list
				m_StackTop = temp->next;
				delete temp;
			}
			else if (previous != NULL && temp->next == NULL)
			{
				// This is the last one.
				previous->next = NULL;
				m_StackBottom = previous;
				delete temp;
			}
			else if (previous != NULL && temp->next != NULL)
			{
				// Somewhere in the list
				previous->next = temp->next;
				delete temp;
			}
		}
		catch (...)
		{
			// Error deleting
		}
	}

	PASS_UNLOCK (&lock);
}

int CTransactionStack::IsHandleAvailable (SCARDHANDLE hCard)
{
    int iRet = 0;
	PASS_LOCK (&lock);

	if (m_StackBottom == NULL)
	{
		iRet = -1;
	}
    else
    {
        // Find existing
        BOOL bFound = FALSE;
	    __TransactionStack *temp = NULL;
	    temp = m_StackTop;
	    while (temp != NULL)
	    {
            if(temp->hCard == hCard)
            {
                // Found One
                bFound = TRUE;
                break;
            }
		    temp = temp->next;
	    }

        if(bFound)
        {
            iRet = m_StackBottom->hCard == hCard ? 1 : 0;
        }
        else
        {
            iRet = -1;
        }
    }

	PASS_UNLOCK (&lock);
    return iRet;
}

void CTransactionStack::Tick (void)
{
	PASS_LOCK (&lock);

#ifdef _WIN32
	        struct _timeb timebuffer;
	        _ftime( &timebuffer );
#else
	        struct timeb timebuffer;
	        ftime( &timebuffer );
#endif

	// Check for message older than MaxTime
	__TransactionStack *temp = NULL;
	temp = m_StackTop;
	__TransactionStack *previous = NULL;

	try 
	{
		while (temp != NULL)
		{
            // Only remove bottom
			if (temp->time > 0 && (temp->time + m_lMaxTime) < timebuffer.time)
			{
				// Found the event, now remove it from the list
				if (previous == NULL && temp->next == NULL)
				{
					// This was the only one in the list
					m_StackTop = NULL;
					m_StackBottom = NULL;
					previous = temp;
					delete temp;
					temp = NULL;
				} 
				else if (previous == NULL && temp->next != NULL)
				{
					// This is the first of the list
					m_StackTop = temp->next;
					previous = NULL;
					delete temp;
					temp = m_StackTop;
				}
				else if (previous != NULL && temp->next == NULL)
				{
					// This is the last one.
					previous->next = NULL;
					m_StackBottom = previous;
	                // Initialize the time stamp
#ifdef _WIN32
	            struct _timeb timebuffer;
	            _ftime( &timebuffer );
#else
	            struct timeb timebuffer;
	            ftime( &timebuffer );
#endif
                    previous->time = timebuffer.time;
					delete temp;
					temp = NULL;
					previous = NULL;
				}
				else if (previous != NULL && temp->next != NULL)
				{
					// Somewhere in the list
					__TransactionStack *localTemp = temp;
					previous->next = temp->next;
					temp = temp->next;
					delete localTemp;
				}
				else
				{
					previous = NULL;
					temp = NULL;
				}
			}
			else
			{
				if (temp != NULL)
				{
					previous = temp;
					temp = temp->next;
				}
			}
		}
	}
	catch (...)
	{
		// Error cleaning message
		// Release lock and exit returning
		PASS_UNLOCK (&lock);
		return;
	}

	PASS_UNLOCK (&lock);
}
