// Verify.cpp: implementation of the CVerify class.
//
//////////////////////////////////////////////////////////////////////

#ifdef WIN32
//#include <windows.h>
    #include <winsock.h>
    #pragma warning(disable:4786)
#else
    #include <sys/socket.h> 
#endif


#include "Verify.h"
#include "eiddefines.h"
#include "Bytearray.h"
#include "Config.h"

#include <string.h>
#include <openssl/ocsp.h>
#include <openssl/bio.h>

#include "wx/filename.h"

#ifndef _WIN32
#define strnicmp strncasecmp
#define stricmp strcasecmp
#endif

/* Old stuff 
// only valid for final cards ... 
// there are now 2 rrn certs used by EIC's ... 
#define RRN_CERT_COUNT  2
#define RRN_CERT_SIZE   20
*/

#define ROOT_CERT_COUNT  1
#define ROOT_CERT_SIZE   920
#define RRN_OID1   "2.16.56.1.1.1.3.1"
#define RRN_OID2   "2.16.56.1.1.1.4"
#define DN_RRN      "RRNRRNBE"  // CN + O + C

//#define CERTIPOST_OID   "0.3.2062.9.6.1.31.6.1"

/* Old stuff 
// Fingerprints of Valid RRN Certificates
static unsigned char _rrn_cert_thumb[RRN_CERT_COUNT][RRN_CERT_SIZE] = 
{
    {0x82, 0xBA, 0x0F, 0xFC, 0xD7, 0x61, 0x36, 0xBE, 0x27, 0x9C, 0x32, 0x90, 0xE7, 0x4C, 0x28, 0xD3, 0x81, 0x63, 0x50, 0x06},
    {0x5B, 0x25 ,0x60 ,0x76 ,0x8F ,0xE0 ,0x9D ,0xC0 ,0xE8 ,0x23 ,0xD7 ,0xA2 ,0x27 ,0x0F ,0xF3 ,0xB3 ,0x0A ,0xD8 ,0x27 ,0xF8}
};
*/

/* Root Certificate */
static unsigned char _root_cert[ROOT_CERT_COUNT][ROOT_CERT_SIZE] = 
{
    {
        0X30,0x82,0x03,0x94,0x30,0x82,0x02,0x7C,0xA0,0x03,0x02,0x01,0x02,0x02,0x10,0x58,0x0B,0x05,0x6C,0x53,0x24,0xDB,0xB2,0x50,0x57,0x18,0x5F,0xF9,0xE5,0xA6,0x50,0x30,0x0D,0x06,0x09,0x2A, 
        0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x05,0x05,0x00,0x30,0x27,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x03,0x13,0x0F,
        0x42,0x65,0x6C,0x67,0x69,0x75,0x6D,0x20,0x52,0x6F,0x6F,0x74,0x20,0x43,0x41,0x30,0x1E,0x17,0x0D,0x30,0x33,0x30,0x31,0x32,0x36,0x32,0x33,0x30,0x30,0x30,0x30,0x5A,0x17,0x0D,0x31,0x34,
        0x30,0x31,0x32,0x36,0x32,0x33,0x30,0x30,0x30,0x30,0x5A,0x30,0x27,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x03,0x13,
        0x0F,0x42,0x65,0x6C,0x67,0x69,0x75,0x6D,0x20,0x52,0x6F,0x6F,0x74,0x20,0x43,0x41,0x30,0x82,0x01,0x22,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,
        0x82,0x01,0x0F,0x00,0x30,0x82,0x01,0x0A,0x02,0x82,0x01,0x01,0x00,0xC8,0xA1,0x71,0xE9,0x1C,0x46,0x42,0x79,0x78,0x71,0x6F,0x9D,0xAE,0xA9,0xA8,0xAB,0x28,0xB7,0x4D,0xC7,0x20,0xEB,0x30,
        0x91,0x5A,0x75,0xF5,0xE2,0xD2,0xCF,0xC8,0x4C,0x14,0x98,0x42,0x58,0xAD,0xC7,0x11,0xC5,0x40,0x40,0x6A,0x5A,0xF9,0x74,0x12,0x27,0x87,0xE9,0x9C,0xE5,0x71,0x4E,0x22,0x2C,0xD1,0x12,0x18,
        0xAA,0x30,0x5E,0xA2,0x21,0xB9,0xD9,0xBB,0xFF,0xF6,0x74,0xEB,0x31,0x01,0xE7,0x3B,0x7E,0x58,0x0F,0x91,0x16,0x4D,0x76,0x89,0xA8,0x01,0x4F,0xAD,0x22,0x66,0x70,0xFA,0x4B,0x1D,0x95,0xC1,
        0x30,0x58,0xEA,0xBC,0xD9,0x65,0xD8,0x9A,0xB4,0x88,0xEB,0x49,0x46,0x52,0xDF,0xD2,0x53,0x15,0x76,0xCB,0x14,0x5D,0x19,0x49,0xB1,0x6F,0x6A,0xD3,0xD3,0xFD,0xBC,0xC2,0x2D,0xEC,0x45,0x3F,
        0x09,0x3F,0x58,0xBE,0xFC,0xD4,0xEF,0x00,0x8C,0x81,0x35,0x72,0xBF,0xF7,0x18,0xEA,0x96,0x62,0x7D,0x2B,0x28,0x7F,0x15,0x6C,0x63,0xD2,0xCA,0xCA,0x7D,0x05,0xAC,0xC8,0x6D,0x07,0x6D,0x32,
        0xBE,0x68,0xB8,0x05,0x40,0xAE,0x54,0x98,0x56,0x3E,0x66,0xF1,0x30,0xE8,0xEF,0xC4,0xAB,0x93,0x5E,0x07,0xDE,0x32,0x8F,0x12,0x74,0xAA,0x5B,0x34,0x23,0x54,0xC0,0xEA,0x6C,0xCE,0xFE,0x36,
        0x92,0xA8,0x09,0x17,0xEA,0xA1,0x2D,0xCF,0x6C,0xE3,0x84,0x1D,0xDE,0x87,0x2E,0x33,0x0B,0x3C,0x74,0xE2,0x21,0x50,0x38,0x95,0x2E,0x5C,0xE0,0xE5,0xC6,0x31,0xF9,0xDB,0x40,0xFA,0x6A,0xA1,
        0xA4,0x8A,0x93,0x9B,0xA7,0x21,0x06,0x87,0x1D,0x27,0xD3,0xC4,0xA1,0xC9,0x4C,0xB0,0x6F,0x02,0x03,0x01,0x00,0x01,0xA3,0x81,0xBB,0x30,0x81,0xB8,0x30,0x0E,0x06,0x03,0x55,0x1D,0x0F,0x01,
        0x01,0xFF,0x04,0x04,0x03,0x02,0x01,0x06,0x30,0x0F,0x06,0x03,0x55,0x1D,0x13,0x01,0x01,0xFF,0x04,0x05,0x30,0x03,0x01,0x01,0xFF,0x30,0x42,0x06,0x03,0x55,0x1D,0x20,0x04,0x3B,0x30,0x39,
        0x30,0x37,0x06,0x05,0x60,0x38,0x01,0x01,0x01,0x30,0x2E,0x30,0x2C,0x06,0x08,0x2B,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,0x20,0x68,0x74,0x74,0x70,0x3A,0x2F,0x2F,0x72,0x65,0x70,0x6F,
        0x73,0x69,0x74,0x6F,0x72,0x79,0x2E,0x65,0x69,0x64,0x2E,0x62,0x65,0x6C,0x67,0x69,0x75,0x6D,0x2E,0x62,0x65,0x30,0x1D,0x06,0x03,0x55,0x1D,0x0E,0x04,0x16,0x04,0x14,0x10,0xF0,0x0C,0x56,
        0x9B,0x61,0xEA,0x57,0x3A,0xB6,0x35,0x97,0x6D,0x9F,0xDD,0xB9,0x14,0x8E,0xDB,0xE6,0x30,0x11,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xF8,0x42,0x01,0x01,0x04,0x04,0x03,0x02,0x00,0x07,0x30,
        0x1F,0x06,0x03,0x55,0x1D,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x10,0xF0,0x0C,0x56,0x9B,0x61,0xEA,0x57,0x3A,0xB6,0x35,0x97,0x6D,0x9F,0xDD,0xB9,0x14,0x8E,0xDB,0xE6,0x30,0x0D,0x06,0x09,
        0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0xC8,0x6D,0x22,0x51,0x8A,0x61,0xF8,0x0F,0x96,0x6E,0xD5,0x20,0xB2,0x81,0xF8,0xC6,0xDC,0xA3,0x16,0x00,
        0xDA,0xCD,0x6A,0xE7,0x6B,0x2A,0xFA,0x59,0x48,0xA7,0x4C,0x49,0x37,0xD7,0x73,0xA1,0x6A,0x01,0x65,0x5E,0x32,0xBD,0xE7,0x97,0xD3,0xD0,0x2E,0x3C,0x73,0xD3,0x8C,0x7B,0x83,0xEF,0xD6,0x42,
        0xC1,0x3F,0xA8,0xA9,0x5D,0x0F,0x37,0xBA,0x76,0xD2,0x40,0xBD,0xCC,0x2D,0x3F,0xD3,0x44,0x41,0x49,0x9C,0xFD,0x5B,0x29,0xF4,0x02,0x23,0x22,0x5B,0x71,0x1B,0xBF,0x58,0xD9,0x28,0x4E,0x2D,
        0x45,0xF4,0xDA,0xE7,0xB5,0x63,0x45,0x44,0x11,0x0D,0x2A,0x7F,0x33,0x7F,0x36,0x49,0xB4,0xCE,0x6E,0xA9,0x02,0x31,0xAE,0x5C,0xFD,0xC8,0x89,0xBF,0x42,0x7B,0xD7,0xF1,0x60,0xF2,0xD7,0x87,
        0xF6,0x57,0x2E,0x7A,0x7E,0x6A,0x13,0x80,0x1D,0xDC,0xE3,0xD0,0x63,0x1E,0x3D,0x71,0x31,0xB1,0x60,0xD4,0x9E,0x08,0xCA,0xAB,0xF0,0x94,0xC7,0x48,0x75,0x54,0x81,0xF3,0x1B,0xAD,0x77,0x9C,
        0xE8,0xB2,0x8F,0xDB,0x83,0xAC,0x8F,0x34,0x6B,0xE8,0xBF,0xC3,0xD9,0xF5,0x43,0xC3,0x64,0x55,0xEB,0x1A,0xBD,0x36,0x86,0x36,0xBA,0x21,0x8C,0x97,0x1A,0x21,0xD4,0xEA,0x2D,0x3B,0xAC,0xBA,
        0xEC,0xA7,0x1D,0xAB,0xBE,0xB9,0x4A,0x9B,0x35,0x2F,0x1C,0x5C,0x1D,0x51,0xA7,0x1F,0x54,0xED,0x12,0x97,0xFF,0xF2,0x6E,0x87,0x7D,0x46,0xC9,0x74,0xD6,0xEF,0xEB,0x3D,0x7D,0xE6,0x59,0x6E,
        0x06,0x94,0x04,0xE4,0xA2,0x55,0x87,0x38,0x28,0x6A,0x22,0x5E,0xE2,0xBE,0x74,0x12,0xB0,0x04,0x43,0x2A
    }
};


static const char *get_ext_sn( X509_EXTENSION *extension ) 
{ 
    ASN1_OBJECT *obj = NULL; 
    int nid = NID_undef; 
    const char *ext_sn; 

    obj = X509_EXTENSION_get_object( extension ); 
    if ( !obj ) return NULL; 

    nid = OBJ_obj2nid( obj ); 
    if ( NID_undef == nid ) return NULL; 

    ext_sn = OBJ_nid2sn( nid ); 
    return ext_sn; 
} 

void CVerify::InitX509Store(spc_x509store_t *spc_store)
{
  //spc_store->cafile = 0;
  //spc_store->capath = 0;
  //spc_store->crlfile = 0;
  spc_store->callback = 0;
  spc_store->certs = sk_X509_new_null();
  spc_store->crls = sk_X509_CRL_new_null();
  spc_store->use_certfile = 0;
  spc_store->use_certs = sk_X509_new_null();
  spc_store->use_keyfile = 0;
  spc_store->use_key = 0;
  spc_store->flags = 0;
}

void CVerify::CleanupX509store(spc_x509store_t *spc_store)
{
    int i = 0;
  //if (spc_store->cafile) free(spc_store->cafile);
  //if (spc_store->capath) free(spc_store->capath);
  //if (spc_store->crlfile) free(spc_store->crlfile);
  if (spc_store->use_certfile) free(spc_store->use_certfile);
  if (spc_store->use_keyfile) free(spc_store->use_keyfile);
  if (spc_store->use_key) EVP_PKEY_free(spc_store->use_key);
  for (i = 0; i < sk_X509_num(spc_store->certs); i++)
  {
     X509 *pX509 = sk_X509_value(spc_store->certs, i);
     if (pX509 != NULL)
     {
         X509_free(pX509);
         pX509 = NULL;
     }
  }
  sk_X509_free(spc_store->certs);
  for (i = 0; i < sk_X509_num(spc_store->crls); i++)
  {
     X509_CRL *pX509CRL = sk_X509_CRL_value(spc_store->crls, i);
     if (pX509CRL != NULL)
     {
         X509_CRL_free(pX509CRL);
         pX509CRL = NULL;
     }
  }
  sk_X509_free(spc_store->crls);
  sk_X509_free(spc_store->use_certs);
}

/*
void spc_x509store_setcafile(spc_x509store_t *spc_store, char *cafile) 
{
  if (spc_store->cafile) free(spc_store->cafile);
  spc_store->cafile = (cafile ? strdup(cafile) : 0);
}
   
void spc_x509store_setcapath(spc_x509store_t *spc_store, char *capath) 
{
  if (spc_store->capath) free(spc_store->capath);
  spc_store->capath = (capath ? strdup(capath) : 0);
}
   
void spc_x509store_setcrlfile(spc_x509store_t *spc_store, char *crlfile) 
{
  if (spc_store->crlfile) free(spc_store->crlfile);
  spc_store->crlfile = (crlfile ? strdup(crlfile) : 0);
}
*/

void CVerify::X509StoreAddcert(spc_x509store_t *spc_store, X509 *cert)
{
    // Check Certificate already in store
    bool bFound = false;
    for (int i = 0; i < sk_X509_num(spc_store->certs); i++)
    {
        X509 *pX509 = sk_X509_value(spc_store->certs, i);
        if (X509_V_OK == X509_cmp(pX509, cert))
        {
            bFound = true;
            X509_free(cert);
            break;
        }
    }
    if(!bFound)
        sk_X509_push(spc_store->certs, cert);
}
   
void CVerify::X509StoreAddCRL(spc_x509store_t *spc_store, X509_CRL *crl)
{
    // Check CRL already in store
    bool bFound = false;
    for (int i = 0; i < sk_X509_num(spc_store->crls); i++)
    {
        X509_CRL *pX509CRL = sk_X509_CRL_value(spc_store->crls, i);
        if (X509_V_OK == X509_CRL_cmp(pX509CRL, crl))
        {
            bFound = true;
            X509_CRL_free(crl);
            break;
        }
    }
    if(!bFound)
        sk_X509_CRL_push(spc_store->crls, crl);
}

void CVerify::X509StoreSetcallback(spc_x509store_t *spc_store, spc_x509verifycallback_t callback)
{
  spc_store->callback = callback;
}
   
void CVerify::X509StoreSetflags(spc_x509store_t *spc_store, int flags)
{
  spc_store->flags |= flags;
}
   
void CVerify::X509StoreClearflags(spc_x509store_t *spc_store, int flags)
{
  spc_store->flags &= ~flags;
}

X509_STORE *CVerify::CreateX509store(spc_x509store_t *spc_store)
{
  int i = 0;
  X509_STORE *store = 0;
  X509_LOOKUP *lookup = 0;
   
  store = X509_STORE_new();
  if (spc_store->callback)
    X509_STORE_set_verify_cb_func(store, spc_store->callback);
   
  if (!(lookup = X509_STORE_add_lookup(store, X509_LOOKUP_file())))
    goto error_exit;

  /* Not Used
  if (!spc_store->cafile) 
  {
    if (!(spc_store->flags & SPC_X509STORE_NO_DEFAULT_CAFILE))
      X509_LOOKUP_load_file(lookup, 0, X509_FILETYPE_DEFAULT);
  } 
  else if (!X509_LOOKUP_load_file(lookup, spc_store->cafile, X509_FILETYPE_PEM))
    goto error_exit;
   
  if (spc_store->crlfile) 
  {
    if (!X509_load_crl_file(lookup, spc_store->crlfile, X509_FILETYPE_PEM))
      goto error_exit;
  }
  */ 

   X509_STORE_set_flags(store, spc_store->flags);

  if (!(lookup = X509_STORE_add_lookup(store, X509_LOOKUP_hash_dir())))
    goto error_exit;

  /* Not Used
  if (!spc_store->capath) 
  {
    if (!(spc_store->flags & SPC_X509STORE_NO_DEFAULT_CAPATH))
      X509_LOOKUP_add_dir(lookup, 0, X509_FILETYPE_DEFAULT);
  } 
  else if (!X509_LOOKUP_add_dir(lookup, spc_store->capath, X509_FILETYPE_PEM))
    goto error_exit;
  */
  
  for (i = 0; i < sk_X509_num(spc_store->certs); i++)
    if (!X509_STORE_add_cert(store, sk_X509_value(spc_store->certs, i)))
      goto error_exit;
  for (i = 0; i < sk_X509_CRL_num(spc_store->crls); i++)
    if (!X509_STORE_add_crl(store, sk_X509_CRL_value(spc_store->crls, i)))
      goto error_exit;
   
  return store;
   
error_exit:
  if (store) X509_STORE_free(store);
  return 0;
}

X509 *CVerify::DownloadCert(X509 *pCert)
{
    X509 *pX509 = NULL;
    X509_NAME * pName = X509_get_issuer_name(pCert);
    const char *pszCertStorePath = eidlib::CConfig::GetCertStorePath();
    const char *pszHttpStorePath = eidlib::CConfig::GetHttpStore();
    if(pName != NULL && pszCertStorePath != NULL && pszHttpStorePath != NULL)
    {
        char *pszName = 0;
        char szLocation[256] = {0};
        char szFile[256] = {0};
        pszName = X509_NAME_oneline(pName, NULL, 0);
        if(NULL != strstr(pszName, "Government CA"))
        {
            int i = 0;
            char **pszGovCerts = eidlib::CConfig::GetGovCerts();
            if(pszGovCerts != NULL)
            {
                while(pszGovCerts[i] != NULL)
                {
                    char *pszDummy = pszGovCerts[i++];
                    sprintf(szLocation, "%s%s.crt", pszHttpStorePath, pszDummy);
                    sprintf(szFile, "%s%s.der", pszCertStorePath, pszDummy);
                    wxString fFile = wxFindFirstFile(szFile);
                    if( fFile.IsEmpty() )
                    {
                        if(NULL != (pX509 = RetrieveCert(szLocation, szFile)))
                        {
                            if(X509_V_OK == X509_check_issued(pX509, pCert))
                            {
                                break;
                            }
                            X509_free(pX509);
                            pX509 = NULL;
                        }
                    }
                }
                free(pszGovCerts[0]);
                free(pszGovCerts);
            }
        }
        else if(NULL != strstr(pszName, "Belgium Root CA"))
        {
            int i = 0;
            char **pszRootCerts = eidlib::CConfig::GetRootCerts();
            if(pszRootCerts != NULL)
            {
                while(pszRootCerts[i] != NULL)
                {
                    char *pszDummy = pszRootCerts[i++];
                    sprintf(szLocation, "%s%s.crt", pszHttpStorePath, pszDummy);
                    sprintf(szFile, "%s%s.der", pszCertStorePath, pszDummy);
                    wxString fFile = wxFindFirstFile(szFile);
                    if( fFile.IsEmpty() )
                    {
                        if(NULL != (pX509 = RetrieveCert(szLocation, szFile)))
                        {
                            if(X509_V_OK == X509_check_issued(pX509, pCert))
                            {
                                break;
                            }
                            X509_free(pX509);
                            pX509 = NULL;
                        }
                    }
                }
                free(pszRootCerts[0]);
                free(pszRootCerts);
            }            
        }
	    OPENSSL_free (pszName);
	    pszName = NULL;
    }
    return pX509;
}

X509 *CVerify::FindLocalIssuer(X509 *pCert)
{
    X509 *pX509 = NULL;
    const char *pszCertStorePath = eidlib::CConfig::GetCertStorePath();
    if(NULL == pszCertStorePath)
        return NULL;

    wxString fFile = wxFindFirstFile(wxString::Format("%s*.der", pszCertStorePath));
    while ( !fFile.IsEmpty() )
    {
        wxFileName fName(fFile);
        FILE *pf = NULL;
        pf = fopen(fName.GetFullPath(), "r+b");
        if(NULL != pf)
        {
            unsigned char szBuffer[2048] = {0};
            CByteArray oByteArray;
            int numread = 0;
            while( !feof(pf) )
            {                    
                numread = fread(szBuffer, sizeof(unsigned char), sizeof(szBuffer), pf);
                if(numread > 0)
                {
                    oByteArray.Append(szBuffer, numread); 
                    numread = 0;
                }
            }
            if(oByteArray.GetSize() > 0)
            {
                unsigned char *pDummy = oByteArray.GetData();
                if (NULL != (pX509 = d2i_X509(&pX509, (unsigned char **)&pDummy, oByteArray.GetSize())))
                {
                    if(X509_V_OK == X509_check_issued(pX509, pCert))
                    {
                        fclose(pf);
                        break;
                    }
                    X509_free(pX509);
                }
                pX509 = NULL;
            }
            fclose(pf);
        }
        fFile = wxFindNextFile();
    }
/*
    // Native Windows way

    WIN32_FIND_DATAA FindFileData = {0};
    HANDLE hFind = 0;
    BOOL bNext = TRUE;
    const char *pszCertStorePath = CConfig::GetCertStorePath();
    if(NULL == pszCertStorePath)
        return NULL;

    char szFind[256] = {0};
    sprintf(szFind, "%s*.der", pszCertStorePath);
    hFind = FindFirstFileA(szFind, &FindFileData);
    while(INVALID_HANDLE_VALUE != hFind && bNext)
    {
        FILE *pf = NULL;
        char szFile[256] = {0};
        sprintf(szFile, "%s%s", pszCertStorePath, FindFileData.cFileName);
        pf = fopen(szFile, "r+b");
        if(NULL != pf)
        {
            unsigned char szBuffer[2048] = {0};
            CByteArray oByteArray;
            int numread = 0;
            while( !feof(pf) )
            {                    
                numread = fread(szBuffer, sizeof(unsigned char), sizeof(szBuffer), pf);
                if(numread > 0)
                {
                    oByteArray.Append(szBuffer, numread); 
                    numread = 0;
                }
            }
            if(oByteArray.GetSize() > 0)
            {
                unsigned char *pDummy = oByteArray.GetData();
                if (NULL != (pX509 = d2i_X509(&pX509, (unsigned char **)&pDummy, oByteArray.GetSize())))
                {
                    if(X509_V_OK == X509_check_issued(pX509, pCert))
                    {
                        fclose(pf);
                        break;
                    }
                    X509_free(pX509);
                }
                pX509 = NULL;
            }
            fclose(pf);
        }
         bNext = FindNextFileA(hFind, &FindFileData);
    }
    FindClose(hFind);
*/
    if(NULL == pX509)
    {
        // Not Found try to download
        pX509 = DownloadCert(pCert);
    }
    return pX509;
}


// Obsolete
char *CVerify::GetExtensionValue(X509 *pCert, char *pszValue, char *pszName)
{
  int extcount, i, j;
  const char *extstr;
  CONF_VALUE *nval = NULL;
  X509_EXTENSION *ext = NULL;
  X509V3_EXT_METHOD *meth = NULL;
  STACK_OF(CONF_VALUE) *val = NULL;
   
  if ((extcount = X509_get_ext_count(pCert)) > 0) 
  {
    for (i = 0; i < extcount; i++) 
    {
      ext = X509_get_ext(pCert, i);
      if(ext != NULL)
      {
          extstr  = get_ext_sn( ext ); 
          //extstr = OBJ_nid2sn(OBJ_obj2nid(X509_EXTENSION_get_object(ext)));
          if (extstr == NULL || 0 != strcmp(extstr, pszValue))
              continue;
          if (!(meth = X509V3_EXT_get(ext))) break;

          val = meth->i2v(meth, X509V3_EXT_d2i(ext), 0);
          for (j = 0;  j < sk_CONF_VALUE_num(val);  j++) 
          {
            nval = sk_CONF_VALUE_value(val, j);
            if (0 == strcmp(nval->name, pszName))
            {
                if(val != NULL)
                {
                    sk_CONF_VALUE_free(val);
                    val = NULL;
                }
                return strdup(nval->value);
            }
          }
          if(val != NULL)
          {
              sk_CONF_VALUE_free(val);
              val = NULL;
          }
        }
    }
  }
  return 0;
}

char *CVerify::GetExtensionValue2(X509 *pCert, int iNID)
{
    int i, j;
    struct stack_st *pStack = NULL;
    const char *pData = NULL;
    BOOL bFound = FALSE;

    pStack = (struct stack_st *) X509_get_ext_d2i(pCert, iNID, NULL, NULL);

    if(pStack != NULL)
    {
        if(iNID == NID_crl_distribution_points)
        {
            for(j = 0; j < sk_DIST_POINT_num(pStack) && !bFound; j++) 
            {
                DIST_POINT *pRes = (DIST_POINT *)sk_DIST_POINT_value(pStack, j);
                if(pRes != NULL)
                {
                    STACK_OF(GENERAL_NAME) *pNames = pRes->distpoint->name.fullname;
                    if(pNames)
                    {
                        for(i = 0; i < sk_GENERAL_NAME_num(pNames) && !bFound; i++) 
                        {
                            GENERAL_NAME *pName = sk_GENERAL_NAME_value(pNames, i);
                            if(pName != NULL && pName->type == GEN_URI )
                            {
                                pData = (const char *)ASN1_STRING_data(pName->d.uniformResourceIdentifier); 
                                bFound = TRUE;
                            }
                         }
                        sk_GENERAL_NAME_free(pNames);
                    }
                }
            } 
            sk_DIST_POINT_free(pStack);
        }
        else if(iNID == NID_info_access)
        {
            for(j = 0; j < sk_ACCESS_DESCRIPTION_num(pStack) && !bFound; j++) 
            {
                ACCESS_DESCRIPTION *pAccess = sk_ACCESS_DESCRIPTION_value(pStack, j);
                if(pAccess != NULL && pAccess->method != NULL && OBJ_obj2nid(pAccess->method) == NID_ad_OCSP)
                {
                    GENERAL_NAME *pName = pAccess->location;
                    if(pName != NULL && pName->type == GEN_URI)
                    {
                        pData = (const char *)ASN1_STRING_data(pName->d.uniformResourceIdentifier); 
                        bFound = TRUE;
                    }
                }
            }
            sk_ACCESS_DESCRIPTION_free(pStack);
        }
    }

    if(bFound)
    {
        return strdup(pData);
    }
    return NULL;
}

int CVerify::VerifyCertHostname(X509 *pCert, char *pszHostname)
{
    struct stack_st *pStack = NULL;
    BOOL bFound = FALSE;

    pStack = (struct stack_st *) X509_get_ext_d2i(pCert, NID_subject_alt_name, NULL, NULL);
    if(pStack != NULL)
    {
        int i, iLen1, iLen2;
	    char *pszDomain = NULL;
	    iLen1 = strlen(pszHostname);
	    pszDomain = strchr(pszHostname,  '.');
	    if (pszDomain)
        {
	    	iLen2 = iLen1 - (pszDomain - pszHostname);
        }

        for(i = 0; i < sk_GENERAL_NAME_num(pStack) && !bFound; i++) 
        {
            GENERAL_NAME *pName = sk_GENERAL_NAME_value(pStack, i);
            if(pName != NULL && pName->type == GEN_DNS )
            {
                char *pData = (char *)ASN1_STRING_data(pName->d.ia5); 
                int iLen = ASN1_STRING_length(pName->d.ia5);

                // Is this an exact match ?
                if ((iLen1 == iLen) && !strnicmp(pszHostname, pData, iLen1))
                {
                    bFound = TRUE;
                }
		        // Is this a wildcard match ?
		        else if ((*pData == '*') && pszDomain && (iLen2 == iLen - 1) && !strnicmp(pszDomain, pData + 1, iLen2))
                {
                    bFound = TRUE;
                }
            }
         }
        sk_GENERAL_NAME_free(pStack);
    }
    if(!bFound)
    {
        char pszName[256] = {0};
        X509_NAME *pxName = NULL;
        pxName = X509_get_subject_name(pCert);
        if(pxName &&  X509_NAME_get_text_by_NID(pxName, NID_commonName, pszName, sizeof(pszName)) > 0) 
        {
            pszName[sizeof(pszName) - 1] = '\0';
            if (!stricmp(pszName, pszHostname)) 
            {
                bFound = TRUE;
            } 
         }
    }

    return bFound;
}


BIO *CVerify::Connect(char *pszHost, int iPort, int iSSL, SSL_CTX **ppSSLCtx) 
{
    BIO *pConn = NULL;
   
    if (iSSL) 
    {
        if (!(pConn = ConnectSSL(pszHost, iPort, NULL, ppSSLCtx))) goto error_exit;
        //SSL *pSSL = NULL;
        //BIO_get_ssl(pConn, &pSSL);
        //if (pSSL && !VerifyCertHostname(SSL_get_peer_certificate(pSSL), pszHost)) goto error_exit;
        //if (SSL_get_verify_result(pSSL) != X509_V_OK) goto error_exit;
        return pConn;
    }

    *ppSSLCtx = 0;
    if (!(pConn = BIO_new_connect(pszHost))) goto error_exit;
    BIO_set_conn_int_port(pConn, &iPort);
    if (BIO_do_connect(pConn) <= 0) goto error_exit;
    return pConn;
   
error_exit:
    if (pConn) BIO_free_all(pConn);
    return 0;
}

BIO *CVerify::ConnectSSL(char *pszHost, int iPort, spc_x509store_t *pStore, SSL_CTX **ppSSLCtx) 
{
    BIO *pConn = NULL;
    OpenSSL_add_all_algorithms();
    *ppSSLCtx = SSL_CTX_new(SSLv23_client_method());
    
    if (!(pConn = BIO_new_ssl_connect(*ppSSLCtx))) goto error_exit;
    BIO_set_conn_hostname(pConn, pszHost);
    BIO_set_conn_int_port(pConn, &iPort);
   
    if (BIO_do_connect(pConn) <= 0) goto error_exit;  
    return pConn;
   
error_exit:
  if (pConn) BIO_free_all(pConn);
  if (*ppSSLCtx) 
  {
      SSL_CTX_free(*ppSSLCtx);
      *ppSSLCtx = NULL;
  }
  return 0;
}


void *CVerify::RetrieveWebdata(char *pszUri, int *piDataLen) 
{
    wxCriticalSectionLocker locker(m_cs);

    int iBytes = 0;
    int iContentLength = 0;
    int iHeaderLen = 0;
    int iSd = 0;
    int iSSL = 0;
    BIO *pConn = 0;
    SSL *pSSL = 0;
    SSL_CTX *pSSLCtx = 0;
    char szBuffer[1024] = {0};
    char *pszHeaders = 0;
    char *pszHost = 0;
    char *pszPath = 0;
    char *pszPort = 0;
    char *pszTmp = 0;
    void  *pvData = 0;
    fd_set  rmask, wmask;
    const char *pszConfigHost = eidlib::CConfig::GetProxyHost();
    const char *pszConfigPort = eidlib::CConfig::GetProxyPort() ;
    
    *piDataLen = 0;

    if(pszConfigHost != NULL && strlen(pszConfigHost) > 0)
    {
        pszHost = (char *)OPENSSL_malloc(strlen(pszConfigHost) + 1);
        memset(pszHost, 0, strlen(pszConfigHost) + 1);
        strcpy(pszHost, pszConfigHost); 
    }
    if(pszConfigPort != NULL && strlen(pszConfigPort) > 0)
    {
        pszPort = (char *)OPENSSL_malloc(strlen(pszConfigPort) + 1);
        memset(pszPort, 0, strlen(pszConfigPort) + 1);
        strcpy(pszPort, pszConfigPort); 
    }

     if(pszHost == NULL || pszPort == NULL)
    {
        if (!OCSP_parse_url(pszUri, &pszHost, &pszPort, &pszPath, &iSSL)) goto end_error;
     }
     else
     {
        pszPath = (char *)OPENSSL_malloc(strlen(pszUri) + 1);
        memset(pszPath, 0, strlen(pszUri) + 1);
        strcpy(pszPath, pszUri); 
     }

    if (!(pConn = Connect(pszHost, atoi(pszPort), iSSL, &pSSLCtx))) goto end_error;
   
    // Send the request for the data
    BIO_printf(pConn, "GET %s HTTP/1.0\r\nConnection: close\r\n\r\n", pszPath);
   
    // Put the socket into non-blocking mode
    BIO_get_fd(pConn, &iSd);
    BIO_socket_nbio(iSd, 1);
    if (iSSL) 
    {
        BIO_get_ssl(pConn, &pSSL);
        SSL_set_mode(pSSL, SSL_MODE_ENABLE_PARTIAL_WRITE | SSL_MODE_ACCEPT_MOVING_WRITE_BUFFER);
    }
   
    // Loop reading data from the socket until we've got all of the headers
    for (;;) 
    {
        FD_ZERO(&rmask);
        FD_SET(iSd, &rmask);
        FD_ZERO(&wmask);
        if (BIO_should_write(pConn)) 
            FD_SET(iSd, &wmask);
        if (select(FD_SETSIZE, &rmask, &wmask, 0, 0) <= 0) continue;
        if (FD_ISSET(iSd, &wmask)) BIO_write(pConn, szBuffer, 0);
        if (FD_ISSET(iSd, &rmask)) 
        {
            if ((iBytes = BIO_read(pConn, szBuffer, sizeof(szBuffer))) <= 0) 
            {
                if (BIO_should_retry(pConn)) continue;
                goto end_error;
            }
            if (!(pszHeaders = (char *)realloc((pszTmp = pszHeaders), iHeaderLen + iBytes))) 
            {
                pszHeaders = pszTmp;
                goto end_error;
            }
            memcpy(pszHeaders + iHeaderLen, szBuffer, iBytes);
            iHeaderLen += iBytes;
            if ((pszTmp = strstr(pszHeaders, "\r\n\r\n")) != 0) 
            {
                *(pszTmp + 2) = '\0';
                *piDataLen = iHeaderLen - ((pszTmp + 4) - pszHeaders);
                iHeaderLen -= (*piDataLen + 2);
                if (*piDataLen > 0) 
                {
                    if (!(pvData = (char *)malloc(*piDataLen))) goto end_error;
                    memcpy(pvData, pszTmp + 4, *piDataLen);
                }
                break;
            }
        }
    }
  
    /* Examine the headers to determine whether or not to continue.  If we are to
    * continue, look for a content-length header to find out how much data we're
    * going to get.  If there is no content-length header, we'll have to read
    * until the remote server closes the connection.
    */
    if (0 == strnicmp(pszHeaders, "HTTP/1.", 7)) 
    {
        if (!(pszTmp = strchr(pszHeaders, ' '))) goto end_error;
        if (strncmp(pszTmp + 1, "200 ", 4) && strncmp(pszTmp + 1, "200\r\n", 5))
            goto end_error;
        for (pszTmp = strstr(pszHeaders, "\r\n");  pszTmp;  pszTmp = strstr(pszTmp + 2, "\r\n")) 
        {
            if (strnicmp(pszTmp + 2, "content-length: ", 16)) continue;
            iContentLength = atoi(pszTmp + 18);
            break;
        }
    } 
    else 
        goto end_error;
   
    /* Continuously read and accumulate data from the remote server.  Finish when
    * we've read up to the content-length that we received.  If we didn't receive
    * a content-length, read until the remote server closes the connection.
    */

    while (!iContentLength || *piDataLen < iContentLength) 
    {
        FD_ZERO(&rmask);
        FD_SET(iSd, &rmask);
        FD_ZERO(&wmask);
        if (BIO_should_write(pConn)) FD_SET(iSd, &wmask);
        if (select(FD_SETSIZE, &rmask, &wmask, 0, 0) <= 0) continue;
        if (FD_ISSET(iSd, &wmask)) BIO_write(pConn, szBuffer, 0);
        if (FD_ISSET(iSd, &rmask))
        if ((iBytes = BIO_read(pConn, szBuffer, sizeof(szBuffer))) <= 0) 
        {
            if (BIO_should_retry(pConn)) continue;
            break;
        }
        if (!(pvData = realloc((pszTmp = (char *)pvData), *piDataLen + iBytes))) 
        {
            pvData = pszTmp;
            goto end_error;
        }

        memcpy((char *)pvData + *piDataLen, szBuffer, iBytes);
        *piDataLen += iBytes;
    }
   
    if (iContentLength && *piDataLen != iContentLength) goto end_error;
        goto end;
   
end_error:
  if (pvData) 
  { 
      free(pvData);  
      pvData = 0;  
      *piDataLen = 0; 
  }
end:
  if (pszHeaders) free(pszHeaders);
  if (pConn) BIO_free_all(pConn);
  if (pszHost) OPENSSL_free(pszHost);
  if (pszPort) OPENSSL_free(pszPort);
  if (pszPath) OPENSSL_free(pszPath);
  if (pSSLCtx) SSL_CTX_free(pSSLCtx);
  return pvData;
}

X509_CRL *CVerify::RetrieveCRL(char *pszUri, char *pszFile) 
{
    BIO *pBio = 0;
    int  iDataLen = 0;
    void  *pvData = 0;
    X509_CRL *pCRL = 0;
    FILE *pf = 0;
    
    if (!(pvData = RetrieveWebdata(pszUri, &iDataLen))) 
    {
        goto end;
    }
   
    // Store on disk
    pf = fopen(pszFile, "w+b");
    if(NULL != pf)
    {
        int iNumWritten = 0;
        iNumWritten = fwrite(pvData, sizeof(unsigned char), iDataLen, pf);
        fclose(pf);
    }
    pBio = BIO_new_mem_buf(pvData, iDataLen);
    pCRL = d2i_X509_CRL_bio(pBio, 0);
    
end:
    if (pBio) BIO_free(pBio);
    if (pvData) free(pvData);
    return pCRL;
}

X509 *CVerify::RetrieveCert(char *pszUri, char *pszFile) 
{
    BIO *pBio = 0;
    int  iDataLen = 0;
    void  *pvData = 0;
    X509 *pX509 = 0;
    FILE *pf = 0;
    
    if (!(pvData = RetrieveWebdata(pszUri, &iDataLen))) 
    {
        goto end;
    }
   
    // Store on disk
    pf = fopen(pszFile, "w+b");
    if(NULL != pf)
    {
        int iNumWritten = 0;
        iNumWritten = fwrite(pvData, sizeof(unsigned char), iDataLen, pf);
        fclose(pf);
    }
    pBio = BIO_new_mem_buf(pvData, iDataLen);
    pX509 = d2i_X509_bio(pBio, 0);
    
end:
    if (pBio) BIO_free(pBio);
    if (pvData) free(pvData);
    return pX509;
}

X509_CRL *CVerify::FindLocalCRL(char *pCRLDistribution)
{
    X509_CRL *pX509CRL = NULL;
    bool bDownload = false;
    const char *pszCRLStorePath = eidlib::CConfig::GetCRLStorePath();

    if(pCRLDistribution != NULL && pszCRLStorePath != NULL)
    {
        char *pTemp; 
        char *pFile = pCRLDistribution;
        while(NULL != (pTemp = strstr(pFile, "/")))
        {
            pFile = ++pTemp;
        }

        FILE *pf = NULL;
        char szFile[256] = {0};
        sprintf(szFile, "%s%s", pszCRLStorePath, pFile);
        pf = fopen(szFile, "r+b");
        if(NULL != pf)
        {
            unsigned char szBuffer[4096] = {0};
            CByteArray oByteArray;
            int numread = 0;
            while( !feof(pf) )
            {                    
                numread = fread(szBuffer, sizeof(unsigned char), sizeof(szBuffer), pf);
                if(numread > 0)
                {
                    oByteArray.Append(szBuffer, numread); 
                    numread = 0;
                }
            }
            if(oByteArray.GetSize() > 0)
            {
                unsigned char *pDummy = oByteArray.GetData();
                pX509CRL = d2i_X509_CRL(&pX509CRL, (unsigned char **)&pDummy, oByteArray.GetSize());
                // Check period
                if(pX509CRL != NULL)
                {
                    int iCheck = 0;
	                iCheck = X509_cmp_time(X509_CRL_get_lastUpdate(pX509CRL), NULL);
	                if (iCheck >= 0)
		            {
                        // X509_V_ERR_CRL_NOT_YET_VALID or X509_V_ERR_ERROR_IN_CRL_LAST_UPDATE_FIELD error
                        bDownload = true;
		            }

	                if(X509_CRL_get_nextUpdate(pX509CRL))
		            {
		                iCheck = X509_cmp_time(X509_CRL_get_nextUpdate(pX509CRL), NULL);
		                if (iCheck <= 0)
			            {
                            // X509_V_ERR_CRL_HAS_EXPIRED or X509_V_ERR_ERROR_IN_CRL_NEXT_UPDATE_FIELD error
                            bDownload = true;
			             }
		            }
                }
                else
                {
                    bDownload = true;
                }
            }
            fclose(pf);
        }
        else
        {
            bDownload = true;
        }
        if(bDownload)
        {
            if(pX509CRL != NULL)
            {
                X509_CRL_free(pX509CRL);
                pX509CRL = NULL;
            }
            // Try to download CRL and check again
            pX509CRL = RetrieveCRL(pCRLDistribution, szFile); 
        }
    }
    return pX509CRL;
}

int CVerify::VerifyCert(X509 *cert, X509_STORE *store, std::vector<std::string> & CRLDistr) 
{
  int  result = -1;
  X509_STORE_CTX *ctx = 0;
  X509_CRL *pX509Crl = NULL;

  
  if ((ctx = X509_STORE_CTX_new()) != 0) 
  {
    if (X509_STORE_CTX_init(ctx, store, cert, 0) == 1)
    {
      // Check all issuers
      bool bOk = true;
      X509 *pCertCheck = cert;
      X509 *pIssuer = NULL;
      do
      {
        if(X509_STORE_CTX_get1_issuer(&pIssuer, ctx, pCertCheck))
        {
            if( 0 == strcmp(pIssuer->name, pCertCheck->name) )
            {
                bOk = false; // Break on Selfsigned
            }
            else
            {
                pCertCheck = pIssuer;
                pIssuer = NULL;
            }
        }
        else
        {
            // Lookup Issuer in additional cert store
            X509 *pFound = FindLocalIssuer(pCertCheck);
            if (pFound == NULL)
            {
                // Issuer certificate not found
                bOk = false;
            }
            else
            {
                // Add certificate to store
                X509_STORE_add_cert(store, pFound);
                // Obsolete
                // char *pCRLDistribution = GetExtensionValue(pFound, "crlDistributionPoints", "URI");
                char *pCRLDistribution = GetExtensionValue2(pFound, NID_crl_distribution_points);
                if(NULL != pCRLDistribution)
                {
                    bool bStop = false;
                    for(unsigned int iNr = 0; iNr < CRLDistr.size(); ++iNr)
                    {
                        if(0 == stricmp(CRLDistr[iNr].c_str(), pCRLDistribution))
                        {
                            bStop = true;
                            break;
                        }
                    }
                    if(!bStop)
                    {
                        CRLDistr.push_back(pCRLDistribution);
                        if(NULL != (pX509Crl = FindLocalCRL(pCRLDistribution)))
                        {
                            X509_STORE_add_crl(store, pX509Crl);
                            pX509Crl = NULL;
                        }
                    }
                    free(pCRLDistribution);
                }
            }
        }
      } while (bOk);
      
      result = (X509_verify_cert(ctx) == 1);
    }
    X509_STORE_CTX_free(ctx);
  }
  return result;
}

int CVerify::VerifyCertroot(X509 *cert, X509_STORE *store) 
{
  int  result = -1;
  X509_STORE_CTX *ctx = 0;
  
  if ((ctx = X509_STORE_CTX_new()) != 0) 
  {
    if (X509_STORE_CTX_init(ctx, store, cert, 0) == 1)
    {
      // Check all issuers
      bool bOk = true;
      X509 *pCertCheck = cert;
      X509 *pIssuer = NULL;
      do
      {
        if(X509_STORE_CTX_get1_issuer(&pIssuer, ctx, pCertCheck))
        {
            if( 0 == strcmp(pIssuer->name, pCertCheck->name) )
            {
                // Now Check against hardcoded Root(s)
                X509 *pX509Root = NULL;
                for (int i = 0; i < ROOT_CERT_COUNT; ++i)
                {
                    unsigned char *pucDummy = (unsigned char*)_root_cert[i];
                    if (NULL == (pX509Root = d2i_X509(&pX509Root, &pucDummy, ROOT_CERT_SIZE)))
                        return BEID_SIGNATURE_PROCESSING_ERROR;
                    if(X509_V_OK == X509_cmp(pX509Root, pIssuer))
                    {
                        result = X509_V_OK;
                    }
                    X509_free(pX509Root);
                    pX509Root = NULL;
                }
                bOk = false;
            }
            else
            {
                pCertCheck = pIssuer;
                pIssuer = NULL;
            }
        }
        else
        {
            // Lookup Issuer in additional cert store
            X509 *pFound = FindLocalIssuer(pCertCheck);
            if (pFound == NULL)
            {
                // Issuer certificate not found
                bOk = false;
            }
            else
            {
                // Add certificate to store
                X509_STORE_add_cert(store, pFound);
            }
        }
      } while (bOk);
    }
    X509_STORE_CTX_free(ctx);
  }
  return result;
}

spc_ocspresult_t CVerify::VerifyViaOCSP(spc_ocsprequest_t *data, spc_x509store_t *pStore) 
{
    BIO *pBio = 0;
    int iRetCode = -1;
    int iReason = 0;
    int iSSL = 0;
    int iStatus = 0;
    char *pszHost = 0;
    const char *pszConfigHost = eidlib::CConfig::GetProxyHost();
    char *pszPath = 0;
    char *pszPort = 0;
    const char *pszConfigPort = eidlib::CConfig::GetProxyPort() ;
    SSL_CTX  *pSSLCtx = 0;
    OCSP_CERTID *pCertID;
    OCSP_REQUEST  *pReq = 0;
    OCSP_RESPONSE *pResp = 0;
    OCSP_BASICRESP *pBasic = 0;
    spc_ocspresult_t  tResult = SPC_OCSPRESULT_ERROR_UNKNOWN;
    ASN1_GENERALIZEDTIME  *producedAt, *thisUpdate, *nextUpdate;
    int iResp = 0;
    unsigned char *pResponse;
    unsigned char *pResponseNext;
   
    if(pszConfigHost != NULL && strlen(pszConfigHost) > 0)
    {
        pszHost = (char *)OPENSSL_malloc(strlen(pszConfigHost) + 1);
        memset(pszHost, 0, strlen(pszConfigHost) + 1);
        strcpy(pszHost, pszConfigHost); 
    }
    if(pszConfigPort != NULL && strlen(pszConfigPort) > 0)
    {
        pszPort = (char *)OPENSSL_malloc(strlen(pszConfigPort) + 1);
        memset(pszPort, 0, strlen(pszConfigPort) + 1);
        strcpy(pszPort, pszConfigPort); 
    }

    tResult = SPC_OCSPRESULT_ERROR_UNKNOWN;
    if(pszHost == NULL || pszPort == NULL)
    {
        if (!OCSP_parse_url(data->url, &pszHost, &pszPort, &pszPath, &iSSL)) 
        {
            tResult = SPC_OCSPRESULT_ERROR_BADOCSPADDRESS;
            goto end;
        }
    }
    else
    {
        pszPath = (char *)OPENSSL_malloc(strlen(data->url) + 1);
        memset(pszPath, 0, strlen(data->url) + 1);
        strcpy(pszPath, data->url); 
    }

    if (!(pReq = OCSP_REQUEST_new())) 
    {
        tResult = SPC_OCSPRESULT_ERROR_OUTOFMEMORY;
        goto end;
    }
   
    pCertID = OCSP_cert_to_id(0, data->cert, data->issuer);
    if (!pCertID || !OCSP_request_add0_id(pReq, pCertID)) goto end;
    OCSP_request_add1_nonce(pReq, 0, -1);
   
    /* sign the request */
    if (data->sign_cert && data->sign_key &&
      !OCSP_request_sign(pReq, data->sign_cert, data->sign_key, EVP_sha1(), 0, 0)) 
    {
        tResult = SPC_OCSPRESULT_ERROR_SIGNFAILURE;
        goto end;
    }
   
    /* establish a connection to the OCSP responder */
    if (!(pBio = Connect(pszHost, atoi(pszPort), iSSL, &pSSLCtx))) 
    {
        tResult = SPC_OCSPRESULT_ERROR_CONNECTFAILURE;
        goto end;
    }
   
    /* send the request and get a response */
    pResp = OCSP_sendreq_bio(pBio, pszPath, pReq);
    if(pResp == NULL)
    {
        tResult = SPC_OCSPRESULT_ERROR_INVALIDRESPONSE;
        goto end;
    }

    if ((iRetCode = OCSP_response_status(pResp)) != OCSP_RESPONSE_STATUS_SUCCESSFUL) 
    {
        switch (iRetCode) 
        {
            case OCSP_RESPONSE_STATUS_MALFORMEDREQUEST:
            tResult = SPC_OCSPRESULT_ERROR_MALFORMEDREQUEST; break;
            case OCSP_RESPONSE_STATUS_INTERNALERROR:
            tResult = SPC_OCSPRESULT_ERROR_INTERNALERROR; break;
            case OCSP_RESPONSE_STATUS_TRYLATER:
            tResult = SPC_OCSPRESULT_ERROR_TRYLATER; break;
            case OCSP_RESPONSE_STATUS_SIGREQUIRED:
            tResult = SPC_OCSPRESULT_ERROR_SIGREQUIRED; break;
            case OCSP_RESPONSE_STATUS_UNAUTHORIZED:
            tResult = SPC_OCSPRESULT_ERROR_UNAUTHORIZED; break;
    }
    goto end;
  }
  
    /* verify the response */
    tResult = SPC_OCSPRESULT_ERROR_INVALIDRESPONSE;
    iResp = i2d_OCSP_RESPONSE(pResp, NULL);
    if(iResp > 0)
    {
        pResponse = pResponseNext = (unsigned char *)malloc(iResp);
        iResp = i2d_OCSP_RESPONSE(pResp, &pResponseNext);
        // pResponse is the result
        free(pResponse);
    }

    if (!(pBasic = OCSP_response_get1_basic(pResp))) goto end;
    if (OCSP_check_nonce(pReq, pBasic) <= 0) goto end;
    if ((iRetCode = OCSP_basic_verify(pBasic, 0, data->store, 0)) <= 0)
    {
        if(!eidlib::CConfig::AllowTestRoot()) 
            goto end;
    }

    if (!OCSP_resp_find_status(pBasic, pCertID, &iStatus, &iReason, &producedAt,
                             &thisUpdate, &nextUpdate))
        goto end;

    /*if (!OCSP_check_validity(thisUpdate, nextUpdate, data->skew, data->maxage))
    {
        goto end;
    }
  */

    /* All done.  Set the return code based on the status from the response. */
    if (iStatus == V_OCSP_CERTSTATUS_REVOKED)
        tResult = SPC_OCSPRESULT_CERTIFICATE_REVOKED;
    else
        tResult = SPC_OCSPRESULT_CERTIFICATE_VALID;
  
end:
    if (pBio) BIO_free_all(pBio);
    if (pszHost) OPENSSL_free(pszHost);
    if (pszPort) OPENSSL_free(pszPort);
    if (pszPath) OPENSSL_free(pszPath);
    if (pReq) OCSP_REQUEST_free(pReq);
    if (pResp) OCSP_RESPONSE_free(pResp);
    if (pBasic) OCSP_BASICRESP_free(pBasic);
    if (pSSLCtx) SSL_CTX_free(pSSLCtx);
    return tResult;
}


X509* CVerify::FindIssuer(X509 *cert, X509_STORE *store)
{
  X509 *pIssuerRet = NULL;
  X509_STORE_CTX *ctx = 0;
  
  if ((ctx = X509_STORE_CTX_new()) != 0) 
  {
    if (X509_STORE_CTX_init(ctx, store, cert, 0) == 1)
    {
      // Check all issuers
      bool bOk = true;
      X509 *pCertCheck = cert;
      X509 *pIssuer = NULL;
      do
      {
        if(X509_STORE_CTX_get1_issuer(&pIssuer, ctx, pCertCheck))
        {
            if(pIssuerRet == NULL)
            {
                pIssuerRet = pIssuer;
            }
            if( 0 == strcmp(pIssuer->name, pCertCheck->name) )
            {
                bOk = false; // Break on Selfsigned
            }
            else
            {
                pCertCheck = pIssuer;
                pIssuer = NULL;
            }

        }
        else
        {
            // Lookup Issuer in additional cert store
            X509 *pFound = FindLocalIssuer(pCertCheck);
            if (pFound == NULL)
            {
                // Issuer certificate not found
                bOk = false;
            }
            else
            {
                // Add certificate to store
                X509_STORE_add_cert(store, pFound);
            }
        }
      } while (bOk);
    }
    X509_STORE_CTX_free(ctx);
  }

    return pIssuerRet;
}

/* Old Helper Functions

int GetFingerprintCert(X509 *pCert, const EVP_MD *pDigest, unsigned char *pucFingerprint,
                        unsigned int *piFingerprintLen) 
{
    if (*piFingerprintLen < (unsigned int)EVP_MD_size((EVP_MD *)pDigest))
        return 0;
    if (!X509_digest(pCert, pDigest, pucFingerprint, piFingerprintLen))
        return 0;
    return *piFingerprintLen;
}
   
int FingerprintEqual(unsigned char *pucFP1, int iFP1len, unsigned char *pucFP2, int iFP2len) 
{
    return (iFP1len == iFP2len && !memcmp(pucFP1, pucFP2, iFP1len));
}
*/

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CCertifManager *CVerify::m_pCertifManager = NULL;
wxCriticalSection CVerify::m_cs;

CVerify::CVerify()
{

}

CVerify::~CVerify()
{

}


int CVerify::VerifySignature(const unsigned char *pucData, unsigned long ulDataLen, 
                                 const unsigned char *pucSig, unsigned long ulSigLen)
{
    int iRet = BEID_SIGNATURE_PROCESSING_ERROR;
    EVP_MD_CTX cmd_ctx;
    EVP_PKEY *pKey = NULL;
    X509 *pX509 = NULL;
    int  iDiffRNCert = 1;
    eidlib::CCertif *pCertRN = NULL;
    unsigned char *pucRNCert = NULL;
    unsigned long ulRNCertLen = 0;
    BEID_Certif_Check tCertifs = {0};

    if(m_pCertifManager == NULL)
    {
        return BEID_SIGNATURE_PROCESSING_ERROR;
    }
    pCertRN = m_pCertifManager->GetCertif("RN");
    if(pCertRN == NULL)
    {
        return BEID_SIGNATURE_PROCESSING_ERROR;
    }
    pucRNCert = pCertRN->GetData(); 
    ulRNCertLen = pCertRN->GetLength(); 

    m_pCertifManager->FillCertifs(&tCertifs);

    // Fill Certificate Store
    OpenSSL_add_all_algorithms();
    spc_x509store_t *pStore = new spc_x509store_t();
    InitX509Store(pStore);

    for(int i = 0; i < tCertifs.certificatesLength; ++i)
    {
        char *pID = tCertifs.certificates[i].certifLabel;
        if(0 != strcmp(pID, "RN"))
        {
            unsigned char *pData = tCertifs.certificates[i].certif;
            unsigned long ulLen = tCertifs.certificates[i].certifLength; 
            if (NULL == (pX509 = d2i_X509(&pX509, &pData, ulLen)))
                return BEID_SIGNATURE_PROCESSING_ERROR;
            X509StoreAddcert(pStore, pX509);
            pX509 = NULL;
        }
    }

    if (NULL == (pX509 = d2i_X509(&pX509, (unsigned char **)&pucRNCert, ulRNCertLen)))
        return BEID_SIGNATURE_PROCESSING_ERROR;

/*
// For testing only : good RRN Certificate
        FILE *pf = fopen("rrn1.der", "r+b");
        unsigned char szBuffer[4096] = {0};
        int iread = 0;
        if(pf != NULL)
        {
            iread = fread(szBuffer, sizeof(unsigned char), sizeof(szBuffer), pf);
            fclose(pf);
        }
        unsigned char *pDummy = szBuffer;  
    if (NULL == (pX509 = d2i_X509(&pX509, (unsigned char **)&pDummy, iread)))
        return SIGNATURE_PROCESSING_ERROR;
*/

    if (NULL == (pKey = X509_get_pubkey(pX509)))
        return BEID_SIGNATURE_PROCESSING_ERROR;

    X509_STORE  *store = 0;
    store = CreateX509store(pStore);
    if(X509_V_OK == VerifyCertroot(pX509, store))
    {
        iDiffRNCert = 0;
    }
    X509_STORE_free(store);
    CleanupX509store(pStore);
    delete pStore;

    // Check OID
    if(0 == iDiffRNCert && X509_V_OK != VerifyRRNDN(pX509))
        return BEID_SIGNATURE_INVALID;

    EVP_VerifyInit(&cmd_ctx, EVP_sha1());

/* Old manner
    // Check fingerprints
    unsigned char ucRRNfing[20] = {0};
    unsigned int iRRNfingLen = 20;
    if(GetFingerprintCert(pX509, cmd_ctx.digest, ucRRNfing, &iRRNfingLen))
    {
         for (int i = 0; i < RRN_CERT_COUNT; ++i)
        {
            if (FingerprintEqual(ucRRNfing, iRRNfingLen, _rrn_cert_thumb[i], RRN_CERT_SIZE))
            {
                iDiffRNCert = 0;
                break;
            }
        }
    }
    else
    {
        return SIGNATURE_PROCESSING_ERROR;
    }
*/

    EVP_VerifyUpdate(&cmd_ctx, pucData, ulDataLen);
    iRet = 2*iDiffRNCert + !EVP_VerifyFinal(&cmd_ctx, (unsigned char *)pucSig, ulSigLen, pKey);
    EVP_PKEY_free(pKey);
    X509_free(pX509);
    return iRet;
}
void CVerify::UpdateCertStatus(const std::vector<eidlib::CCertif *> & Certifs, void *pCertX509, long lStatus)
{
    bool bFound = false;
    X509 *pX509 = NULL;
    for(int unsigned i = 0; i < Certifs.size() && !bFound; ++i)
    {
        eidlib::CCertif *pCertif = Certifs[i];
        if(pCertif != NULL)
        {
            unsigned char *pData = pCertif->GetData();
            if (NULL != (pX509 = d2i_X509(&pX509, &pData, pCertif->GetLength())))
            {
                if (X509_V_OK == X509_cmp((X509 *)pCertX509, pX509))
                {
                    bFound = true;
                    if(pCertif->GetCertStatus() == BEID_CERTSTATUS_CERT_NOT_VALIDATED)
                    {
                        pCertif->SetCertStatus(lStatus);
                    }
                }
                X509_free(pX509);
                pX509 = NULL;
            }
        }
    }
}

int CVerify::VerifyCallback(int ok, X509_STORE_CTX *store)
{
    X509 *pCert = store->current_cert;
    int iStatus = store->error;
#ifdef _DEBUG
    if (!ok)
    {
        printf("Error: %s\n", X509_verify_cert_error_string(store->error));
    }
#endif

    // Update Status in Certif
    if(NULL != m_pCertifManager)
    {
        m_pCertifManager->UpdateCertStatus(pCert, iStatus); 
    }
    return ok;
}


int CVerify::VerifyHash(const unsigned char *pucData, unsigned long ulDataLen, const unsigned char *pucHash)
{
    int iRet = BEID_SIGNATURE_PROCESSING_ERROR;
    EVP_MD_CTX cmd_ctx;
    X509 *pX509 = NULL;
    int  iDiffRNCert = 1;
    unsigned char md_value[EVP_MAX_MD_SIZE] = {0};
    unsigned int md_len = 0;
    eidlib::CCertif *pCertRN = NULL;
    unsigned char *pucRNCert = NULL;
    unsigned long ulRNCertLen = 0;
    BEID_Certif_Check tCertifs = {0};

    if(m_pCertifManager == NULL)
    {
        return BEID_SIGNATURE_PROCESSING_ERROR;
    }
    pCertRN = m_pCertifManager->GetCertif("RN");
    if(pCertRN == NULL)
    {
        return BEID_SIGNATURE_PROCESSING_ERROR;
    }
    pucRNCert = pCertRN->GetData(); 
    ulRNCertLen = pCertRN->GetLength(); 

    m_pCertifManager->FillCertifs(&tCertifs);

    OpenSSL_add_all_algorithms();
    spc_x509store_t *pStore = new spc_x509store_t();
    InitX509Store(pStore);

    for(int i = 0; i < tCertifs.certificatesLength; ++i)
    {
        char *pID = tCertifs.certificates[i].certifLabel;
        if(0 != strcmp(pID, "RN"))
        {
            unsigned char *pData = tCertifs.certificates[i].certif;
            unsigned long ulLen = tCertifs.certificates[i].certifLength; 
            if (NULL == (pX509 = d2i_X509(&pX509, &pData, ulLen)))
                return BEID_SIGNATURE_PROCESSING_ERROR;
            X509StoreAddcert(pStore, pX509);
            pX509 = NULL;
        }
    }

    if (NULL == (pX509 = d2i_X509(&pX509, (unsigned char **)&pucRNCert, ulRNCertLen)))
        return BEID_SIGNATURE_PROCESSING_ERROR;


    X509_STORE  *store = 0;
    store = CreateX509store(pStore);
    if(X509_V_OK == VerifyCertroot(pX509, store))
    {
        iDiffRNCert = 0;
    }
    X509_STORE_free(store);
    CleanupX509store(pStore);
    delete pStore;

    // Check OID
    if(0 == iDiffRNCert && X509_V_OK != VerifyRRNDN(pX509))
        return BEID_SIGNATURE_INVALID;

    EVP_DigestInit(&cmd_ctx, EVP_sha1());
    EVP_DigestUpdate(&cmd_ctx, pucData, ulDataLen);
    EVP_DigestFinal(&cmd_ctx, md_value, &md_len);

    X509_free(pX509);

    iRet =  2*iDiffRNCert + (memcmp(md_value, pucHash, md_len) == 0 ? BEID_SIGNATURE_VALID : BEID_SIGNATURE_INVALID);
    return iRet;
}


long CVerify::VerifyCRL(const std::vector<eidlib::CCertif *> & Certifs)
{
    long lRet = 0;

    OpenSSL_add_all_algorithms();
    unsigned int i = 0;
    std::vector<std::string> CRLDistr;
    X509 *pX509 = NULL;
    X509_CRL *pX509Crl = NULL;
    spc_x509store_t *pStore = new spc_x509store_t();
    InitX509Store(pStore);
    X509StoreSetflags(pStore, X509_V_FLAG_CRL_CHECK | X509_V_FLAG_CRL_CHECK_ALL);
    X509StoreSetcallback(pStore, VerifyCallback);

    // Fill Certificate Store
    for(i = 0; i < Certifs.size(); ++i)
    {
        eidlib::CCertif *pCertif = Certifs[i];
        if(pCertif != NULL && BEID_CERTSTATUS_CERT_NOT_VALIDATED == pCertif->GetCertStatus())
        {
            unsigned char *pData = pCertif->GetData();
            //char *pID = pCertif->GetID();
            if (NULL == (pX509 = d2i_X509(&pX509, &pData, pCertif->GetLength())))
                return BEID_SIGNATURE_PROCESSING_ERROR;
            X509StoreAddcert(pStore, pX509);
            // Obsolete
            // char *pCRLDistribution = GetExtensionValue(pX509, "crlDistributionPoints", "URI");
            char *pCRLDistribution = GetExtensionValue2(pX509, NID_crl_distribution_points);
            
            if(NULL != pCRLDistribution)
            {
                bool bStop = false;
                for(unsigned int iNr = 0; iNr < CRLDistr.size(); ++iNr)
                {
                    if(0 == stricmp(CRLDistr[iNr].c_str(), pCRLDistribution))
                    {
                        bStop = true;
                        break;
                    }
                }
                if(!bStop)
                {
                    CRLDistr.push_back(pCRLDistribution);
                    if(NULL != (pX509Crl = FindLocalCRL(pCRLDistribution)))
                    {
                        X509StoreAddCRL(pStore, pX509Crl);
                        pX509Crl = NULL;
                    }
                }
                free(pCRLDistribution);
            }
            pX509 = NULL;
        }
    }

    //////////////////////////////////////////////////
    /// For testing only

   /* unsigned char *pDummy = NULL;
    // Add Root Cert
    for (i = 0; i < ROOT_CERT_COUNT; ++i)
    {
        pDummy = (unsigned char*)_root_cert[i];
        if (NULL == (pX509 = d2i_X509(&pX509, &pDummy, ROOT_CERT_SIZE)))
            return BEID_SIGNATURE_PROCESSING_ERROR;
        X509StoreAddcert(pStore, pX509);
        pX509 = NULL;
    }
*/

    //////////////////////////////////////////////////
    /// For testing only
    //Add rrncerts
 /*   {    
    unsigned char *pDummy = NULL;
        FILE *pf = fopen("rrn1.der", "r+b");
        unsigned char szBuffer[4096] = {0};
        int iread = 0;
        if(pf != NULL)
        {
            iread = fread(szBuffer, sizeof(unsigned char), sizeof(szBuffer), pf);
            fclose(pf);
        }
        pDummy = szBuffer;
    if (NULL == (pX509 = d2i_X509(&pX509, (unsigned char **)&pDummy, iread)))
        return BEID_SIGNATURE_PROCESSING_ERROR;
            X509StoreAddcert(pStore, pX509);
            
            char *pCRLDistribution = GetExtensionValue(pX509, "crlDistributionPoints", "URI");
            if(NULL != pCRLDistribution)
            {
                bool bStop = false;
                for(int iNr = 0; iNr < CRLDistr.size(); ++iNr)
                {
                    if(0 == stricmp(CRLDistr[iNr].c_str(), pCRLDistribution))
                    {
                        bStop = true;
                        break;
                    }
                }
                if(!bStop)
                {
                    CRLDistr.push_back(pCRLDistribution);
                    if(NULL != (pX509Crl = FindLocalCRL(pCRLDistribution)))
                    {
                        X509StoreAddCRL(pStore, pX509Crl);
                        pX509Crl = NULL;
                    }
                }
                free(pCRLDistribution);
            }
            pX509 = NULL;
    }
*/

  X509_STORE  *store = 0;
  store = CreateX509store(pStore);

    for(i = 0; i < (unsigned int)sk_X509_num(pStore->certs); ++i)
    {
        X509 *pValue = sk_X509_value(pStore->certs, i);
        if(pValue != NULL)
        {
            VerifyCert(pValue, store, CRLDistr);
            pValue = NULL;
        }
    }
   X509_STORE_free(store);
   CRLDistr.clear();

    CleanupX509store(pStore);
    delete pStore;

    return lRet;
}


long CVerify::VerifyOCSP(const std::vector<eidlib::CCertif *> & Certifs)
{
    long lRet = 0;

    OpenSSL_add_all_algorithms();
    unsigned int i = 0;

    X509 *pX509 = NULL;
    spc_x509store_t *pStore = new spc_x509store_t();
    InitX509Store(pStore);
    X509StoreSetcallback(pStore, VerifyCallback);

    // Fill Certificate Store
    for(i = 0; i < Certifs.size(); ++i)
    {
        eidlib::CCertif *pCertif = Certifs[i];
        if(pCertif != NULL && BEID_CERTSTATUS_CERT_NOT_VALIDATED == pCertif->GetCertStatus())
        {
            unsigned char *pData = pCertif->GetData();
            if (NULL == (pX509 = d2i_X509(&pX509, &pData, pCertif->GetLength())))
                return BEID_SIGNATURE_PROCESSING_ERROR;
            X509StoreAddcert(pStore, pX509);
            pX509 = NULL;
        }
    }

    //////////////////////////////////////////////////
    /// For testing only
/*
    unsigned char *pDummy = NULL;
    // Add Root Cert
    for (i = 0; i < ROOT_CERT_COUNT; ++i)
    {
        pDummy = (unsigned char*)_root_cert[i];
        if (NULL == (pX509 = d2i_X509(&pX509, &pDummy, ROOT_CERT_SIZE)))
            return SIGNATURE_PROCESSING_ERROR;
        spc_x509store_addcert(pStore, pX509);
        pX509 = NULL;
    }
*/

    //////////////////////////////////////////////////
    /// For testing only
    //Add rrncerts
/*    {   
        unsigned char *pDummy = NULL;
        FILE *pf = fopen("rrn1.der", "r+b");
        unsigned char szBuffer[4096] = {0};
        int iread = 0;
        if(pf != NULL)
        {
            iread = fread(szBuffer, sizeof(unsigned char), sizeof(szBuffer), pf);
            fclose(pf);
        }
        pDummy = szBuffer;
    if (NULL == (pX509 = d2i_X509(&pX509, (unsigned char **)&pDummy, iread)))
        return SIGNATURE_PROCESSING_ERROR;
            spc_x509store_addcert(pStore, pX509);
        pX509 = NULL;
    }
*/
  /*          if(NULL != (pX509Crl = FindLocalCRL(pX509)))
            {
                spc_x509store_addcrl(pStore, pX509Crl);
            }
            pX509Crl = NULL;
    }
*/

    X509_STORE  *store = 0;
    store = CreateX509store(pStore);

    spc_ocspresult_t tResult;
    spc_ocsprequest_t tRequest = {0};
    tRequest.store = store;
    tRequest.skew = 5;
    tRequest.maxage = 1;

    bool bContinue = true;

    for(i = 0; i < (unsigned int)sk_X509_num(pStore->certs) && bContinue; ++i)
    {
        X509 *pValue = sk_X509_value(pStore->certs, i);
        if(pValue != NULL)
        {
            int iStatus = BEID_CERTSTATUS_CERT_NOT_VALIDATED;
            X509 *pIssuer = FindIssuer(pValue, store);
            if(pIssuer != NULL)
            {
                char *pOCSPResponder = NULL;
                // Get OCSP Responder
                // Obsolete
                // pOCSPResponder = GetExtensionValue(pValue, "authorityInfoAccess", "OCSP - URI");
                pOCSPResponder = GetExtensionValue2(pValue, NID_info_access);
                
                if(NULL != pOCSPResponder)
                {
                    tRequest.url = pOCSPResponder;
                    tRequest.cert = pValue;
                    tRequest.issuer = pIssuer;
                    tResult = VerifyViaOCSP(&tRequest, pStore);
                    // Update result
                    switch(tResult)
                    {
                        case SPC_OCSPRESULT_CERTIFICATE_VALID: iStatus = BEID_CERTSTATUS_CERT_VALIDATED_OK; break;
                        case SPC_OCSPRESULT_CERTIFICATE_REVOKED: iStatus = BEID_CERTSTATUS_CERT_REVOKED; break;
                        default: lRet = -1; bContinue = false; // Error occured
                    }
                    m_pCertifManager->UpdateCertStatus(pValue, iStatus); 
                    free(pOCSPResponder);
                }
                X509_free(pIssuer);
            }
            else
            {
                m_pCertifManager->UpdateCertStatus(pValue, BEID_CERTSTATUS_UNABLE_TO_GET_ISSUER_CERT_LOCALLY); 
            }
            pValue = NULL;
        }
    }
    X509_STORE_free(store);

    CleanupX509store(pStore);
    delete pStore;

    return lRet;
}

int CVerify::VerifyOID(X509 *pCert)
{
    int iRet = -1;
    int j = 0;
    STACK_OF(POLICYINFO) *pol = NULL;
    POLICYINFO *pinfo = NULL;
    BOOL bFound = FALSE;

    pol = (struct stack_st *) X509_get_ext_d2i(pCert, NID_certificate_policies, NULL, NULL);

    if(pol != NULL)
    {
        for(j = 0; j < sk_POLICYINFO_num(pol); j++) 
        {
            pinfo = sk_POLICYINFO_value(pol, j);
            if(pinfo != NULL)
            {
                char szBuffer[64] = {0};
                OBJ_obj2txt(szBuffer, 64, pinfo->policyid, 0);
                if (0 == strcmp(szBuffer, RRN_OID1) || 0 == strcmp(szBuffer, RRN_OID2))
                {
                    iRet = X509_V_OK;
                    bFound = TRUE;
                    break;
                }
            }
        }
    }

    if(pol !=NULL)
    {
        sk_POLICYINFO_free(pol);
    }
    return iRet;
}

int CVerify::VerifyRRNDN(X509 *pCert)
{
    int iRet = -1;

    if(pCert != NULL)
    {
        char szName[256] = {0};
        char szTemp[128] = {0};
        X509_NAME_get_text_by_NID(X509_get_subject_name(pCert), NID_commonName, szTemp, sizeof(szTemp));
        strcat(szName, szTemp);
        memset(szTemp, 0, sizeof(szTemp));
        X509_NAME_get_text_by_NID(X509_get_subject_name(pCert), NID_organizationName, szTemp, sizeof(szTemp));
        strcat(szName, szTemp);
        memset(szTemp, 0, sizeof(szTemp));
        X509_NAME_get_text_by_NID(X509_get_subject_name(pCert), NID_countryName, szTemp, sizeof(szTemp));
        strcat(szName, szTemp);
        if (0 == strcmp(szName, DN_RRN))
        {
            iRet = X509_V_OK;
        }
    }
    return iRet;
}

void *CVerify::RetrieveData(char *pszUri, int *piDataLen)
{
    return RetrieveWebdata(pszUri, piDataLen);
}

