//////////////////////////////////////////////////////////////////////
// OpenSC Error Mapping
//////////////////////////////////////////////////////////////////////

#ifndef _EID_ERRORS_H
#define _EID_ERRORS_H

#include <opensc/errors.h>
#ifndef DWORD
typedef unsigned long DWORD;
#endif
#ifdef _WIN32
#include <SCardErr.h>
#else
#include <pcsclite.h>
#endif

typedef struct
{
    int iOpenSC;
    int iMapped;
    int iType;
    unsigned char ucSW[2];
} tOpenSCError;

const static tOpenSCError OpenSCErrors[] = {
/* Errors related to reader operation */
    {SC_ERROR_READER, SCARD_E_READER_UNAVAILABLE, BEID_E_PCSC, {0x00, 0x00}},
    {SC_ERROR_NO_READERS_FOUND, SCARD_E_UNKNOWN_READER, BEID_E_PCSC, {0x00, 0x00}},
    {SC_ERROR_CARD_NOT_PRESENT, SCARD_E_NO_SMARTCARD, BEID_E_PCSC, {0x00, 0x00}},
    {SC_ERROR_CARD_REMOVED, SCARD_W_REMOVED_CARD, BEID_E_PCSC, {0x00, 0x00}},
    {SC_ERROR_CARD_RESET, SCARD_W_RESET_CARD, BEID_E_PCSC, {0x00, 0x00}},
    {SC_ERROR_KEYPAD_TIMEOUT, 0, BEID_E_KEYPAD_TIMEOUT, {0x00, 0x00}},
    {SC_ERROR_KEYPAD_CANCELLED, 0, BEID_E_KEYPAD_CANCELLED, {0x00, 0x00}},
    {SC_ERROR_KEYPAD_PIN_MISMATCH, 0, BEID_E_KEYPAD_PIN_MISMATCH, {0x00, 0x00}},
    {SC_ERROR_KEYPAD_MSG_TOO_LONG, 0, BEID_E_KEYPAD_MSG_TOO_LONG, {0x00, 0x00}},
    {SC_ERROR_EVENT_TIMEOUT, SCARD_E_TIMEOUT, BEID_E_PCSC, {0x00, 0x00}},
    {SC_ERROR_CARD_UNRESPONSIVE, SCARD_W_UNRESPONSIVE_CARD, BEID_E_PCSC, {0x00, 0x00}},

/* Resulting from a card command or related to the card*/
    {SC_ERROR_MEMORY_FAILURE, 0, BEID_E_CARD, {0x62, 0x00}},
    {SC_ERROR_CARD_CMD_FAILED, 0, BEID_E_CARD, {0x62, 0x82}},
    {SC_ERROR_WRONG_LENGTH, 0, BEID_E_CARD, {0x67, 0x00}},
    {SC_ERROR_NO_CARD_SUPPORT, 0, BEID_E_CARD, {0x68, 0x00}},
    {SC_ERROR_NOT_ALLOWED, 0, BEID_E_CARD, {0x69, 0x00}},
    {SC_ERROR_SECURITY_STATUS_NOT_SATISFIED, 0, BEID_E_CARD, {0x69, 0x82}},
    {SC_ERROR_AUTH_METHOD_BLOCKED, 0, BEID_E_CARD, {0x69, 0x83}},
    {SC_ERROR_INCORRECT_PARAMETERS, 0, BEID_E_CARD, {0x6A, 0x00}},
    {SC_ERROR_NO_CARD_SUPPORT, 0, BEID_E_CARD, {0x6A, 0x81}},
    {SC_ERROR_FILE_NOT_FOUND, 0, BEID_E_CARD, {0x6A, 0x82}},
    {SC_ERROR_RECORD_NOT_FOUND, 0, BEID_E_CARD, {0x6A, 0x83}},
    {SC_ERROR_CARD_CMD_FAILED, 0, BEID_E_CARD, {0x6A, 0x84}},
    {SC_ERROR_INS_NOT_SUPPORTED, 0, BEID_E_CARD, {0x6D, 0x00}},
    {SC_ERROR_CLASS_NOT_SUPPORTED, 0, BEID_E_CARD, {0x6E, 0x00}},
    {SC_ERROR_CARD_CMD_FAILED, 0, BEID_E_CARD, {0x6F, 0x00}},
    {SC_ERROR_PIN_CODE_INCORRECT, 0, BEID_E_CARD, {0x63, 0xC0}},

/* Returned by OpenSC library when called with invalid arguments */
    {SC_ERROR_INVALID_ARGUMENTS, 0, BEID_E_BAD_PARAM, {0x00, 0x00}},
    {SC_ERROR_BUFFER_TOO_SMALL, 0, BEID_E_INSUFFICIENT_BUFFER, {0x00, 0x00}},
    {SC_ERROR_INVALID_PIN_LENGTH, 0, BEID_E_INVALID_PIN_LENGTH, {0x00, 0x00}},

/* Resulting from OpenSC internal operation */
    {SC_ERROR_INTERNAL, 0, BEID_E_INTERNAL, {0x00, 0x00}},
    {SC_ERROR_UNKNOWN, 0, BEID_E_UNKNOWN, {0x00, 0x00}}
};

static BOOL ConvertOpenSCError(long ulError, BEID_Status *ptStatus)
{
    if(SC_NO_ERROR == ulError)
    {
        ptStatus->general = BEID_OK;
        ptStatus->pcsc = BEID_OK;
        ptStatus->system = BEID_OK;
        memset(ptStatus->cardSW, 0, 2); 
        return TRUE;
    }

	int iErrCount = sizeof(OpenSCErrors)/sizeof(OpenSCErrors[0]);
	for (int i = 0; i < iErrCount; i++)
    {
		if (OpenSCErrors[i].iOpenSC == ulError) 
        {
            ptStatus->general = OpenSCErrors[i].iType;
            ptStatus->pcsc = OpenSCErrors[i].iMapped;
            memcpy(ptStatus->cardSW, OpenSCErrors[i].ucSW, 2); 
			return TRUE;
		}
    }
    /* Error Not Found, check if other OpenSC error */
    if(-1099 > ulError && ulError > -1999)
    {
        ptStatus->general = BEID_E_UNKNOWN;
        return TRUE;
    }
    return FALSE;
}

#endif /* _EID_ERRORS_H */
