#include "group.h"

/**
 *
 * Beryl group plugin
 *
 * option.c
 *
 * Copyright : (C) 2006 by Patrick Niklaus, Roi Cohen, Danny Baumann
 * Authors: Patrick Niklaus <patrick.niklaus@googlemail.com>
 *          Roi Cohen       <roico@beryl-project.org>
 *          Danny Baumann   <maniac@beryl-project.org>
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 **/

/*
 * Defaults
 *
 */
#define GROUP_SELECT_MODIFIERS_DEFAULT          (CompSuperMask)
#define GROUP_SELECT_BUTTON_DEFAULT             Button1

#define GROUP_SELECT_SINGLE_MODIFIERS_DEFAULT   (CompSuperMask)
#define GROUP_SELECT_SINGLE_KEY_DEFAULT         "s"

#define GROUP_GROUPING_MODIFIERS_DEFAULT        (CompSuperMask)
#define GROUP_GROUPING_KEY_DEFAULT              "g"

#define GROUP_UNGROUPING_MODIFIERS_DEFAULT      (CompSuperMask)
#define GROUP_UNGROUPING_KEY_DEFAULT            "u"

#define GROUP_TABMODE_MODIFIERS_DEFAULT         (CompSuperMask)
#define GROUP_TABMODE_KEY_DEFAULT               "t"

#define GROUP_CHANGE_TAB_LEFT_MODIFIERS_DEFAULT (CompSuperMask)
#define GROUP_CHANGE_TAB_LEFT_KEY_DEFAULT       "Left"

#define GROUP_CHANGE_TAB_RIGHT_MODIFIERS_DEFAULT (CompSuperMask)
#define GROUP_CHANGE_TAB_RIGHT_KEY_DEFAULT       "Right"

#define GROUP_REMOVING_MODIFIERS_DEFAULT       (CompSuperMask)
#define GROUP_REMOVING_KEY_DEFAULT             "r"

#define GROUP_CLOSING_MODIFIERS_DEFAULT        (CompSuperMask)
#define GROUP_CLOSING_KEY_DEFAULT              "c"

#define GROUP_IGNORE_MODIFIERS_DEFAULT          (CompSuperMask)
#define GROUP_IGNORE_KEY_DEFAULT                "x"

#define GROUP_CHANGE_COLOR_MODIFIERS_DEFAULT	(CompSuperMask)
#define GROUP_CHANGE_COLOR_KEY_DEFAULT          "y"

#define GROUP_OPACITY_DEFAULT           80
#define GROUP_OPACITY_MIN               0
#define GROUP_OPACITY_MAX               100

#define GROUP_SATURATION_DEFAULT        20
#define GROUP_SATURATION_MIN            0
#define GROUP_SATURATION_MAX            100

#define GROUP_BRIGHTNESS_DEFAULT        70
#define GROUP_BRIGHTNESS_MIN            0
#define GROUP_BRIGHTNESS_MAX            100

#define GROUP_PRECISION_DEFAULT         25
#define GROUP_PRECISION_MIN             1
#define GROUP_PRECISION_MAX             100

#define GROUP_THUMB_SIZE_DEFAULT	128
#define GROUP_THUMB_SIZE_MIN		16
#define GROUP_THUMB_SIZE_MAX		256

#define GROUP_BORDER_WIDTH_DEFAULT	10
#define GROUP_BORDER_WIDTH_MIN		1
#define GROUP_BORDER_WIDTH_MAX		20

#define GROUP_BORDER_RADIUS_DEFAULT	10
#define GROUP_BORDER_RADIUS_MIN		1
#define GROUP_BORDER_RADIUS_MAX		20

#define GROUP_GLOW_SIZE_DEFAULT		64
#define GROUP_GLOW_SIZE_MIN		1
#define GROUP_GLOW_SIZE_MAX		300

#define GROUP_TABBAR_FONTSIZE_DEFAULT	12
#define GROUP_TABBAR_FONTSIZE_MIN	6
#define GROUP_TABBAR_FONTSIZE_MAX	24

#define GROUP_TAB_DRAG_Y_DISTANCE_DEFAULT   400
#define GROUP_TAB_DRAG_Y_DISTANCE_MIN	    10
#define GROUP_TAB_DRAG_Y_DISTANCE_MAX	    2500

#define GROUP_TAB_DRAG_SPEED_LIMIT_DEFAULT  800
#define GROUP_TAB_DRAG_SPEED_LIMIT_MIN	    0
#define GROUP_TAB_DRAG_SPEED_LIMIT_MAX	    2000

#define GROUP_MOVE_DEFAULT			TRUE
#define GROUP_RESIZE_DEFAULT			FALSE
#define GROUP_RAISE_DEFAULT			TRUE
#define GROUP_AUTO_UNGROUP_DEFAULT		FALSE
#define GROUP_AUTO_GROUP_DEFAULT		TRUE
#define GROUP_RELATIVE_DISTANCE_DEFAULT		FALSE
#define GROUP_TAB_CREATE_MIPMAPS_DEFAULT	FALSE
#define GROUP_GLOW_DEFAULT			TRUE
#define GROUP_UNTAB_ON_CLOSE_DEFAULT		FALSE
#define GROUP_SPRING_MODEL_ON_MOVE_DEFAULT	FALSE
#define GROUP_AUTOTAB_DEFAULT			FALSE
#define GROUP_DND_UNGROUP_WINDOW_DEFAULT	TRUE
#define GROUP_RESIZE_UNMAXIMIZE_DEFAULT		FALSE
#define GROUP_MINIMIZE_ALL_DEFAULT		TRUE
#define GROUP_SHADE_ALL_DEFAULT			FALSE

#define GROUP_COLOR_SELECTION_RED_DEFAULT       0x0000
#define GROUP_COLOR_SELECTION_GREEN_DEFAULT     0x0000
#define GROUP_COLOR_SELECTION_BLUE_DEFAULT      0x0000
#define GROUP_COLOR_SELECTION_ALPHA_DEFAULT     0x9999

#define GROUP_COLOR_LINE_RED_DEFAULT            0x0000
#define GROUP_COLOR_LINE_GREEN_DEFAULT          0x0000
#define GROUP_COLOR_LINE_BLUE_DEFAULT           0x0000
#define GROUP_COLOR_LINE_ALPHA_DEFAULT          0xABAB

#define GROUP_COLOR_TAB_FILL_RED_DEFAULT	0x0000
#define GROUP_COLOR_TAB_FILL_GREEN_DEFAULT	0x0000
#define GROUP_COLOR_TAB_FILL_BLUE_DEFAULT	0x0000
#define GROUP_COLOR_TAB_FILL_ALPHA_DEFAULT	0x9999

#define GROUP_COLOR_TAB_BORDER_RED_DEFAULT	0x0000
#define GROUP_COLOR_TAB_BORDER_GREEN_DEFAULT	0x0000
#define GROUP_COLOR_TAB_BORDER_BLUE_DEFAULT	0x0000
#define GROUP_COLOR_TAB_BORDER_ALPHA_DEFAULT	0xABAB

#define GROUP_COLOR_TABBAR_FONT_RED_DEFAULT	0xFFFF
#define GROUP_COLOR_TABBAR_FONT_BLUE_DEFAULT	0xFFFF
#define GROUP_COLOR_TABBAR_FONT_GREEN_DEFAULT	0xFFFF
#define GROUP_COLOR_TABBAR_FONT_ALPHA_DEFAULT	0xFFFF

#define GROUP_GLOW_TYPE_DEFAULT 	GlowTextureRectangular

#define GROUP_FADE_TIME_MIN			0.0f
#define GROUP_FADE_TIME_MAX			5.0f
#define GROUP_FADE_TIME_PRECISION		0.1f
#define GROUP_FADE_TIME_DEFAULT			0.2f

#define GROUP_FADE_TEXT_TIME_MIN		0.00f
#define GROUP_FADE_TEXT_TIME_MAX		5.00f
#define GROUP_FADE_TEXT_TIME_PRECISION		0.05f
#define GROUP_FADE_TEXT_TIME_DEFAULT		0.25f

#define GROUP_CHANGE_ANIMATION_TIME_DEFAULT	0.5f
#define GROUP_CHANGE_ANIMATION_TIME_MIN		0.0f
#define GROUP_CHANGE_ANIMATION_TIME_MAX		5.0f
#define GROUP_CHANGE_ANIMATION_TIME_PRECISION	0.05f

#define GROUP_TABBING_SPEED_DEFAULT		1.2f
#define GROUP_TABBING_SPEED_MIN			0.1f
#define GROUP_TABBING_SPEED_MAX			50.0f
#define GROUP_TABBING_SPEED_PRECISION		0.1f

#define GROUP_TABBING_TIMESTEP_DEFAULT		0.1f
#define GROUP_TABBING_TIMESTEP_MIN		0.1f
#define GROUP_TABBING_TIMESTEP_MAX		50.0f
#define GROUP_TABBING_TIMESTEP_PRECISION	0.1f

#define GROUP_VISIBILITY_TIME_MIN		0.0f
#define GROUP_VISIBILITY_TIME_MAX		10.0f
#define GROUP_VISIBILITY_TIME_PRECISION		0.05f
#define GROUP_VISIBILITY_TIME_DEFAULT		0.0f

#define GROUP_TAB_DRAG_HOVER_TIME_DEFAULT	0.5f
#define GROUP_TAB_DRAG_HOVER_TIME_MIN		0.0f
#define GROUP_TAB_DRAG_HOVER_TIME_MAX		5.0f
#define GROUP_TAB_DRAG_HOVER_TIME_PRECISION	0.1f

#define GROUP_TAB_DRAG_SPRING_K_DEFAULT		8.0f
#define GROUP_TAB_DRAG_SPRING_K_MIN		0.1f
#define GROUP_TAB_DRAG_SPRING_K_MAX		30.0f
#define GROUP_TAB_DRAG_SPRING_K_PRECISION	0.1f

#define GROUP_TAB_DRAG_FRICTION_DEFAULT		35.0f
#define GROUP_TAB_DRAG_FRICTION_MIN		0.1f
#define GROUP_TAB_DRAG_FRICTION_MAX		100.0f
#define GROUP_TAB_DRAG_FRICTION_PRECISION	0.1f

char *groupDefaultTypes[] = {
	N_("Normal"),
	N_("Dialog"),
	N_("ModalDialog")
};

/*
 * groupScreenInitOptions
 *
 */
void groupScreenInitOptions(GroupScreen * gs)
{

	CompOption *o;
	int i;

	o = &gs->opt[GROUP_SCREEN_OPTION_TYPES];
	o->name = "mask";
	o->group = N_("General");
	o->subGroup = N_("");
	o->displayHints = "";
	o->advanced = True;
	o->shortDesc = N_("Window Types");
	o->longDesc = N_("The types of windows which will be grouped");
	o->type = CompOptionTypeList;
	o->value.list.type = CompOptionTypeString;
	o->value.list.nValue = N_WIN_TYPE;
	o->value.list.value = malloc(sizeof(CompOptionValue) * N_WIN_TYPE);
	for (i = 0; i < N_WIN_TYPE; i++)
		o->value.list.value[i].s = strdup(groupDefaultTypes[i]);
	o->rest.s.string = (char **) windowTypeString;
	o->rest.s.nString = nWindowTypeString;

	o = &gs->opt[GROUP_SCREEN_OPTION_OPACITY];
	o->name = "opacity";
	o->group = N_("Selection");
	o->subGroup = N_("Options");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Opacity");
	o->longDesc = N_("Opacity of selected windows");
	o->type = CompOptionTypeInt;
	o->value.i = GROUP_OPACITY_DEFAULT;
	o->rest.i.min = GROUP_OPACITY_MIN;
	o->rest.i.max = GROUP_OPACITY_MAX;

	o = &gs->opt[GROUP_SCREEN_OPTION_SATURATION];
	o->name = "saturation";
	o->group = N_("Selection");
	o->subGroup = N_("Options");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Saturation");
	o->longDesc = N_("Saturation of selected windows");
	o->type = CompOptionTypeInt;
	o->value.i = GROUP_SATURATION_DEFAULT;
	o->rest.i.min = GROUP_SATURATION_MIN;
	o->rest.i.max = GROUP_SATURATION_MAX;

	o = &gs->opt[GROUP_SCREEN_OPTION_BRIGHTNESS];
	o->name = "brightness";
	o->group = N_("Selection");
	o->subGroup = N_("Options");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Brightness");
	o->longDesc = N_("Brightness of selected windows");
	o->type = CompOptionTypeInt;
	o->value.i = GROUP_BRIGHTNESS_DEFAULT;
	o->rest.i.min = GROUP_BRIGHTNESS_MIN;
	o->rest.i.max = GROUP_BRIGHTNESS_MAX;

	o = &gs->opt[GROUP_SCREEN_OPTION_PRECISION];
	o->name = "precision";
	o->group = N_("Selection");
	o->subGroup = N_("Options");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Precision");
	o->longDesc = N_("Precision of the seletion, which means its the percent "
					 "of the window area that has to be visible and in the selection.");
	o->type = CompOptionTypeInt;
	o->value.i = GROUP_PRECISION_DEFAULT;
	o->rest.i.min = GROUP_PRECISION_MIN;
	o->rest.i.max = GROUP_PRECISION_MAX;

	o = &gs->opt[GROUP_SCREEN_OPTION_TABBING_SPEED];
	o->name = "tabbing_speed";
	o->group = N_("Tabbing");
	o->subGroup = N_("Animation Options");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Tabbing Speed");
	o->longDesc =
	    N_("Tabbing Speed");
	o->type = CompOptionTypeFloat;
	o->value.f = GROUP_TABBING_SPEED_DEFAULT;
	o->rest.f.min = GROUP_TABBING_SPEED_MIN;
	o->rest.f.max = GROUP_TABBING_SPEED_MAX;
	o->rest.f.precision = GROUP_TABBING_SPEED_PRECISION;

	o = &gs->opt[GROUP_SCREEN_OPTION_TABBING_TIMESTEP];
	o->name = "tabbing_timestep";
	o->group = N_("Tabbing");
	o->subGroup = N_("Animation Options");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Tabbing Timestep");
	o->longDesc =
	    N_("The duration (in s) of the tabbing animation.");
	o->type = CompOptionTypeFloat;
	o->value.f = GROUP_TABBING_TIMESTEP_DEFAULT;
	o->rest.f.min = GROUP_TABBING_TIMESTEP_MIN;
	o->rest.f.max = GROUP_TABBING_TIMESTEP_MAX;
	o->rest.f.precision = GROUP_TABBING_TIMESTEP_PRECISION;

	o = &gs->opt[GROUP_SCREEN_OPTION_THUMB_SIZE];
	o->name = "thumb_size";
	o->group = N_("Tabbing");
	o->subGroup = N_("Misc. Options");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Thumb Size");
	o->longDesc =
	    N_("The size of the window thumbs in the tab bar.");
	o->type = CompOptionTypeInt;
	o->value.i = GROUP_THUMB_SIZE_DEFAULT;
	o->rest.i.min = GROUP_THUMB_SIZE_MIN;
	o->rest.i.max = GROUP_THUMB_SIZE_MAX;

	o = &gs->opt[GROUP_SCREEN_OPTION_BORDER_WIDTH];
	o->name = "border_width";
	o->group = N_("Tabbing");
	o->subGroup = N_("Misc. Options");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Border Width");
	o->longDesc =
	    N_("The width of the border around the tab bar.");
	o->type = CompOptionTypeInt;
	o->value.i = GROUP_BORDER_WIDTH_DEFAULT;
	o->rest.i.min = GROUP_BORDER_WIDTH_MIN;
	o->rest.i.max = GROUP_BORDER_WIDTH_MAX;

	o = &gs->opt[GROUP_SCREEN_OPTION_BORDER_RADIUS];
	o->name = "border_radius";
	o->group = N_("Tabbing");
	o->subGroup = N_("Misc. Options");
	o->displayHints = "";
	o->advanced = True;
	o->shortDesc = N_("Border Radius");
	o->longDesc =
	    N_("The radius for the edges of the tab bar.");
	o->type = CompOptionTypeInt;
	o->value.i = GROUP_BORDER_RADIUS_DEFAULT;
	o->rest.i.min = GROUP_BORDER_RADIUS_MIN;
	o->rest.i.max = GROUP_BORDER_RADIUS_MAX;

	o = &gs->opt[GROUP_SCREEN_OPTION_TABBAR_FONTSIZE];
	o->name = "tabbar_font_size";
	o->group = N_("Tabbing");
	o->subGroup = N_("Window title font");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Font Size");
	o->longDesc =
	    N_("The size of window title font in the tab bar.");
	o->type = CompOptionTypeInt;
	o->value.i = GROUP_TABBAR_FONTSIZE_DEFAULT;
	o->rest.i.min = GROUP_TABBAR_FONTSIZE_MIN;
	o->rest.i.max = GROUP_TABBAR_FONTSIZE_MAX;

	o = &gs->opt[GROUP_SCREEN_OPTION_TABBAR_FONTCOLOR];
	o->name = "tabbar_font_color";
	o->group = N_("Tabbing");
	o->subGroup = N_("Window title font");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Font Color");
	o->longDesc = N_("The color of the window title in the tab bar.");
	o->type = CompOptionTypeColor;
	o->value.c[0] = GROUP_COLOR_TABBAR_FONT_RED_DEFAULT;
	o->value.c[1] = GROUP_COLOR_TABBAR_FONT_GREEN_DEFAULT;
	o->value.c[2] = GROUP_COLOR_TABBAR_FONT_BLUE_DEFAULT;
	o->value.c[3] = GROUP_COLOR_TABBAR_FONT_ALPHA_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_MOVE];
	o->name = "move";
	o->group = N_("General");
	o->subGroup = N_("");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Move every window in the group");
	o->longDesc = N_("If one window in the group gets moved, "
			 "every other window in the group gets moved as well.");
	o->type = CompOptionTypeBool;
	o->value.b = GROUP_MOVE_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_RESIZE];
	o->name = "resize";
	o->group = N_("General");
	o->subGroup = N_("");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Resize every window in the group");
	o->longDesc = N_("If one window in the group gets resized, "
			 "every other window in the group gets resized as well.");
	o->type = CompOptionTypeBool;
	o->value.b = GROUP_RESIZE_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_RAISE];
	o->name = "raise";
	o->group = N_("General");
	o->subGroup = N_("");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Raise every window in the group");
	o->longDesc = N_("If one window in the group gets selected, "
			 "every window in the group gets raised.");
	o->type = CompOptionTypeBool;
	o->value.b = GROUP_RAISE_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_AUTO_GROUP];
	o->name = "auto_group";
	o->group = N_("General");
	o->subGroup = N_("");
	o->displayHints = "";
	o->advanced = True;
	o->shortDesc = N_("Group the windows after selection");
	o->longDesc = N_("If you have selected your windows,"
			 "this automatically groups them. "
			 "(Doesn't work with selection mode 'normal')");
	o->type = CompOptionTypeBool;
	o->value.b = GROUP_AUTO_GROUP_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_AUTO_UNGROUP];
	o->name = "auto_ungroup";
	o->group = N_("General");
	o->subGroup = N_("");
	o->displayHints = "";
	o->advanced = True;
	o->shortDesc =
	    N_("Ungroup the windows if only one window is left");
	o->longDesc =
	    N_
	    ("If there is only 1 window in the group left, it will be ungrouped.");
	o->type = CompOptionTypeBool;
	o->value.b = GROUP_AUTO_UNGROUP_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_RELATIVE_DISTANCE];
	o->name = "relative_distance";
	o->group = N_("General");
	o->subGroup = N_("");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Compute distance relative");
	o->longDesc =
	    N_
	    ("The distance between the windows is computed relative to the window size. "
	     "This allows you to have windows staying next to eachother.");
	o->type = CompOptionTypeBool;
	o->value.b = GROUP_RELATIVE_DISTANCE_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_SELECTION_COLOR];
	o->name = "fill_color";
	o->group = N_("Selection");
	o->subGroup = N_("Colors");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Selection Color");
	o->longDesc = N_("Fill color of the selection.");
	o->type = CompOptionTypeColor;
	o->value.c[0] = GROUP_COLOR_SELECTION_RED_DEFAULT;
	o->value.c[1] = GROUP_COLOR_SELECTION_GREEN_DEFAULT;
	o->value.c[2] = GROUP_COLOR_SELECTION_BLUE_DEFAULT;
	o->value.c[3] = GROUP_COLOR_SELECTION_ALPHA_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_LINE_COLOR];
	o->name = "line_color";
	o->group = N_("Selection");
	o->subGroup = N_("Colors");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Line Color");
	o->longDesc = N_("Line color of the selection.");
	o->type = CompOptionTypeColor;
	o->value.c[0] = GROUP_COLOR_LINE_RED_DEFAULT;
	o->value.c[1] = GROUP_COLOR_LINE_GREEN_DEFAULT;
	o->value.c[2] = GROUP_COLOR_LINE_BLUE_DEFAULT;
	o->value.c[3] = GROUP_COLOR_LINE_ALPHA_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_TAB_FILL_COLOR];
	o->name = "tab_color";
	o->group = N_("Tabbing");
	o->subGroup = N_("Colors");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Tab Color");
	o->longDesc = N_("Fill color of the tab bar.");
	o->type = CompOptionTypeColor;
	o->value.c[0] = GROUP_COLOR_TAB_FILL_RED_DEFAULT;
	o->value.c[1] = GROUP_COLOR_TAB_FILL_GREEN_DEFAULT;
	o->value.c[2] = GROUP_COLOR_TAB_FILL_BLUE_DEFAULT;
	o->value.c[3] = GROUP_COLOR_TAB_FILL_ALPHA_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_TAB_BORDER_COLOR];
	o->name = "border_color";
	o->group = N_("Tabbing");
	o->subGroup = N_("Colors");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Tab Border Color");
	o->longDesc = N_("Border color of the tab bar.");
	o->type = CompOptionTypeColor;
	o->value.c[0] = GROUP_COLOR_TAB_BORDER_RED_DEFAULT;
	o->value.c[1] = GROUP_COLOR_TAB_BORDER_GREEN_DEFAULT;
	o->value.c[2] = GROUP_COLOR_TAB_BORDER_BLUE_DEFAULT;
	o->value.c[3] = GROUP_COLOR_TAB_BORDER_ALPHA_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_TAB_CREATE_MIPMAPS];
	o->name                 = "mipmaps";
	o->shortDesc        	= N_("Create mipmaps for thumbnails.");
	o->group                = N_("Tabbing");
	o->subGroup             = N_("Misc. Options");
	o->displayHints         = "";
	o->advanced 		= True;
	o->longDesc             = N_("Create mipmaps for thumbnails in the tab-bar.");
	o->type                 = CompOptionTypeBool;
	o->value.b              = GROUP_TAB_CREATE_MIPMAPS_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_GLOW];
	o->name                 = "glow";
	o->shortDesc        	= N_("Enable Glow");
	o->group                = N_("Glow");
	o->subGroup             = N_("");
	o->displayHints         = "";
	o->advanced 		= False;
	o->longDesc             = N_("Enable grouped window glowing.");
	o->type                 = CompOptionTypeBool;
	o->value.b              = GROUP_GLOW_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_GLOW_SIZE];
	o->name = "glow_size";
	o->group = N_("Glow");
	o->subGroup = N_("");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Glow size");
	o->longDesc =
	    N_("The size of the grouped window glow.");
	o->type = CompOptionTypeInt;
	o->value.i = GROUP_GLOW_SIZE_DEFAULT;
	o->rest.i.min = GROUP_GLOW_SIZE_MIN;
	o->rest.i.max = GROUP_GLOW_SIZE_MAX;

	o = &gs->opt[GROUP_SCREEN_OPTION_GLOW_TYPE];
	o->name = "glow_type";
	o->group = N_("Glow");
	o->subGroup = N_("");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Glow type");
	o->longDesc =
	    N_("The type of the glow.");
	o->type = CompOptionTypeString;
	o->value.s = strdup(glowTextureName[GROUP_GLOW_TYPE_DEFAULT]);
	o->rest.s.string = (char **) glowTextureName;
	o->rest.s.nString = nGlowTextureNames; 

	/* initialize glowType here to prevent code duplication in init.c */
	gs->glowType = GROUP_GLOW_TYPE_DEFAULT;
	
	o = &gs->opt[GROUP_SCREEN_OPTION_FADE_TIME];
	o->name = "fade_time";
	o->group = N_("Tabbing");
	o->subGroup = N_("Animation Options");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Fade time for tab-bar animations.");
	o->longDesc = N_("Duration (in s) of the fading animation of the tab-bar when unmapping / mapping it.");
	o->type = CompOptionTypeFloat;
	o->value.f = GROUP_FADE_TIME_DEFAULT;
	o->rest.f.min = GROUP_FADE_TIME_MIN;
	o->rest.f.max = GROUP_FADE_TIME_MAX;
	o->rest.f.precision = GROUP_FADE_TIME_PRECISION;

	o = &gs->opt[GROUP_SCREEN_OPTION_FADE_TEXT_TIME];
	o->name = "fade_text_time";
	o->group = N_("Tabbing");
	o->subGroup = N_("Animation Options");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Fade time for text animations.");
	o->longDesc = N_("Duration (in s) of the fading animation of the text when showing / hiding it.");
	o->type = CompOptionTypeFloat;
	o->value.f = GROUP_FADE_TEXT_TIME_DEFAULT;
	o->rest.f.min = GROUP_FADE_TEXT_TIME_MIN;
	o->rest.f.max = GROUP_FADE_TEXT_TIME_MAX;
	o->rest.f.precision = GROUP_FADE_TEXT_TIME_PRECISION;

	o = &gs->opt[GROUP_SCREEN_OPTION_VISIBILITY_TIME];
	o->name = "visibility_time";
	o->group = N_("Tabbing");
	o->subGroup = N_("Misc. Options");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Tab bar visibility time after tab change");
	o->longDesc = N_("Time (in s) the tab bar is visible after a tab change.");
	o->type = CompOptionTypeFloat;
	o->value.f = GROUP_VISIBILITY_TIME_DEFAULT;
	o->rest.f.min = GROUP_VISIBILITY_TIME_MIN;
	o->rest.f.max = GROUP_VISIBILITY_TIME_MAX;
	o->rest.f.precision = GROUP_VISIBILITY_TIME_PRECISION;
	
	o = &gs->opt[GROUP_SCREEN_OPTION_UNTAB_ON_CLOSE];
	o->name = "untab_on_close";
	o->group = N_("Tabbing");
	o->subGroup = N_("Misc. Options");
	o->displayHints = "";
	o->advanced = True;
	o->shortDesc = N_("Untab when closing the window.");
	o->longDesc = N_("Untab the group when closing the tabbed window, "
			 "instead of changing tab.");
	o->type = CompOptionTypeBool;
	o->value.b = GROUP_UNTAB_ON_CLOSE_DEFAULT;
	
	o = &gs->opt[GROUP_SCREEN_OPTION_CHANGE_ANIMATION_TIME];
	o->name = "change_animation_time";
	o->group = N_("Tabbing");
	o->subGroup = N_("Animation Options");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Change Tab Animation Duration");
	o->longDesc =
	    N_("The duration (in s) of the animation that happens when changing tabs.");
	o->type = CompOptionTypeFloat;
	o->value.f = GROUP_CHANGE_ANIMATION_TIME_DEFAULT;
	o->rest.f.min = GROUP_CHANGE_ANIMATION_TIME_MIN;
	o->rest.f.max = GROUP_CHANGE_ANIMATION_TIME_MAX;
	o->rest.f.precision = GROUP_CHANGE_ANIMATION_TIME_PRECISION;

	o = &gs->opt[GROUP_SCREEN_OPTION_TAB_DRAG_HOVER_TIME];
	o->name = "drag_hover_time";
	o->group = N_("Tabbing");
	o->subGroup = N_("Slot dragging");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Hover time for slot dragging");
	o->longDesc = N_("Timespan (in s) after which a grouped window is activated if " 
		"a window of another group is dragged over it.");
	o->type = CompOptionTypeFloat;
	o->value.f = GROUP_TAB_DRAG_HOVER_TIME_DEFAULT;
	o->rest.f.min = GROUP_TAB_DRAG_HOVER_TIME_MIN;
	o->rest.f.max = GROUP_TAB_DRAG_HOVER_TIME_MAX;
	o->rest.f.precision = GROUP_TAB_DRAG_HOVER_TIME_PRECISION;

	o = &gs->opt[GROUP_SCREEN_OPTION_TAB_DRAG_SPRING_K];
	o->name = "drag_spring_k";
	o->group = N_("Tabbing");
	o->subGroup = N_("Slot dragging");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Slot drag Spring K");
	o->longDesc = N_("Spring Konstant used for slot dragging.");
	o->type = CompOptionTypeFloat;
	o->value.f = GROUP_TAB_DRAG_SPRING_K_DEFAULT;
	o->rest.f.min = GROUP_TAB_DRAG_SPRING_K_MIN;
	o->rest.f.max = GROUP_TAB_DRAG_SPRING_K_MAX;
	o->rest.f.precision = GROUP_TAB_DRAG_SPRING_K_PRECISION;

	o = &gs->opt[GROUP_SCREEN_OPTION_TAB_DRAG_FRICTION];
	o->name = "drag_friction";
	o->group = N_("Tabbing");
	o->subGroup = N_("Slot dragging");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Slot drag friction");
	o->longDesc = N_("Spring friction for slot dragging.");
	o->type = CompOptionTypeFloat;
	o->value.f = GROUP_TAB_DRAG_FRICTION_DEFAULT;
	o->rest.f.min = GROUP_TAB_DRAG_FRICTION_MIN;
	o->rest.f.max = GROUP_TAB_DRAG_FRICTION_MAX;
	o->rest.f.precision = GROUP_TAB_DRAG_FRICTION_PRECISION;

	o = &gs->opt[GROUP_SCREEN_OPTION_TAB_DRAG_Y_DISTANCE];
	o->name = "drag_y_distance";
	o->group = N_("Tabbing");
	o->subGroup = N_("Slot dragging");
	o->displayHints = "";
	o->advanced = True;
	o->shortDesc = N_("Y distance for spring model");
	o->longDesc = N_("Distance (in pixels) between the tab bar and the"
		"dragged slot for applying the spring model. If the distance"
		"is larger than that value, the model isn't applied.");
	o->type = CompOptionTypeInt;
	o->value.i = GROUP_TAB_DRAG_Y_DISTANCE_DEFAULT;
	o->rest.i.min = GROUP_TAB_DRAG_Y_DISTANCE_MIN;
	o->rest.i.max = GROUP_TAB_DRAG_Y_DISTANCE_MAX;

	o = &gs->opt[GROUP_SCREEN_OPTION_TAB_DRAG_SPEED_LIMIT];
	o->name = "drag_speed_limit";
	o->group = N_("Tabbing");
	o->subGroup = N_("Slot dragging");
	o->displayHints = "";
	o->advanced = True;
	o->shortDesc = N_("Speed limit");
	o->longDesc = N_("Mouse pointer movement speed limit (in pixels/secons)"
		"for the spring model.");
	o->type = CompOptionTypeInt;
	o->value.i = GROUP_TAB_DRAG_SPEED_LIMIT_DEFAULT;
	o->rest.i.min = GROUP_TAB_DRAG_SPEED_LIMIT_MIN;
	o->rest.i.max = GROUP_TAB_DRAG_SPEED_LIMIT_MAX;

	o = &gs->opt[GROUP_SCREEN_OPTION_SPRING_MODEL_ON_MOVE];
	o->name = "spring_model_on_move";
	o->group = N_("Tabbing");
	o->subGroup = N_("Slot dragging");
	o->displayHints = "";
	o->advanced = True;
	o->shortDesc = N_("Spring model on move");
	o->longDesc = N_("Use spring model for the tab-bar when moving the window.");
	o->type = CompOptionTypeBool;
	o->value.b = GROUP_SPRING_MODEL_ON_MOVE_DEFAULT;
	
	o=&gs->opt[GROUP_SCREEN_OPTION_AUTOTAB];
	o->name="autotab_create";
	o->group = N_("Tabbing");
	o->subGroup = N_("Misc. Options");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Autotab windows on creation");
	o->longDesc = N_("Autotab windows with themselves on window creation");
	o->type = CompOptionTypeBool;
	o->value.b = GROUP_AUTOTAB_DEFAULT;
	
	o = &gs->opt[GROUP_SCREEN_OPTION_DND_UNGROUP_WINDOW];
	o->name = "dnd_ungroup_window";
	o->group = N_("Tabbing");
	o->subGroup = N_("Slot dragging");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Remove window from group after dropping it outside a tab-bar.");
	o->longDesc = N_("Remove window from group after droping it outside a tab-bar.");
	o->type = CompOptionTypeBool;
	o->value.b = GROUP_DND_UNGROUP_WINDOW_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_RESIZE_UNMAXIMIZE];
	o->name = "resize_unmaximize";
	o->group = N_("General");
	o->subGroup = N_("");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Maximize/unmaximize with group.");
	o->longDesc = N_("Maximize/unmaximize with group.");
	o->type = CompOptionTypeBool;
	o->value.b = GROUP_RESIZE_UNMAXIMIZE_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_MINIMIZE_ALL];
	o->name = "minimize_all";
	o->group = N_("General");
	o->subGroup = N_("");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Minimize with group.");
	o->longDesc = N_("Minimize with group.");
	o->type = CompOptionTypeBool;
	o->value.b = GROUP_MINIMIZE_ALL_DEFAULT;

	o = &gs->opt[GROUP_SCREEN_OPTION_SHADE_ALL];
	o->name = "shade_all";
	o->group = N_("General");
	o->subGroup = N_("");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Shade with group.");
	o->longDesc = N_("Shade with group.");
	o->type = CompOptionTypeBool;
	o->value.b = GROUP_SHADE_ALL_DEFAULT;
}

/*
 * groupGetScreenOptions
 *
 */
CompOption *groupGetScreenOptions(CompScreen * s, int *count)
{
	if (s) {
		GROUP_SCREEN(s);

		*count = NUM_OPTIONS(gs);
		return gs->opt;
	} else {
		GroupScreen *gs = malloc(sizeof(GroupScreen));
		groupScreenInitOptions(gs);
		*count = NUM_OPTIONS(gs);
		return gs->opt;
	}
}


/*
 * groupSetScreenOption
 *
 */
Bool
groupSetScreenOption(CompScreen * s, char *name, CompOptionValue * value)
{

	CompOption *o;
	int index;

	GROUP_SCREEN(s);

	o = compFindOption(gs->opt, NUM_OPTIONS(gs), name, &index);
	if (!o)
		return FALSE;

	switch (index) {
	case GROUP_SCREEN_OPTION_TYPES:
		if (compSetOptionList(o, value)) {
			gs->wMask =
			    compWindowTypeMaskFromStringList(&o->value);
			return TRUE;
		}
		break;

	case GROUP_SCREEN_OPTION_SELECTION_COLOR:
	case GROUP_SCREEN_OPTION_LINE_COLOR:
	case GROUP_SCREEN_OPTION_TAB_FILL_COLOR:
	case GROUP_SCREEN_OPTION_TAB_BORDER_COLOR:
		if (compSetColorOption(o, value)) {
			return TRUE;
		}
		break;

	case GROUP_SCREEN_OPTION_TABBAR_FONTCOLOR:
		if (compSetColorOption(o, value)) {
			GroupSelection *group;

			for (group = gs->groups; group; group = group->next) 
				groupRenderWindowTitle(group);

			return TRUE;
		}
		break;

	case GROUP_SCREEN_OPTION_GLOW_TYPE:
		if (compSetStringOption(o, value)) {
			int i;
			for (i = 0; i < GlowTextureNum; i++) {
				if (strcmp (o->value.s, glowTextureName[i]) == 0) {
					if (i != gs->glowType) {
						gs->glowType = i;

						finiTexture (s, &gs->glowTexture);
 						initTexture (s, &gs->glowTexture);

 						RGBAimageToTexture (s, &gs->glowTexture, 
							glowTextureProperties[gs->glowType].textureData,
							glowTextureProperties[gs->glowType].textureSize,
							glowTextureProperties[gs->glowType].textureSize);

						if (gs->opt[GROUP_SCREEN_OPTION_GLOW].value.b && gs->groups) {
							groupRecomputeGlow (s);
							damageScreen (s);
						}
					}
					return TRUE;
				}
			}
		}
		break;

	case GROUP_SCREEN_OPTION_OPACITY:
	case GROUP_SCREEN_OPTION_BRIGHTNESS:
	case GROUP_SCREEN_OPTION_SATURATION:
	case GROUP_SCREEN_OPTION_PRECISION:
	case GROUP_SCREEN_OPTION_TAB_DRAG_Y_DISTANCE:
	case GROUP_SCREEN_OPTION_TAB_DRAG_SPEED_LIMIT:
		if (compSetIntOption(o, value)) {
			return TRUE;
		}
		break;
	
	case GROUP_SCREEN_OPTION_THUMB_SIZE:
	case GROUP_SCREEN_OPTION_BORDER_WIDTH:
		if (compSetIntOption(o, value)) {
			GroupSelection *group;
			
			for (group = gs->groups; group; group = group->next)
			{
				if(group->tabBar)
					groupRecalcTabBarPos(group, (group->tabBar->region->extents.x1 + group->tabBar->region->extents.x2) / 2,
					                            group->tabBar->region->extents.x1, group->tabBar->region->extents.x2);
			}
			
			return TRUE;
		}
		break;
	
	case GROUP_SCREEN_OPTION_BORDER_RADIUS:
		if (compSetIntOption(o, value)) {
			GroupSelection *group;
			
			for (group = gs->groups; group; group = group->next)
			{
				if(group->tabBar)
					groupRenderTabBarBackground(group);		
			}
			
			return TRUE;
		}
		break;

	case GROUP_SCREEN_OPTION_TABBAR_FONTSIZE:
		if (compSetIntOption(o, value)) {
			GroupSelection *group;
			for (group = gs->groups; group; group = group->next) {
				groupRenderWindowTitle(group);
			}
			return TRUE;
		}
		break;

	case GROUP_SCREEN_OPTION_TABBING_SPEED:
	case GROUP_SCREEN_OPTION_TABBING_TIMESTEP:
	case GROUP_SCREEN_OPTION_FADE_TIME:
	case GROUP_SCREEN_OPTION_FADE_TEXT_TIME:
	case GROUP_SCREEN_OPTION_VISIBILITY_TIME:
	case GROUP_SCREEN_OPTION_CHANGE_ANIMATION_TIME:
	case GROUP_SCREEN_OPTION_TAB_DRAG_HOVER_TIME:
	case GROUP_SCREEN_OPTION_TAB_DRAG_SPRING_K:
	case GROUP_SCREEN_OPTION_TAB_DRAG_FRICTION:
		if (compSetFloatOption(o, value)) {
			return TRUE;
		}
		break;

	case GROUP_SCREEN_OPTION_GLOW_SIZE:
		if (compSetIntOption(o, value)) {
			if (gs->opt[GROUP_SCREEN_OPTION_GLOW].value.b) {
				CompWindow *w;

				groupRecomputeGlow (s);

				for (w = s->windows; w; w = w->next) {
					GROUP_WINDOW (w);

					if (gw->glowQuads) {
						damageWindowOutputExtents (w);
						updateWindowOutputExtents (w);
						damageWindowOutputExtents (w);
					}
				}
			}
			return TRUE;
		}
		break;

	case GROUP_SCREEN_OPTION_GLOW:
		if (compSetBoolOption(o, value)) {
			CompWindow *w;

			groupRecomputeGlow (s);

			for (w = s->windows; w; w = w->next) {
				GROUP_WINDOW (w);

				if (gw->glowQuads) {
					damageWindowOutputExtents (w);
					updateWindowOutputExtents (w);
					damageWindowOutputExtents (w);
				}
			}
			return TRUE;
		}
		break;

	case GROUP_SCREEN_OPTION_MOVE:
	case GROUP_SCREEN_OPTION_RESIZE:
	case GROUP_SCREEN_OPTION_RAISE:
	case GROUP_SCREEN_OPTION_AUTO_UNGROUP:
	case GROUP_SCREEN_OPTION_AUTO_GROUP:
	case GROUP_SCREEN_OPTION_RELATIVE_DISTANCE:
	case GROUP_SCREEN_OPTION_TAB_CREATE_MIPMAPS:
	case GROUP_SCREEN_OPTION_UNTAB_ON_CLOSE:
	case GROUP_SCREEN_OPTION_SPRING_MODEL_ON_MOVE:
	case GROUP_SCREEN_OPTION_AUTOTAB:
	case GROUP_SCREEN_OPTION_DND_UNGROUP_WINDOW:
	case GROUP_SCREEN_OPTION_RESIZE_UNMAXIMIZE:
	case GROUP_SCREEN_OPTION_MINIMIZE_ALL:
	case GROUP_SCREEN_OPTION_SHADE_ALL:
		if (compSetBoolOption(o, value)) {
			return TRUE;
		}
		break;

	default:
		break;
	}

	return FALSE;
}

/*
 * groupDisplayInitOptions
 *
 */
void groupDisplayInitOptions(GroupDisplay * gd)
{
	CompOption *o;

	o = &gd->opt[GROUP_DISPLAY_OPTION_SELECT];
	o->name = "select";
	o->group = N_("Selection");
	o->subGroup = N_("Key bindings");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Select");
	o->longDesc = N_("The key for starting selecting windows.");
	o->type = CompOptionTypeAction;
	o->value.action.initiate = groupSelect;
	o->value.action.terminate = groupSelectTerminate;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitButton;
	o->value.action.type = CompBindingTypeButton;
	o->value.action.button.modifiers = GROUP_SELECT_MODIFIERS_DEFAULT;
	o->value.action.button.button = GROUP_SELECT_BUTTON_DEFAULT;

	o = &gd->opt[GROUP_DISPLAY_OPTION_SELECT_SINGLE];
	o->name = "select_single";
	o->group = N_("Selection");
	o->subGroup = N_("Key bindings");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Select single window");
	o->longDesc = N_("The key for selecting the current window.");
	o->type = CompOptionTypeAction;
	o->value.action.initiate = groupSelectSingle;
	o->value.action.terminate = 0;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers =
	    GROUP_SELECT_SINGLE_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
	    XStringToKeysym(GROUP_SELECT_SINGLE_KEY_DEFAULT);

	o = &gd->opt[GROUP_DISPLAY_OPTION_GROUPING];
	o->name = "group";
	o->group = N_("Grouping");
	o->subGroup = N_("Key bindings");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Group");
	o->longDesc = N_("The key for grouing windows.");
	o->type = CompOptionTypeAction;
	o->value.action.initiate = groupGroupWindows;
	o->value.action.terminate = 0;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = GROUP_GROUPING_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
	    XStringToKeysym(GROUP_GROUPING_KEY_DEFAULT);

	o = &gd->opt[GROUP_DISPLAY_OPTION_UNGROUPING];
	o->name = "ungroup";
	o->group = N_("Grouping");
	o->subGroup = N_("Key bindings");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Ungroup");
	o->longDesc = N_("The key for ungrouing windows.");
	o->type = CompOptionTypeAction;
	o->value.action.initiate = groupUnGroupWindows;
	o->value.action.terminate = 0;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = GROUP_UNGROUPING_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
	    XStringToKeysym(GROUP_UNGROUPING_KEY_DEFAULT);

	o = &gd->opt[GROUP_DISPLAY_OPTION_TABMODE];
	o->name = "tabmode";
	o->group = N_("Tabbing");
	o->subGroup = N_("Key bindings");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Tab");
	o->longDesc = N_("The key for entering the tab mode.");
	o->type = CompOptionTypeAction;
	o->value.action.initiate = groupInitTab;
	o->value.action.terminate = 0;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = GROUP_TABMODE_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
	    XStringToKeysym(GROUP_TABMODE_KEY_DEFAULT);

	o = &gd->opt[GROUP_DISPLAY_OPTION_CHANGE_TAB_LEFT];
	o->name = "change_tab_left";
	o->group = N_("Tabbing");
	o->subGroup = N_("Key bindings");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Change Tab Left");
	o->longDesc = N_("The key for changing the tab to the left.");
	o->type = CompOptionTypeAction;
	o->value.action.initiate = groupChangeTabLeft;
	o->value.action.terminate = 0;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = GROUP_CHANGE_TAB_LEFT_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
	    XStringToKeysym(GROUP_CHANGE_TAB_LEFT_KEY_DEFAULT);

	o = &gd->opt[GROUP_DISPLAY_OPTION_CHANGE_TAB_RIGHT];
	o->name = "change_tab_right";
	o->group = N_("Tabbing");
	o->subGroup = N_("Key bindings");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Change Tab Right");
	o->longDesc = N_("The key for changing the tab to the right.");
	o->type = CompOptionTypeAction;
	o->value.action.initiate = groupChangeTabRight;
	o->value.action.terminate = 0;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = GROUP_CHANGE_TAB_RIGHT_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
	    XStringToKeysym(GROUP_CHANGE_TAB_RIGHT_KEY_DEFAULT);

	o = &gd->opt[GROUP_DISPLAY_OPTION_REMOVING];
	o->name = "remove";
	o->group = N_("Grouping");
	o->subGroup = N_("Key bindings");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Remove Window");
	o->longDesc = N_("The key for removing the selected window.");
	o->type = CompOptionTypeAction;
	o->value.action.initiate = groupRemoveWindow;
	o->value.action.terminate = 0;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = GROUP_REMOVING_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
	    XStringToKeysym(GROUP_REMOVING_KEY_DEFAULT);

	o = &gd->opt[GROUP_DISPLAY_OPTION_CLOSING];
	o->name = "close";
	o->group = N_("Grouping");
	o->subGroup = N_("Key bindings");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Close Windows");
	o->longDesc = N_("The key for closing all windows in the group.");
	o->type = CompOptionTypeAction;
	o->value.action.initiate = groupCloseWindows;
	o->value.action.terminate = 0;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = GROUP_CLOSING_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
	    XStringToKeysym(GROUP_CLOSING_KEY_DEFAULT);

	o = &gd->opt[GROUP_DISPLAY_OPTION_IGNORE];
	o->name = "ignore";
	o->group = N_("Grouping");
	o->subGroup = N_("Key bindings");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Ignore Group");
	o->longDesc = N_("The key for ignoring the group."
			 "If this key is pressed you can resize/move a single"
			 "window in the group.");
	o->type = CompOptionTypeAction;
	o->value.action.initiate = groupSetIgnore;
	o->value.action.terminate = groupUnsetIgnore;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	o->value.action.type = CompBindingTypeKey;
	o->value.action.key.modifiers = GROUP_IGNORE_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
	    XStringToKeysym(GROUP_IGNORE_KEY_DEFAULT);

	o = &gd->opt[GROUP_DISPLAY_OPTION_CHANGE_COLOR];
	o->name = "change_color";
	o->group = N_("Glow");
	o->subGroup = N_("Key bindings");
	o->displayHints = "";
	o->advanced = False;
	o->shortDesc = N_("Change the glow color");
	o->longDesc =
	    N_("If you don't like the current color of the glow, with this key you can change it.");
	o->type = CompOptionTypeAction;
	o->value.action.initiate = groupChangeColor;
	o->value.action.terminate = 0;
	o->value.action.bell = FALSE;
	o->value.action.edgeMask = 0;
	o->value.action.state = CompActionStateInitKey;
	//o->value.action.type = CompBindingTypeKey;
	o->value.action.type = 0;
	o->value.action.key.modifiers =
	    GROUP_CHANGE_COLOR_MODIFIERS_DEFAULT;
	o->value.action.key.keysym =
	    XStringToKeysym(GROUP_CHANGE_COLOR_KEY_DEFAULT);
}

/*
 * groupGetDisplayOptions
 *
 */
CompOption *groupGetDisplayOptions(CompDisplay * d, int *count)
{
	if (d) {
		GROUP_DISPLAY(d);

		*count = NUM_OPTIONS(gd);
		return gd->opt;
	} else {
		GroupDisplay *gd = malloc(sizeof(GroupDisplay));
		groupDisplayInitOptions(gd);
		*count = NUM_OPTIONS(gd);
		return gd->opt;
	}
}

/*
 * groupSetDisplayOption
 *
 */
Bool
groupSetDisplayOption(CompDisplay * d, char *name, CompOptionValue * value)
{
	CompOption *o;
	int index;

	GROUP_DISPLAY(d);

	o = compFindOption(gd->opt, NUM_OPTIONS(gd), name, &index);
	if (!o)
		return FALSE;

	switch (index) {

	case GROUP_DISPLAY_OPTION_SELECT:
	case GROUP_DISPLAY_OPTION_SELECT_SINGLE:
	case GROUP_DISPLAY_OPTION_GROUPING:
	case GROUP_DISPLAY_OPTION_UNGROUPING:
	case GROUP_DISPLAY_OPTION_REMOVING:
	case GROUP_DISPLAY_OPTION_CLOSING:
	case GROUP_DISPLAY_OPTION_CHANGE_COLOR:
	case GROUP_DISPLAY_OPTION_IGNORE:
	case GROUP_DISPLAY_OPTION_TABMODE:
	case GROUP_DISPLAY_OPTION_CHANGE_TAB_LEFT:
	case GROUP_DISPLAY_OPTION_CHANGE_TAB_RIGHT:
		if (setDisplayAction(d, o, value))
			return TRUE;
		break;

	default:
		break;
	}

	return FALSE;
}
