/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef FLASHINSPECTOR_H_
#define FLASHINSPECTOR_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "types.h"
#include "SplittableBuffer.h"
#include <stddef.h>
#include <map>
#include <memory>

class DeflateDecompressor;

class FlashInspector
{
public:
	enum Status { IN_PROGRESS, COMPLETE, FAILED };
	
	FlashInspector();
	
	~FlashInspector();
	
	void consumeDataChunk(SplittableBuffer& data, bool eof);
	
	void reset();
	
	SplittableBuffer& accumulatedData() { return m_incomingData; }
	
	Status getStatus() const { return m_status; }
	
	unsigned int getWidth() const { return m_width; }
	
	unsigned int getHeight() const { return m_height; }
	
	unsigned int getFrameRate() const { return m_frameRate; }
	
	bool isClickable() const { return m_isClickable; }
private:
	typedef SplittableBuffer::ByteIterator DataIter;	
	
	enum State {
		ST_HEADER,
		ST_HEADER2,
		ST_TAG_HEADER,
		ST_TAG_BODY
	};
	
	class OOB {}; // out of bounds exception
	struct Rect;
	struct Matrix;
	struct Context;
	
	struct Position
	{
		DataIter bytepos;
		int bitpos;
		
		Position(DataIter const& bytep, int bitp = 0) : bytepos(bytep), bitpos(bitp) {}
	};
	
	struct Shape
	{
		int32_t width, height;
		int32_t btn_width, btn_height;
		// We fill btn_width and btn_height only for buttons that have a getUrl action.
		// btn_width * btn_height is always less or equal than width * height
		
		Shape() : width(0), height(0), btn_width(0), btn_height(0) {}
		
		Shape(int32_t w, int32_t h) : width(w), height(h), btn_width(0), btn_height(0) {}
		
		double getArea() const { return double(width) * double(height); }
	
		double getButtonArea() const { return double(btn_width) * double(btn_height); }
	};
	
	void processNewData(bool eof);
	
	void processButton(Position& pos, DataIter const& end);
	
	void processButton2(Position& pos, DataIter const& end);
	
	static bool processButtonActions(Position& pos, DataIter const& end);
	
	static bool processActionRecords(Position& pos, DataIter const& end);
	
	static bool processUrlAction(Position& pos, DataIter const& end, uint8_t code);
	
	bool processButtonRecords(Shape& hit_area, Position& pos, DataIter const& end, bool has_cxform);
	
	void processPlaceObject(Position& pos, DataIter const& end, Shape* sprite = 0);
	
	void processPlaceObject2(Position& pos, DataIter const& end, Shape* sprite = 0);
	
	bool processClipActions(Position& pos, DataIter const& end);
	
	uint32_t readClipEventFlags(Position& pos, DataIter const& end);
	
	void processObjectPlacement(Shape const& object, Matrix const* mat, Shape* sprite = 0);
	
	void processSprite(Position& pos, DataIter const& end);
	
	static void readRect(Rect& rect, Position& pos, DataIter const& end);
	
	static void readMatrix(Matrix& mat, Position& pos, DataIter const& end);
	
	static void skipColorTransformAlpha(Position& pos, DataIter const& end);
	
	static void byteAlign(Position& pos);
	
	static uint8_t readU8(DataIter& pos, DataIter const& end);
	
	static uint16_t readU16(DataIter& pos, DataIter const& end);
	
	static uint32_t readU32(DataIter& pos, DataIter const& end);
	
	static uint32_t readUBits(Position& pos, DataIter const& end, int nbits);
	
	static int32_t readSBits(Position& pos, DataIter const& end, int nbits);
	
	static void readBytes(DataIter& pos, DataIter const& end, size_t size, uint8_t* dest);
	
	static void skipBytes(DataIter& pos, DataIter const& end, size_t size);
	
	static void skipString(DataIter& pos, DataIter const& end);
	
	static int const INSPECTION_FRAME_LIMIT = 3;
	static size_t const INSPECTION_SIZE_LIMIT = 70*1024;
	static double const BUTTON_AREA_THRESHOLD;
	
	Status m_status;
	State m_state;
	SplittableBuffer m_incomingData;
	SplittableBuffer m_data;
	std::auto_ptr<DeflateDecompressor> m_pDecomp;
	std::map<uint16_t, Shape> m_shapes;
	unsigned int m_flashVersion;
	unsigned int m_width;
	unsigned int m_height;
	double m_frameArea;
	uint16_t m_frameRate;
	uint16_t m_tagCode;
	int m_curFrame;
	size_t m_tagLength;
	bool m_isClickable;
};

#endif
