/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "HttpStreamWriter.h"
#include "AbstractDataConsumer.h"
#include "HttpRequestMetadata.h"
#include "HttpResponseMetadata.h"
#include "HttpVersion.h"
#include "HttpRequestLine.h"
#include "HttpStatusLine.h"
#include "SplittableBuffer.h"
#include "DataChunk.h"
#include "BString.h"
#include "SBOutStream.h"
#include "HttpHeadersCollection.h"
#include "HttpHeader.h"
#include "HttpHeaderStructure.h"
#include "HttpHeaderElement.h"
#include "StringUtils.h"
#include "URI.h"
#include <string>
#include <algorithm>
#include <list>
#include <cassert>

using namespace std;

HttpStreamWriter::HttpStreamWriter(AbstractDataConsumer& consumer)
:	m_rConsumer(consumer),
	m_state(WAITING_FOR_METADATA),
	m_isEOF(false),
	m_isConnClose(false)
{
}

void
HttpStreamWriter::startRequest(HttpRequestMetadata& metadata,
	bool set_conn_close, bool through_proxy, bool chunked_ok)
{
	reset();
	
	// We always send HTTP/1.1 requests, although we keep them
	// HTTP/1.0 compatible.
	HttpVersion const& http_ver = HttpVersion::HTTP_1_1;
	HttpHeadersCollection& headers = metadata.headers();
	processRequestLine(metadata.requestLine(), http_ver, through_proxy);
	prepareRequestHeaders(headers, metadata.requestLine().getURI());
	
	
	if (metadata.requestLine().getHttpVersion() < HttpVersion::HTTP_1_0) {
		m_state = FLAT_BODY;
		m_isConnClose = true;
		return;
	}
	
	switch (metadata.getBodyStatus()) {
		case HttpMessageMetadata::BODY_FORBIDDEN: {
			prepareForbiddenBodyMode(headers, http_ver, set_conn_close);
			break;
		}
		case HttpMessageMetadata::BODY_SIZE_KNOWN: {
			prepareSizedFlatMode(
				headers, http_ver,
				set_conn_close, metadata.getBodySize()
			);
			break;
		}
		case HttpMessageMetadata::BODY_SIZE_UNKNOWN: {
			if (chunked_ok) {
				prepareChunkedMode(
					headers, http_ver, set_conn_close
				);
			} else {
				prepareBufferedFlatMode(
					headers, http_ver, set_conn_close
				);
			}
			break;
		}
	}
	
	if (m_state != BUFFERED_FLAT_BODY) {
		output();
	}
}

void
HttpStreamWriter::startResponse(HttpResponseMetadata& metadata,
	HttpVersion const& max_http_ver, bool set_conn_close)
{
	reset();
	
	if (max_http_ver < HttpVersion::HTTP_1_0) {
		m_state = FLAT_BODY;
		return;
	}
	
	HttpHeadersCollection& headers = metadata.headers();
	HttpVersion http_ver = std::min(max_http_ver, HttpVersion::HTTP_1_1);
	processStatusLine(metadata.statusLine(), http_ver);
	prepareResponseHeaders(headers);
	
	switch (metadata.getBodyStatus()) {
		case HttpMessageMetadata::BODY_FORBIDDEN: {
			prepareForbiddenBodyMode(headers, http_ver, set_conn_close);
			break;
		}
		case HttpMessageMetadata::BODY_SIZE_KNOWN: {
			prepareSizedFlatMode(
				headers, http_ver,
				set_conn_close, metadata.getBodySize()
			);
			break;
		}
		case HttpMessageMetadata::BODY_SIZE_UNKNOWN: {
			if (http_ver >= HttpVersion::HTTP_1_1) {
				prepareChunkedMode(headers, http_ver, set_conn_close);
			} else {
				prepareUnsizedFlatMode(headers, http_ver, true);
			}
			break;
		}
	}
	
	output();
}

void
HttpStreamWriter::appendBodyData(SplittableBuffer& data, bool eof)
{
	if (m_isEOF) {
		assert(data.empty() && eof);
		return;
	}
	m_isEOF = eof;
	
	switch (m_state) {
		case WAITING_FOR_METADATA: {
			assert(false && "body data before metadata");
			return;
		}
		case FLAT_BODY: {
			m_outStream.data().appendDestructive(data);
			break;
		}
		case BUFFERED_FLAT_BODY: {
			m_bufferedBodyPart.appendDestructive(data);
			if (!eof) {
				return;
			}
			m_outStream << "Content-Length: " << m_bufferedBodyPart.size() << "\r\n\r\n";
			m_outStream.data().appendDestructive(m_bufferedBodyPart);
			break;
		}
		case CHUNKED_BODY: {
			m_bufferedBodyPart.appendDestructive(data);
			if (!m_bufferedBodyPart.empty()) {
				size_t size = m_bufferedBodyPart.size();
				if (size < BUFFER_CHUNKS_LESS_THAN && !eof) {
					return;
				} else {
					m_outStream << std::hex << size << std::dec << "\r\n";
					m_outStream.data().appendDestructive(m_bufferedBodyPart);
					m_outStream << "\r\n";
				}
			}
			if (eof) {
				m_outStream << "0\r\n\r\n";
			}
			break;
		}
	}
	
	output(eof);
}

void
HttpStreamWriter::reset()
{
	m_outStream.clear();
	m_bufferedBodyPart.clear();
	m_state = WAITING_FOR_METADATA;
	m_isEOF = false;
	m_isConnClose = false;
}

void
HttpStreamWriter::output(bool eof)
{
	SplittableBuffer data;
	m_outStream.swapData(data);
	m_rConsumer.processNewData(data, eof);
}

void
HttpStreamWriter::processRequestLine(
	HttpRequestLine& request_line, HttpVersion const& http_ver,
	bool through_proxy)
{
	request_line.setHttpVersion(http_ver);
	request_line.toStream(m_outStream, !through_proxy);
}

void
HttpStreamWriter::processStatusLine(HttpStatusLine& status_line, HttpVersion const& http_ver)
{
	status_line.setHttpVersion(http_ver);
	if (http_ver < HttpVersion::HTTP_1_1) {
		if (status_line.getCode() == 303 || status_line.getCode() == 307) {
			status_line.setCodeAndMessage(302);
		}
	}
	status_line.toStream(m_outStream);
}

void
HttpStreamWriter::prepareCommonHeaders(HttpHeadersCollection& headers)
{
	BString const connection("Connection");
	if (HttpHeader* conn_hdr = headers.getHeaderPtr(connection)) {
		HttpHeaderStructure structure(*conn_hdr);
		for (; !structure.empty(); structure.elements().pop_front()) {
			headers.removeHeader(structure.elements().front().getName());
		}
		headers.removeHeader(connection);
	}
	headers.removeHeader(BString("Proxy-Connection"));
	headers.removeHeader(BString("Keep-Alive"));
}

void
HttpStreamWriter::prepareRequestHeaders(HttpHeadersCollection& headers, URI const& request_uri)
{
	prepareCommonHeaders(headers);
	BString const host("Host");
	//headers.setHeader(HttpHeader(BString("Connection"), BString("te")));
	//headers.setHeader(HttpHeader(BString("TE"), BString("gzip,deflate")));
	headers.setHeader(HttpHeader(BString("Accept-Encoding"), BString("gzip,deflate")));
	if (!headers.hasHeader(host)) {
		if (request_uri.getPort() == -1) {
			headers.setHeader(HttpHeader(host, request_uri.getHost()));
		} else {
			SBOutStream strm(50);
			strm << request_uri.getHost() << ':' << request_uri.getPort();
			headers.setHeader(HttpHeader(host, strm.data().toBString()));
		}
	}
}

void
HttpStreamWriter::prepareForbiddenBodyMode(
	HttpHeadersCollection& headers, HttpVersion const& http_ver,
	bool set_conn_close)
{
	m_state = FLAT_BODY;
	m_isConnClose = set_conn_close;
	handleConnClose(headers, http_ver, set_conn_close);
	m_outStream << headers << "\r\n";
}

void
HttpStreamWriter::prepareSizedFlatMode(
	HttpHeadersCollection& headers, HttpVersion const& http_ver,
	bool set_conn_close, uintmax_t body_size)
{
	m_state = FLAT_BODY;
	m_isConnClose = set_conn_close;
	handleConnClose(headers, http_ver, set_conn_close);
	headers.setHeader(HttpHeader(
		BString("Content-Length"),
		BString(StringUtils::fromNumber(body_size))
	));
	headers.removeHeader(BString("Transfer-Encoding"));
	m_outStream << headers << "\r\n";
}

void
HttpStreamWriter::prepareUnsizedFlatMode(
	HttpHeadersCollection& headers, HttpVersion const& http_ver,
	bool set_conn_close)
{
	m_state = FLAT_BODY;
	m_isConnClose = true;
	handleConnClose(headers, http_ver, set_conn_close);
	headers.removeHeader(BString("Content-Length"));
	headers.removeHeader(BString("Transfer-Encoding"));
	m_outStream << headers << "\r\n";
}

void
HttpStreamWriter::prepareBufferedFlatMode(
	HttpHeadersCollection& headers, HttpVersion const& http_ver,
	bool set_conn_close)
{
	m_state = BUFFERED_FLAT_BODY;
	m_isConnClose = set_conn_close;
	handleConnClose(headers, http_ver, set_conn_close);
	headers.removeHeader(BString("Content-Length"));
	headers.removeHeader(BString("Transfer-Encoding"));
	m_outStream << headers; // Content-Length will be appended later
}

void
HttpStreamWriter::prepareChunkedMode(
	HttpHeadersCollection& headers, HttpVersion const& http_ver,
	bool set_conn_close)
{
	m_state = CHUNKED_BODY;
	m_isConnClose = set_conn_close;
	handleConnClose(headers, http_ver, set_conn_close);
	headers.removeHeader(BString("Content-Length"));
	BString const transfer_encoding("Transfer-Encoding");
	BString const chunked("chunked");
	if (HttpHeader* teh = headers.getHeaderPtr(transfer_encoding)) {
		HttpHeaderStructure structure(*teh);
		structure.appendElement(HttpHeaderElement(chunked));
		structure.commitChanges(*teh);
	} else {
		headers.setHeader(HttpHeader(transfer_encoding, chunked));
	}
	m_outStream <<  headers << "\r\n";
}

void
HttpStreamWriter::handleConnClose(
	HttpHeadersCollection& headers, HttpVersion const& http_ver,
	bool set_conn_close)
{
	BString const connection("Connection");
	BString const close("close");
	BString const keep_alive("keep-alive");
	HttpHeader* ch = headers.getHeaderPtr(BString("Connection"));
	if (!ch) {
		if (set_conn_close) {
			headers.setHeader(HttpHeader(connection, close));
		} else {
			if (http_ver < HttpVersion::HTTP_1_1) {
				headers.setHeader(HttpHeader(connection, keep_alive));
			}
		}
	} else {
		HttpHeaderStructure structure(*ch);
		if (set_conn_close) {
			structure.appendElement(HttpHeaderElement(close));
		} else {
			structure.removeElementsByName(close);
			if (http_ver < HttpVersion::HTTP_1_1) {
				structure.appendElement(keep_alive);
			} else if (structure.empty()) {
				headers.removeHeader(connection);
				return;
			}
		}
		structure.commitChanges(*ch);
	}
}
