/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "WorkerThreadPool.h"
#include "ServiceContext.h"
#include "Reactor.h"
#include "ReactorFactory.h"
#include "SynchFactory.h"
#include "TimeDelta.h"
#include "MonotonicTimer.h"
#include "Client.h"
#include "ScopedIncDec.h"
#include "Debug.h"
#include <ace/config-lite.h>
#include <ace/Thread_Manager.h>
#include <ace/Time_Value.h>
#include <ace/Reverse_Lock_T.h>
#include <ace/OS_NS_sys_time.h>
#include <memory>
#include <stdexcept>

using namespace std;

class WorkerThreadPool::ReactorRegistration : public IntrusiveListNode
{
public:
	ReactorRegistration(Reactor* reactor, ReactorList& list);
	
	Reactor* reactor() { return m_pReactor; }
private:
	Reactor* m_pReactor;
};


WorkerThreadPool::WorkerThreadPool()
:	m_cond(m_mutex),
#if defined(ACE_WIN32) && !ACE_HAS_WINNT4
	// CriticalSection-based mutexes don't support trylock() on win9x
	m_leaderMutex(USYNC_PROCESS),
#else
	m_leaderMutex(USYNC_THREAD),
#endif
	m_connAcceptor(ConnAcceptor::DONT_STOP_ON_SIGNALS),
	m_numWaiters(0),
	m_isClosing(false)
{
}

WorkerThreadPool::~WorkerThreadPool()
{
	{
		ACE_Guard<Mutex> guard(m_mutex);
		
		m_isClosing = true;
		
		m_connAcceptor.abort();
		
		ReactorList::iterator it = m_runningReactors.begin();
		ReactorList::iterator const end = m_runningReactors.end();
		for (; it != end; ++it) {
			it->reactor()->stop();
		}
		
		m_cond.broadcast();
	}
	m_threadManager.wait();
}

void
WorkerThreadPool::start()
{
	prepareNewThread();
}

bool
WorkerThreadPool::addAcceptor(ConnAcceptor::AcceptorPtr const& acceptor)
{
	return m_connAcceptor.add(acceptor);
}

void
WorkerThreadPool::removeAcceptor(ConnAcceptor::AcceptorPtr const& acceptor)
{
	m_connAcceptor.remove(acceptor);
}

void
WorkerThreadPool::removeAllAcceptors()
{
	m_connAcceptor.removeAll();
}

WorkerThreadPool::NewThreadStatus
WorkerThreadPool::prepareNewThread()
{
	ACE_Guard<Mutex> guard(m_mutex);
	
	if (m_isClosing) {
		return CLOSING;
	}
	
	int res = 0;
	if (m_numWaiters > 0) {
		res = m_cond.signal();
	} else {
		m_lastThreadCreateTime = MonotonicTimer::getTimestamp();
		res = m_threadManager.spawn(
			&WorkerThreadPool::runThread,
			this, THR_SCOPE_SYSTEM|THR_DETACHED
		);
	}
	return res == 0 ? READY : FAILURE;
}

int
WorkerThreadPool::wait()
{
	// m_mutex is locked by our caller
	
	ScopedIncDec<int> waiters_manager(m_numWaiters);
	
	ACE_Time_Value tv(WAIT_FOR_NEW_TASK);
	tv += ACE_OS::gettimeofday();
	return m_cond.wait(&tv);
}

void
WorkerThreadPool::runService()
{
	auto_ptr<Reactor> reactor;
	try {
		reactor = ReactorFactory::createBestReactor(SynchFactory<ACE_NULL_SYNCH>());
	} catch (Reactor::Exception& e) {
		DEBUGLOG2("Could not initialize the reactor: " << e.what());
		return;
	}
	ServiceContext service_context(*reactor);
	
	ACE_Guard<Mutex> guard(m_mutex);
	
	ReactorRegistration reactor_registration(reactor.get(), m_runningReactors);
	
	while (!m_isClosing) {
		if (tryBecomingLeader(service_context)) {
			continue;
		}
		
		if (m_numWaiters >= WAITERS_KEEP_ALIVE && MonotonicTimer::getTimestamp()
		    > m_lastThreadCreateTime + TimeDelta::fromSec(WAIT_FOR_NEW_TASK)) {
			break;
		}
		
		if (wait() == -1) {
			if (m_numWaiters >= WAITERS_KEEP_ALIVE) {
				break;
			}
		}
	}
}

/*
Leader is a worker that is currently waiting for connections.
For details, search google for Leader/Followers pattern.
*/
bool
WorkerThreadPool::tryBecomingLeader(ServiceContext& service_context)
{
	// m_mutex is locked by our caller
	
	typedef ACE_Reverse_Lock<Mutex> ReverseLock;
	ReverseLock reverse_lock(m_mutex);
	ACE_Guard<ReverseLock> unguard(reverse_lock);
	
	// m_mutex is unlocked
	
	Client::SocketPtr client_sock(new RefCountableSAP<ACE_SOCK_Stream>);
	
	{ // leader lock scope
		ACE_Guard<LeaderMutex> leader_guard(m_leaderMutex, /* block = */0);
		if (!leader_guard.locked()) {
			return false;
		}
		
		ConnAcceptor::Status const conn_status = m_connAcceptor.accept(
			*client_sock, service_context.clientAddr()
		);
		if (conn_status != ConnAcceptor::ACCEPTED) {
			return true;
		}
	}
	
	// We need a new leader.
	if (prepareNewThread() == CLOSING) {
		return true;
	}
	
	Client client(service_context, client_sock);
	Reactor& reactor = service_context.reactor();
	Reactor::Status const reactor_status = reactor.runEventLoop();
	if (reactor_status == Reactor::STOPPED) {
		reactor.restart();
	}
	service_context.reset(); // should be called after the event loop, not before it
	
	return true;
}

ACE_THR_FUNC_RETURN
WorkerThreadPool::runThread(void* arg)
{
	static_cast<WorkerThreadPool*>(arg)->runService();
	return 0;
}


/*================ WorkerThreadPool::ReactorRegistration ===============*/

WorkerThreadPool::ReactorRegistration::ReactorRegistration(
	Reactor* reactor, ReactorList& list)
:	m_pReactor(reactor)
{
	list.push_back(*this);
}
