/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "JsFilterContext.h"
#include "JsRuntime.h"
#include "JsRequestScope.h"
#include "FilterJsLogger.h"
#include "FilterTag.h"
#include "FilterGroupTag.h"
#include "BString.h"
#include <ace/config-lite.h>
#include <ace/OS_NS_stdio.h>
#include <jsapi.h>
#include <map>

using namespace std;

class JsFilterContext::JsException
{
};


class JsFilterContext::Impl
{
public:
	Impl();
	
	~Impl();
	
	std::auto_ptr<std::string> performAction(
		FilterTag const& filter_tag, FilterGroupTag const& group_tag,
		std::string const& js_code, std::vector<BString> const& params);
private:
	enum { BRANCH_LIMIT = 400 };
	enum { MAX_MESSAGES = 12 };
	
	bool paramsToJsval(
		std::vector<BString> const& source,
		std::vector<jsval>& target);
	
	bool requestSendingMessage();
	
	static void errorReporter(JSContext *cx, const char *msg, JSErrorReport *report);
	
	static JSBool branchCallback(JSContext *cx, JSScript *script);
	
	static JSBool log(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval);
	
	std::map<FilterTag, JSFunction*> m_compiledFunctions;
	// A null JSFunction* means the function didn't compile because of errors.
	
	FilterTag const* m_pCurFilterTag;
	FilterGroupTag const* m_pCurGroupTag;
	JSContext* m_pContext;
	JSObject* m_pRoot;
	int m_branchCount;
	int m_functionCount;
	int m_messageCount;
};


JsFilterContext::JsFilterContext()
{
	try {
		m_ptrImpl.reset(new Impl);
	} catch (JsException const&) {}
}

JsFilterContext::~JsFilterContext()
{
}

std::auto_ptr<std::string>
JsFilterContext::performAction(
	FilterTag const& filter_tag, FilterGroupTag const& group_tag,
	std::string const& js_code, std::vector<BString> const& params)
{
	if (m_ptrImpl.get()) {
		return m_ptrImpl->performAction(filter_tag, group_tag, js_code, params);
	} else {
		return std::auto_ptr<std::string>();
	}
}


/*======================= JsFilterContext::Impl =========================*/

JsFilterContext::Impl::Impl()
:	m_pCurFilterTag(0),
	m_pCurGroupTag(0),
	m_pContext(0),
	m_pRoot(0),
	m_branchCount(0),
	m_functionCount(0),
	m_messageCount(0)
{
	JSRuntime* runtime = JsRuntime::rep();
	m_pContext = JS_NewContext(runtime, STACK_SIZE);
	if (!m_pContext) {
		throw JsException();
	}
	JS_SetContextPrivate(m_pContext, this);
	
	JsRequestScope rscope(m_pContext);
	
	JS_SetErrorReporter(m_pContext, &errorReporter);
	JS_SetBranchCallback(m_pContext, &branchCallback);
	
	m_pRoot = JS_NewObject(m_pContext, 0, 0, 0);
	if (!m_pRoot) {
		throw JsException();
	}
	if (!JS_InitStandardClasses(m_pContext, m_pRoot)) {
		throw JsException();
	}
	
	JSFunction* log_fn = JS_DefineFunction(
		m_pContext, m_pRoot, "log", &log, 1,
		JSPROP_ENUMERATE|JSPROP_PERMANENT
	);
	if (!log_fn) {
		throw JsException();
	}
}

JsFilterContext::Impl::~Impl()
{
	JS_DestroyContext(m_pContext);
}

std::auto_ptr<std::string>
JsFilterContext::Impl::performAction(
	FilterTag const& filter_tag, FilterGroupTag const& group_tag,
	std::string const& js_code, std::vector<BString> const& params)
{
	auto_ptr<string> res;
	
	m_pCurFilterTag = &filter_tag;
	m_pCurGroupTag = &group_tag;
	JSFunction*& fun = m_compiledFunctions[filter_tag];
	
	JsRequestScope rscope(m_pContext);
	
	if (!fun) {
		char name[20];
		++m_functionCount;
		ACE_OS::snprintf(name, sizeof(name), "__f%d__", m_functionCount);
		fun = JS_CompileFunction(
			m_pContext, m_pRoot, name,
			0, 0, // zero arguments and null pointer to arguments
			js_code.c_str(), js_code.size(),
			0, 1 // null file name and line number
		);
	}
	
	if (fun) {
		m_branchCount = 0;
		std::vector<jsval> jsval_params;
		if (paramsToJsval(params, jsval_params)) {
			jsval rval;
			JSBool r = JS_CallFunction(
				m_pContext, m_pRoot, fun,
				jsval_params.size(), &jsval_params[0],
				&rval
			);
			if (r == JS_TRUE && !JSVAL_IS_NULL(rval) && !JSVAL_IS_VOID(rval)) {
				JSString* str = JS_ValueToString(m_pContext, rval);
				if (str) {
					res.reset(new string(JS_GetStringBytes(str)));
				}
			}
		}
	}
	
	// m_pCurFilterTag = 0; // not really necessary
	// m_pCurGroupTag = 0;  //
	
	return res;
}

bool
JsFilterContext::Impl::paramsToJsval(
	std::vector<BString> const& source, std::vector<jsval>& target)
{
	target.reserve(source.size());
	std::vector<BString>::const_iterator it = source.begin();
	std::vector<BString>::const_iterator const end = source.end();
	for (; it != end; ++it) {
		JSString* jstr = JS_NewStringCopyN(m_pContext, it->begin(), it->size());
		if (!jstr) {
			return false;
		}
		target.push_back(STRING_TO_JSVAL(jstr));
	}
	return true;
}

bool
JsFilterContext::Impl::requestSendingMessage()
{
	if (m_messageCount == MAX_MESSAGES) {
		char const* msg = "<too many messages from a single context>";
		FilterJsLogger::logMessage(*m_pCurFilterTag, *m_pCurGroupTag, msg);
	}
	return (++m_messageCount <= MAX_MESSAGES);
}

void
JsFilterContext::Impl::errorReporter(
	JSContext* cx, char const* msg, JSErrorReport* report)
{
	Impl* impl = static_cast<Impl*>(JS_GetContextPrivate(cx));
	if (impl->requestSendingMessage()) {
		FilterJsLogger::logError(
			*impl->m_pCurFilterTag, *impl->m_pCurGroupTag,
			msg, report->lineno, report->linebuf,
			report->tokenptr - report->linebuf
		);
	}
}

JSBool
JsFilterContext::Impl::branchCallback(JSContext *cx, JSScript *script)
{
	Impl* impl = static_cast<Impl*>(JS_GetContextPrivate(cx));
	if (++impl->m_branchCount < BRANCH_LIMIT) {
		return JS_TRUE;
	} else {
		// infinite loop?
		return JS_FALSE; // terminate the script
	}
}

JSBool
JsFilterContext::Impl::log(
	JSContext* cx, JSObject* obj, uintN argc, jsval* argv, jsval* rval)
{
	*rval = JSVAL_VOID;
	if (argc > 0) {
		Impl* impl = static_cast<Impl*>(JS_GetContextPrivate(cx));
		if (impl->requestSendingMessage()) {
			JSString* str = JS_ValueToString(cx, argv[0]);
			if (str) {
				char const* data = JS_GetStringBytes(str);
				if (data) {
					FilterJsLogger::logMessage(
						*impl->m_pCurFilterTag,
						*impl->m_pCurGroupTag, data
					);
				}
			}
		}
	}
	return JS_TRUE;
}
