/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ErrorResponse.h"
#include "CraftedResponse.h"
#include "SplittableBuffer.h"
#include "SBOutStream.h"
#include "BString.h"
#include "Date.h"
#include "StringUtils.h"
#include "HttpStatusLine.h"
#include "ArraySize.h"
#include "HtmlEscaper.h"
#include <ace/config-lite.h>
#include <ace/OS_NS_netdb.h>
#include <ace/OS_NS_unistd.h>
#include <stddef.h>
#include <cstring>

using namespace std;

auto_ptr<CraftedResponse>
ErrorResponse::errConnectionFailed(string url, string host, int port, char const* syserr)
{
	static char const* templ[] =
#	include "ERR_CONNECT_FAIL.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url;
	m[BString("@HOST@")] = host;
	m[BString("@PORT@")] = StringUtils::fromNumber(port);
	m[BString("@SYSERR@")] = syserr;
	auto_ptr<CraftedResponse> response(
		new CraftedResponse(HttpStatusLine::SC_SERVICE_UNAVAILABLE)
	);
	prepareContent(*response, templ, m);
	return response;
}

auto_ptr<CraftedResponse>
ErrorResponse::errConnectionTimeout(string url, string host, int port)
{
	static char const* templ[] =
#	include "ERR_CONNECT_TIMEOUT.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url;
	m[BString("@HOST@")] = host;
	m[BString("@PORT@")] = StringUtils::fromNumber(port);
	auto_ptr<CraftedResponse> response(
		new CraftedResponse(HttpStatusLine::SC_GATEWAY_TIMEOUT)
	);
	prepareContent(*response, templ, m);
	return response;
}

auto_ptr<CraftedResponse>
ErrorResponse::errHostNameResolution(string url, string host)
{
	static char const* templ[] =
#	include "ERR_HOST_NAME_RESOLUTION.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url;
	m[BString("@HOST@")] = host;
	auto_ptr<CraftedResponse> response(
		new CraftedResponse(HttpStatusLine::SC_SERVICE_UNAVAILABLE)
	);
	prepareContent(*response, templ, m);
	return response;
}

auto_ptr<CraftedResponse>
ErrorResponse::errServerTimeout(string url)
{
	static char const* templ[] =
#	include "ERR_SERVER_TIMEOUT.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url;
	auto_ptr<CraftedResponse> response(
		new CraftedResponse(HttpStatusLine::SC_GATEWAY_TIMEOUT)
	);
	prepareContent(*response, templ, m);
	return response;
}

auto_ptr<CraftedResponse>
ErrorResponse::errParsingRequest(string url, string problem)
{
	static char const* templ[] =
#	include "ERR_PARSING_REQUEST.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url;
	m[BString("@PROBLEM@")] = problem;
	auto_ptr<CraftedResponse> response(
		new CraftedResponse(HttpStatusLine::SC_BAD_REQUEST)
	);
	prepareContent(*response, templ, m);
	return response;
}

auto_ptr<CraftedResponse>
ErrorResponse::errParsingResponse(string url, string problem)
{
	static char const* templ[] =
#	include "ERR_PARSING_RESPONSE.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url;
	m[BString("@PROBLEM@")] = problem;
	auto_ptr<CraftedResponse> response(
		new CraftedResponse(HttpStatusLine::SC_SERVICE_UNAVAILABLE)
	);
	prepareContent(*response, templ, m);
	return response;
}

auto_ptr<CraftedResponse>
ErrorResponse::errUnsupportedProtocol(string url, string protocol)
{
	static char const* templ[] =
#	include "ERR_UNSUPPORTED_PROTOCOL.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url;
	m[BString("@PROTOCOL@")] = protocol;
	auto_ptr<CraftedResponse> response(
		new CraftedResponse(HttpStatusLine::SC_NOT_IMPLEMENTED)
	);
	prepareContent(*response, templ, m);
	return response;
}

auto_ptr<CraftedResponse>
ErrorResponse::errUnsupportedMethod(string url, string method)
{
	static char const* templ[] =
#	include "ERR_UNSUPPORTED_METHOD.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url;
	m[BString("@METHOD@")] = method;
	auto_ptr<CraftedResponse> response(
		new CraftedResponse(HttpStatusLine::SC_NOT_IMPLEMENTED)
	);
	prepareContent(*response, templ, m);
	return response;
}

auto_ptr<CraftedResponse>
ErrorResponse::errExpectationFailed(string url, string expectation)
{
	static char const* templ[] =
#	include "ERR_EXPECTATION_FAILED.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url;
	m[BString("@EXPECTATION@")] = expectation;
	auto_ptr<CraftedResponse> response(
		new CraftedResponse(HttpStatusLine::SC_EXPECTATION_FAILED)
	);
	prepareContent(*response, templ, m);
	return response;
}

auto_ptr<CraftedResponse>
ErrorResponse::errGenericError(string url, string error, int status_code)
{
	static char const* templ[] =
#	include "ERR_GENERIC.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url;
	m[BString("@ERROR@")] = error;
	auto_ptr<CraftedResponse> response(
		new CraftedResponse(status_code)
	);
	prepareContent(*response, templ, m);
	return response;
}

auto_ptr<CraftedResponse>
ErrorResponse::errUrlForbidden(string url)
{
	static char const* templ[] =
#	include "ERR_URL_FORBIDDEN.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url;
	auto_ptr<CraftedResponse> response(
		new CraftedResponse(HttpStatusLine::SC_FORBIDDEN)
	);
	prepareContent(*response, templ, m);
	return response;
}

auto_ptr<CraftedResponse>
ErrorResponse::errRemoteProxyConfig(string url, string problem)
{
	static char const* templ[] =
#	include "ERR_REMOTE_PROXY_CONFIG.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url;
	m[BString("@PROBLEM@")] = problem;
	auto_ptr<CraftedResponse> response(
		new CraftedResponse(HttpStatusLine::SC_INTERNAL_SERVER_ERROR)
	);
	prepareContent(*response, templ, m);
	return response;
}

auto_ptr<CraftedResponse>
ErrorResponse::errForwardingThroughProxy(
	std::string url, std::string problem,
	std::string proxy_type, std::string proxy_host, int proxy_port)
{
	static char const* templ[] =
#	include "ERR_FORWARDING_THROUGH_PROXY.inc"
	MacroMap m;
	prepareCommonMacros(m);
	m[BString("@URL@")] = url;
	m[BString("@PROBLEM@")] = problem;
	m[BString("@PROXY_TYPE@")] = proxy_type;
	m[BString("@PROXY_HOST@")] = proxy_host;
	m[BString("@PROXY_PORT@")] = StringUtils::fromNumber(proxy_port);
	auto_ptr<CraftedResponse> response(
		new CraftedResponse(HttpStatusLine::SC_SERVICE_UNAVAILABLE)
	);
	prepareContent(*response, templ, m);
	return response;
}

auto_ptr<CraftedResponse>
ErrorResponse::errDummyError()
{
	return auto_ptr<CraftedResponse>(
		new CraftedResponse(HttpStatusLine::SC_INTERNAL_SERVER_ERROR)
	);
}


void
ErrorResponse::prepareCommonMacros(MacroMap& m)
{
	m[BString("@DATE@")] = Date::formatCurrentTime().toStdString();
	m[BString("@HOSTNAME@")] = getFullHostName();
	m[BString("@VERSION@")] = BFILTER_VERSION;
}

void
ErrorResponse::prepareContent(
	CraftedResponse& response, char const* templ[], MacroMap const& macros)
{
	BString const empty;
	SBOutStream strm;
	for (char const* str; (str = *templ); ++templ) {
		size_t len = strlen(str);
		if (len >= 2 && str[0] == '@' && str[len-1] == '@') {
			MacroMap::const_iterator fit =
				macros.find(BString(empty, str, str + len));
			if (fit != macros.end()) {
				strm << HtmlEscaper::escape(fit->second);
			}
		} else {
			strm.write(str, len);
		}
	}

	HttpHeadersCollection& headers = response.metadata().headers();
	headers.setHeader(BString("Content-Type"), BString("text/html"));
	headers.setHeader(BString("Cache-Control"), BString("no-cache, must-revalidate"));
	headers.setHeader(BString("X-BFilter-Content"), BString("generated"));
	strm.swapData(response.body());
}

std::string
ErrorResponse::getFullHostName()
{
	char hname[MAXHOSTNAMELEN + 1];
	if (ACE_OS::hostname(hname, ARRAY_SIZE(hname)) < 0) {
		return string();
	}
	
	struct hostent h_ent, *h_entp;
	ACE_HOSTENT_DATA buf;
	int h_err = 0;
	h_entp = ACE_OS::gethostbyname_r(hname, &h_ent, buf, &h_err);
	if (!h_entp || h_err < 0) {
		return hname;
	}
	
	return string(h_entp->h_name);
}
