/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "UrlsFile.h"
#include "UrlPatterns.h"
#include "UrlsFileStructure.h"
#include "ConfigErrorHandler.h"
#include "OperationLog.h"
#include "Log.h"
#include "FileVersion.h"
#include "FileOps.h"
#include "GlobalState.h"
#include <stddef.h>

namespace wxGUI
{

UrlsFile::UrlsFile(Which which, wxFileName const& file_path)
:	m_filePath(file_path),
	m_which(which)
{
}

UrlsFile::~UrlsFile()
{
}

bool
UrlsFile::loadAndForceApply()
{
	Log* const log = OperationLog::instance();
	
	log->appendRecord(
		m_which == URLS_LOCAL
		? _T("Loading local url patterns ... ")
		: _T("Loading standard url patterns ... ")
	);
	size_t const num_records = log->getNumRecords();
	
	std::string text;
	FileVersion version;
	if (!FileOps::readFile(m_filePath, text, &version)) {
		return false;
	}
	
	UrlPatterns new_patterns;
	UrlsFileStructure new_structure;
	ConfigErrorHandler eh(getFileName());
	new_structure.load(text, new_patterns, eh);
	
	apply(new_patterns, version);
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord(_T("done"), log->getSuccessStyle());
	}
	
	return true;
}

bool
UrlsFile::applyAndSave(std::string const& text)
{
	Log* const log = OperationLog::instance();
	
	log->appendRecord(
		m_which == URLS_LOCAL
		? _T("Applying local url patterns ... ")
		: _T("Applying standard url patterns ... ")
	);
	size_t const num_records = log->getNumRecords();
	
	UrlPatterns new_patterns;
	UrlsFileStructure new_structure;
	ConfigErrorHandler eh(getFileName());
	new_structure.load(text, new_patterns, eh);
	if (eh.numErrors() != 0) {
		return false;
	}
	
	FileVersion version;
	if (!FileOps::writeFile(m_filePath, text, &version)) {
		return false;
	}
	
	apply(new_patterns, version);
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord(_T("done"), log->getSuccessStyle());
	}
	
	return true;
}

FileVersion
UrlsFile::readFileVersion() const
{
	return FileOps::readFileVersion(m_filePath);
}

wxString
UrlsFile::getFileName() const
{
	return m_which == URLS_LOCAL ? _T("urls.local") : _T("urls");
}

void
UrlsFile::apply(UrlPatterns& new_patterns, FileVersion const& version)
{
	GlobalState::WriteAccessor global_state;
	if (m_which == URLS) {
		global_state->swapStandardUrlPatterns(new_patterns);
	} else {
		global_state->swapLocalUrlPatterns(new_patterns);
		global_state->setUrlsLocalFileVersion(version);
	}
}

} // namespace wxGUI
