/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FilterOrderingTestSuite.h"
#include "ContentFilterList.h"
#include "AvailableContentFilters.h"
#include "AvailableFiltersOrdered.h"
#include "RegexFilterDescriptor.h"
#include "IntrusivePtr.h"
#include "FilterTag.h"
#include "FilterGroupTag.h"
#include "TextPattern.h"
#include <string>
#include <iostream>

using boost::unit_test_framework::test_suite;
using boost::unit_test_framework::test_case;

static void test();
static void check_ordered_filters(AvailableFiltersOrdered const& filters);

static void test()
{
	std::string const fname1("file1");
	std::string const fname2("file2");
	FilterGroupTag const group_tag1(fname1);
	FilterGroupTag const group_tag2(fname2);
	IntrusivePtr<TextPattern const> const pattern(
		new TextPattern("*", 0)
	);
	BOOST_REQUIRE(pattern->isValid());
	ContentFilterList group1;
	ContentFilterList group2;
	
	RegexFilterDescriptor filter11(FilterTag(), group_tag1, "filter11");
	filter11.searchPattern() = pattern;
	filter11.replacement().reset(new std::string());
	filter11.setEnabled();
	BOOST_REQUIRE(filter11.isValid());
	BOOST_REQUIRE(filter11.isEnabled());
	group1.push_back(filter11);
	
	RegexFilterDescriptor filter12(FilterTag(), group_tag1, "filter12");
	filter12.searchPattern() = pattern;
	filter12.replacement().reset(new std::string());
	filter12.setEnabled();
	BOOST_REQUIRE(filter12.isValid());
	BOOST_REQUIRE(filter12.isEnabled());
	group1.push_back(filter12);
	
	RegexFilterDescriptor filter13(FilterTag(), group_tag1, "filter13");
	filter13.searchPattern() = pattern;
	filter13.replacement().reset(new std::string());
	BOOST_REQUIRE(filter13.isValid());
	BOOST_REQUIRE(filter13.isEnabled() == false);
	group1.push_back(filter13);
	
	RegexFilterDescriptor filter21(FilterTag(), group_tag2, "filter21");
	filter21.searchPattern() = pattern;
	filter21.replacement().reset(new std::string());
	filter21.setEnabled();
	BOOST_REQUIRE(filter21.isValid());
	BOOST_REQUIRE(filter21.isEnabled());
	group2.push_back(filter21);
	
	RegexFilterDescriptor filter22(FilterTag(), group_tag2, "filter22");
	filter22.searchPattern() = pattern;
	filter22.replacement().reset(new std::string());
	filter22.order() = -10;
	filter22.setEnabled();
	BOOST_REQUIRE(filter22.isValid());
	BOOST_REQUIRE(filter22.isEnabled());
	group2.push_back(filter22);
	
	RegexFilterDescriptor filter23(FilterTag(), group_tag2, "filter23");
	filter23.replacement().reset(new std::string());
	filter23.setEnabled();
	BOOST_REQUIRE(filter23.isValid() == false);
	BOOST_REQUIRE(filter23.isEnabled());
	group2.push_back(filter23);
	
	AvailableContentFilters avail1;
	avail1.add(group1);
	avail1.add(group2);
	check_ordered_filters(AvailableFiltersOrdered(avail1));
	
	AvailableContentFilters avail2;
	avail2.add(group2);
	avail2.add(group1);
	check_ordered_filters(AvailableFiltersOrdered(avail2));
}

static void check_ordered_filters(AvailableFiltersOrdered const& filters)
{
	BOOST_REQUIRE(filters.end() - filters.begin() == 4);
	AvailableFiltersOrdered::iterator it(filters.begin());
	BOOST_REQUIRE((*it)->name() == "filter22");
	++it;
	BOOST_REQUIRE((*it)->name() == "filter11");
	++it;
	BOOST_REQUIRE((*it)->name() == "filter12");
	++it;
	BOOST_REQUIRE((*it)->name() == "filter21");
}

/*========================= FilterOrderingTestSuite ======================*/

FilterOrderingTestSuite::FilterOrderingTestSuite()
{
	add(BOOST_TEST_CASE(&test));
}
