/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ForwardingTestSuite.h"
#include "Forwarding.h"
#include "ConfErrorHandler.h"
#include "ConfError.h"
#include "BString.h"
#include "URI.h"
#include "ProxyDescriptor.h"
#include "SymbolicInetAddr.h"
#include <boost/test/unit_test.hpp>
#include <sstream>
#include <string>
#include <memory>
#include <iostream>

using boost::unit_test_framework::test_suite;
using boost::unit_test_framework::test_case;

static char const xml_input[] =
"<?xml version='1.0' encoding='UTF-8'?>\n"
"<forwarding>\n"
"\n"
"<option name='Direct'>\n"
"</option>\n"
"\n"
"<option name='Corporate Proxy' selected='selected'>\n"
"  <bypass>\n"
"    <host-mask>*.corp.com</host-mask>\n"
"    <host-mask>127.*</host-mask>\n"
"  </bypass>\n"
"  <proxy-chain>\n"
"    <proxy>\n"
"      <type>http</type>\n"
"      <host>proxy.corp.com</host>\n"
"      <port>3128</port>\n"
"    </proxy>\n"
"  </proxy-chain>\n"
"  <proxy-chain/><!-- error: multiple <proxy-chain> tags -->\n"
"  <option><!-- error: unexpected tag --><more><stuff/></more></option>\n"
"</option>\n"
"\n"
"<option name='Tor'>\n"
"  <bypass>\n"
"    <simple-hostnames/>\n"
"    <host-mask>127.*.*.*</host-mask>\n"
"    <host-mask>172.16.*.*</host-mask>\n"
"  </bypass>\n"
"  <proxy-chain>\n"
"    <proxy>\n"
"      <type>socks4a</type>\n"
"      <host>localhost</host>\n"
"      <port>9050</port>\n"
"    </proxy>\n"
"  </proxy-chain>\n"
"</option>\n"
"\n"
"<option name='Tor -&gt; Other Proxy'>\n"
"  <bypass>\n"
"    <host-mask>corp.com</host-mask>\n"
"    <host-mask>*.ad*.corp.*</host-mask>\n"
"  </bypass>\n"
"  <proxy-chain>\n"
"    <proxy>\n"
"      <type>socks4a</type>\n"
"      <host>localhost</host>\n"
"      <port>9050</port>\n"
"    </proxy>\n"
"    <proxy>\n"
"      <type>socks5</type>\n"
"      <host>1.2.3.4</host>\n"
"      <port>3128</port>\n"
"      <user>john</user>\n"
"      <pass><![CDATA[1234]]></pass>\n"
"    </proxy>\n"
"  </proxy-chain>\n"
"</option>\n"
"</forwarding>\n";


static void test();
static void check_config(Forwarding::Config const& config);
static void check_option_1(Forwarding::Option const& opt);
static void check_option_2(Forwarding::Option const& opt);
static void check_option_3(Forwarding::Option const& opt);
static void check_option_4(Forwarding::Option const& opt);
static void test_resolver(Forwarding::Config const& config);
static void check_empty_chain(Forwarding::ProxyChainConstPtr const& chain);
static void check_system_chain(Forwarding::ProxyChainConstPtr const& chain);
static void check_chain_2(Forwarding::ProxyChainConstPtr const& chain);
static void check_chain_3(Forwarding::ProxyChainConstPtr const& chain);
static void check_chain_4(Forwarding::ProxyChainConstPtr const& chain);
static void check_corp_proxy(ProxyDescriptor const& proxy);
static void check_tor_proxy(ProxyDescriptor const& proxy);
static void check_other_proxy(ProxyDescriptor const& proxy);
static void check_system_proxy(ProxyDescriptor const& proxy);

namespace
{

class ErrorHandler : public ConfErrorHandler
{
public:
	ErrorHandler() : m_numErrors(0) {}
	
	int getErrorCount() const { return m_numErrors; }
	
	virtual bool handleError(ConfError const& err) {
		/*
		if (err.getLine() > 0) {
			std::cerr << '[' << err.getLine();
			if (err.getCol() > 0) {
				std::cerr << ':' << err.getCol();
			}
			std::cerr << "] ";
		}
		std::cerr << err.getMessage() << std::endl;
		*/
		++m_numErrors;
		return true;
	}
private:
	int m_numErrors;
};

}


static void test()
{
	std::istringstream strm1(xml_input);
	Forwarding::Config config1;
	ErrorHandler eh1;
	config1.fromStream(strm1, eh1);
	
	BOOST_CHECK(eh1.getErrorCount() == 2);
	check_config(config1);
	test_resolver(config1);
	
	std::stringstream strm2;
	config1.toStream(strm2);
	
	Forwarding::Config config2;
	ErrorHandler eh2;
	config2.fromStream(strm2, eh2);
	
	BOOST_REQUIRE(eh2.getErrorCount() == 0);
	check_config(config2);
	test_resolver(config2);
}

static void check_config(Forwarding::Config const& config)
{
	BOOST_REQUIRE(config.options().size() == 4);
	
	Forwarding::OptionList::const_iterator it(config.options().begin());
	check_option_1(*it);
	++it;
	check_option_2(*it);
	++it;
	check_option_3(*it);
	++it;
	check_option_4(*it);
}

static void check_option_1(Forwarding::Option const& opt)
{
	BOOST_CHECK(opt.getName() == Forwarding::Utf8String("Direct"));
	BOOST_CHECK(opt.isSelected() == false);
	BOOST_CHECK(opt.bypassList().size() == 0);
	BOOST_CHECK(opt.proxyChain().size() == 0);
}

static void check_option_2(Forwarding::Option const& opt)
{
	BOOST_CHECK(opt.getName() == Forwarding::Utf8String("Corporate Proxy"));
	BOOST_CHECK(opt.isSelected() == true);
	BOOST_REQUIRE(opt.bypassList().size() == 2);
	BOOST_REQUIRE(opt.proxyChain().size() == 1);
	
	// <host-mask>*.corp.com</host-mask>
	Forwarding::MatcherConstPtr m1(opt.bypassList().front()->getMatcher());
	BOOST_CHECK(!m1->matches(URI(BString("http://corp.com"))));
	BOOST_CHECK(!m1->matches(URI(BString("http://corp.com."))));
	BOOST_CHECK(m1->matches(URI(BString("http://sub.corp.com"))));
	BOOST_CHECK(m1->matches(URI(BString("http://sub.corp.com."))));
	BOOST_CHECK(m1->matches(URI(BString("http://sub.corp.com/zzz"))));
	BOOST_CHECK(!m1->matches(URI(BString("http://sub.corp.net"))));
	BOOST_CHECK(!m1->matches(URI(BString("http://othercorp.com"))));
	
	// <host-mask>127.*</host-mask>
	Forwarding::MatcherConstPtr m2(opt.bypassList().back()->getMatcher());
	BOOST_CHECK(m2->matches(URI(BString("http://127.0.0.1"))));
	BOOST_CHECK(m2->matches(URI(BString("http://127.0.0.2/zzz"))));
	BOOST_CHECK(!m2->matches(URI(BString("http://128.0.0.1/"))));
	
	check_corp_proxy(opt.proxyChain().front());
}

static void check_option_3(Forwarding::Option const& opt)
{
	BOOST_CHECK(opt.getName() == Forwarding::Utf8String("Tor"));
	BOOST_CHECK(opt.isSelected() == false);
	BOOST_REQUIRE(opt.bypassList().size() == 3);
	BOOST_REQUIRE(opt.proxyChain().size() == 1);
	
	// <simple-hostnames/>
	Forwarding::MatcherConstPtr m1(opt.bypassList().front()->getMatcher());
	BOOST_CHECK(m1->matches(URI(BString("http://no-dots-in-host/zzz"))));
	BOOST_CHECK(!m1->matches(URI(BString("http://has.dots.in.host/zzz"))));
	
	// <host-mask>172.16.*.*</host-mask>
	Forwarding::MatcherConstPtr m2(opt.bypassList().back()->getMatcher());
	BOOST_CHECK(m2->matches(URI(BString("http://172.16.1.1/"))));
	BOOST_CHECK(!m2->matches(URI(BString("http://172.17.1.1/zzz"))));
	
	check_tor_proxy(opt.proxyChain().front());
}

static void check_option_4(Forwarding::Option const& opt)
{
	BOOST_CHECK(opt.getName() == Forwarding::Utf8String("Tor -> Other Proxy"));
	BOOST_CHECK(opt.isSelected() == false);
	BOOST_REQUIRE(opt.bypassList().size() == 2);
	BOOST_REQUIRE(opt.proxyChain().size() == 2);
		
	// <host-mask>corp.com</host-mask>
	Forwarding::MatcherConstPtr m1(opt.bypassList().front()->getMatcher());
	BOOST_CHECK(m1->matches(URI(BString("http://corp.com"))));
	BOOST_CHECK(m1->matches(URI(BString("http://corp.com."))));
	BOOST_CHECK(!m1->matches(URI(BString("http://sub.corp.com"))));
	BOOST_CHECK(!m1->matches(URI(BString("http://corp.net"))));
	BOOST_CHECK(!m1->matches(URI(BString("http://othercorp.com"))));
	
	
	// <host-mask>*.ad*.corp.*</host-mask>
	Forwarding::MatcherConstPtr m3(opt.bypassList().back()->getMatcher());
	BOOST_CHECK(m3->matches(URI(BString("http://us.ad02.corp.com"))));
	BOOST_CHECK(m3->matches(URI(BString("http://us.ad02.corp.net"))));
	BOOST_CHECK(m3->matches(URI(BString("http://us.ad.corp.com."))));
	BOOST_CHECK(!m3->matches(URI(BString("http://ad.corp.com"))));
	BOOST_CHECK(!m3->matches(URI(BString("http://us.az02.corp.com"))));
	BOOST_CHECK(!m3->matches(URI(BString("http://www.corp.com"))));
	BOOST_CHECK(!m3->matches(URI(BString("http://othercorp.com"))));
	
	check_tor_proxy(opt.proxyChain().front());
	check_other_proxy(opt.proxyChain().back());
}

static void test_resolver(Forwarding::Config const& config)
{
	Forwarding::Resolver r(config);
	
	check_chain_2(r.resolve(URI(BString("http://somesite.org/zzz"))));
	check_chain_2(r.resolve(URI(BString("http://corp.com/zzz"))));
	check_empty_chain(r.resolve(URI(BString("http://zzz.corp.com/"))));
	
	BOOST_CHECK_NO_THROW(r.selectOption(Forwarding::Utf8String("Tor")));
	check_chain_3(r.resolve(URI(BString("http://somesite.org/zzz"))));
	
	BOOST_CHECK_THROW(r.selectOption(Forwarding::Utf8String("123")), Forwarding::ResolverException);

	Forwarding::SystemOption sys_opt(Forwarding::Utf8String("Unrelated"));
	ProxyDescriptor proxy;
	proxy.setType(proxy.SOCKS4);
	proxy.setAddr(SymbolicInetAddr("1.1.1.1", 1111));
	sys_opt.proxyChain().push_back(proxy);
	
	r.selectSystemOption(sys_opt);
	check_system_chain(r.resolve(URI(BString("http://somesite.org/zzz"))));
	
	sys_opt.setErrorMessage("Wrong configuration");
	r.selectSystemOption(sys_opt);
	BOOST_CHECK_THROW(r.resolve(URI(BString("http://somesite.org/forum/zzz"))), Forwarding::ResolverException);
}

static void check_empty_chain(Forwarding::ProxyChainConstPtr const& chain)
{
	BOOST_REQUIRE(chain.get());
	BOOST_CHECK(chain->empty());
}

static void check_system_chain(Forwarding::ProxyChainConstPtr const& chain)
{
	BOOST_REQUIRE(chain.get());
	BOOST_REQUIRE(chain->size() == 1);
	check_system_proxy(chain->front());
}

static void check_chain_2(Forwarding::ProxyChainConstPtr const& chain)
{
	BOOST_REQUIRE(chain.get());
	BOOST_REQUIRE(chain->size() == 1);
	check_corp_proxy(chain->front());
}

static void check_chain_3(Forwarding::ProxyChainConstPtr const& chain)
{
	BOOST_REQUIRE(chain.get());
	BOOST_REQUIRE(chain->size() == 1);
	check_tor_proxy(chain->front());
}

static void check_chain_4(Forwarding::ProxyChainConstPtr const& chain)
{
	BOOST_REQUIRE(chain.get());
	BOOST_REQUIRE(chain->size() == 2);
	check_tor_proxy(chain->front());
	check_other_proxy(chain->back());
}

static void check_corp_proxy(ProxyDescriptor const& proxy)
{
	BOOST_CHECK(proxy.getType() == ProxyDescriptor::HTTP);
	BOOST_CHECK(proxy.getAddr().getHost() == "proxy.corp.com");
	BOOST_CHECK(proxy.getAddr().getPort() == 3128);
	BOOST_CHECK(proxy.getUserName().empty());
	BOOST_CHECK(proxy.getPassword().empty());
}

static void check_tor_proxy(ProxyDescriptor const& proxy)
{
	BOOST_CHECK(proxy.getType() == ProxyDescriptor::SOCKS4A);
	BOOST_CHECK(proxy.getAddr().getHost() == "localhost");
	BOOST_CHECK(proxy.getAddr().getPort() == 9050);
	BOOST_CHECK(proxy.getUserName().empty());
	BOOST_CHECK(proxy.getPassword().empty());
}

static void check_other_proxy(ProxyDescriptor const& proxy)
{
	BOOST_CHECK(proxy.getType() == ProxyDescriptor::SOCKS5);
	BOOST_CHECK(proxy.getAddr().getHost() == "1.2.3.4");
	BOOST_CHECK(proxy.getAddr().getPort() == 3128);
	BOOST_CHECK(proxy.getUserName() == "john");
	BOOST_CHECK(proxy.getPassword() == "1234");
}

static void check_system_proxy(ProxyDescriptor const& proxy)
{
	BOOST_CHECK(proxy.getType() == ProxyDescriptor::SOCKS4);
	BOOST_CHECK(proxy.getAddr().getHost() == "1.1.1.1");
	BOOST_CHECK(proxy.getAddr().getPort() == 1111);
	BOOST_CHECK(proxy.getUserName().empty());
	BOOST_CHECK(proxy.getPassword().empty());
}


/*=========================== ForwardingTestSuite ========================*/

ForwardingTestSuite::ForwardingTestSuite()
{
	add(BOOST_TEST_CASE(&test));
}
