/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ChainAsyncConnector.h"
#include "ChainConnectorListener.h"
#include "SymbolicInetAddr.h"
#include "ProxyDescriptor.h"
#include "ConnectionRoute.h"
#include <string>
#include <stddef.h>
#include <assert.h>

class ChainAsyncConnector::Hops
{
public:
	Hops(ConnectionRoute const& route);
	
	~Hops();
	
	ProxyDescriptor const* getHopOrigin() const;
	
	SymbolicInetAddr const& getHopTarget() const;
	
	bool nextHop();
private:
	std::vector<ProxyDescriptor> m_proxies;
	SymbolicInetAddr m_targetAddr;
	size_t m_curHop; // 0 means connecting to m_proxies[0]
};


/*========================= ChainAsyncConnector ========================*/

ChainAsyncConnector::ChainAsyncConnector()
:	m_pReactor(0)
{
}

ChainAsyncConnector::~ChainAsyncConnector()
{
}

void
ChainAsyncConnector::initiate(
	Listener& listener, Reactor& reactor,
	ConnectionRoute const& route,
	TimeDelta const* hop_timeout)
{
	abort();
	
	try {
		/*
		Make isInProgress() return true.
		This must be done before anything else, because otherwise
		abort() would consider there is nothing to abort.
		*/
		m_pReactor = &reactor;
		
		m_observerLink.setObserver(&listener);
		m_ptrHops.reset(new Hops(route));
		m_hopTimeout = hop_timeout ? *hop_timeout : TimeDelta::max();
		
		// Note: this may result in onConnectionFailed() to be called.
		m_directConnector.initiate(
			*this, reactor,
			m_ptrHops->getHopTarget(), hop_timeout
		);
	} catch (...) {
		abort();
		throw;
	}
}

void
ChainAsyncConnector::abort()
{
	if (isInProgress()) {
		m_observerLink.setObserver(0);
		m_directConnector.abort();
		m_socksConnector.abort();
		// m_hopTimeout = TimeDelta();
		m_ptrHops.reset(0);
		m_pReactor = 0;
	}
}

void
ChainAsyncConnector::onConnectionEstablished(
	AutoClosingSAP<ACE_SOCK_Stream>& conn)
{
	onHopDone(conn);
}

void
ChainAsyncConnector::onConnectionFailed(AsyncConnectorError const& err)
{
	// Save necessary data before abort()
	Listener* const listener = m_observerLink.getObserver();
	assert(listener);
	SymbolicInetAddr const target(m_ptrHops->getHopTarget());
	
	abort();
	
	listener->onConnectionFailed(err, target);
}

void
ChainAsyncConnector::onConnThroughSocksDone(
	AutoClosingSAP<ACE_SOCK_Stream>& conn)
{
	onHopDone(conn);
}

void
ChainAsyncConnector::onConnThroughSocksFailed(SocksError const& err)
{
	// Save necessary data before abort()
	Listener* const listener = m_observerLink.getObserver();
	assert(listener);
	ProxyDescriptor const proxy(*m_ptrHops->getHopOrigin());
	
	abort();
	
	listener->onConnectionFailed(err, proxy);
}

void
ChainAsyncConnector::onHopDone(AutoClosingSAP<ACE_SOCK_Stream>& conn)
{
	if (!m_ptrHops->nextHop()) {
		Listener* const listener = m_observerLink.getObserver();
		assert(listener);
		abort();
		listener->onConnectionEstablished(conn);
		return;
	}
	
	ProxyDescriptor const* proxy = m_ptrHops->getHopOrigin();
	assert(proxy);
	
	SocksAsyncConnector::SocksType socks_type = SocksAsyncConnector::SOCKS4;
	if (!getSocksType(*proxy, socks_type)) {
		Listener* const listener = m_observerLink.getObserver();
		ProxyDescriptor const proxy_saved(*proxy);
		assert(listener);
		abort();
		listener->onConnectionFailed(
			"unsupported proxy type in proxy chain",
			proxy_saved
		);
		return;
	}
	
	m_socksConnector.initiate(
		socks_type, *this, *m_pReactor, conn,
		m_ptrHops->getHopTarget(),
		proxy->getUserName(), proxy->getPassword(),
		m_hopTimeout == TimeDelta::max() ? 0 : &m_hopTimeout
	);
}

bool
ChainAsyncConnector::getSocksType(
	ProxyDescriptor const& proxy,
	SocksAsyncConnector::SocksType& type)
{
	switch (proxy.getType()) {
		case ProxyDescriptor::SOCKS4: {
			type =  SocksAsyncConnector::SOCKS4;
			break;
		}
		case ProxyDescriptor::SOCKS4A: {
			type = SocksAsyncConnector::SOCKS4A;
			break;
		}
		case ProxyDescriptor::SOCKS5: {
			type = SocksAsyncConnector::SOCKS5;
			break;
		}
		default: {
			return false;
		}
	}
	return true;
}


/*===================== ChainAsyncConnector::Hops ======================*/

ChainAsyncConnector::Hops::Hops(ConnectionRoute const& route)
:	m_proxies(route.getProxies()),
	m_targetAddr(route.getDestination()),
	m_curHop(0)
{
}

ChainAsyncConnector::Hops::~Hops()
{
}

ProxyDescriptor const*
ChainAsyncConnector::Hops::getHopOrigin() const
{
	if (m_curHop == 0) {
		return 0;
	} else {
		return &m_proxies[m_curHop - 1];
	}
}

SymbolicInetAddr const&
ChainAsyncConnector::Hops::getHopTarget() const
{
	if (m_curHop < m_proxies.size()) {
		return m_proxies[m_curHop].getAddr();
	} else {
		return m_targetAddr;
	}
}

bool
ChainAsyncConnector::Hops::nextHop()
{
	if (m_curHop < m_proxies.size()) {
		++m_curHop;
		return true;
	} else {
		return false;
	}
}
