/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef FILTERGROUPTAG_H_
#define FILTERGROUPTAG_H_

#include "Tag.h"
#include "RefCountable.h"
#include "RefCounter.h"
#include <typeinfo>
#include <assert.h>

class ACE_MT_SYNCH;

/**
 * \brief Implementation detail of FilterGroupTag.
 */
class FilterGroupAbstractOrderer :
	public RefCountable<RefCounter<ACE_MT_SYNCH> >
{
public:
	virtual bool before(FilterGroupAbstractOrderer const& rhs) const = 0;
};


/**
 * \brief An id for filter groups.
 *
 * This class also provides ordering for filter groups.  A group of filters
 * is comprised of filters within the same file.
 */
class FilterGroupTag : public Tag<FilterGroupAbstractOrderer, TagNoCompareOps>
{
private:
	typedef Tag<FilterGroupAbstractOrderer, TagNoCompareOps> Base;
public:
	class EquivComp;
	class OrderComp;
	
	// Member-wise copying is OK.
	
	/**
	 * \brief Construct from an ordering token.
	 *
	 * Typically used like this:
	 * \code
	 * std::string file_name = ...;
	 * FilterGroupTag tag(file_name);
	 * \endcode
	 * In this case we order FilterGroupTag's by their file names.
	 * Note that we could use any type, not just std::string.
	 */
	template<typename OrderToken>
	FilterGroupTag(OrderToken const& token);
	
	bool operator==(FilterGroupTag const& rhs) const {
		return &**this == &*rhs;
	}
	
	bool operator!=(FilterGroupTag const& rhs) const {
		return &**this != &*rhs;
	}
private:
	template<typename T> class TemplateOrderer;
	
	FilterGroupTag(); // forbidden
};

/**
 * \brief An std::less<>-like comparator for equivalency relationship.
 */
class FilterGroupTag::EquivComp
{
public:
	/**
	 * \return true if \p lhs precedes \p rhs, false otherwise.
	 */
	bool operator()(FilterGroupTag const& lhs,
	                FilterGroupTag const& rhs) const {
		return &*lhs < &*rhs; // order by address
	}
};


/**
 * \brief An std::less<>-like comparator for ordering relationship.
 */
class FilterGroupTag::OrderComp
{
public:
	/**
	 * \return true if \p lhs precedes \p rhs, false otherwise.
	 */
	bool operator()(FilterGroupTag const& lhs,
	                FilterGroupTag const& rhs) const {
		return lhs->before(*rhs);
	}
};


template<typename T>
class FilterGroupTag::TemplateOrderer : public FilterGroupAbstractOrderer
{
public:
	TemplateOrderer(T const& token) : m_token(token) {}
	
	virtual bool before(FilterGroupAbstractOrderer const& rhs) const;
private:
	T const m_token;
};


template<typename T>
bool
FilterGroupTag::TemplateOrderer<T>::before(
	FilterGroupAbstractOrderer const& rhs) const
{
	typedef TemplateOrderer<T> const* ThisType;
	ThisType rhs_thistype = dynamic_cast<ThisType>(&rhs);
	if (rhs_thistype) {
		return this->m_token < rhs_thistype->m_token;
	} else {
#if 1
		assert(!"FilterGroupTag's with different orderers");
#endif
		// This gives consistent ordering.
		return typeid(*this).before(typeid(rhs));
	}
}


template<typename OrderToken>
FilterGroupTag::FilterGroupTag(OrderToken const& token)
:	Base(new TemplateOrderer<OrderToken>(token))
{
}

#endif
