/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "IPv4SubnetParser.h"
#include "StringUtils.h"
#include "BString.h"
#include <assert.h>

IPv4SubnetParser::IPv4SubnetParser()
:	m_netAddr(0),
	m_netBits(0),
	m_pErrorPos(0)
{
}

bool
IPv4SubnetParser::parse(char const* begin, char const* end)
{
	m_pErrorPos = parseSubnet(begin, end);
	return (m_pErrorPos == 0);
}

bool
IPv4SubnetParser::parse(std::string const& str)
{
	m_pErrorPos = parseSubnet(str.c_str(), str.c_str() + str.size());
	return (m_pErrorPos == 0);
}

bool
IPv4SubnetParser::parse(BString const& str)
{
	m_pErrorPos = parseSubnet(str.begin(), str.end());
	return (m_pErrorPos == 0);
}

/**
 * \return Returns the error position, or null if no error.
 */
char const*
IPv4SubnetParser::parseSubnet(char const* begin, char const* end)
{
	uint32_t ip = 0;
	unsigned netbits = 0;
	
	char const* p = begin;
	for (;;) {
		if (netbits > 32) {
			return p;
		}
		
		uint8_t octet = 0;
		if (parseOctet(p, end, octet)) {
			ip |= uint32_t(octet) << (24 - netbits);
			netbits += 8;
		} else {
			return p;
		}
		
		if (p == end) {
			m_netAddr = ip;
			m_netBits = netbits;
			return 0;
		}
		
		if (*p == '.') {
			++p;
		} else if (*p == '/') {
			break;
		} else {
			return p;
		}
	}
	
	assert(p != end);
	assert(*p == '/');
	++p;
	if (!parseNetBits(p, end, netbits)) {
		return p;
	}
	if (p != end) {
		return p;
	}
	
	m_netAddr = ip;
	m_netBits = netbits;
	return 0;
}

/**
 * On success, true is returned and \a p is moved to the next token.\n
 * On failure, false is returned and \a p is moved to the error position.
 */
bool
IPv4SubnetParser::parseOctet(char const*& p, char const* end, uint8_t& octet)
{
	char const* e = end;
	unsigned val = StringUtils::parseUnsigned<unsigned>(p, e);
	if (e == p) {
		return false;
	} else if (e - p > 3) {
		return false;
	} else if (val > 255) {
		return false;
	} else {
		octet = val;
		p = e;
		return true;
	}
}

/**
 * On success, true is returned, and \a p is moved to the next token.\n
 * On failure, false is returned and \a p is moved to the error position.
 */
bool
IPv4SubnetParser::parseNetBits(char const*& p, char const* end, unsigned& netbits)
{
	char const* e = end;
	unsigned val = StringUtils::parseUnsigned<unsigned>(p, e);
	if (e == p) {
		return false;
	} else if (e - p > 2) {
		return false;
	} else if (val > 32) {
		return false;
	} else {
		netbits = val;
		p = e;
		return true;
	}
}
