/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "RequestLogWindow.h"
#include "Application.h"
#include "MvcListView.h"
#include "RequestLogHandler.h"
#include <ace/config-lite.h>
#include <wx/panel.h>
#include <wx/event.h>
#include <wx/sizer.h>
#include <wx/button.h>
#include <wx/tglbtn.h>
#include <wx/listctrl.h>
#include <wx/colour.h>
#include <wx/icon.h>
#include <wx/font.h>
#include <wx/gdicmn.h>

using namespace std;

namespace wxGUI
{

class RequestLogWindow::LogView : public MvcListView
{
public:
	LogView(wxWindow* parent, wxWindowID id, long);
private:
	virtual void onRowCountChanging();

	virtual void onRowCountChanged();

	void onKeyDown(wxKeyEvent& evt);

	void onDoubleClick(wxMouseEvent& evt);

	bool m_isAtBottom;
	DECLARE_EVENT_TABLE()
};

BEGIN_EVENT_TABLE(RequestLogWindow, wxFrame)
	EVT_CLOSE(onWindowClose)
	EVT_BUTTON(wxID_CLOSE, onClose)
	EVT_BUTTON(wxID_CLEAR, onClear)
	EVT_TOGGLEBUTTON(wxID_HELP, onLegend)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(RequestLogWindow::LogView, MvcListView)
	EVT_KEY_DOWN(onKeyDown)
	EVT_LEFT_DCLICK(onDoubleClick)
END_EVENT_TABLE()

RequestLogWindow* RequestLogWindow::m_spInstance = 0;

RequestLogWindow::RequestLogWindow()
:	wxFrame(wxGetApp().GetTopWindow(), wxID_ANY, _T("BFilter Request Log")),
	m_pTopPanel(0),
	m_pLogSizer(0),
	m_pLogView(0),
	m_pLegendView(0),
	m_pLegendButton(0)
{
	SetExtraStyle(wxWS_EX_BLOCK_EVENTS);
	SetIcon(wxIcon(_T("AppIcon"), wxBITMAP_TYPE_ICO_RESOURCE, 16, 16));

	wxBoxSizer* top_vbox = new wxBoxSizer(wxVERTICAL);
	SetSizer(top_vbox);

	m_pTopPanel = new wxPanel(this);
	top_vbox->Add(m_pTopPanel, 1, wxGROW);

	wxBoxSizer* panel_vbox = new wxBoxSizer(wxVERTICAL);
	m_pTopPanel->SetSizer(panel_vbox);
	
	m_pLogSizer = new wxBoxSizer(wxVERTICAL);
	panel_vbox->Add(m_pLogSizer, 1, wxGROW);
	m_pLogSizer->SetMinSize(500, 250);
	
	wxBoxSizer* hbox1 = new wxBoxSizer(wxHORIZONTAL);
	panel_vbox->Add(hbox1, 0, wxGROW|wxALL, 5);
	
	wxBoxSizer* clearlog_vbox = new wxBoxSizer(wxVERTICAL);
	hbox1->Add(clearlog_vbox, 1);
	
	wxButton* clearlog_btn = new wxButton(m_pTopPanel, wxID_CLEAR, _T("Clear"));
	clearlog_vbox->Add(clearlog_btn, 0, wxALIGN_LEFT);
	
	wxButton* close_btn = new wxButton(m_pTopPanel, wxID_CLOSE, _T("Close"));
	hbox1->Add(close_btn, 0);

	wxBoxSizer* legend_vbox = new wxBoxSizer(wxVERTICAL);
	hbox1->Add(legend_vbox, 1);

	m_pLegendButton = new wxToggleButton(m_pTopPanel, wxID_HELP, _T("Legend"));
	legend_vbox->Add(m_pLegendButton, 0, wxALIGN_RIGHT);
	
	showLog();
	
	panel_vbox->SetSizeHints(m_pTopPanel);
	top_vbox->SetSizeHints(this);
}

RequestLogWindow::~RequestLogWindow()
{
}

void
RequestLogWindow::show()
{
	if (!m_spInstance) {
		m_spInstance = new RequestLogWindow;
		m_spInstance->Show();
	} else {
		m_spInstance->Show();
		m_spInstance->Raise();
	}
}

void
RequestLogWindow::onWindowClose(wxCloseEvent& evt)
{
	if (m_spInstance == this) {
		m_spInstance = 0;
	}
	Destroy();
}

void
RequestLogWindow::onClose(wxCommandEvent& evt)
{
	Close();
}

void
RequestLogWindow::onClear(wxCommandEvent& evt)
{
	wxGetApp().getRequestLogHandler().clearLog();
	if (m_pLogView) {
		m_pLogView->SetFocus();
	}
}

void
RequestLogWindow::onLegend(wxCommandEvent& evt)
{
	if (m_pLegendButton->GetValue()) {
		showLegend();
	} else {
		showLog();
	}
}

void
RequestLogWindow::showLog()
{
	if (m_pLogView) {
		return;
	}
	
	m_pLogSizer->Detach(0);
	delete m_pLegendView;
	m_pLegendView = 0;
	
	m_pLogView = new LogView(
		m_pTopPanel, wxID_ANY,
		wxLC_REPORT|wxLC_VIRTUAL|wxLC_NO_HEADER|wxLC_SINGLE_SEL
	);
	m_pLogSizer->Add(m_pLogView, 1, wxGROW);
	m_pLogView->SetBackgroundColour(*wxWHITE);
	m_pLogView->SetImageList(&m_icons.getImageList(), wxIMAGE_LIST_SMALL);
	m_pLogView->SetFocus();
	m_pLogSizer->Layout();
	
	wxGetApp().getRequestLogHandler().attachView(*m_pLogView);
}

void
RequestLogWindow::showLegend()
{
	if (m_pLegendView) {
		return;
	}

	m_pLogSizer->Detach(0);
	delete m_pLogView;
	m_pLogView = 0;

	m_pLegendView = new wxListCtrl(
		m_pTopPanel, wxID_ANY, wxDefaultPosition, wxDefaultSize,
		wxLC_REPORT|wxLC_NO_HEADER|wxLC_SINGLE_SEL
	);
	m_pLogSizer->Add(m_pLegendView, 1, wxGROW);
	m_pLegendView->SetBackgroundColour(*wxWHITE);
	m_pLegendView->SetImageList(&m_icons.getImageList(), wxIMAGE_LIST_SMALL);
	m_pLogSizer->Layout();

	wxSize size = m_pLegendView->GetClientSize();
	m_pLegendView->InsertColumn(0, wxEmptyString, wxLIST_FORMAT_LEFT, size.x);
	
	typedef RequestLogHandler RLH;
	wxColour clr_normal(RLH::getColorFor(RequestLog::NORMAL_REQUEST));
	wxColour clr_subst(RLH::getColorFor(RequestLog::SUBST_REQUEST));
	wxColour clr_analyze(RLH::getColorFor(RequestLog::ANALYZE_REQUEST));
	wxColour clr_internal(RLH::getColorFor(RequestLog::INTERNAL_REQUEST));

	wxListItem item;

	item.SetText(_T(" Normal request"));
	item.SetId(m_pLegendView->GetItemCount());
	item.SetImage(RequestLogIcons::ICON_NONE);
	item.SetBackgroundColour(clr_normal);
	m_pLegendView->InsertItem(item);

	item.SetText(_T(" Ad substitution request"));
	item.SetId(m_pLegendView->GetItemCount());
	item.SetBackgroundColour(clr_subst);
	m_pLegendView->InsertItem(item);

	item.SetText(_T(" Analyze request (response is to be analyzed)"));
	item.SetId(m_pLegendView->GetItemCount());
	item.SetBackgroundColour(clr_analyze);
	m_pLegendView->InsertItem(item);
	
	item.SetText(_T(" External script fetch during HTML processing"));
	item.SetId(m_pLegendView->GetItemCount());
	item.SetBackgroundColour(clr_internal);
	m_pLegendView->InsertItem(item);
	
	item.SetText(_T(" Normal response"));
	item.SetId(m_pLegendView->GetItemCount());
	item.SetBackgroundColour(clr_normal);
	item.SetImage(RequestLogIcons::ICON_OK);
	m_pLegendView->InsertItem(item);

	item.SetText(_T(" Error response"));
	item.SetId(m_pLegendView->GetItemCount());
	item.SetImage(RequestLogIcons::ICON_ERROR);
	m_pLegendView->InsertItem(item);

	item.SetText(_T(" Request cancelled (client has disconnected)"));
	item.SetId(m_pLegendView->GetItemCount());
	item.SetImage(RequestLogIcons::ICON_CANCEL);
	m_pLegendView->InsertItem(item);

	item.SetText(_T(" Redirect response"));
	item.SetId(m_pLegendView->GetItemCount());
	item.SetImage(RequestLogIcons::ICON_REDIRECT);
	m_pLegendView->InsertItem(item);

	item.SetText(_T(" Ad substitution response"));
	item.SetId(m_pLegendView->GetItemCount());
	item.SetImage(RequestLogIcons::ICON_AD);
	m_pLegendView->InsertItem(item);

	item.SetText(_T(" Cached response"));
	item.SetId(m_pLegendView->GetItemCount());
	item.SetImage(RequestLogIcons::ICON_CACHE);
	m_pLegendView->InsertItem(item);
	
	item.SetText(wxEmptyString);
	item.SetId(m_pLegendView->GetItemCount());
	item.SetImage(RequestLogIcons::ICON_NONE);
	m_pLegendView->InsertItem(item);
	
	/*
	wxFont font = m_pLegendView->GetFont();
	font.SetPointSize(font.GetPointSize()+1);
	item.SetFont(font);
	*/

	item.SetText(_T(" Hint #1:  Double-clicking a URL copies it to clipboard."));
	item.SetId(m_pLegendView->GetItemCount());
	item.SetTextColour(wxColor(0x57, 0x24, 0xff));
	m_pLegendView->InsertItem(item);

	item.SetText(_T(" Hint #2:  Ctrl+C copies the whole list."));
	item.SetId(m_pLegendView->GetItemCount());
	m_pLegendView->InsertItem(item);
}


/*======================= RequestLogWindow::LogView =====================*/

RequestLogWindow::LogView::LogView(wxWindow* parent, wxWindowID id, long style)
:	MvcListView(parent, id, wxDefaultPosition, wxDefaultSize, style),
	m_isAtBottom(true)
{
}

void
RequestLogWindow::LogView::onRowCountChanging()
{
	int delta = GetScrollRange(wxVERTICAL) -
		GetScrollPos(wxVERTICAL) - GetCountPerPage();
	m_isAtBottom = (delta <= 0);
}

void
RequestLogWindow::LogView::onRowCountChanged()
{
	if (!m_isAtBottom) {
		return;
	}
	
	int delta = GetScrollRange(wxVERTICAL) -
		GetScrollPos(wxVERTICAL) - GetCountPerPage();
	if (delta > 0) {
		ScrollLines(delta);
	}
}

void
RequestLogWindow::LogView::onKeyDown(wxKeyEvent& evt)
{
	if (evt.CmdDown() && evt.GetKeyCode() == 'C') {
		wxGetApp().getRequestLogHandler().copyToClipboard();
	} else {
		evt.Skip();
	}
}

void
RequestLogWindow::LogView::onDoubleClick(wxMouseEvent& evt)
{
	int flags = 0;
	long row = HitTest(evt.GetPosition(), flags);
	if (row != wxNOT_FOUND && (flags & wxLIST_HITTEST_ONITEM)) {
		wxGetApp().getRequestLogHandler().copyUrlAtRow(row);
	}
}

} // namespace wxGUI
