/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "HtmlLexer.h"
#include "SplittableBuffer.h"
#include "BString.h"
#include <boost/test/auto_unit_test.hpp>
#include <deque>

class MyHtmlLexer : public HtmlLexer
{
public:
	virtual void processDocType(Iterator const& begin, Iterator const& end) {}
	
	virtual void processText(Iterator const& begin, Iterator const& end);
	
	virtual void processComment(Iterator const& begin, Iterator const& end) {}
	
	virtual void processCDATA(Iterator const& begin, Iterator const& end);
	
	virtual void processOpeningTagName(Iterator const& begin, Iterator const& end);
	
	virtual void processOpeningTag(
		Iterator const& begin, Iterator const& end, bool explicit_empty) {
	}
	
	virtual void processClosingTagName(Iterator const& begin, Iterator const& end) {}
	
	virtual void processClosingTag(
		Iterator const& begin, Iterator const& end,
		bool noscript_follows = false) {
	}
	
	virtual bool processAttrName(Iterator const& begin, Iterator const& end) {
		return false;
	}
	
	virtual void processAttrValue(Iterator const& begin, Iterator const& end) {}
	
	virtual void processAttrNullValue() {}
	
	virtual bool isCDATAStarting() const;
	
	virtual bool isCDATAEnding(Iterator const& begin, Iterator const& end) const;
	
	virtual bool isNoscriptToBeExpected() const { return false; }
	
	SplittableBuffer concatenateTextBlocks() const;
	
	SplittableBuffer concatenateCdataBlocks() const;
	
	BString m_openingTagName;
	std::deque<BString> m_textBlocks;
	std::deque<BString> m_cdataBlocks;
};


BOOST_AUTO_UNIT_TEST(test_broken_cdata_1)
{
	SplittableBuffer input;
	input.append(BString("<script>123</script <other>456</other>"));
	MyHtmlLexer lexer;
	lexer.consume(input, /* eof = */ true);
	
	BOOST_REQUIRE(lexer.m_cdataBlocks.size() == 1);
	BOOST_CHECK(lexer.m_cdataBlocks[0] == BString("123"));
	BOOST_REQUIRE(!lexer.m_textBlocks.empty());
	BOOST_CHECK(*lexer.m_textBlocks.rbegin() == BString("456"));
}

BOOST_AUTO_UNIT_TEST(test_broken_cdata_2)
{
	SplittableBuffer input;
	input.append(BString("<script><!-- 123</script>456"));
	MyHtmlLexer lexer;
	lexer.consume(input, /* eof = */ true);
	
	SplittableBuffer cdata(lexer.concatenateCdataBlocks());
	BOOST_CHECK(cdata.toBString() == BString("<!-- 123"));
	BOOST_REQUIRE(lexer.m_textBlocks.size() == 1);
	BOOST_CHECK(lexer.m_textBlocks[0] == BString("456"));
}


/*============================== MyHtmlLexer ============================*/

void
MyHtmlLexer::processText(Iterator const& begin, Iterator const& end)
{
	if (begin != end) {
		m_textBlocks.push_back(SplittableBuffer::toBString(begin, end));
	}
}

void
MyHtmlLexer::processCDATA(Iterator const& begin, Iterator const& end)
{
	if (begin != end) {
		m_cdataBlocks.push_back(SplittableBuffer::toBString(begin, end));
	}
}

void
MyHtmlLexer::processOpeningTagName(Iterator const& begin, Iterator const& end)
{
	m_openingTagName = SplittableBuffer::toBString(begin, end);
}

bool
MyHtmlLexer::isCDATAStarting() const
{
	return m_openingTagName == BString("script");
}

bool
MyHtmlLexer::isCDATAEnding(Iterator const& begin, Iterator const& end) const
{
	return SplittableBuffer::toBString(begin, end) == BString("script");
}

SplittableBuffer
MyHtmlLexer::concatenateTextBlocks() const
{
	SplittableBuffer buf;
	std::deque<BString>::const_iterator it = m_textBlocks.begin();
	std::deque<BString>::const_iterator end = m_textBlocks.end();
	for (; it != end; ++it) {
		buf.append(*it);
	}
	return buf;
}

SplittableBuffer
MyHtmlLexer::concatenateCdataBlocks() const
{
	SplittableBuffer buf;
	std::deque<BString>::const_iterator it = m_cdataBlocks.begin();
	std::deque<BString>::const_iterator end = m_cdataBlocks.end();
	for (; it != end; ++it) {
		buf.append(*it);
	}
	return buf;
}
