/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "cache/LimitedSpaceManager.h"
#include "IntrusivePtr.h"
#include "AbstractCommand.h"
#include <boost/test/auto_unit_test.hpp>
#include <string>
#include <memory>
#include <stddef.h>

using namespace HttpCache;

typedef LimitedSpaceManager<std::string> Manager;

static bool free_space_is(Manager& manager, size_t space)
{
	std::string const object_key("free_space_check");
	if (manager.newObject(object_key, space + 1).get()) {
		manager.eraseObject(object_key);
		return false;
	}
	if (manager.newObject(object_key, space).get()) {
		manager.eraseObject(object_key);
		return true;
	}
	return false;
}

BOOST_AUTO_UNIT_TEST(test1)
{
	IntrusivePtr<Manager> manager(Manager::create(200, IntrusivePtr<AbstractCommand>()));
	IntrusivePtr<LimitedSpaceObject> instance1(manager->newObject("file1", 100));
	BOOST_REQUIRE(instance1.get());
	IntrusivePtr<LimitedSpaceObject> instance2(manager->openObject("file1"));
	BOOST_REQUIRE(instance2.get());
	BOOST_CHECK(free_space_is(*manager, 100));
}

BOOST_AUTO_UNIT_TEST(test2)
{
	IntrusivePtr<Manager> manager(Manager::create(200, IntrusivePtr<AbstractCommand>()));
	IntrusivePtr<LimitedSpaceObject> instance1(manager->newObject("file1", 100));
	BOOST_REQUIRE(instance1.get());
	IntrusivePtr<LimitedSpaceObject> instance2(manager->openObject("file1"));
	BOOST_REQUIRE(instance2.get());
	
	{
		ResizeRequest resize1to150(instance1, 150); // will be destroyed last
		BOOST_REQUIRE(resize1to150.requestGranted());
		ResizeRequest resize2to200(instance2, 200);
		BOOST_REQUIRE(resize2to200.requestGranted());
		BOOST_REQUIRE(free_space_is(*manager, 0));
		resize1to150.confirm();
		resize2to200.confirm();
	}
	
	BOOST_CHECK(free_space_is(*manager, 50));
}

BOOST_AUTO_UNIT_TEST(test3)
{
	IntrusivePtr<Manager> manager(Manager::create(200, IntrusivePtr<AbstractCommand>()));
	IntrusivePtr<LimitedSpaceObject> instance1(manager->newObject("file1", 100));
	BOOST_REQUIRE(instance1.get());
	IntrusivePtr<LimitedSpaceObject> instance2(manager->openObject("file1"));
	BOOST_REQUIRE(instance2.get());
	
	{
		ResizeRequest resize1to150(instance1, 150); // will be destroyed last
		BOOST_REQUIRE(resize1to150.requestGranted());
		ResizeRequest resize2to200(instance2, 200);
		BOOST_REQUIRE(resize2to200.requestGranted());
		BOOST_REQUIRE(free_space_is(*manager, 0));
		resize2to200.confirm();
	}
	
	BOOST_CHECK(free_space_is(*manager, 0));
}

BOOST_AUTO_UNIT_TEST(test4)
{
	IntrusivePtr<Manager> manager(Manager::create(200, IntrusivePtr<AbstractCommand>()));
	IntrusivePtr<LimitedSpaceObject> instance1(manager->newObject("file1", 100));
	BOOST_REQUIRE(instance1.get());
	IntrusivePtr<LimitedSpaceObject> instance2(manager->openObject("file1"));
	BOOST_REQUIRE(instance2.get());
	
	{
		GrowRequest grow1to150(instance1, 150); // will be destroyed last
		BOOST_REQUIRE(grow1to150.requestGranted());
		ResizeRequest resize2to80(instance2, 80);
		BOOST_REQUIRE(resize2to80.requestGranted());
		BOOST_REQUIRE(free_space_is(*manager, 50));
		grow1to150.setNewSize(140);
		resize2to80.confirm();
	}
	
	BOOST_CHECK(free_space_is(*manager, 60));
}

BOOST_AUTO_UNIT_TEST(test5)
{
	IntrusivePtr<Manager> manager(Manager::create(200, IntrusivePtr<AbstractCommand>()));
	
	{
		IntrusivePtr<LimitedSpaceObject> instance1(manager->newObject("file1", 100));
		BOOST_REQUIRE(instance1.get());
	}
	
	manager->eraseObject("file1");
	BOOST_CHECK(free_space_is(*manager, 200));
}

BOOST_AUTO_UNIT_TEST(test6)
{
	IntrusivePtr<Manager> manager(Manager::create(200, IntrusivePtr<AbstractCommand>()));
	IntrusivePtr<LimitedSpaceObject> instance1(manager->newObject("file1", 100));
	BOOST_REQUIRE(instance1.get());
	
	manager->eraseObject("file1");
	BOOST_CHECK(free_space_is(*manager, 100));
	instance1.reset(0);
	BOOST_CHECK(free_space_is(*manager, 200));
}
