/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "TempDir.h"
#include <ace/config-lite.h>
#include <ace/Dirent.h>
#include <ace/Lib_Find.h> // for ACE::get_temp_dir()
#include <ace/OS_NS_sys_stat.h> // for ACE_OS::mkdir(), ACE_OS::stat()
#include <ace/OS_NS_stdio.h> // for ACE_OS::snprintf()
#include <ace/OS_NS_unistd.h> // for ACE_OS::unlink(), ACE_OS::rmdir()
#include <ace/os_include/os_limits.h> // for PATH_MAX
#include <stdexcept>
#include <string.h>
#include <stddef.h>
#include <stdlib.h>
#include <assert.h>

TempDir::TempDir()
{
	// Our directory will have the form of "${TEMP}/bfilterXXXXXX",
	// where X is a digit.
	
	char path[PATH_MAX];
	if (ACE::get_temp_dir(path, sizeof(path)) == -1) {
		throw std::runtime_error("could not get the temp directory");
	}
	size_t const base_len = strlen(path);

	// get_temp_dir() returns a trailing slash.
	assert(path[base_len - 1] == '/' || path[base_len - 1] == '\\');
	
	char const component1[] = "bfilter";
	size_t const component1_len = sizeof(component1) - 1;
	size_t const component2_len = 6;
	size_t const total_len = base_len + component1_len + component2_len + 1;
	if (total_len > sizeof(path)) {
		throw std::runtime_error("path length exceedes PATH_MAX");
	}
	
	memcpy(&path[base_len], component1, component1_len);
	path[total_len - 1] = '\0';
	
	for (;;) {
		fillRandomDigits(&path[base_len + component1_len], component2_len);
		if (ACE_OS::mkdir(path, 0700) != -1) {
			break;
		} else if (errno != EEXIST) {
			throw std::runtime_error("mkdir() failed for unknown reason");
		}
	}
	
	m_tempDir.assign(path);
}

TempDir::~TempDir()
{
	rmdirRecursive(m_tempDir);
}

void
TempDir::fillRandomDigits(char* const str, size_t const len)
{
		static char const digits[] = "0123456789";
		for (size_t i = 0; i < len; ++i) {
			str[i] = digits[rand() % 10];
		}
}

void
TempDir::rmdirRecursive(std::string const& dir)
{	
	ACE_Dirent dh;
	if (dh.open(dir.c_str()) != -1) {
		for (dirent* ent; (ent = dh.read()); ) {
			if (!strcmp(ent->d_name, ".") || !strcmp(ent->d_name, "..")) {
				continue;
			}
			
			std::string new_path(dir);
			new_path += '/';
			new_path += ent->d_name;
			
#ifdef DT_DIR
			if (ent->d_type == DT_DIR) {
				rmdirRecursive(new_path);
			} else {
				ACE_OS::unlink(new_path.c_str());
			}
#else
			ACE_stat st;
			if (ACE_OS::stat(new_path.c_str(), &st) == 0) {
				if (st.st_mode & S_IFDIR) {
					rmdirRecursive(new_path);
				} else {
					ACE_OS::unlink(new_path.c_str());
				}
			}
#endif
		}
	}
	ACE_OS::rmdir(dir.c_str());
}
