/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "bible.h"
#include "utilities.h"
#include "scripture.h"
#include <fnmatch.h>


char *USFM_IDs[NUMBER_OF_BIBLEBOOKS] =
  { "GEN", "EXO", "LEV", "NUM", "DEU", "JOS", "JDG", "RUT", "1SA", "2SA",
    "1KI", "2KI", "1CH", "2CH", "EZR", "NEH", "EST", "JOB", "PSA", "PRO",
    "ECC", "SNG", "ISA", "JER", "LAM", "EZK", "DAN", "HOS", "JOL", "AMO", "OBA",
    "JON", "MIC", "NAM", "HAB", "ZEP", "HAG", "ZEC", "MAL",
    "MAT", "MRK", "LUK", "JHN", "ACT", "ROM", "1CO", "2CO", "GAL", "EPH", "PHP",
    "COL", "1TH", "2TH", "1TI", "2TI", "TIT", "PHM", "HEB", "JAS",
    "1PE", "2PE", "1JN", "2JN", "3JN", "JUD", "REV"
};
char *BibleWorksIDs[NUMBER_OF_BIBLEBOOKS] =
  { "Gen", "Exo", "Lev", "Num", "Deu", "Jos", "Jdg", "Rut", "1Sa", "2Sa",
    "1Ki", "2Ki", "1Ch", "2Ch", "Ezr", "Neh", "Est", "Job", "Psa", "Pro",
    "Ecc", "Sol", "Isa", "Jer", "Lam", "Eze", "Dan", "Hos", "Joe", "Amo", "Oba",
    "Jon", "Mic", "Nah", "Hab", "Zep", "Hag", "Zec", "Mal",
    "Mat", "Mar", "Luk", "Joh", "Act", "Rom", "1Co", "2Co", "Gal", "Eph", "Phi",
    "Col", "1Th", "2Th", "1Ti", "2Ti", "Tit", "Phm", "Heb", "Jam",
    "1Pe", "2Pe", "1Jo", "2Jo", "3Jo", "Jud", "Rev"
};
char *OSIS_IDs[NUMBER_OF_BIBLEBOOKS] =
  { "Gen", "Exod", "Lev", "Num", "Deut", "Josh", "Judg", "Ruth", "1Sam",
    "2Sam", "1Kgs", "2Kgs", "1Chr", "2Chr", "Ezra", "Neh", "Esth", "Job",
    "Ps", "Prov", "Eccl", "Song", "Isa", "Jer", "Lam", "Ezek", "Dan", "Hos",
    "Joel", "Amos", "Obad", "Jonah", "Mic", "Nah", "Hab", "Zeph",
    "Hag", "Zech", "Mal",
    "Matt", "Mark", "Luke", "John", "Acts", "Rom", "1Cor", "2Cor", "Gal", "Eph",
    "Phil", "Col", "1Thess", "2Thess", "1Tim", "2Tim", "Titus",
    "Phlm", "Heb", "Jas", "1Pet", "2Pet", "1John", "2John", "3John", "Jude",
    "Rev"
};
char *EnglishBookNames[NUMBER_OF_BIBLEBOOKS] =
  { "Genesis", "Exodus", "Leviticus", "Numbers", "Deuteronomy", "Joshua",
    "Judges", "Ruth", "1 Samuel", "2 Samuel", "1 Kings", "2 Kings",
    "1 Chronicles", "2 Chronicles", "Ezra", "Nehemiah", "Esther", "Job",
    "Psalms", "Proverbs", "Ecclesiastes", "Song of Solomon", "Isaiah",
    "Jeremiah", "Lamentations", "Ezekiel", "Daniel", "Hosea", "Joel", "Amos",
    "Obadiah", "Jonah", "Micah", "Nahum", "Habakkuk", "Zephaniah",
    "Haggai", "Zechariah", "Malachi",
    "Matthew", "Mark", "Luke", "John", "Acts", "Romans", "1 Corinthians",
    "2 Corinthians", "Galatians", "Ephesians", "Philippians", "Colossians",
    "1 Thessalonians", "2 Thessalonians", "1 Timothy", "2 Timothy", "Titus",
    "Philemon", "Hebrews", "James", "1 Peter", "2 Peter", "1 John",
    "2 John", "3 John", "Jude", "Revelation"
};


ustring id_to_biblebook_english (const ustring & id)
{
  // Takes id and returns an English bookname.
  // E.g. GEN returns "Genesis".
  for (unsigned int i = 0; i < NUMBER_OF_BIBLEBOOKS; i++)
    {
      // Note: Allow for mixed case IDs for extra robustness.
      if (upperCase (id) == USFM_IDs[i])
        return EnglishBookNames[i];
    }
  return "Unknown";
}


ustring index_to_paratext_id (unsigned int index)
{
  return USFM_IDs[index];
}


unsigned paratext_id_to_index (const ustring & id)
{
  // Returns the index of the Paratext ID. So e.g. EXO returns 1, and GEN 0
  for (unsigned int i = 0; i < NUMBER_OF_BIBLEBOOKS; i++)
    {
      // Note: Allow for mixed case IDs for extra robustness.
      if (upperCase (id) == USFM_IDs[i])
        return i;
    }
  return string::npos;
}


unsigned bibleworks_id_to_index (const ustring & id)
{
  // Returns the index of the BibleWorks ID. So e.g. Exo returns 1, and Gen 0
  ustring id2 = upperCase (id);
  for (unsigned int i = 0; i < NUMBER_OF_BIBLEBOOKS; i++) {
    if (id2 == upperCase (BibleWorksIDs[i]))
      return i;
  }
  return string::npos;
}


ustring index_to_bibleworks_id (unsigned int index)
{
  return BibleWorksIDs[index];
}


unsigned osis_id_to_index (const ustring & id)
{
  // Returns the index of the OSIS ID. So e.g. Exod returns 1, and Gen 0
  ustring id2 = upperCase (id);
  for (unsigned int i = 0; i < NUMBER_OF_BIBLEBOOKS; i++)
  {
    if (id2 == upperCase (OSIS_IDs[i]))
      return i;
  }
  return string::npos;
}


ustring index_to_osis_id (unsigned int index)
// Returns the OSIS id.
{
  return OSIS_IDs [index];
}


unsigned  english_id_to_index (const ustring & id)
{
  // Returns the index of the English name, e.g. "1Corinthians" or "1 Corinthians"
  ustring id2 = upperCase (id);
  for (unsigned int i = 0; i < NUMBER_OF_BIBLEBOOKS; i++) {
    if (id2 == upperCase (EnglishBookNames[i]))
      return i;
  }
  return string::npos;
}


void sort_references (vector <ustring>& references)
/*
Sorts all references from Genesis to Revelation. 
Sorts on book first, then chapter, and finally verse.
*/
{
  // When there are less than two entries no sorting is needed.
  if (references.size () < 2) return;
  try 
  {
    // Make a vector that contains the numerical equivalent of the references.
    vector <unsigned int> numerical;
    for (unsigned int i = 0; i < references.size (); i++) {
      ustring book, chapter, verse;
      decode_reference (references[i], book, chapter, verse);
      numerical.push_back (reference_to_numerical_equivalent (book, chapter, verse));
    }
    // Sort the references.
    quick_sort (numerical, references, 0, numerical.size());
  }
  catch (exception & ex)
  {
    cerr << "Sorting references: " << ex.what () << endl;
  }
}


ustring index_to_biblebook (unsigned int index)
{
  return EnglishBookNames[index];
}


void decode_reference (const ustring & reference, ustring & book, ustring & chapter, ustring & verse)
/*
 * Decodes "reference" and provides:
 * - book
 * - chapter
 * - verse
 * 
 * E.g. "Song of Solomon 1:1" becomes "Song of Solomon", 
 * chapter "1" and verse "1".
 */
{
  try
  {
    ustring ref (reference);
    // Extract the book.
    // Deal with books like "1 Samuel" or "Song of Solomon".
    int booklength;
    booklength = ref.rfind (" ");
    book = ref.substr (0, booklength);
    ref.erase (0, booklength);
    ref = trim (ref);
    // Extract chapter.
    chapter = number_in_string (ref);
    ref.erase (0, chapter.length () + 1);
    ref = trim (ref);
    // Extract verse.
    verse = ref;
  }
  catch (exception & ex)
  {
    cerr << ex.what () << endl;
  }
}


bool reference_discover_internal (const ustring& oldbook, const ustring& oldchapter, 
                         const ustring& oldverse,  const ustring& reference,
                         ustring& newbook, ustring& newchapter, ustring& newverse)
{
  /*
    This interprets "reference" as a valid reference.
    If needed it will use information of the current reference to complete the info.
    For example, when "reference" is "1", it is interpreted as verse one of the current
    chapter of the current book. If "reference" is "21 10" it is interpreted as the current
    book, chapter 21 verse 10. And so forth.
  */
  // Filter the response.
  string response = trim (reference);
  // Change a colon to a space.
  unsigned int position = response.find (":");
  while (position != string::npos) {
    response[position] = ' ';
    position = response.find (":");
  }
  // Change a dot to a space.
  // Dots are in OSIS references.
  position = response.find (".");
  while (position != string::npos) {
    response[position] = ' ';
    position = response.find (".");
  }
  // Change to upper case.
  response = upperCase (response);
  // Trim again.
  response = trim (response);
  // Divide the response in parts.
  istringstream r (response);
  ustring s;
  vector <ustring> input;
  for (unsigned int i = 0; i < 5; i++) {
    s.clear ();
    r >> s;
    if (!s.empty ())
      input.push_back (s);
  }
  // Deal with cases when the user enters e.g. "1 Corinthians 10:1".
  if (input.size () >= 2) {
    // This is the case when the first one is a number ...
    if (number_in_string (input[0]) == input[0]) {
      // ... and the second one contains no numbers at all.
      if (number_in_string (input[1]).empty ()) {
        input[1] = input[0] + " " + input[1];
        input.erase (input.begin ());
      }
    }
  }
  // Deal with Song of Solomon
  if (input.size() >= 3) {
    if (input[0] == "SONG") {
      if (input [1] == "OF") {
        input [2] = input[0] + " " + input[1] + " " + input[2];
        input.erase (input.begin ());
        input.erase (input.begin ());
      }
    }
  }
  // See whether the first one is a name or a number.
  // If it's a name, then this will be the bookname.
  // If it's a number, there will be no bookname, but we have chapter and/or verse only.
  // If it contains a hyphen (-) or a comma (,) it is a verse number.
  bool book_was_given = false;
  if (input.size () > 0) {
    if (looks_like_verse (input[0])) {
      // It's a verse, so no bookname given. Use the current book.
      newbook = oldbook;
    }
    else {
      // It's a bookname, find out which.
      book_was_given = true;
      newbook = book_find_valid (input[0]);
      if (newbook.empty ()) {
        // If something like "1 co" has been added, deal with that by making it "1co", and checking again.
        ustring s = input[0];
        if (s.length () > 2) {
          if (s[1] == ' ') {
            s.erase (1, 1);
            newbook = book_find_valid (s);
          }
        }
        if (newbook.empty ()) {
          // Bad book: bail out.
          return false;
        }
      }
      // Remove the book, because it's no longer needed.
      input.erase (input.begin ());
    }
  }
  else {
    return false;
  }
  // As from here on we can be sure of this one thing: We've got a proper bookname.
  // Also as the bookname has been removed from the input data, only chapter and/or verse remain to be dealt with.
  if (input.size () >= 2)
  {
    // Two variabeles left, so that will be chapter and verse.
    newchapter = number_in_string (input[0]);
    newverse = lowerCase (input[1]);
    if (newchapter.empty () || newverse.empty ())
    {
      return false;
    }
    return true;
  }
  // Here we have only one variabele left.
  // This is either chapter (as in "Genesis 1") or verse (as in "1").
  // Or we have no variabele left, then we take chapter one verse one.
  if (book_was_given)
  {
    // Here deal with the fact that a book was given, but nothing else.
    if (input.size () == 0)
    {
      newchapter = "1";
      newverse = "1";
      return true;
    }
  }
  // One variabele left, so we've either book or verse.
  if (book_was_given)
  {
    // Book was given, so it is a chapter.
    newchapter = number_in_string (input[0]);
    newverse = "1";
    if (newchapter.empty ())
    {
      return false;
    }
    return true;
  }
  else
  {
    // It is a verse.
    newverse = lowerCase (input[0]);
    newchapter = oldchapter;
    if (newverse.empty ())
    {
      return false;
    }
    return true;
  }
}


bool reference_discover (const ustring& oldbook, const ustring& oldchapter, 
                         const ustring& oldverse,  const ustring& reference,
                         ustring& newbook, ustring& newchapter, ustring& newverse)
{
/* 
This is the new function "reference_discover". It uses the previous one which
has now been renamed "reference_discover_internal".
This new function iterates even more over a references, and is able to cut
off bits at the beginning that would not be a references. This occurs when 
loading a file with references saved by BibleWorks. It has a format as 
shown here:
 
BWRL 1

KJV 2Ki 25:18
KJV 1Ch 6:36

In this example the "KJV" needs to be taken out and then the reference will 
appear cleanly.
*/
  bool result;
  result = reference_discover_internal (oldbook, oldchapter, oldverse, reference, newbook, newchapter, newverse);
  if (!result) {
    if (reference.length() >= 11) {
      ustring adaptedreference (reference);
      adaptedreference.erase (0, 4);
      result = reference_discover_internal (oldbook, oldchapter, oldverse, adaptedreference, newbook, newchapter, newverse);
    }
  }
  return result;
}


ustring book_find_valid (const ustring & rawbook)
{
  /*
     This sees if variabele "rawbook" can be interpreted as a book in any way.
   */
  unsigned int index;
  // Check on names entered like Genesis or 1 Corinthians, the full English name
  // A bug was discovered so that "Judges" was interpreted as "Jude", because
  // of the three letters "JUD". Solved by checking on full English name first.
  index = english_id_to_index (rawbook);
  if (index != string::npos) {
    return index_to_biblebook (index);
  }
  // Recognise the abbreviations used by Paratext.
  index = paratext_id_to_index (rawbook);
  if (index != string::npos) {
    return index_to_biblebook (paratext_id_to_index (rawbook));
  }
  // Try the abbreviations defined by the OSIS project.
  index = osis_id_to_index (rawbook);
  if (index != string::npos)
  {
    return index_to_biblebook (index);
  }
  // Try the abbreviations of BibleWorks.
  index = bibleworks_id_to_index (rawbook);
  if (index != string::npos)
  {
    return index_to_biblebook (index);
  }
  // Abbreviations in Paratext: See if shortening the bookname helps.
  if (rawbook.length () >= 3)
    {
      ustring s = rawbook.substr (0, 3);
      index = paratext_id_to_index (s);
      if (index != string::npos)
        {
          return index_to_biblebook (index);
        }
    }
  // BibleWorks: See if shortening the bookname helps.
  if (rawbook.length () >= 3)
    {
      ustring s = rawbook.substr (0, 3);
      index = bibleworks_id_to_index (s);
      if (index != string::npos)
        {
          return index_to_biblebook (index);
        }
    }
  // OSIS. See if shortening the bookname helps.
  // The shortest abbreviation is 2 characters long,
  // and the longest 6. So we've to try them all.
  // Length: 2
  if (rawbook.length () >= 2)
    {
      ustring s = rawbook.substr (0, 2);
      index = osis_id_to_index (s);
      if (index != string::npos)
        {
          return index_to_biblebook (index);
        }
    }
  // Length: 3
  if (rawbook.length () >= 3)
    {
      ustring s = rawbook.substr (0, 3);
      index = osis_id_to_index (s);
      if (index != string::npos)
        {
          return index_to_biblebook (index);
        }
    }
  // Length: 4
  if (rawbook.length () >= 4)
    {
      ustring s = rawbook.substr (0, 4);
      index = osis_id_to_index (s);
      if (index != string::npos)
        {
          return index_to_biblebook (index);
        }
    }
  // Length: 5
  if (rawbook.length () >= 5) {
    ustring s = rawbook.substr (0, 5);
    index = osis_id_to_index (s);
    if (index != string::npos) {
      return index_to_biblebook (index);
    }
  }
  // Length: 6
  if (rawbook.length () >= 6) {
    ustring s = rawbook.substr (0, 6);
    index = osis_id_to_index (s);
    if (index != string::npos) {
      return index_to_biblebook (index);
    }
  }
  // If the book has not yet been found, then it's getting tough.
  // Not found yet, check on names like "1Corinthians".
  if (rawbook.length () >= 1) {
    ustring s = rawbook.substr (0, 1);
    ustring s2 = rawbook;
    if (s == "1" || s == "2" || s == "3")
      s2.insert (1, " ");
    index = english_id_to_index (s2);
    if (index != string::npos) {
      return index_to_biblebook (index);
    }
  }
  return "";
}


unsigned int reference_to_numerical_equivalent (const ustring& book, const ustring& chapter, const ustring& verse)
/*
Produces the numerical equivalent of a reference.
Supports half verses, like 10a, and 11b.
Genesis 1:1 becomes 1001002
Genesis 1:2 becomes 1001004
Exodus  2:1 becomes 2001002
Etc.
*/
{
  unsigned int i;
  i = english_id_to_index (book) * 1000000;
  i = i + (convert_to_int (chapter) * 1000);
  vector<int> verses = verses_encode (verse);
  i = i + verses[0];
  return i;
}


ustring book_chapter_verse_to_reference (int book, int chapter, const ustring& verse)
/*
Changes a book number, with a chapter number, and a verse number, 
to a full references, e.g. "Genesis 1:1a-4".
*/
{
  ustring reference;
  reference = index_to_biblebook (book);
  reference.append (" ");
  reference.append (convert_to_string(chapter));
  reference.append (":");
  reference.append (verse);
  return reference;
}


ustring book_chapter_verse_to_reference_paratext (int book, int chapter, int verse)
/*
Changes a book number, with a chapter number, and a verse number, 
to a references as used in Paratext, e.g. "GEN 1:1".
*/
{
  ustring reference;
  reference = index_to_paratext_id (book);
  reference.append (" ");
  reference.append (convert_to_string(chapter));
  reference.append (":");
  reference.append (convert_to_string(verse));
  return reference;
}


ustring book_selection_information (Session * session, const ustring& project)
/*
Returns text to be displayed typically beside the "select books" button.
The text contains a bit of information about the current selection.
It returns a maximum of three units, e.g.:
No books (1 unit)
John, Hebrew (2 units)
Genesis, Exodus, Leviticus (3 units)
Old Testament, Mark (2 units)
etc.
*/
{
  // Storage for the units.
  vector<ustring> units;

  // Message to return.
  ustring message;

  // Deal with empty selection.
  if (session->selected_books.empty()) {
    message = "No books";
    return message;
  }
  
  // See whether OT / NT is in the selection.
  set<ustring> selection (session->selected_books.begin(), session->selected_books.end());

  /*
  At first it only gave "Old Testament" and/or "New Testament" when all 39
  and/or 27 books were there, using the following code:
  
  // Is the whole OT in the selection?
  bool ot_found = false;
  for (unsigned int i = 0; i < 39; i++) {
    ustring book = index_to_biblebook (i);
    ot_found = (selection.find (book) != selection.end());
    if (!ot_found)
      break;
  }

  // If we found the whole OT, deal with that.
  if (ot_found) {
    units.push_back ("Old Testament");
    for (unsigned int i = 0; i < 39; i++) {
      ustring book = index_to_biblebook (i);
      set<ustring>::iterator iter;
      iter = selection.find(book);
      selection.erase (iter);
    }
  }

  // Is the whole NT in the selection?
  bool nt_found = false;
  for (unsigned int i = 39; i < 66; i++) {
    ustring book = index_to_biblebook (i);
    nt_found = (selection.find (book) != selection.end());
    if (!nt_found)
      break;
  }

  // If we found the whole NT, deal with that.
  if (nt_found) {
    units.push_back ("New Testament");
    for (unsigned int i = 39; i < 66; i++) {
      ustring book = index_to_biblebook (i);
      set<ustring>::iterator iter;
      iter = selection.find(book);
      selection.erase (iter);
    }
  }
  
  But it was preferrable that e.g. "Old Testament" would be given, not only when
  the whole OT was there, but also when all OT books that are currently in the 
  project were there. E.g. if the project contains Genesis and Exodus, and both
  are selected, it should also give "Old Testament". See the following code.
  */
  vector<ustring> selectable_books;
  Scripture scripture (project);
  for (unsigned int i = 0; i < scripture.books.size (); i++) {
    selectable_books.push_back (scripture.books[i]);
  }

  // Is the OT in the selection? And the NT?
  unsigned int selectable_ot_books = 0;
  unsigned int selectable_nt_books = 0;
  for (unsigned int i = 0; i < selectable_books.size(); i++) {
    unsigned index = english_id_to_index (selectable_books[i]);
    if ((index >= 0) && (index < 39))
      selectable_ot_books++;
    if ((index >= 39) && (index < 66))
      selectable_nt_books++;
  }
  unsigned int selected_ot_books = 0;
  for (unsigned int i = 0; i < 39; i++) {
    ustring book = index_to_biblebook (i);
    if (selection.find (book) != selection.end())
      selected_ot_books++;
  }
  if ((selected_ot_books > 1) && (selectable_ot_books == selected_ot_books)) {
    units.push_back ("Old Testament");
    for (unsigned int i = 0; i < 39; i++) {
      ustring book = index_to_biblebook (i);
      set<ustring>::iterator iter;
      iter = selection.find(book);
      if (iter != selection.end())
        selection.erase (iter);
    }
  }
  unsigned int selected_nt_books = 0;
  for (unsigned int i = 39; i < 66; i++) {
    ustring book = index_to_biblebook (i);
    if (selection.find (book) != selection.end())
      selected_nt_books++;
  }
  if ((selected_nt_books > 1) && (selectable_nt_books == selected_nt_books)) {
    units.push_back ("New Testament");
    for (unsigned int i = 39; i < 66; i++) {
      ustring book = index_to_biblebook (i);
      set<ustring>::iterator iter;
      iter = selection.find(book);
      if (iter != selection.end())
        selection.erase (iter);
    }
  }

  // Deal with any remaining books.  
  if ((selection.size() + units.size()) > 3) {
    units.push_back (convert_to_string (int (selection.size())) + " books");
  }
  else {
    for (unsigned int i = 0; i < NUMBER_OF_BIBLEBOOKS; i++) {
      ustring book = index_to_biblebook (i);
      if (selection.find (book) != selection.end()) {
        units.push_back (book);
      }
    }
  }
  
  // Assemble the message.
  for (unsigned int i = 0; i < units.size(); i++) {
    if (!message.empty())
      message.append (" + ");
    message.append (units[i]);      
  }

  return message;
}


bool looks_like_verse (const ustring& text)
// This checks the text given and sees whether it looks like a verse. If so
// it returns true.
{
  // If it is a number only, it looks like a verse.
  if (number_in_string (text) == text)
    return true;
  // If it contains a hyphen (-) or a comma (,) it is a verse number.
  if (text.find_first_of (",-") != string::npos)
    return true;
  // If it contains a digit followed by either an "a" or a "b", it's a verse.
  // Note everything is capitalized, so we check for "A" or "B".
  ustring pattern = "*[0-9][A,B]*";
  if (fnmatch (pattern.c_str(), text.c_str(), 0) == 0)
    return true;
  return false;
}


void verses_encode_internal (const ustring& verse, vector<int>& expanded_verses)
{
  int expanded_verse;
  expanded_verse = 2 * (convert_to_int (verse));
  if (verse.find_first_of ("aA") != string::npos) {
    expanded_verses.push_back (expanded_verse);
  } else if (verse.find_first_of ("bB") != string::npos) {
    expanded_verses.push_back (++expanded_verse);
  } else {
    expanded_verses.push_back (expanded_verse);
    expanded_verses.push_back (++expanded_verse);
  }
}


vector<int> verses_encode (const ustring& verse)
/*
This encodes a verse into a number of integers. As we may have ranges of verses,
like 1b-5, or 1b,2, we handle these ranges or sequences by converting them to
a series of integers values, each integer value representing half of a verse.
So verse 0 becomes then "0, 1", and verse 1 will be "2, 3". Verse 1a will be 
"2".
*/
{
  // Storage.
  vector<int> expanded_verses;
  // Work on a copy of the verse;
  ustring vs = verse;
  // If there is a range, take the beginning and the end and fill up in between.
  if (vs.find ("-") != string::npos) {
    size_t position;
    position = vs.find ("-");
    ustring start_range, end_range;
    start_range = vs.substr (0, position);
    vs.erase (0, ++position);
    end_range = vs;
    int start_expanded_verse = 2 * convert_to_int (number_in_string (start_range));
    if (start_range.find_first_of ("bB") != string::npos)
      start_expanded_verse++;
    int end_expanded_verse = 2 * convert_to_int (number_in_string (end_range));
    if (end_range.find_first_of ("aA") == string::npos)
      end_expanded_verse++;
    // Sometimes people give start higher than the end, so swap them here.
    {
      int min = MIN (start_expanded_verse, end_expanded_verse);
      int max = MAX (start_expanded_verse, end_expanded_verse);
      start_expanded_verse = min;
      end_expanded_verse = max;
    }
    for (int i2 = start_expanded_verse; i2 <= end_expanded_verse; i2++) {
      expanded_verses.push_back (i2);
    }
  } 
  // If there is a sequence, take each verse in the sequence, and store it.
  else if (vs.find (",") != string::npos) {
    int iterations = 0;
    do {
      // In case of an unusual range formation, do not hang, but give message.
      iterations++;
      if (iterations > 50) {
        break;
      }
      size_t position = vs.find (",");
      ustring localverse;
      if (position == string::npos) {
        localverse = vs;
        vs.clear();
      } else {
        localverse = vs.substr (0, position);
        vs.erase (0, ++position);
      }
      verses_encode_internal (localverse, expanded_verses);
    } while (!vs.empty());
  }
  // No range and no sequence: a "normal" verse.
  else {
    verses_encode_internal (vs, expanded_verses);
  }
  // Return result.
  return expanded_verses;
}
