/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include "categorize_chapter.h"
#include "constants.h"
#include "gwrappers.h"
#include "shell.h"


CategorizeChapter::CategorizeChapter (vector<ustring>& lines)
{
  // Filenames for input and categorized output.
  // We use Scripture Checks to produce this data from the lines we have.
  ustring inputfile = temporary_file ("export_inputfile");
  unlink (inputfile.c_str());
  ustring categorized_outputfile = temporary_file ("export_categorized_outputfile");
  unlink (categorized_outputfile.c_str());
  // Save the lines and produce the output: categorized data.
  write_lines (inputfile, lines);
  ustring scripturechecks = "sc-input-usfm ";
  ustring command (scripturechecks);
  command.append (shell_quote_space (inputfile) + "--categorize >");
  command.append (shell_quote_space (categorized_outputfile));
  system (command.c_str());
  parse (categorized_outputfile);
}


CategorizeChapter::~CategorizeChapter ()
{
}


void CategorizeChapter::parse (const ustring& filename)
{
  // Some variables for parsing.
  bool was_error = false;
  gchar *contents;
  gsize  length;
  GError *gerror;
  gerror = NULL;
  // Read the xml file.
  if (!g_file_get_contents (filename.c_str(), &contents, &length, &gerror)) {
    gw_critical (gerror->message);
    g_error_free (gerror);
    was_error = true;
  }
  // If length is (about) zero, don't parse it, because it does not have data,
  // and parsing an empty file gives a segmentation error.
  if (length < 10)
    was_error = true;
  // Set up parser.
  GMarkupParseContext *context;
  GMarkupParser parser = {
    start_element_handler,
    end_element_handler,
    text_handler,
    passthrough_handler,
    error_handler
  };
  // Parse xml file.
  if (!was_error) {
    context = g_markup_parse_context_new (&parser, GMarkupParseFlags (0), gpointer (this), NULL);
    if (!g_markup_parse_context_parse (context, contents, length, NULL)) {
      g_markup_parse_context_free (context);
      was_error = true;
      gw_critical (gerror->message);
    }
  }
  if (!was_error) {
    if (!g_markup_parse_context_end_parse (context, NULL)) {
      g_markup_parse_context_free (context);
      was_error = true;
      gw_critical (gerror->message);
    }
  }
  if (!was_error)
    g_markup_parse_context_free (context);
}


void CategorizeChapter::start_element_handler (GMarkupParseContext *context,
                                   const gchar         *element_name,
                                   const gchar        **attribute_names,
                                   const gchar        **attribute_values,
                                   gpointer             user_data,
                                   GError             **error)
{
  ((CategorizeChapter *) user_data)->start_element_handler (element_name, attribute_values);
}


void CategorizeChapter::end_element_handler (GMarkupParseContext *context,
                                 const gchar         *element_name,
                                 gpointer             user_data,
                                 GError             **error)
{
  ((CategorizeChapter *) user_data)->end_element_handler (element_name);
}


void CategorizeChapter::text_handler (GMarkupParseContext *context,
                          const gchar         *text,
                          gsize                text_len,
                          gpointer             user_data,
                          GError             **error)
{
  ((CategorizeChapter *) user_data)->text_handler (text);
}


void CategorizeChapter::passthrough_handler (GMarkupParseContext *context,
                                    const gchar         *passthrough_text,
                                    gsize                text_len,
                                    gpointer             user_data,
                                    GError             **error)
{
}


void CategorizeChapter::error_handler (GMarkupParseContext *context,
                                  GError              *error,
                                  gpointer             user_data)
{
  gw_critical (error->message);
}


void CategorizeChapter::start_element_handler (const gchar  *element_name,
                                          const gchar **attribute_values)
{
  currentelement = element_name;
  if (currentelement == "verse") {
    verse = attribute_values[0];
  }
}


void CategorizeChapter::end_element_handler (const gchar *element_name)
{
  ustring element (element_name);
  if (element == "verse") {
    // Store data.
    verses.push_back (verse);
    id_texts.push_back (id_text);
    intro_texts.push_back (intro_text);
    head_texts.push_back (head_text);
    chap_texts.push_back (chap_text);
    study_texts.push_back (study_text);
    note_texts.push_back (note_text);
    ref_texts.push_back (ref_text);
    verse_texts.push_back (verse_text);
    // Clear text.    
    id_text.clear();
    intro_text.clear();
    head_text.clear();
    chap_text.clear();
    study_text.clear();
    note_text.clear();
    ref_text.clear();
    verse_text.clear();
  }
}


void CategorizeChapter::text_handler (const gchar *text)
{
  ustring utext = trim (text);
  if (utext.empty())
    return;
  if       (currentelement == IDENTIFIER_TEXT_TAG) {
    if (!id_text.empty())
      id_text.append (" ");
    id_text.append (utext);
  } else if (currentelement == INTRODUCTION_TEXT_TAG) {
    if (!intro_text.empty())
      intro_text.append (" ");
    intro_text.append (utext);
  } else if (currentelement == HEADING_TEXT_TAG) {
    if (!head_text.empty())
      head_text.append (" ");
    head_text.append (utext);
  } else if (currentelement == CHAPTER_TEXT_TAG) {
    if (!chap_text.empty())
      chap_text.append (" ");
    chap_text.append (utext);
   } else if (currentelement == STUDY_NOTE_TEXT_TAG) {
    if (!study_text.empty())
      study_text.append (" ");
    study_text.append (utext);
  } else if (currentelement == NOTE_TEXT_TAG) {
    if (!note_text.empty())
      note_text.append (" ");
    note_text.append (utext);
  } else if (currentelement == CROSSREFERENCE_TEXT_TAG) {
    if (!ref_text.empty())
      ref_text.append (" ");
    ref_text.append (utext);
  } else if (currentelement == VERSE_TEXT_TAG) {
    if (!verse_text.empty())
      verse_text.append (" ");
    verse_text.append (utext);
  }
}
