/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include "usfm.h"
#include "chapter.h"
#include "bible.h"
#include "usfmtools.h"


Chapter::Chapter (vector <ustring>& inputlines)
{
  for (unsigned int i = 0; i < inputlines.size (); i++)
    lines.push_back (inputlines[i]);
}


Chapter::~Chapter ()
{
}


void Chapter::get_verse (const ustring& verse, vector <ustring>& outputlines)
{
  outputlines.clear ();
  unsigned int begin, end;
  get_verse_boundaries (verse, begin, end);
  for (unsigned int i = begin; i <= end; i++) {
    outputlines.push_back (lines[i]);
  }
  // Keep versenumber to accept the verse later.
  previously_presented_verse = verse;
}


void Chapter::set_verse (vector<ustring>& inputlines)
{
  // Get the two boundaries in the vector of strings, where the
  // verse begins and where it ends.
  unsigned int begin, end;
  get_verse_boundaries (previously_presented_verse, begin, end);
  // Using an temporal vector of lines, start filling it with
  // 1. the set of lines before the current verse starts.
  // 2. the current edited verse.
  // 3. the lines following the current verse.
  vector<ustring> temporal_lines;
  for (unsigned int i = 0; i < lines.size(); i++) {
    if ((i < begin) || (i > end)) {
      temporal_lines.push_back(lines[i]);
    } else {
      i = end;
      for (unsigned int i2 = 0; i2 < inputlines.size(); i2++) {
        temporal_lines.push_back(inputlines[i2]);
      }
    }
  }
  // Copy the temporal lines back to the book.
  lines.clear();
  for (unsigned int i = 0; i < temporal_lines.size(); i++) {
    lines.push_back(temporal_lines[i]);
  }
}


void Chapter::get_verse_boundaries (const ustring& verse, unsigned int &begin, unsigned int &end)
{
  // Initialize the begin and end.
  begin = 0;
  end = lines.size() - 1;
  // Look up the beginning and the end of the requested verse.
  ustring current_verse = "0";
  bool first_boundary_found = false;
  for (unsigned int i = 0; i < lines.size (); i++) {
    ustring line = lines[i];
    ustring marker = usfm_extract_marker (line);
    if (usfm_is_verse (marker)) {
      size_t position = line.find (" ");
      position = CLAMP (position, 0, line.length());
      current_verse = line.substr (0, position);
      if (first_boundary_found || (verse == "0")) {
        end = i - 1;
        break;
      }        
      if (!first_boundary_found) {
        if (current_verse == verse) {
          begin = i;
          first_boundary_found = true;
        }
      }
    }
  }
  // If the verse was not found, use the extended search mechanism to retrieve it.
  if (!first_boundary_found)
    get_verse_boundaries_extended (verse, begin, end);
}


void Chapter::get_verse_boundaries_extended (const ustring& verse, unsigned int &begin, unsigned int &end)
/*
This function will be called when the exact verse was not found in the text.
E.g. it searched for verse "1-2", and the text happens to have verse "1" and
verse "2", but not the combined verse. This extended function handles these
cases.
This is roughly the strategy to be followed to get the right boundaries.
- The requested verse may have the a or b suffix, e.g. verse 4a.
- Therefore use a system of "half" verses.
- Create a vector with all the half verses requested.
- Create sets with all the half verses that are in a particular verse, e.g.
  v 1-2 will contain 1a and 1b, and v 3-5 contains 3a, 3b, 4a, 4b, 5a, 5b.
- Take any set that contains any of the requested half verses and set the
  boundaries accordingly.
*/
{
  // Get the requested half verses.
  vector<int> requested_verses = verses_encode (verse);
  // Store the sets with all available half verses.
  vector< set<int> > available_verses_sets;
  // Store the linenumbers where each verse starts.
  vector<int> available_verses_start_lines;
  // Go through all the lines of this chapter to look for verses.
  for (unsigned int i = 0; i < lines.size (); i++) {
    // Extract marker.
    ustring line = lines[i];
    ustring marker = usfm_extract_marker (line);
    // Handle it if this is a verse.
    if (usfm_is_verse (marker)) {
      // Find the verse number.
      size_t position = line.find (" ");
      position = CLAMP (position, 0, line.length() - 1);
      ustring current_verse = line.substr (0, position);
      // Encode the verses and put them in the set.
      vector<int> verses_v = verses_encode (current_verse);
      set<int> verses_s (verses_v.begin(), verses_v.end());
      available_verses_sets.push_back (verses_s);
      available_verses_start_lines.push_back (i);
    }
  }
  // Look for the starting and ending line number.
  bool start_found = false;
  bool end_found = false;
  for (unsigned int a = 0; a < available_verses_sets.size(); a++) {
    bool match = false;
    for (unsigned int r = 0; r < requested_verses.size(); r++) {
      if (available_verses_sets[a].find (requested_verses[r]) != available_verses_sets[a].end()) {
        match = true;
      }
    }
    if (match) {
      if (!start_found) {
        start_found = true;
        begin = available_verses_start_lines[a];
      }
    } else {
      if (start_found) {
        if (!end_found) {
          end_found = true;
          end = available_verses_start_lines[a] - 1;
        }
      }
    }
  }
}
