/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include <libgen.h>
#include <glib.h>
#include "highlight.h"
#include "utilities.h"
#include "gwrappers.h"
#include "usfmtools.h"


Highlight::Highlight (GtkTextBuffer * buffer)
{
  // Save and initialize variables.
  mybuffer = buffer;
}


Highlight::~Highlight ()
{
}


void Highlight::line_at_cursor (GtkTextTag * tag)
// Highlights the line the cursor is on.
{
  // Remove previous highlight, if any.
  GtkTextIter startiter;
  GtkTextIter enditer;
  gtk_text_buffer_get_start_iter (mybuffer, &startiter);
  gtk_text_buffer_get_end_iter (mybuffer, &enditer);
  gtk_text_buffer_remove_tag (mybuffer, tag, &startiter, &enditer);
  // Hightlight the line number the cursor is on.
  gtk_text_buffer_get_iter_at_mark (mybuffer, &startiter, gtk_text_buffer_get_insert (mybuffer));
  int line_of_cursor = gtk_text_iter_get_line (&startiter);
  gtk_text_buffer_get_iter_at_line (mybuffer, &startiter, line_of_cursor);
  gtk_text_buffer_get_iter_at_line (mybuffer, &enditer, ++line_of_cursor);
  gtk_text_buffer_apply_tag (mybuffer, tag, &startiter, &enditer);
}


void Highlight::searchwords (GtkTextTag * tag, Session * session, const ustring& verse)
/*
  This highlights all the words in the verse that agree to the word
  we search for.
*/
{
  // Remove any previous highlights.
  GtkTextIter startiter;
  GtkTextIter enditer;
  gtk_text_buffer_get_start_iter (mybuffer, &startiter);
  gtk_text_buffer_get_end_iter (mybuffer, &enditer);
  gtk_text_buffer_remove_tag (mybuffer, tag, &startiter, &enditer);
  // The word(s) to search for.
  for (unsigned int i = 0; i < session->highlight_words.size (); i++) {
    ustring searchword = session->highlight_words[i];
    bool casesensitive = session->highlight_casesensitives[i];
    bool globbing = session->highlight_globbings[i];
    bool startmatch = session->highlight_matchbegins[i];
    bool endmatch = session->highlight_matchends[i];
    // Highlight words in the line the cursor is on.
    bool cursor_set = false;
    GtkTextIter iterator;
    gtk_text_buffer_get_iter_at_mark (mybuffer, &iterator, gtk_text_buffer_get_insert (mybuffer));
    int line_of_cursor = gtk_text_iter_get_line (&iterator);
    bool is_verse;
    searchwords_internal (line_of_cursor, convert_to_int (verse), searchword, cursor_set, is_verse, casesensitive, globbing, startmatch, endmatch, tag);
    // See whether we have lines before this verse that is not a verse number,
    // and highlight words there too. As introductions can be quite lengthy,
    // look many lines back.
    int lower_bound = line_of_cursor - 50;
    lower_bound = CLAMP (lower_bound, 0, line_of_cursor);
    int higher_bound = line_of_cursor - 1;
    higher_bound = CLAMP (higher_bound, 0, line_of_cursor);
    is_verse = false;
    for (int i = higher_bound; i >= lower_bound; i--)
      if (!is_verse)
        searchwords_internal (i, convert_to_int (verse), searchword, cursor_set, is_verse, casesensitive, globbing, startmatch, endmatch, tag);
    // Do the same for many lines after this verse.
    int hi_line = gtk_text_buffer_get_line_count (mybuffer) - 1;
    lower_bound = line_of_cursor + 1;
    lower_bound = CLAMP (lower_bound, line_of_cursor, hi_line);
    higher_bound = line_of_cursor + 50;
    higher_bound = CLAMP (higher_bound, line_of_cursor, hi_line);
    is_verse = false;
    for (int i = lower_bound; i <= higher_bound; i++) {
      if (!is_verse) {
        searchwords_internal (i, convert_to_int (verse), searchword, cursor_set, is_verse, casesensitive, globbing, startmatch, endmatch, tag);
      }
    }
  }
}


void Highlight::searchwords_internal (unsigned int linenumber, unsigned int versenumber, 
                                      const ustring & searchword, bool & cursor_set, bool & is_verse, 
                                      bool casesensitive, bool globbing, bool matchbegin, bool matchend,
                                      GtkTextTag * tag)
/*
Searches for words to highlight.
Problem when case insensitive searching:
  Character ﬃ was changed to ffi after casefolding, and as that one is 2
  characters longer than the original ﬃ, we ran in problems of searching
  past the line, which gave an exception in Gtk.
The solution was to determine the length of the word from the ones that are 
to highlight, not from the casefolded searchword, but the original one.
*/
{
  // Extract the line.
  GtkTextIter begin;
  gtk_text_buffer_get_iter_at_line (mybuffer, &begin, linenumber);
  GtkTextIter end;
  gtk_text_buffer_get_iter_at_line (mybuffer, &end, linenumber + 1);
  ustring line = gtk_text_buffer_get_text (mybuffer, &begin, &end, false);
  line = trim (line);
  // Get the verse number.
  ustring this_verse;
  ustring marker (line);
  marker = usfm_extract_marker (marker);
  is_verse = usfm_is_verse (marker);
  if (is_verse)
    this_verse = number_in_string (line);
  // If there is any verse number and it is different from the one we have, return.
  if (is_verse)
    if (convert_to_int (this_verse) != versenumber)
      return;
  // Find all places in this line that have the search word.
  /*
  To do that properly for glob-style pattern matching, for begin/end word
  matching and case (in)sensitivity, we need to open the box of tricks.
  We produce all possible combinations for characters and lengths, e.g.
  We have this text:
    he is
  We then make the following strings from it, and see whether they match:
    "h"
    "he"
    "he "
    "he i"
    "he is"
    "e"
    "e "
    "e i"
    "e is"
    " "
    " i"
    " is"
    "i"
    "is"
    "s"
  Anyone matching will then be highlighted.
  */
  // Deal with case sensitivity.
  ustring case_considerate_search_word (searchword);
  if (!casesensitive)
    case_considerate_search_word = case_considerate_search_word.casefold();
  // Make the glob-style pattern matching pattern specification.
  GPatternSpec * gpatternspec;
  gpatternspec = g_pattern_spec_new (case_considerate_search_word.c_str());
  for (unsigned int i = 0; i < line.length(); i++) {
    ustring line2 (line.substr (0, i + 1));
    for (unsigned int offposition = 0; offposition < line2.length(); offposition++) {
      // Get the line as described above.
      // We use optimization here to get the speed acceptable when we have 
      // long lines. But when globbing is done, because of the characters of 
      // glob-style matching, we don't know how long the searchword might be,
      // we do not use that optimization.
      unsigned int linelength = line2.length() - offposition;
      if (!globbing)
        if (linelength > searchword.length()) 
          continue;
      ustring compareline (line2.substr (offposition, linelength));
      // Deal with case sensitivity.
      if (!casesensitive)
        compareline = compareline.casefold();
      // Now compare.
      bool match = false;
      if (globbing) {
        if (g_pattern_match_string (gpatternspec, compareline.c_str()))
          match = true;        
      } else {
        if (case_considerate_search_word == compareline)
          match = true;
      }
      // Get the iterators in the textbuffer that belong to this possible match.
      if (match) {
        gtk_text_buffer_get_iter_at_line_offset (mybuffer, &begin, linenumber, offposition);
        gtk_text_buffer_get_iter_at_line_offset (mybuffer, &end, linenumber, offposition + searchword.length ());
      }
      // Deal with begin-word matching.
      if (match) {
        if (matchbegin) {
          if (!gtk_text_iter_starts_word (&begin))
            match = false;
        }
      }
      // Deal with end-word matching.
      if (match) {
        if (matchend) {
          if (!gtk_text_iter_ends_word (&end))
            match = false;
        }
      }
      // Highlight the word.
      if (match) {
        gtk_text_buffer_apply_tag (mybuffer, tag, &begin, &end);
        if (!cursor_set) {
          gtk_text_buffer_place_cursor (mybuffer, &begin);
          cursor_set = true;
        }
      }
    }
  }
  // Free the memory for the glob-style matching pattern.
  g_pattern_spec_free (gpatternspec);
}


GtkTextTag * Highlight::get_tag (const ustring& tagname)
/*
This returns the tag named "tagname", and this tagname is created if it wasn't
there.
This tag is used to retrieve verse location information from the editor, i.e.
"on which verse are we now?".
*/
{
  // Get the tables of tags.
  GtkTextTagTable *table;
  table = gtk_text_buffer_get_tag_table (mybuffer);
  GtkTextTag *tag;
  // Get the named tag.
  tag = gtk_text_tag_table_lookup (table, tagname.c_str());
  // If the tag was not there, create it.
  if (!tag) {
    tag = gtk_text_buffer_create_tag (mybuffer, tagname.c_str(), NULL);
    // Note this int value of "1" is needed later to indicate to us this is the tag we need.
    g_object_set_data (G_OBJECT (tag), "verse-tag", GINT_TO_POINTER (1));
  }    
  return tag;
}
