/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include "indexer.h"
#include <config.h>
#include "bible.h"
#include "book.h"
#include "scripture.h"
#include <glib.h>
#include "project.h"
#include "gwrappers.h"
#include <sqlite3.h>
#include "sqlite_reader.h"
#include "indexchapter.h"
#include "bookmetrics.h"
#include "directories.h"
#include "shell.h"


#define TABLE_FILE_SIGNATURE "file_signature"
#define TABLE_LINES "lines"


#define IDENTIFIER_TEXT_TAG "identifier-text"
#define INTRODUCTION_TEXT_TAG "introduction-text"
#define HEADING_TEXT_TAG "heading-text"
#define CHAPTER_TEXT_TAG "chapter-text"
#define STUDY_NOTE_TEXT_TAG "studynote-text"
#define NOTE_TEXT_TAG "note-text"
#define CROSSREFERENCE_TEXT_TAG "crossreference-text"
#define VERSE_TEXT_TAG "verse-text"


Indexer::Indexer (int dummy)
{
  shutdown_flag = false;
}


Indexer::~Indexer ()
{
}


void Indexer::start ()
// Starts the indexer and it keeps running on its own as a thread.
{
  g_thread_create (GThreadFunc (run_thread), gpointer (this), false, NULL);
}


void Indexer::initialize_books (const ustring& project)
{
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Connect to the database and create the table(s) if needed.
    rc = sqlite3_open(dbfilename (project).c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    sqlite3_busy_timeout (db, 2000);
    SqliteReader reader (0);
    char * sql;
    // Check for the table that stores the lines
    sql = g_strdup_printf ("select count(*) from sqlite_master where name='%s';", TABLE_LINES);
    rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (reader.ustring0[0] == "0") {
      sql = g_strdup_printf ("create table '%s' (book integer, chapter integer, verse text, line text, casefolded text, id text, idcf text, intro text, introcf text, head text, headcf text, chap text, chapcf text, study text, studycf text, note text, notecf text, ref text, refcf text, vs text, vscf text);", TABLE_LINES);
      rc = sqlite3_exec (db, sql, NULL, NULL, &error);
      g_free (sql);
      if (rc != SQLITE_OK) {
        throw runtime_error (error);
      }
    }
    reader.ustring0.clear();
    // Check for the table that stores the file signature
    sql = g_strdup_printf ("select count(*) from sqlite_master where name='%s';",  TABLE_FILE_SIGNATURE);
    rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (reader.ustring0[0] == "0") {
      sql = g_strdup_printf ("create table %s (filename text, signature text);", TABLE_FILE_SIGNATURE);
      rc = sqlite3_exec (db, sql, NULL, NULL, &error);
      g_free (sql);
      if (rc != SQLITE_OK) {
        throw runtime_error (error);
      }
    }
    // Get books in project.
    Scripture scripture (project);
    // Check each book.
    for (unsigned int i = 0; i < scripture.paths.size (); i++) {
      IndexStorage indexstorage (project, scripture.paths[i]);
      indexobjects.push_back (indexstorage);
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.  
  sqlite3_close (db);
}


void Indexer::update_chapter (const ustring& project, unsigned int book, unsigned int chapter, const vector <ustring>& lines)
{
  // Schedule this chapter for later execution. First come, first serve :)
  IndexStorage indexstorage (project, book, chapter, lines);
  indexobjects.push_back (indexstorage);
}


void Indexer::vacuum (const ustring& project)
// Vacuum the tables through external program.
{
  string filename = dbfilename (project);
  ustring command = BIBLEDIT_VACUUM;
  command.append (shell_quote_space (filename) + "&");
  system (command.c_str());
}


void Indexer::log (const ustring & message)
{
  gw_message ("Indexer: " + message);
}


void Indexer::check_book_thread (gpointer data)
{
  ((Indexer *) data)->check_book ();
}


void Indexer::check_book ()
{
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    // Connect to the database
    rc = sqlite3_open(dbfilename (indexobjects[0].project).c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    sqlite3_busy_timeout (db, 2000);
    // Get the actual signature for the file.
    string actual_signature;
    /* 
     * At first we got the signature through the ls -l command, but this caused 
     * a problem. It gives the modification time of the file with a resolution
     * of a minute. But if changes are applied to a file, and after the change
     * it still shows the same minute, then it did not get updates in the words
     * database. We now use the stat system call with higher accuracy of time.
     * This solves the problem 
     */
    struct stat statbuf;
    stat (indexobjects[0].filename.c_str(), &statbuf);
    actual_signature = convert_to_string (int(statbuf.st_size));
    actual_signature.append (convert_to_string (int(statbuf.st_mtime)));
    actual_signature.append (convert_to_string (int(statbuf.st_ctime)));
    // See if the filename exists in the table anyway.
    bool update = false;
    SqliteReader reader (0);
    char * sql;
    sql = g_strdup_printf ("select count(*) from '%s' where filename='%s';", TABLE_FILE_SIGNATURE, indexobjects[0].filename.c_str ());
    rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    if (reader.ustring0[0] == "0") {
      // Doesn't exist, so schedule the file for updating the database.
      update = true;
    }
    reader.ustring0.clear();
    if (!update) {
      // The filename is in the database - now check its signature.
      char * sql;
      sql = g_strdup_printf ("select signature from '%s' where filename='%s';", TABLE_FILE_SIGNATURE, indexobjects[0].filename.c_str ());
      rc = sqlite3_exec(db, sql, reader.callback, &reader, &error);
      g_free (sql);
      if (rc != SQLITE_OK) {
        throw runtime_error (error);
      }
      if (reader.ustring0.size() > 0) {
        string stored_signature = reader.ustring0[0];
        if (stored_signature != actual_signature) {
          update = true;
        }
      }
    }
    if (update) {
      log ("Scheduled: " + indexobjects[0].filename);
      // Store data about this filename.
      Book wbook (indexobjects[0].filename, false, "");
      BookMetrics bookmetrics (indexobjects[0].filename);
      for (unsigned int chapter = 0; chapter < bookmetrics.get_chapters().size(); chapter++) {
        vector <ustring> lines;
        wbook.get_chapter (chapter, lines);
        IndexStorage indexstorage (indexobjects[0].project, wbook.get_id_index (), chapter, lines);
        indexobjects.push_back (indexstorage);
      }
      // Schedule its signature for later storage.
      IndexStorage indexstorage (indexobjects[0].project, indexobjects[0].filename, actual_signature);
      indexobjects.push_back (indexstorage);
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.  
  sqlite3_close (db);
}


void Indexer::index_chapter_thread (gpointer data)
{
  ((Indexer *) data)->index_chapter ();
}


void Indexer::index_chapter ()
{
  // Entry in log.
  log (index_to_biblebook (indexobjects[0].book) + " " + convert_to_string (indexobjects[0].chapter));
  // Index chapter.
  ustring filename (dbfilename (indexobjects[0].project));
  IndexChapter indexchapter (filename, indexobjects[0].book, indexobjects[0].chapter, indexobjects[0].lines);
}


void Indexer::delete_book (const ustring& project, unsigned int book)
{
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(dbfilename (project).c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    sqlite3_busy_timeout (db, 2000);
    rc = sqlite3_exec (db, "begin;", NULL, NULL, &error);
    char * sql;
    sql = g_strdup_printf ("delete from '%s' where book = %d;", TABLE_LINES, book);
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    rc = sqlite3_exec (db, "commit;", NULL, NULL, &error);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  // Close connection.  
  sqlite3_close (db);
}


ustring Indexer::dbfilename (const ustring& project)
{
  // Returns full path and name of the word database.
  ustring s = project + ".idx3";
  return gw_build_filename (directories_get_temp(), s);
}


void Indexer::store_signature_thread (gpointer data)
{
  ((Indexer *) data)->store_signature ();
}


void Indexer::store_signature ()
{
  // Store signature of the book.
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try
  {
    rc = sqlite3_open(dbfilename (indexobjects[0].project).c_str (), &db);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    sqlite3_busy_timeout (db, 2000);
    rc = sqlite3_exec (db, "begin;", NULL, NULL, &error);
    if (rc) {
      throw runtime_error (sqlite3_errmsg(db));
    }
    char * sql;
    sql = g_strdup_printf("delete from '%s' where filename = '%s';", TABLE_FILE_SIGNATURE, indexobjects[0].filename.c_str ());
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    sql = g_strdup_printf("insert into '%s' values('%s', '%s');", TABLE_FILE_SIGNATURE, indexobjects[0].filename.c_str (), indexobjects[0].signature.c_str ());
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
    rc = sqlite3_exec (db, "commit;", NULL, NULL, &error);
    if (rc != SQLITE_OK) {
      throw runtime_error (error);
    }
  }
  catch (exception & ex)
  {
    gw_critical (ex.what ());
  }
  sqlite3_close (db);
}


bool Indexer::is_ready ()
{
  return (indexobjects.size() == 0);
}


void Indexer::shutdown ()
{
  // In some cases, e.g. when we shut down bibledit just after opening a freshly
  // created project, there are manu index objects to be done, and it can take
  // more than five minutes on a common machine to finish these index object.
  // This is too long. Therefore we just finish indexing, and so bibledit
  // will shut down much faster. Next time bibledit starts, it will do these 
  // objects again.
  shutdown_flag = true;
}


void Indexer::run_thread (gpointer data)
{
  ((Indexer *) data)->run ();  
}


void Indexer::run ()
{
  while (!shutdown_flag) {
    // Continue if there are no objects to be dealt with.
    if (indexobjects.size() == 0) {
      g_usleep (10000);
      continue;
    }
    // Deal with this object.
    switch (indexobjects[0].type) {
      case isCheckBook :
      {
        check_book ();
        break;
      }
      case isIndexChapter :
      {
        index_chapter ();
        break;
      }
      case isStoreSignature :
      {
        store_signature ();
        break;
      }
    }
    // Erase the object.
    indexobjects.erase (indexobjects.begin());
  }
  // Clear any remaining objects, so that the ready function works properly.
  indexobjects.clear();
}
