/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include <glib.h>
#include "printreferences.h"
#include "utilities.h"
#include "bible.h"
#include "book.h"
#include "chapter.h"
#include "usfm.h"
#include "usfmtools.h"
#include <config.h>
#include "scripture.h"
#include "pdfviewer.h"
#include "xmlutils.h"
#include "paper.h"
#include "formatter.h"
#include "xmlfo-utils.h"
#include "constants.h"
#include "project.h"
#include "gwrappers.h"
#include "gtkwrappers.h"
#include "directories.h"
#include "fontutils.h"
#include "notecaller.h"
#include "xslfofootnote.h"
#include "xslfoendnote.h"
#include "xslfoxref.h"


void view_references_pdf (Configuration * configuration, const vector<ustring>& references,
                          Session * session, Progress * progress)
/*
Formats the references in "references", and highlights all words in
"session->highlights*" and shows them in a pdf viewer.
*/
{
  // Log.
  gw_message ("Starting to print the references");
  
  // Progress system.
  progress->start();
  progress->set_text ("Starting");
  progress->set_iterate (0, 0.5, references.size());
  if (progress->cancel) {
    progress->finish();
    return;
  }

  // Prepare for the inline text markers.
  Project project (configuration);
  Usfm usfm (project.get_stylesheet());
  UsfmInlineMarkers usfm_inline_markers (usfm);

  // Prepare for leaving out footnotes, endnotes and crossreferences.
  XslFoFootnote xslfofootnote (usfm, configuration, false);
  XslFoEndnote xslfoendnote (usfm, configuration, false);
  XslFoXref xslfoxref (usfm, configuration, false);
  
  // Storage for final xsl-fo file.
  vector<ustring> xslfo_lines;

  // Start building the xml-fo file.
  // The various elements of this file are created using objects.
  // When the object goes out of scope, it writes the closing elements.
  {
    XmlFoRoot xmlforoot (&xslfo_lines, configuration);
    {
      XmlFoLayoutMasterSet layoutmasterset (&xslfo_lines, configuration, false);
    }      
    {
      XmlFoPageSequence pagesequence (&xslfo_lines, false);
      {
        XmlFoStaticContent staticcontent (&xslfo_lines, configuration);
      }
      {
        XmlFoFlow flow (&xslfo_lines);
        {
          // The scripture where to extract the references from.
          Scripture scripture (configuration->project);
          
          // Book related data.
          Book *book = NULL;
          int previous_book_index = -1;
  
          // Produce chunks for the xsl-fo file for all references.
          for (unsigned int i = 0; i < references.size(); i++) {
            // Update progress bar.
            progress->iterate();
            progress->set_text ("Collecting references");
            if (progress->cancel) {
              progress->finish();
              return;
            }
            // Get the reference decoded, and get the book number.
            ustring bookref, chapterref, verse;
            decode_reference (references[i], bookref, chapterref, verse);
            int book_index = scripture.get_index_of_book (bookref);
            // Book exists?
            if (book_index >= 0) {
              // Open the book.
              // To speed things up we do not destroy and recreate a book object 
              // if the book is still the same. Only new books destroy/create the object.
              // A test showed that this works more than three times as fast as when 
              // recreating the book object for each references.
              if (book_index != previous_book_index) {
                if (book) delete (book);
                book = new Book (scripture.paths[book_index], false, "");
                previous_book_index = book_index;
              }
              // Get the chapter and the verse text.
              vector <ustring> lines;
              book->get_chapter (convert_to_int (chapterref), lines);
              Chapter chapter (lines);
              vector <ustring> verses;
              chapter.get_verse (verse, verses);
              // As a verse may consist of more lines, deal with each.
              ustring line;
              for (unsigned int i2 = 0; i2 < verses.size (); i2++) {
                ustring s = verses[i2];
                usfm_extract_marker (s);
                line.append (" " + s);
              }
              // Take out footnotes, endnotes, crossreferences.
              xslfofootnote.transform (NULL, line);
              xslfoendnote.transform (NULL, line);
              xslfoxref.transform (NULL, line);              
              // Positions in the line, and lengths to highlight.
              vector <size_t> highlight_positions;
              vector <size_t> highlight_lengths;
              // Go through all the words to highlight.
              for (unsigned int i2 = 0; i2 < session->highlight_casesensitives.size(); i2++) {
                // Word to highlight
                ustring highlightword;
                if (session->highlight_casesensitives[i2]) 
                  highlightword = session->highlight_words[i2];
                else 
                  highlightword = session->highlight_words[i2].casefold ();
                // Variabele s holds a shadow string.
                ustring s;
                if (session->highlight_casesensitives[i2]) 
                  s = line;
                else 
                  s = line.casefold ();
                // Find positions for highlighting.
                size_t offposition = s.find (highlightword);
                while (offposition != string::npos) {
                  // Store position and length.
                  highlight_positions.push_back (offposition);
                  highlight_lengths.push_back (highlightword.length());
                  // Look for possible next word to highlight.
                  offposition = offposition + highlightword.length () + 1;
                  // There is something like a bug in s.find. If the offposition
                  // is greater than the length of s, then s.find will return
                  // a value below offposition, instead of string::npos as is
                  // expected. Workaround.
                  if (offposition > s.length())
                    break;
                  offposition = s.find (highlightword, offposition);
                }
              }
              // Sort the positions from small to bigger.
              xml_sort_positions (highlight_positions, highlight_lengths);
              // Combine overlapping positions.
              xml_combine_overlaps (highlight_positions, highlight_lengths);
              // Change <, > and & to their corresponding entities.
              xml_handle_entities (line, highlight_positions);
              // Insert the code for highlighting.
              xml_fo_insert_emphasis (line, highlight_positions, highlight_lengths);
              // In case this prints changes, deal with strike-through and bold.
              replace_text (line, STRIKE_THROUGH_BEGIN, "<fo:inline text-decoration=\"line-through\">");
              replace_text (line, STRIKE_THROUGH_END, "</fo:inline>");
              replace_text (line, BOLD_BEGIN, "<fo:inline font-weight=\"bold\">");
              replace_text (line, BOLD_END, "</fo:inline>");
              // Deal with inline markers, so that they do not appear in the output
              // as markers, but format the text instead.
              // At times there is an overlap of fo:inline markers, caused in a 
              // situation where there are changes at a position where an inline
              // format marker is active too. This overlap is not allowed in the
              // XSL standard, and therefore must be avoided. It is avoided so:
              // 1. See if there were any replacements due to changes.
              // 2. If so, do not handle the inline text, but just remove their
              //    corresponding usfm markers.
              usfm_handle_inline_text (line, &usfm_inline_markers, NULL);
              // Add the block, and this line, to the xsl-fo file.
              // Measures are taken, within the capabilities of fop 0.20.5, that
              // any reference stays wholly on one page. FOP 0.20.5 does that only
              // within a table row.
              {
                xslfo_lines.push_back ("      <fo:table table-layout=\"fixed\" width=\"100%\">");
                xslfo_lines.push_back ("        <fo:table-column column-width=\"proportional-column-width(1)\"/>");
                xslfo_lines.push_back ("        <fo:table-body>");
                // Next line has keep-together.within-page="always", rather than
                // keep-together="always", as the latter one causes things to be 
                // kept together, in in a line, which causes the line to overflow
                // the right margin.
                xslfo_lines.push_back ("          <fo:table-row keep-together.within-page=\"always\">");
                xslfo_lines.push_back ("            <fo:table-cell>");
                // XSLFormatter was better than FOP in that it does honour space conditionality,
                // which is initially set at "discard" for the beginning of a 
                // references area, as here. So to get the distance between the 
                // lines right, this is inserted: space-before.conditionality="retain".
                xslfo_lines.push_back ("              <fo:block space-before=\"2mm\" space-before.conditionality=\"retain\">");
                xslfo_lines.push_back (references[i] + line);
                xslfo_lines.push_back ("              </fo:block>");
                xslfo_lines.push_back ("            </fo:table-cell>");
                xslfo_lines.push_back ("          </fo:table-row>");
                xslfo_lines.push_back ("        </fo:table-body>");
                xslfo_lines.push_back ("      </fo:table>");
              }
            }
          }
          // If object book is still alive, destroy it.
          if (book) delete (book);
          
        }
      }
    }
  }
  
  // Make a temporary directory where to put the working files and the resulting
  // .pdf file.information. This directory is not removed, because the pdf viewer 
  // needs the .pdf file to be there during viewing or printing.
  string working_directory = gw_build_filename(directories_get_temp(), "references");
  create_directory (working_directory);
  // Produce filename of .fo file.
  string fofilename = gw_build_filename(working_directory, "document.fo");
  // Write the document.
  write_lines (fofilename, xslfo_lines);
  // Tell user to be patient while we pdfxmltex runs.  
  progress->set_text ("Typesetting pages ...");
  if (progress->cancel) {
    progress->finish();
    return;
  }
  // Convert the xsl-fo document to .pdf.
  NoteCaller dummy (nntNumerical, "", false);
  string pdffilename = gw_build_filename(working_directory, "document.pdf");
  int conversion_result = formatter_convert_to_pdf (configuration, fofilename, pdffilename, progress, &dummy, &dummy);
  // Progressbar: ready.
  progress->set_fraction (1);
  progress->set_text ("Ready");
  if (progress->cancel) {
    progress->finish();
    return;
  }
  // View the .pdf document.
  pdfviewer (pdffilename);
  // Finalize progress bar.
  progress->finish();
  // Give message if there were errors.
  if (conversion_result != 0) {
    string message = "While formatting the text problems were encountered.\n"
                     "See menu Help - System log for more details.\n"
                     "See the helpfile for a possible solution.";
    gtkw_dialog_error (NULL, message);
  };
  // Log: ready.
  gw_message ("Ready printing the references");
}
