/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "utilities.h"
#include "track.h"
#include "bible.h"


Track::Track (unsigned int dummy)
{
  clear ();
}


Track::~Track ()
{
}


void Track::tick (const ustring & book, const ustring & chapter, const ustring & verse)
{
/* This is how the tracker stores its references, through "tick".
 * If the counter for a stable reference is high enough, and a new reference
 * comes in, it stores the old reference.
 * Every time a new reference comes in it resets the counter for a stable reference.
 */
  time_elapsed++;
  string current_reference = reference (book, chapter, verse);
  // The recorded reference variable contains the reference that has been recorded.
  // This is usually the last reference that was recorded,
  // except in cases that the forward/back arrows were clicked.
  // In those cases it shows the reference that was than recorded, a while ago.
  // This prevents the trackers from recording the previously recorded reference again.
  if (current_reference != previous_reference) {
    // The number of seconds after which the reference gets recorded.
    if (time_elapsed >= 5) {
      store_reference (previous_reference);
      // When a new reference is recorded, the pointer goes straight to the end of the list.
      // Even if it was not pointing at the end after the forward/back buttons were clicked.
      reference_pointer = recorded_references.size ();
    }
    previous_reference = current_reference;
    time_elapsed = 0;
  }
}


void Track::get_next_reference (ustring& book, ustring& chapter, ustring& verse)
{
  if (we_went_back)
    reference_pointer++;
  decode_reference (recorded_references[reference_pointer], book, chapter, verse);
  // Adjust "previous reference" so that no new record is made of this change.
  previous_reference = recorded_references[reference_pointer];
  reference_pointer++;
  we_went_back = false;
  we_went_forward = true;
}


void Track::get_previous_reference (ustring& book, ustring& chapter, ustring& verse)
{
  // Record the current reference if we go back the first step.
  if (reference_pointer == int (recorded_references.size())) {
    store_reference (previous_reference);
  }
  if (we_went_forward)
    reference_pointer--;
  reference_pointer--;
  decode_reference (recorded_references[reference_pointer], book, chapter, verse);
  // Adjust "previous reference" so that no new record is made of this change.
  previous_reference = recorded_references[reference_pointer];
  we_went_back = true;
  we_went_forward = false;
}


bool Track::next_reference_available ()
{
  return (reference_pointer < int (recorded_references.size ()));
}


bool Track::previous_reference_available ()
{
  return (reference_pointer > 0);
}


ustring Track::reference (const ustring & book, const ustring & chapter, const ustring & verse)
{
  return book + " " + chapter + ":" + verse;
}


void Track::clear ()
{
  time_elapsed = 0;
  reference_pointer = 0;
  previous_reference.clear ();
  recorded_references.clear ();
  we_went_back = false;
  we_went_forward = false;
}


void Track::store_reference (const ustring& reference) {
/*
 * Store the reference, but only when the last reference on the stack
 * is different. That is, do not store the same reference twice in a row.
 */
  if (recorded_references.size() == 0) {
    recorded_references.push_back (reference);
    return;
  }
  if (recorded_references[recorded_references.size() - 1] != reference)
    recorded_references.push_back (reference);
}
