/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "exception.h"
#include <gtk/gtk.h>
#include <config.h>
#include "utilities.h"
#include "directories.h"
#include "gwrappers.h"
#include "stylesheetutils.h"
#include "shell.h"
#include "projectutils.h"
#include "scripting.h"
#include "import.h"
#include "categorize.h"
#include "progresswindow.h"
#include "books.h"
#include "generalconfig.h"
#include "synchronize.h"
#include "notes_utils.h"
#include "export_utils.h"
#include "projectconfig.h"
#include "checks.h"
#include "check_validate_usfm.h"
#include "check_count_words.h"
#include "opendocument.h"
#include "check_count_usfms.h"
#include "check_count_characters.h"
#include "check_chapters_verses.h"
#include "check_compare_usfm.h"
#include "check_unwanted_patterns.h"
#include "check_unwanted_words.h"
#include "check_repetition.h"
#include "check_capitalization.h"
#include "check_matching_pairs.h"
#include "mapping.h"
#include "versification.h"
#include "templates.h"
#include "originreferences.h"
#include "upgrade.h"


void scripting_usage (GOptionEntry entries[], bool subscript);
ustring scripting_get_from_gchar (gchar * argument);
void scripting_handle_error (GError * error);
void scripting_ensure_project (const ustring& project);
void initialize_gui (bool gui, int argc, char *argv[]);
int export_stylesheet (int argc, char *argv[]);
int upgrade_data (int argc, char *argv[]);
int import_book (int argc, char *argv[]);
int export_book (int argc, char *argv[]);
int shutdown_actions (int argc, char *argv[]);
int export_project_usfm (int argc, char *argv[]);
int export_project_sword (int argc, char *argv[]);
int export_project_opendocument (int argc, char *argv[]);
int general_configuration (int argc, char *argv[]);
int project_configuration (int argc, char *argv[]);
int create_project (int argc, char *argv[]);
int delete_project (int argc, char *argv[]);
int validate_markers (int argc, char *argv[]);
int count_words (int argc, char *argv[]);
int count_usfms (int argc, char *argv[]);
int count_characters (int argc, char *argv[]);
int check_chapters_verses (int argc, char *argv[]);
int compare_usfms (int argc, char *argv[]);
int unwanted_patterns (int argc, char *argv[]);
int unwanted_words (int argc, char *argv[]);
int repeating_words (int argc, char *argv[]);
int check_capitalization (int argc, char *argv[]);
int check_matching_pairs (int argc, char *argv[]);
int import_verse_system (int argc, char *argv[]);
int map_ (int argc, char *argv[]);
int test_ (int argc, char *argv[]);


int scripting (int argc, char *argv[])
{
  // Parse options.
  bool version = false;
  bool upgradedata = false;
  bool importbook = false;
  bool exportbook = false;
  bool exportprojectusfm = false;
  bool exportprojectsword = false;
  bool exportprojectopendocument = false;
  bool exportstylesheet = false;
  bool shutdownactions = false;
  bool generalconfiguration = false;
  bool projectconfiguration = false;
  bool createproject = false;
  bool deleteproject = false;
  bool validatemarkers = false;
  bool countwords = false;
  bool countusfms = false;
  bool countcharacters = false;
  bool checkchaptersverses = false;
  bool compareusfms = false;
  bool unwantedpatterns = false;
  bool unwantedwords = false;
  bool repeatingwords = false;
  bool checkcapitalization = false;
  bool checkmatchingpairs = false;
  bool importversesystem = false;
  bool map = false;
  bool test = false;
  GOptionEntry entries[] = 
  {
    { "version", gchar (NULL), 0, G_OPTION_ARG_NONE, &version, "Version number", NULL },
    { "upgrade-data", gchar (NULL), 0, G_OPTION_ARG_NONE, &upgradedata, "Upgrade all relevant data", NULL },
    { "import-book", gchar (NULL), 0, G_OPTION_ARG_NONE, &importbook, "Import a book", NULL },
    { "export-book", gchar (NULL), 0, G_OPTION_ARG_NONE, &exportbook, "Export a book", NULL },
    { "export-project-usfm", gchar (NULL), 0, G_OPTION_ARG_NONE, &exportprojectusfm, "Export a project to USFM", NULL },
    { "export-project-sword", gchar (NULL), 0, G_OPTION_ARG_NONE, &exportprojectsword, "Export a project to SWORD", NULL },
    { "export-project-odt", gchar (NULL), 0, G_OPTION_ARG_NONE, &exportprojectopendocument, "Export a project to OpenDocument text", NULL },
    { "export-stylesheet", gchar (NULL), 0, G_OPTION_ARG_NONE, &exportstylesheet, "Export a stylesheet", NULL },
    { "shutdown-actions", gchar (NULL), 0, G_OPTION_ARG_NONE, &shutdownactions, "Called on Bibledit shutdown", NULL },
    { "general-configuration", gchar (NULL), 0, G_OPTION_ARG_NONE, &generalconfiguration, "Access Bibledit's general configuration", NULL },
    { "project-configuration", gchar (NULL), 0, G_OPTION_ARG_NONE, &projectconfiguration, "Access Bibledit's project configuration", NULL },
    { "create-project", gchar (NULL), 0, G_OPTION_ARG_NONE, &createproject, "Create a project", NULL },
    { "delete-project", gchar (NULL), 0, G_OPTION_ARG_NONE, &deleteproject, "Delete a project", NULL },
    { "validate-markers", gchar (NULL), 0, G_OPTION_ARG_NONE, &validatemarkers, "Validate USFM markers", NULL },
    { "count-words", gchar (NULL), 0, G_OPTION_ARG_NONE, &countwords, "Count words", NULL },
    { "count-usfms", gchar (NULL), 0, G_OPTION_ARG_NONE, &countusfms, "Count USFMs", NULL },
    { "count-characters", gchar (NULL), 0, G_OPTION_ARG_NONE, &countcharacters, "Count characters", NULL },
    { "check-chapters-verses", gchar (NULL), 0, G_OPTION_ARG_NONE, &checkchaptersverses, "Check chapters and verses", NULL },
    { "compare-usfms", gchar (NULL), 0, G_OPTION_ARG_NONE, &compareusfms, "Compare USFMs", NULL },
    { "check-unwanted-patterns", gchar (NULL), 0, G_OPTION_ARG_NONE, &unwantedpatterns, "Look for unwanted patterns", NULL },
    { "check-unwanted-words", gchar (NULL), 0, G_OPTION_ARG_NONE, &unwantedwords, "Look for unwanted words", NULL },
    { "check-repeating-words", gchar (NULL), 0, G_OPTION_ARG_NONE, &repeatingwords, "Look for repeating words", NULL },
    { "check-capitalization", gchar (NULL), 0, G_OPTION_ARG_NONE, &checkcapitalization, "Check capitalization", NULL },
    { "check-matching-pairs", gchar (NULL), 0, G_OPTION_ARG_NONE, &checkmatchingpairs, "Check matching pairs of punctuation", NULL },
    { "import-verse-system", gchar (NULL), 0, G_OPTION_ARG_NONE, &importversesystem, "Import a verse system", NULL },
    { "map", gchar (NULL), 0, G_OPTION_ARG_NONE, &map, "Map a verse to another versification", NULL },
    { "test", gchar (NULL), 0, G_OPTION_ARG_NONE, &test, "Test Bibledit", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_set_ignore_unknown_options (context, true);
  g_option_context_set_help_enabled (context, false);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  if (error) {
    gw_error (error->message);
    g_error_free (error);
  }

  // Take action depending on options.
  if (version) {
    cout << VERSION << endl;
    return 0;
  } else if (upgradedata) {
    return upgrade_data (argc, argv);
  } else if (importbook) {
    return import_book (argc, argv);
  } else if (exportbook) {
    export_book (argc, argv);
  } else if (exportprojectusfm) {
    export_project_usfm (argc, argv);
  } else if (exportprojectsword) {
    export_project_sword (argc, argv);
  } else if (exportprojectopendocument) {
    export_project_opendocument (argc, argv);
  } else if (exportstylesheet) {
    export_stylesheet (argc, argv);
  } else if (shutdownactions) {
    shutdown_actions (argc, argv);
  } else if (generalconfiguration) {
    general_configuration (argc, argv);
  } else if (projectconfiguration) {
    project_configuration (argc, argv);
  } else if (createproject) {
    create_project (argc, argv);
  } else if (deleteproject) {
    delete_project (argc, argv);
  } else if (validatemarkers) {
    validate_markers (argc, argv);
  } else if (countwords) {
    count_words (argc, argv);
  } else if (countusfms) {
    count_usfms (argc, argv);
  } else if (countcharacters) {
    count_characters (argc, argv);
  } else if (checkchaptersverses) {
    check_chapters_verses (argc, argv);
  } else if (compareusfms) {
    compare_usfms (argc, argv);
  } else if (unwantedpatterns) {
    unwanted_patterns (argc, argv);
  } else if (unwantedwords) {
    unwanted_words (argc, argv);
  } else if (repeatingwords) {
    repeating_words (argc, argv);
  } else if (checkcapitalization) {
    check_capitalization (argc, argv);
  } else if (checkmatchingpairs) {
    check_matching_pairs (argc, argv);
  } else if (importversesystem) {
    import_verse_system (argc, argv);
  } else if (map) {
    map_ (argc, argv);
  } else if (test) {
    test_ (argc, argv);
  } else {
    scripting_usage (entries, false);
  }
  
  // Ready.
  return 0;
}


void scripting_usage (GOptionEntry entries[], bool subscript)
{
  int pointer = 0;
  cout << "Possible options:" << endl << endl;
  while (entries[pointer].long_name) {
    cout << "--" << entries[pointer].long_name;
    int length = strlen (entries[pointer].long_name);
    for (unsigned int i = length; i < 25; i++) cout << " ";
    cout << entries[pointer].description << endl;
    pointer++;
  }
  cout << endl;
  if (subscript) {
    ustring s = "Bibledit needs more information";
    gw_error (s.c_str());
  } else {
    cout << "Add --help to any of the above to get specific help, e.g. --" << entries[3].long_name << " --help" << endl << endl;
  }
}


ustring scripting_get_from_gchar (gchar * argument)
{
  ustring value;
  if (argument) { 
    value = argument; 
    g_free (argument);
  }
  return value;
}


void scripting_handle_error (GError * error)
{
  if (error) {
    ustring s = error->message;
    g_error_free (error); 
    gw_error (s.c_str()); 
  }
}


void scripting_ensure_project (const ustring& project)
{
  if (!project_exists (project)) {
    cerr << "Project " << project << " does not exist" << endl;
    exit (EXIT_FAILURE);
  }
}


void initialize_gui (bool gui, int argc, char *argv[])
{
  if (gui)
    gtk_init (&argc, &argv);
}


int export_stylesheet (int argc, char *argv[])
// Exports a stylesheet.
{
  // Parse options.
  ustring stylesheet;
  ustring file;
  gchar * g_stylesheet = NULL;
  gchar * g_file = NULL;
  GOptionEntry entries[] = 
  {
    { "stylesheet", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_stylesheet, "Name of the stylesheet to be exported", NULL },
    { "file", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_file, "Name of the file to write to", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  stylesheet = scripting_get_from_gchar (g_stylesheet);
  file = scripting_get_from_gchar (g_file);

  // Usage.
  if (stylesheet.empty() || file.empty()) scripting_usage (entries, true);

  // Check stylesheet.
  if (!stylesheet_exists (stylesheet)) {
    cerr << "Stylesheet " << stylesheet << " not found." << endl;
    return 1;
  }

  // Do the export.
  stylesheet_export (stylesheet, file);
  
  return 0;
}


int upgrade_data (int argc, char *argv[])
// Upgrades older data to the currently used formats.
{
  // Parse options.
  bool gui = false;
  GOptionEntry entries[] = 
  {
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Show graphical progressbar", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (gui, argc, argv);

  // Upgrade data.
  upgrade (gui);
  
  return 0;
}


int import_book (int argc, char *argv[])
// Imports a USFM file into Bibledit's database, with good error checking.
{
  // Parse options.
  ustring project;
  ustring file;
  ustring book;
  ustring encoding;  
  gchar * g_project = NULL;
  gchar * g_file = NULL;
  gchar * g_book = NULL;
  gchar * g_encoding = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project to import the book into", NULL },
    { "file", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_file, "Name of the USFM file which should be imported", NULL },
    { "book", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_book, "Optional: the name of the book, e.g. Matthew", NULL },
    { "encoding", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_encoding, "Optional: the encoding of the file, e.g. iso-8859-15", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  project = scripting_get_from_gchar (g_project);
  file = scripting_get_from_gchar (g_file);
  book = scripting_get_from_gchar (g_book);
  encoding = scripting_get_from_gchar (g_encoding);

  // Usage.
  if (project.empty() || file.empty()) scripting_usage (entries, true);

  // Check project.
  scripting_ensure_project (project);
    
  // Read the book.
  ImportBookRead ibr (file, encoding);
  ibr.usfm ();
  if (ibr.lines.size() == 0) {
    cerr << "Could not read file " << file << "." << endl;
    return 1;
  }
  
  // Handle bookname.
  if (book.empty()) {
    book = ibr.bookname;
  }
  if (book.empty()) {
    cerr << "Could not determine the bookname." << endl;
    return 1;
  }
  
  // Each line gets associated chapter and verse information.
  CategorizeChapterVerse ccv (ibr.lines);

  // Store it in the databases.
  project_store_book (project, books_english_to_id (book), ccv);

  // Success.  
  return 0;
}


int export_book (int argc, char *argv[])
// Exports a USFM file from Bibledit's database, with good error checking.
{
  // Parse options.
  ustring project;
  ustring book;
  ustring file;
  gchar * g_project = NULL;
  gchar * g_book = NULL;
  gchar * g_file = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project to be exported", NULL },
    { "book", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_book, "Name of the book, e.g. Matthew", NULL },
    { "file", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_file, "Name of the file to which should be exported", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  project = scripting_get_from_gchar (g_project);
  book = scripting_get_from_gchar (g_book);
  file = scripting_get_from_gchar (g_file);

  // Usage.
  if (project.empty() || file.empty() || book.empty()) 
    scripting_usage (entries, true);
  
  // Project is there?
  scripting_ensure_project (project);

  // Book is there?
  if (!project_book_exists (project, books_english_to_id (book))) {
    cerr << "Book " << book << " does not exist." << endl;
    return 1;
  }
  
  // Get and write book.
  vector <ustring> lines = project_retrieve_book (project, books_english_to_id (book));
  write_lines (file, lines);

  // Success.  
  return 0;
}


int shutdown_actions (int argc, char *argv[])
// Takes certain actions when Bibledit shuts down.
{
  // Parse options.
  GOptionEntry entries[] = 
  {
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);

  {
    // Open a configuration to read parameters from.
    GeneralConfiguration configuration (0);
    
    // Synchronize all projects on shutdown?
    if (configuration.synchronize_on_shutdown ()) {
      SynchronizeProjects sp (0);      
    }

    unsigned int startuptime = configuration.startup_time ();
    // Set about to vacuum the sqlite databases.
    // Vacuuming a database is done only when it got changed. Saves time.
    // Project related databases.
    vector <ustring> projects = projects_get_all ();
    for (unsigned int i = 0; i < projects.size(); i++) {
      project_vacuum (projects[i], startuptime);
      project_configuration_vacuum (projects[i], startuptime);
    }
    // Stylesheets.
    vector <ustring> stylesheets;
    stylesheet_get_ones_available (stylesheets);
    for (unsigned int i = 0; i < stylesheets.size(); i++) {
      stylesheet_vacuum (stylesheets[i], startuptime);
    }
    // Notes.
    notes_vacuum (startuptime);
  }

  // After configuration is no longer accessed, vacuum it.
  configuration_vacuum ();
  
  // Success.  
  return 0;
}


int export_project_usfm (int argc, char *argv[])
// Exports a project to USFM files, one per book
// Does good error checking.
{
  // Parse options.
  ustring project;
  ustring location;
  bool zip = false;
  bool gui = false;
  gchar * g_project = NULL;
  gchar * g_location = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project to be exported", NULL },
    { "location", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_location, "Where to store the output", NULL },
    { "zip", gchar (NULL), 0, G_OPTION_ARG_NONE, &zip, "Whether to zip the files", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Show graphical progressbar", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (gui, argc, argv);
  project = scripting_get_from_gchar (g_project);
  location = scripting_get_from_gchar (g_location);

  // Usage.
  if (project.empty() || location.empty()) scripting_usage (entries, true);

  // Check project.
  scripting_ensure_project (project);

  // (Temporal) output directory.
  ustring tempdir = gw_build_filename (directories_get_temp(), "usfm-export");
  remove_directory (tempdir);
  create_directory (tempdir);
  if (!zip) create_directory (location);
    
  // Get books.
  vector <unsigned int> books = project_get_books (project);

  // Progress information.
  ProgressWindow * progresswindow = NULL;
  if (gui) {
    progresswindow = new ProgressWindow ("Exporting project", true);
    progresswindow->set_iterate (0, 1, books.size());
  }

  // Export all books to usfm.
  for (unsigned int i = 0; i < books.size(); i++) {
    // Progress info.
    if (gui) {
      progresswindow->iterate ();
      if (progresswindow->cancel)
        return 1;
    }
    vector <ustring> lines = project_retrieve_book (project, books[i]);
    ustring filename = books_id_to_english (books[i]) + ".usfm";
    if (zip) filename = gw_build_filename (tempdir, filename);
    else filename = gw_build_filename (location, filename);
    write_lines (filename, lines);
  }

  // Zip them?
  if (zip) {
    if (!g_str_has_suffix (location.c_str(), ".zip")) location.append (".zip");
    unlink (location.c_str());
    ustring command = "cd" + shell_quote_space (tempdir) + "; zip -r zip.zip *.usfm; mv zip.zip" + shell_quote_space (location);
    system (command.c_str());
  }
  
  // Destroy window.
  if (progresswindow)
    delete progresswindow;

  // Success.  
  return 0;
}


int export_project_sword (int argc, char *argv[])
// Exports a project to a file fit for compiling by the Sword compiler.
// Does good error checking.
{
  // Parse options.
  ustring project;
  bool gui = false;
  ustring directory;
  gchar * g_project = NULL;
  gchar * g_directory = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project to be exported", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Show graphical progressbar", NULL },
    { "directory", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_directory, "Optional: where to store the module", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (gui, argc, argv);
  project = scripting_get_from_gchar (g_project);
  directory = scripting_get_from_gchar (g_directory);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Check project.
  scripting_ensure_project (project);

  // Take the action.
  export_to_sword_script (project, directory, gui);
  
  // Success.  
  return 0;
}


int export_project_opendocument (int argc, char *argv[])
// Exports a project to an OpenDocument text file.
{
  // Parse options.
  ustring project;
  bool gui = false;
  ustring filename;
  gchar * g_project = NULL;
  gchar * g_filename = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project to be exported", NULL },
    { "file", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_filename, "Where to save the file", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Show graphical progressbar", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (gui, argc, argv);
  project = scripting_get_from_gchar (g_project);
  filename = scripting_get_from_gchar (g_filename);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);
  if (filename.empty()) scripting_usage (entries, true);

  // Check project.
  scripting_ensure_project (project);

  // Take the action.
  OpenDocument odt (project, filename, gui);
  
  // Success.  
  return 0;
}


int general_configuration (int argc, char *argv[])
// Reads or writes to the general configuration.
{
  // Parse options.
  bool read;
  bool write;
  bool reset;
  ustring key;
  ustring value;
  gchar * g_key = NULL;
  gchar * g_value = NULL;
  GOptionEntry entries[] = 
  {
    { "read", gchar (NULL), 0, G_OPTION_ARG_NONE, &read, "Read a value from the configuration", NULL },
    { "write", gchar (NULL), 0, G_OPTION_ARG_NONE, &write, "Write a value to the configuration", NULL },
    { "reset", gchar (NULL), 0, G_OPTION_ARG_NONE, &reset, "Reset the configuration to default values", NULL },
    { "key", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_key, "When reading/writing: The key to select the value", NULL },
    { "value", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_value, "When writing: The value to be stored", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  key = scripting_get_from_gchar (g_key);
  value = scripting_get_from_gchar (g_value);

  // Usage.
  if (read && key.empty()) scripting_usage (entries, true);
  if (write && (key.empty() || value.empty())) scripting_usage (entries, true);
  if (!read && !write && !reset) scripting_usage (entries, true);

  if (read) {
    GeneralConfiguration configuration (0);
    cout << configuration.value_get (key, "") << endl;
  } else if (write) {
    GeneralConfiguration configuration (0);
    configuration.value_set (key, value);
  } else if (reset) {
    unlink (configuration_filename ().c_str());
  }
  
  // Ready.
  return 0;
}


int project_configuration (int argc, char *argv[])
// Reads or writes to the project configuration.
{
  // Parse options.
  bool read;
  bool write;
  bool reset;
  ustring project;
  ustring key;
  ustring value;
  gchar * g_project = NULL;
  gchar * g_key = NULL;
  gchar * g_value = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Which project's configuration to access", NULL },
    { "read", gchar (NULL), 0, G_OPTION_ARG_NONE, &read, "Read a value from the configuration", NULL },
    { "write", gchar (NULL), 0, G_OPTION_ARG_NONE, &write, "Write a value to the configuration", NULL },
    { "reset", gchar (NULL), 0, G_OPTION_ARG_NONE, &reset, "Reset the configuration to default values", NULL },
    { "key", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_key, "When reading/writing: The key to select the value", NULL },
    { "value", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_value, "When writing: The value to be stored", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  project = scripting_get_from_gchar (g_project);
  key = scripting_get_from_gchar (g_key);
  value = scripting_get_from_gchar (g_value);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);
  if (read && key.empty()) scripting_usage (entries, true);
  if (write && (key.empty() || value.empty())) scripting_usage (entries, true);
  if (!read && !write && !reset) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  if (read) {
    ProjectConfiguration projectconfig (project);
    cout << projectconfig.value_get ((char *) key.c_str(), "") << endl;
  } else if (write) {
    ProjectConfiguration projectconfig (project);
    projectconfig.value_set ((char *) key.c_str(), value);
  } else if (reset) {
    unlink (project_configuration_filename (project).c_str());
  }
  
  // Ready.
  return 0;
}


int create_project (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project to create", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  project = scripting_get_from_gchar (g_project);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);
    
  // Execute task.
  if (project_exists (project)) {
    cerr << "Project already exists" << endl;
    return 1;
  }
  project_create (project);
  return 0;  
}


int delete_project (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project to delete", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  project = scripting_get_from_gchar (g_project);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Execute task.
  project_delete (project);
  return 0;  
}


int validate_markers (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  bool gui = false;
  bool checksheet = false;
  gchar * g_project = NULL;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Optional: Show graphical progressbar", NULL },
    { "check-stylesheet", gchar (NULL), 0, G_OPTION_ARG_NONE, &checksheet, "Optional: Are the markers in the stylesheet?", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (gui, argc, argv);
  project = scripting_get_from_gchar (g_project);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  // Execute task.
  vector <unsigned int> books;
  CheckValidateUsfm check (project, books, gui, checksheet);
  checks_output_references_comments (check.references, check.comments);

  return 0;  
}


int count_words (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  bool sortword = false;
  bool sortcount = false;
  int excludecount = 0;
  ustring extracharacters;
  gchar * g_extracharacters = NULL;
  bool gui = false;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project", NULL },
    { "sort-word", gchar (NULL), 0, G_OPTION_ARG_NONE, &sortword, "Optional: Sort on word", NULL },
    { "sort-count", gchar (NULL), 0, G_OPTION_ARG_NONE, &sortcount, "Optional: Sort on count", NULL },
    { "exclude-count", gchar (NULL), 0, G_OPTION_ARG_INT, &excludecount, "Optional: Exclude words with a count of n or higher", NULL },
    { "extra-characters", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_extracharacters, "Optional: Extra characters that are part of a word", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Optional: Show graphical progressbar", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (true, argc, argv);
  project = scripting_get_from_gchar (g_project);
  extracharacters = scripting_get_from_gchar (g_extracharacters);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  // Sort on only one thing.
  if (sortword && sortcount) {
    cout << "Cant't sort on more than one key" << endl;
    return 1;
  }
  
  // Execute task.
  vector <unsigned int> books;
  CheckCountWords check (project, books, extracharacters, sortword, sortcount, excludecount, gui);
  checks_output_two_columns (check.words, check.counts);

  return 0;  
}


int count_usfms (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  bool sortmarker = false;
  bool sortcount = false;
  bool gui = false;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project", NULL },
    { "sort-marker", gchar (NULL), 0, G_OPTION_ARG_NONE, &sortmarker, "Optional: Sort on marker", NULL },
    { "sort-count", gchar (NULL), 0, G_OPTION_ARG_NONE, &sortcount, "Optional: Sort on count", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Optional: Show graphical progressbar", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (true, argc, argv);
  project = scripting_get_from_gchar (g_project);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  // Execute task.
  vector <unsigned int> books;
  CheckSortType sorttype = cstSort0;
  if (sortmarker) sorttype = cstSort1;
  if (sortcount) sorttype = cstSort2;
  CheckCountUsfms check (project, books, sorttype, gui);
  checks_output_two_columns (check.markers, check.counts);

  return 0;  
}


int count_characters (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  bool sortcharacter = false;
  bool sortcount = false;
  bool gui = false;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project", NULL },
    { "sort-character", gchar (NULL), 0, G_OPTION_ARG_NONE, &sortcharacter, "Optional: Sort on character", NULL },
    { "sort-count", gchar (NULL), 0, G_OPTION_ARG_NONE, &sortcount, "Optional: Sort on count", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Optional: Show graphical progressbar", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (true, argc, argv);
  project = scripting_get_from_gchar (g_project);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  // Execute task.
  vector <unsigned int> books;
  CheckCountCharacters check (project, books, sortcharacter, sortcount, gui);
  checks_output_two_columns (check.characters, check.counts);
  return 0;  
}


int check_chapters_verses (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  bool gui = false;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Optional: Show graphical progressbar", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (true, argc, argv);
  project = scripting_get_from_gchar (g_project);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  // Execute task.
  vector <unsigned int> books;
  CheckChaptersVerses check (project, books, gui);
  checks_output_references_comments (check.references, check.comments);
  return 0;  
}


int compare_usfms (int argc, char *argv[])
{
  // Parse options.
  ustring project1;
  gchar * g_project1 = NULL;
  ustring project2;
  gchar * g_project2 = NULL;
  bool allmarkers = false;
  ustring includeonly;
  gchar * g_includeonly = NULL;
  ustring ignore;
  gchar * g_ignore = NULL;
  bool ignoreverse0 = false;
  bool gui = false;
  GOptionEntry entries[] = 
  {
    { "project1", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project1, "Name of first project", NULL },
    { "project2", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project2, "Name of second project", NULL },
    { "all-markers", gchar (NULL), 0, G_OPTION_ARG_NONE, &allmarkers, "Optional: Check all the markers", NULL },
    { "include-only", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_includeonly, "Optional: Only include markers given", NULL },
    { "ignore", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_ignore, "Optional: Markers to ignore", NULL },
    { "ignore-verse-0", gchar (NULL), 0, G_OPTION_ARG_NONE, &ignoreverse0, "Optional: Ignore verse 0", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Optional: Show graphical progressbar", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (true, argc, argv);
  project1 = scripting_get_from_gchar (g_project1);
  project2 = scripting_get_from_gchar (g_project2);
  includeonly = scripting_get_from_gchar (g_includeonly);
  ignore = scripting_get_from_gchar (g_ignore);

  // Usage.
  if (project1.empty()) scripting_usage (entries, true);
  if (project2.empty()) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project1);
  scripting_ensure_project (project2);

  // Execute task.
  vector <unsigned int> books;
  CheckCompareUsfms check (project1, project2, books, gui, allmarkers, includeonly, ignore, ignoreverse0);
  checks_output_references_comments (check.references, check.comments);
  return 0;  
}


int unwanted_patterns (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  ustring file;
  gchar * g_file = NULL;
  bool gui = false;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project", NULL },
    { "file", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_file, "Optional: File with patterns", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Optional: Show graphical progressbar", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (true, argc, argv);
  project = scripting_get_from_gchar (g_project);
  file = scripting_get_from_gchar (g_file);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  // Execute task.
  vector <unsigned int> books;
  CheckUnwantedPatterns check (project, books, file, gui);
  checks_output_references_comments (check.references, check.comments);
  return 0;  
}


int unwanted_words (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  ustring file;
  gchar * g_file = NULL;
  bool gui = false;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project", NULL },
    { "file", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_file, "Optional: File with words", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Optional: Show graphical progressbar", NULL },
    { NULL }
  };  
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (true, argc, argv);
  project = scripting_get_from_gchar (g_project);
  file = scripting_get_from_gchar (g_file);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  // Execute task.
  vector <unsigned int> books;
  CheckUnwantedWords check (project, books, file, gui);
  checks_output_references_comments (check.references, check.comments);
  return 0;  
}


int repeating_words (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  bool ignorecase = false;
  ustring onlythese;
  gchar * g_onlythese = NULL;
  ustring ignorethese;
  gchar * g_ignorethese = NULL;
  bool gui = false;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project", NULL },
    { "ignore-case", gchar (NULL), 0, G_OPTION_ARG_NONE, &ignorecase, "Optional: Whether to ignore case", NULL },
    { "only-these", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_onlythese, "Optional: File with words to only include", NULL },
    { "ignore-these", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_ignorethese, "Optional: File with words to ignore", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Optional: Show graphical progressbar", NULL },
    { NULL }
  };
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (true, argc, argv);
  project = scripting_get_from_gchar (g_project);
  onlythese = scripting_get_from_gchar (g_onlythese);
  ignorethese = scripting_get_from_gchar (g_ignorethese);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  // Execute task.
  vector <unsigned int> books;
  CheckRepetition check (project, books, ignorecase, onlythese, ignorethese, gui);
  checks_output_references_comments (check.references, check.comments);
  return 0;  
}


int check_capitalization (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  ustring punctuation_followed_by_capitals;
  gchar * g_punctuation_followed_by_capitals = NULL;
  ustring ignore_lower_case_following;
  gchar * g_ignore_lower_case_following = NULL;
  ustring abbreviations;
  gchar * g_abbreviations = NULL;
  bool anyprefixes = false;
  ustring uncapitalizedprefixes;
  gchar * g_uncapitalizedprefixes = NULL;
  ustring capitalizedsuffixes;
  gchar * g_capitalizedsuffixes = NULL;
  bool gui = false;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project", NULL },
    { "pfbc", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_punctuation_followed_by_capitals, "Optional: Punctuation followed by capitals", NULL },
    { "ilcf", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_ignore_lower_case_following, "Optional: Ignore lower case following these", NULL },
    { "abbreviations", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_abbreviations, "Optional: File with abbreviations", NULL },
    { "any-prefixes", gchar (NULL), 0, G_OPTION_ARG_NONE, &anyprefixes, "Optional: Allow anhy prefixes", NULL },
    { "uncapitalized-prefixes", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_uncapitalizedprefixes, "Optional: File with uncapitalized prefixes", NULL },
    { "capitalized-suffixes", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_capitalizedsuffixes, "Optional: File with capitalized suffixes", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Optional: Show graphical progressbar", NULL },
    { NULL }
  };
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (true, argc, argv);
  project = scripting_get_from_gchar (g_project);
  punctuation_followed_by_capitals = scripting_get_from_gchar (g_punctuation_followed_by_capitals);
  ignore_lower_case_following = scripting_get_from_gchar (g_ignore_lower_case_following);
  abbreviations = scripting_get_from_gchar (g_abbreviations);
  uncapitalizedprefixes = scripting_get_from_gchar (g_uncapitalizedprefixes);
  capitalizedsuffixes = scripting_get_from_gchar (g_capitalizedsuffixes);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  // Execute task.
  vector <unsigned int> books;
  CheckCapitalization check (project, books, 
                             punctuation_followed_by_capitals,
                             ignore_lower_case_following,
                             abbreviations,
                             anyprefixes,
                             uncapitalizedprefixes,
                             capitalizedsuffixes,
                             gui);
  checks_output_references_comments (check.references, check.comments);
  return 0;  
}


int check_matching_pairs (int argc, char *argv[])
{
  // Parse options.
  ustring project;
  gchar * g_project = NULL;
  ustring ignore;
  gchar * g_ignore = NULL;
  bool gui = false;
  GOptionEntry entries[] = 
  {
    { "project", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_project, "Name of project", NULL },
    { "ignore", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_ignore, "Optional: Pairs to ignore", NULL },
    { "gui", gchar (NULL), 0, G_OPTION_ARG_NONE, &gui, "Optional: Show graphical progressbar", NULL },
    { NULL }
  };
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (true, argc, argv);
  project = scripting_get_from_gchar (g_project);
  ignore = scripting_get_from_gchar (g_ignore);

  // Usage.
  if (project.empty()) scripting_usage (entries, true);

  // Project is there?
  scripting_ensure_project (project);

  // Execute task.
  vector <unsigned int> books;
  CheckMatchingPairs check (project, books, ignore, gui);
  checks_output_references_comments (check.references, check.comments);
  return 0;  
}


int import_verse_system (int argc, char *argv[])
{
  // Parse options.
  ustring name;
  gchar * g_name = NULL;
  ustring versification;
  gchar * g_versification = NULL;
  ustring mapping;
  gchar * g_mapping = NULL;
  GOptionEntry entries[] = 
  {
    { "name", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_name, "Name of the verse system", NULL },
    { "versification", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_versification, "Versification textfile", NULL },
    { "mapping", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_mapping, "Mapping textfile", NULL },
    { NULL }
  };
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (true, argc, argv);
  name = scripting_get_from_gchar (g_name);
  versification = scripting_get_from_gchar (g_versification);
  mapping = scripting_get_from_gchar (g_mapping);

  if (name.empty() || versification.empty() || mapping.empty()) 
    scripting_usage (entries, true);

  versification_import_textfile (name, versification);
  mapping_import_textfile (name, mapping);
  
  return 0;  
}


int map_ (int argc, char *argv[])
{
  // Parse options.
  ustring name;
  gchar * g_name = NULL;
  ustring book;
  gchar * g_book = NULL;
  int chapter = 0;
  ustring verse;
  gchar * g_verse = NULL;
  bool to_original = false;
  bool to_me = false;
  GOptionEntry entries[] = 
  {
    { "name", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_name, "Versification system", NULL },
    { "book", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_book, "Book", NULL },
    { "chapter", gchar (NULL), 0, G_OPTION_ARG_INT, &chapter, "Chapter", NULL },
    { "verse", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_verse, "Verse", NULL },
    { "to-original", gchar (NULL), 0, G_OPTION_ARG_NONE, &to_original, "Map data to original versification", NULL },
    { "to-me", gchar (NULL), 0, G_OPTION_ARG_NONE, &to_me, "Map data to my versification", NULL },
    { NULL }
  };
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  initialize_gui (true, argc, argv);
  name = scripting_get_from_gchar (g_name);
  book = scripting_get_from_gchar (g_book);
  verse = scripting_get_from_gchar (g_verse);

  if ((name.empty()) || book.empty() || (chapter == 0) || (verse.empty()) || (to_original == to_me))
    scripting_usage (entries, true);

  Mapping mapping (name, books_english_to_id (book));
  vector <int> chapters;
  vector <int> verses;
  if (to_original) mapping.me_to_original (chapter, verse, chapters, verses);
  if (to_me) mapping.original_to_me (chapter, verse, chapters, verses);
  for (unsigned int i = 0; i < chapters.size(); i++) {
    cout << book << " " << chapters[i] << ":" << verses[i] << endl;
  }    
    
  return 0;  
}


int test_ (int argc, char *argv[])
{
  // Parse options.
  int testnumber = 0;
  ustring string1;
  gchar * g_string1 = NULL;
  ustring string2;
  gchar * g_string2 = NULL;
  ustring string3;
  gchar * g_string3 = NULL;
  int integer1 = 0;
  int integer2 = 0;
  int integer3 = 0;
  bool boolean1 = false;
  bool boolean2 = false;
  bool boolean3 = false;
  bool boolean4 = false;
  bool boolean5 = false;
  GOptionEntry entries[] = 
  {
    { "test-number", gchar (NULL), 0, G_OPTION_ARG_INT, &testnumber, "Test number", NULL },
    { "string1", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_string1, "String 1", NULL },
    { "string2", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_string2, "String 2", NULL },
    { "string3", gchar (NULL), 0, G_OPTION_ARG_STRING, &g_string3, "String 3", NULL },
    { "integer1", gchar (NULL), 0, G_OPTION_ARG_INT, &integer1, "Integer 1", NULL },
    { "integer2", gchar (NULL), 0, G_OPTION_ARG_INT, &integer2, "Integer 2", NULL },
    { "integer3", gchar (NULL), 0, G_OPTION_ARG_INT, &integer3, "Integer 3", NULL },
    { "boolean1", gchar (NULL), 0, G_OPTION_ARG_NONE, &boolean1, "Boolean 1", NULL },
    { "boolean2", gchar (NULL), 0, G_OPTION_ARG_NONE, &boolean2, "Boolean 2", NULL },
    { "boolean3", gchar (NULL), 0, G_OPTION_ARG_NONE, &boolean3, "Boolean 3", NULL },
    { "boolean4", gchar (NULL), 0, G_OPTION_ARG_NONE, &boolean4, "Boolean 4", NULL },
    { "boolean5", gchar (NULL), 0, G_OPTION_ARG_NONE, &boolean5, "Boolean 5", NULL },
    { NULL }
  };
  GError *error = NULL;
  GOptionContext * context = g_option_context_new ("");
  g_option_context_add_main_entries (context, entries, NULL);
  g_option_context_parse (context, &argc, &argv, &error);  
  g_option_context_free (context);
  scripting_handle_error (error);
  string1 = scripting_get_from_gchar (g_string1);
  string2 = scripting_get_from_gchar (g_string2);
  string3 = scripting_get_from_gchar (g_string3);

  ustring project ("__test__");
  
  switch (testnumber) {
    case 1: 
    {
      scripting_ensure_project (project);
      // affectfootnotes = boolean1
      // affectendnotes = boolean2
      // affectxrefs = boolean3
      // action = integer1
      // includechapter = boolean4
      // separator = string1
      // includeverse = boolean5
      // suffix = string2
      OriginReferences originreferences (project, boolean1, boolean2, boolean3, (OriginReferencesActionType) integer1, boolean4, string1, boolean5, string2, false);
      break;      
    }
    default :
    {
      scripting_usage (entries, true);
    }
  }
    
  return 0;  
}
