/*
 * Bickley - a meta data management framework.
 * Copyright © 2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <stdlib.h>

#include "bkl-entry.h"
#include "bkl-item-extended.h"
#include "bkl-utils.h"

enum {
    PROP_0,
    PROP_THUMBNAIL,
    PROP_LAST_USED,
    PROP_USE_COUNT,
    PROP_RATING,
    PROP_PINNED,
    PROP_DATE_INDEXED,
    PROP_TAGS
};

struct _BklItemExtendedPrivate {
    char *thumbnail;
    glong last_used;
    glong use_count;
    glong date_indexed;
    int rating;
    gboolean pinned;
    GPtrArray *tags;
};

#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), BKL_TYPE_ITEM_EXTENDED, BklItemExtendedPrivate))
G_DEFINE_TYPE (BklItemExtended, bkl_item_extended, BKL_TYPE_ITEM);

static void
free_string_array (GPtrArray *array)
{
    int i;

    for (i = 0; i < array->len; i++) {
        g_free (array->pdata[i]);
    }

    g_ptr_array_free (array, TRUE);
}

static void
bkl_item_extended_finalize (GObject *object)
{
    BklItemExtended *self = (BklItemExtended *) object;
    BklItemExtendedPrivate *priv = self->priv;

    if (priv->tags) {
        g_print ("Priv->tags: %p\n", priv->tags);
        free_string_array (priv->tags);
        priv->tags = NULL;
    }

    g_signal_handlers_destroy (object);
    G_OBJECT_CLASS (bkl_item_extended_parent_class)->finalize (object);
}

static void
bkl_item_extended_dispose (GObject *object)
{
    G_OBJECT_CLASS (bkl_item_extended_parent_class)->dispose (object);
}

static void
bkl_item_extended_set_property (GObject      *object,
                                guint         prop_id,
                                const GValue *value,
                                GParamSpec   *pspec)
{
    switch (prop_id) {
    case PROP_THUMBNAIL:
        break;

    case PROP_LAST_USED:
        break;

    case PROP_USE_COUNT:
        break;

    case PROP_RATING:
        break;

    case PROP_PINNED:
        break;

    case PROP_DATE_INDEXED:
        break;

    case PROP_TAGS:
        break;

    default:
        break;
    }
}

static void
bkl_item_extended_get_property (GObject    *object,
                                guint       prop_id,
                                GValue     *value,
                                GParamSpec *pspec)
{
    switch (prop_id) {
    case PROP_THUMBNAIL:
        break;

    case PROP_LAST_USED:
        break;

    case PROP_USE_COUNT:
        break;

    case PROP_RATING:
        break;

    case PROP_PINNED:
        break;

    case PROP_DATE_INDEXED:
        break;

    case PROP_TAGS:
        break;

    default:
        break;
    }
}

static void
bkl_item_extended_class_init (BklItemExtendedClass *klass)
{
    GObjectClass *o_class = (GObjectClass *)klass;

    o_class->dispose = bkl_item_extended_dispose;
    o_class->finalize = bkl_item_extended_finalize;
    o_class->set_property = bkl_item_extended_set_property;
    o_class->get_property = bkl_item_extended_get_property;

    g_type_class_add_private (klass, sizeof (BklItemExtendedPrivate));

    g_object_class_install_property (o_class, PROP_THUMBNAIL,
                                     g_param_spec_string ("thumbnail",
                                                          "", "", "",
                                                          G_PARAM_READWRITE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_LAST_USED,
                                     g_param_spec_long ("last-used", "", "",
                                                        0, G_MAXLONG, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_USE_COUNT,
                                     g_param_spec_long ("use-count", "", "",
                                                        0, G_MAXLONG, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_RATING,
                                     g_param_spec_int ("rating", "", "",
                                                       0, G_MAXINT, 0,
                                                       G_PARAM_READWRITE |
                                                       G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_PINNED,
                                     g_param_spec_boolean ("pinned", "", "",
                                                           FALSE,
                                                           G_PARAM_READWRITE |
                                                           G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_DATE_INDEXED,
                                     g_param_spec_long ("date-indexed", "", "",
                                                        0, G_MAXLONG, 0,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_TAGS,
                                     g_param_spec_pointer ("tags", "", "",
                                                           G_PARAM_READWRITE |
                                                           G_PARAM_STATIC_STRINGS));
}

static void
bkl_item_extended_init (BklItemExtended *self)
{
    BklItemExtendedPrivate *priv;

    priv = self->priv = GET_PRIVATE (self);
}

#define IS_EMPTY_STRING(s) ((s) == NULL || *(s) == '\0')
void
bkl_item_extended_set_from_field (BklItemExtended *item,
                                  KozoField       *field)
{
    const char *data;
    int number;
    glong l;

    data = kozo_field_get_value_string (field, BKL_EXTENDED_FIELD_THUMBNAIL);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_extended_set_thumbnail (item, data);
    }

    data = kozo_field_get_value_string (field, BKL_EXTENDED_FIELD_LAST_USED);
    if (!IS_EMPTY_STRING (data)) {
        l = strtol (data, NULL, 10);
        bkl_item_extended_set_last_used (item, l);
    }

    data = kozo_field_get_value_string (field, BKL_EXTENDED_FIELD_USE_COUNT);
    if (!IS_EMPTY_STRING (data)) {
        l = strtol (data, NULL, 10);
        bkl_item_extended_set_use_count (item, l);
    }

    data = kozo_field_get_value_string (field, BKL_EXTENDED_FIELD_RATING);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_extended_set_rating (item, number);
    }

    data = kozo_field_get_value_string (field, BKL_EXTENDED_FIELD_PINNED);
    if (!IS_EMPTY_STRING (data)) {
        number = atoi (data);
        bkl_item_extended_set_pinned (item, number == 0 ? FALSE : TRUE);
    }

    data = kozo_field_get_value_string (field, BKL_EXTENDED_FIELD_DATE_INDEXED);
    if (!IS_EMPTY_STRING (data)) {
        l = strtol (data, NULL, 10);
        bkl_item_extended_set_date_indexed (item, l);
    }

    data = kozo_field_get_value_string (field, BKL_EXTENDED_FIELD_TAGS);
    if (!IS_EMPTY_STRING (data)) {
        bkl_item_extended_set_tags (item, _bkl_string_to_string_array (data));
    }
}

GSList *
bkl_item_extended_get_fields (BklItemExtended *item)
{
    BklItemExtendedPrivate *priv = item->priv;
    GSList *fields = NULL;

    fields = g_slist_prepend (fields,
                              _bkl_string_array_to_string (priv->tags));
    fields = g_slist_prepend (fields, g_strdup_printf ("%ld", priv->date_indexed));
    fields = g_slist_prepend (fields, g_strdup_printf ("%d", priv->pinned));
    fields = g_slist_prepend (fields, g_strdup_printf ("%d", priv->rating));
    fields = g_slist_prepend (fields, g_strdup_printf ("%ld", priv->use_count));
    fields = g_slist_prepend (fields, g_strdup_printf ("%ld", priv->last_used));
    fields = g_slist_prepend (fields, g_strdup (priv->thumbnail));

    return fields;
}

void
bkl_item_extended_set_thumbnail (BklItemExtended *item,
                                 const char      *thumbnail)
{
    g_free (item->priv->thumbnail);
    item->priv->thumbnail = g_strdup (thumbnail);
}

const char *
bkl_item_extended_get_thumbnail (BklItemExtended *item)
{
    return item->priv->thumbnail;
}

void
bkl_item_extended_set_last_used (BklItemExtended *item,
                                 glong            last_used)
{
    item->priv->last_used = last_used;
}

glong
bkl_item_extended_get_last_used (BklItemExtended *item)
{
    return item->priv->last_used;
}

void
bkl_item_extended_set_use_count (BklItemExtended *item,
                                 glong            use_count)
{
    item->priv->use_count = use_count;
}

glong
bkl_item_extended_get_use_count (BklItemExtended *item)
{
    return item->priv->use_count;
}

void
bkl_item_extended_set_rating (BklItemExtended *item,
                              int              rating)
{
    item->priv->rating = rating;
}

int
bkl_item_extended_get_rating (BklItemExtended *item)
{
    return item->priv->rating;
}

void
bkl_item_extended_set_pinned (BklItemExtended *item,
                              gboolean         pinned)
{
    item->priv->pinned = pinned;
}

gboolean
bkl_item_extended_get_pinned (BklItemExtended *item)
{
    return item->priv->pinned;
}

void
bkl_item_extended_set_tags (BklItemExtended *item,
                            GPtrArray       *tags)
{
    if (item->priv->tags) {
        free_string_array (item->priv->tags);
    }
    item->priv->tags = tags;
}

GPtrArray *
bkl_item_extended_get_tags (BklItemExtended *item)
{
    return item->priv->tags;
}

void
bkl_item_extended_set_date_indexed (BklItemExtended *item,
                                    glong            date_indexed)
{
    item->priv->date_indexed = date_indexed;
}

glong
bkl_item_extended_get_date_indexed (BklItemExtended *item)
{
    return item->priv->date_indexed;
}
