/**
 * @namespace   biewlib
 * @file        biewlib/sysdep/ia16/cpu_info.c
 * @brief       This file contains function for retrieving CPU information for
 *              16-bits Intel x86 compatible platform
 * @version     -
 * @remark      this source file is part of Binary vIEW project (BIEW).
 *              The Binary vIEW (BIEW) is copyright (C) 1995 Nick Kurshev.
 *              All rights reserved. This software is redistributable under the
 *              licence given in the file "Licence.en" ("Licence.ru" in russian
 *              translation) distributed in the BIEW archive.
 * @note        Requires POSIX compatible development system
 * @remark      I used such form of this file for compatibility with ia32
 *              strategy. (See remarks for biewlib/sysdep/ia32/cpu_info.c). If
 *              somebody want ported it under other 16-bits compilers (like
 *              Watcom, Borland, MSC or other), then he can use preprocessor
 *              directives: #ifdef #endif for expanding this file. But I think
 *              too few peoples have'll interest to 16-bits systems. Their time
 *              is over.
 *
 * @author      Nick Kurshev
 * @since       1995
 * @note        Development, fixes and improvements
**/
#include <stdio.h>
#include <string.h>

#include "biewlib/biewlib.h"

#ifndef __TSC__
 #error Sorry! This source file required TopSpeed C development system. Please port it.
#endif

#define CPU_CLONE     0x000F
#define __HAVE_FPU    0x8000
#define __HAVE_CPUID  0x4000
#define __HAVE_MMX    0x2000
#define __HAVE_SSE    0x1000

static unsigned __NEAR__ __FASTCALL__ __cpu_type( void ) =
{
   0x52,                     /* push     dx */
   0x53,                     /* push     bx */
   0x51,                     /* push     cx */
   0x54,                     /* push     sp */
   0x9C,                     /* pushf       */

   0xB8, 0xFF, 0xFF,         /* mov      ax, 0FFFFh */
   0xB1, 0x42,               /* mov      cl, 42h */
   0xD3, 0xE8,               /* shr      ax, cl */
   0xA9, 0xFF, 0x3F,         /* test     ax, 03FFFh */
   0x75, 0x05,               /* jnz      next1 ; it high than 8086 */
   0x33, 0xC0,               /* xor      ax, ax */
   0xE9, 0xA9, 0x00,         /* jmp near ret_label ; return 0 */
/*next1:*/
   0x54,                     /* push     sp */
   0x5B,                     /* pop      bx */
   0x39, 0xE3,               /* cmp      bx, sp */
   0x74, 0x06,               /* jnz      next2 ; it high than 80186 */
   0xB8, 0x01, 0x00,         /* mov      ax, 1 */
   0xE9, 0x9D, 0x00,         /* jmp near ret_label ; return 1 */
 /*next2:*/
   0xBB, 0x00, 0x70,         /* mov      bx, 7000h */
   0x53,                     /* push     bx */
   0x9D,                     /* popf */
   0x9C,                     /* pushf */
   0x5B,                     /* pop      bx */
   0x81, 0xE3, 0x00, 0x70,   /* and      bx, 7000h */
   0x75, 0x06,               /* jnz      next3 ; it high than 80286 */
   0xB8, 0x02, 0x00,         /* mov      ax, 2 */
   0xE9, 0x8A, 0x00,         /* jmp short ret_val ; return 2 */
/*next3:*/
   0x66, 0x51,               /* push     ecx */
   0x66, 0x9C,               /* pushfd */
   0x66, 0x89, 0xE2,         /* mov      edx, esp */
   0x66, 0x83, 0xE4, 0xFC,   /* and      esp, ~3 */
   0x66, 0x9C,               /* pushfd */
   0x66, 0x58,               /* pop      eax */
   0x66, 0x89, 0xC1,         /* mov      ecx, eax */
   0x66, 0x35, 0x00, 0x00, 0x04, 0x00, /* xor eax, 40000h */
   0x66, 0x50,               /* push     eax */
   0x66, 0x9D,               /* popfd */
   0x66, 0x9C,               /* pushfd */
   0x66, 0x58,               /* pop      eax */
   0x66, 0x31, 0xC8,         /* xor      eax, ecx */
   0x66, 0xC1, 0xE8, 0x12,   /* shr      eax, 12h */
   0x66, 0x83, 0xE0, 0x01,   /* and      eax, 1 */
   0x66, 0x51,               /* push     ecx */
   0x66, 0x9D,               /* popfd */
   0x66, 0x89, 0xD4,         /* mov      esp, edx */
   0x66, 0x3D, 0x00, 0x00, 0x00, 0x00, /* cmp eax, 0 */
   0x75, 0x05,               /* jnz     next4 ; it high than 80386 */
   0xB8, 0x03, 0x00,         /* mov     ax , 3 */
   0xEB, 0x47,               /* jmp short ret_label32; return 3 */
/*next4:*/
   0x66, 0x9C,               /* pushfd */
   0x66, 0x58,               /* pop     eax */
   0x66, 0x89, 0xC1,         /* mov     ecx, eax */
   0x66, 0x35, 0x00, 0x00, 0x20, 0x00, /* xor eax, 200000h */
   0x66, 0x50,               /* push    eax */
   0x66, 0x9D,               /* popfd */
   0x66, 0x9C,               /* pushfd */
   0x66, 0x58,               /* pop     eax */
   0x66, 0x31, 0xC8,         /* xor     eax, ecx */
   0x75, 0x05,               /* jnz next5 ; it high than 80486 */
   0xB8, 0x04, 0x00,         /* mov     ax, 4 */
   0xEB, 0x28,               /* jmp short ret_label32 ; return 4 */
/*next5:*/
   0x66, 0xB8, 0x01, 0x00, 0x00, 0x00, /* mov eax, 1*/
   0x0F, 0xA2,               /* cpuid */
   0x88, 0xE0,               /* mov     al, ah */
   0x83, 0xE0, 0x0F,         /* and     ax, 000Fh */
   0x0D, 0x00, 0x40,         /* or      ax, 4000h */
   0x66, 0xF7, 0xC2, 0x00, 0x00, 0x80, 0x00, /* test edx, 800000h */
   0x74, 0x03,               /* jz      test_SSE */
   0x0D, 0x00, 0x20,         /* or      ax, 2000h */
/*test_SSE:*/
   0x66, 0xF7, 0xC2, 0x00, 0x00, 0x00, 0x02, /* test edx, 2000000h */
   0x74, 0x03,               /* jz      ret_label32 */
   0x0D, 0x00, 0x10,         /* or      ax, 1000h */
/*ret_label32:*/
   0x66, 0x9D,               /* popfd */
   0x66, 0x59,               /* pop     ecx */
/*ret_label:*/
   0x9D,                     /* popf */
   0x5C,                     /* pop     sp */
   0x59,                     /* pop     cx */
   0x5B,                     /* pop     bx */
   0x5A,                     /* pop     dx */
   0xC3                      /* ret near 0 */
};

/* buff pass through bx:ax */
static void __NEAR__ __FASTCALL__ __cpu_name(char *buff) =
{
  0x06,                      /* push    es */
  0x60,                      /* pusha */
  0x8E, 0xC3,                /* mov     es, bx */
  0x89, 0xC7,                /* mov     di, ax */
  0x66, 0x33, 0xC0,          /* xor     eax, eax */
  0x0F, 0xA2,                /* cpuid */
  0x66, 0x89, 0xD8,          /* mov     eax, ebx */
  0x66, 0xAB,                /* stosd */
  0x66, 0x89, 0xD0,          /* mov     eax, edx */
  0x66, 0xAB,                /* stosd */
  0x66, 0x89, 0xC8,          /* mov     eax, ecx */
  0x66, 0xAB,                /* stosd */
  0x30, 0xC0,                /* xor     al, al */
  0xAA,                      /* stosb */
  0x61,                      /* popa */
  0x07,                      /* pop     es */
  0xC3                       /* ret near 0 */
};

static void __NEAR__ __FASTCALL__ __extended_name(char *buff) =
{
  0x06,                      /* push es */
  0x60,                      /* pusha */
  0x8E, 0xC3,                /* mov  es, bx */
  0x89, 0xC7,                /* mov  di, ax */
  0x66, 0xB8, 0x02, 0x00, 0x00, 0x80, /* mov eax, 80000002h */
  0x0F, 0xA2,                /* cpuid */
  0x66, 0xAB,                /* stosd */
  0x66, 0x89, 0xD8,          /* mov  eax, ebx */
  0x66, 0xAB,                /* stosd */
  0x66, 0x89, 0xC8,          /* mov  eax, ecx */
  0x66, 0xAB,                /* stosd */
  0x66, 0x89, 0xD0,          /* mov  eax, edx */
  0x66, 0xAB,                /* stosd */
  0x66, 0xB8, 0x03, 0x00, 0x00, 0x80, /* mov eax, 80000003h */
  0x0F, 0xA2,                /* cpuid */
  0x66, 0xAB,                /* stosd */
  0x66, 0x89, 0xD8,          /* mov  eax, edx */
  0x66, 0xAB,                /* stosd */
  0x66, 0x89, 0xC8,          /* mov  eax, ecx */
  0x66, 0xAB,                /* stosd */
  0x66, 0x89, 0xD0,          /* mov  eax, edx */
  0x66, 0xAB,                /* stosd */
  0x66, 0xB8, 0x04, 0x00, 0x00, 0x80, /* mov eax, 80000004h */
  0x0F, 0xA2,                /* cpuid */
  0x66, 0xAB,                /* stosd */
  0x66, 0x89, 0xD8,          /* mov  eax, ebx */
  0x66, 0xAB,                /* stosd */
  0x66, 0x89, 0xC8,          /* mov  eax, ecx */
  0x66, 0xAB,                /* stosd */
  0x66, 0x89, 0xD0,          /* mov  eax, edx */
  0x66, 0xAB,                /* stosd */
  0x30, 0xC0,                /* xor  al, al */
  0xAA,                      /* stosb */
  0x61,                      /* popa */
  0x07,                      /* pop  es */
  0xC3                       /* ret near 0 */
};

/* arg pass through bx:ax */
/* On input __eax = cpuid function */
/* On output __eax filled. __edx returned as unsigned long */
static unsigned long  __NEAR__ __FASTCALL__ __cpuid_edx(unsigned long *__eax) =
{
   0x1E,                     /* push ds */
   0x06,                     /* push es */
   0x56,                     /* push si */
   0x57,                     /* push di */
   0x53,                     /* push bx */
   0x51,                     /* push cx */
   0x8E, 0xDB,               /* mov  ds, bx */
   0x8E, 0xC3,               /* mov  es, bx */
   0x89, 0xC6,               /* mov  si, ax */
   0x89, 0xC7,               /* mov  di, ax */
   0x66, 0xAD,               /* lodsd */
   0x0F, 0xA2,               /* cpuid */
   0x66, 0xAB,               /* stosd */
   0x89, 0xD0,               /* mov  ax, dx */
   0x66, 0xC1, 0xEA, 0x10,   /* shr  edx, 16 */
   0x59,                     /* pop  cx */
   0x5B,                     /* pop  bx */
   0x5F,                     /* pop  di */
   0x5E,                     /* pop  si */
   0x07,                     /* pop  es */
   0x1F,                     /* pop  ds */
   0xC3                      /* ret near 0 */
};

/* arg pass through bx:ax */
/* On input __eax = cpuid function */
/* On output __eax filled with __ecx and __ebx returned as unsigned long */
static unsigned long  __NEAR__ __FASTCALL__ __cpuid_ebxecx(unsigned long *__eax) =
{
   0x1E,                     /* push ds */
   0x06,                     /* push es */
   0x56,                     /* push si */
   0x57,                     /* push di */
   0x53,                     /* push bx */
   0x51,                     /* push cx */
   0x8E, 0xDB,               /* mov  ds, bx */
   0x8E, 0xC3,               /* mov  es, bx */
   0x89, 0xC6,               /* mov  si, ax */
   0x89, 0xC7,               /* mov  di, ax */
   0x66, 0xAD,               /* lodsd */
   0x0F, 0xA2,               /* cpuid */
   0x66, 0x8B, 0xC1,         /* mov  eax, ecx */
   0x66, 0xAB,               /* stosd */
   0x8B, 0xC3,               /* mov  ax, bx */
   0x66, 0xC1, 0xEB, 0x10,   /* shr  ebx, 16 */
   0x8B, 0xD3,               /* mov  dx, bx */
   0x59,                     /* pop  cx */
   0x5B,                     /* pop  bx */
   0x5F,                     /* pop  di */
   0x5E,                     /* pop  si */
   0x07,                     /* pop  es */
   0x1F,                     /* pop  ds */
   0xC3                      /* ret near 0 */
};

static unsigned __NEAR__ __FASTCALL__ __fpu_type( void ) =
{
  0x55,                      /* push bp */
  0x89, 0xE5,                /* mov  bp, sp */
  0x51,                      /* push cx */
  0x53,                      /* push bx */
  0x83, 0xEC, 0x02,          /* sub sp, 2 */

  0xDB, 0xE3,                /* fninit        ;initialize 8087 (no WAIT) */
  0xB9, 0x20, 0x00,          /* mov cx, 20h */
  0xE2, 0xFE,                /* L1: loop L1   ;wait for it to complete */
  0xD9, 0x7E, 0xF8,          /* fnstcw [bp-8] ;store control word */
  0xB9, 0x10, 0x00,          /* mov cx, 10h */
  0xE2, 0xFE,                /* L2: loop L2   ;wait for it to complete */
                             /* ;Determine if we have an 8087, 80287, or 80387 */
  0x31, 0xDB,                /* xor bx, bx    ;assume no NDP on return */
  0x8B, 0x46, 0xF8,          /* mov ax, [bp-8] */
  0x80, 0xE4, 0x0F,          /* and ah, 0Fh */
  0x80, 0xFC, 0x03,          /* cmp ah, 3 */
  0x75, 0x44,                /* jnz L6        ;if no 80x87 */
  0x9B, 0xD9, 0x7E, 0xF8,    /* fstcw [bp-8] */
  0x43,                      /* inc bx        ;could be 8087, 80287 or 80387 */
  0x9B,                      /* wait */
  0x83, 0x66, 0xF8, 0x7F,    /* and word [bp-8], 07Fh ;turn off interrupt mask bit */
  0xD9, 0x6E, 0xF8,          /* fldcw [bp-8] */
  0xDB, 0xE1,                /* fdisi         ;disable interrupts (works on 8087 only) */
  0x9B, 0xD9, 0x7E, 0xF8,    /* fstcw [bp-8] */
  0x9B,                      /* wait */
  0xF6, 0x46, 0xF8, 0x80,    /* test byte [bp-8],80h ;see if bit is back on */
  0x75, 0x2A,                /* jnz L6               ;yes, then 8087 */
  0x43,                      /* inc bx               ;287 or 387 */
  0x80, 0x4E, 0xF8, 0xBF,    /* or byte [bp-8],0BFh  ;disable interrupts, mask exceptions */
  0x81, 0x66, 0xF8, 0xFF, 0xEF,/* and word [bp-8],0EFFFh ;turn off infinity bit */
  0xD9, 0x6E, 0xF8,          /* fldcw [bp-8] */
  0xD9, 0xE8,                /* fld1 */
  0xD9, 0xEE,                /* fldz */
  0xDE, 0xF9,                /* fdivp st1, st0   ;divide by 0 to get infinity */
  0xD9, 0xC0,                /* fld st0 */
  0xD9, 0xE0,                /* fchs st0         ;create +infinity and -infinity */
  0xDE, 0xD9,                /* fcompp           ;and see if they're the same */
  0x9B, 0xDF, 0xE0,          /* fstsw ax */
  0x81, 0x4E, 0xF8, 0x00, 0x10,/* or word [bp-8],01000h ;turn on infinity bit */
  0xD9, 0x6E, 0xF8,          /* fldcw [bp-8] */
  0x9E,                      /* sahf */
  0x74, 0x01,                /* jz L3            ;equal, so it's a 287 */
  0x43,                      /* inc bx           ;bx = 3 for 80387 */
  0xDB, 0xE2,                /* L3: fclex        ;clear error bits caused by divide-by-0 */
  0x89, 0x5E, 0xF8,          /* L6: mov [bp-8],bx ;set flag */
/* L15:	*/
  0x8B, 0x46, 0xF8,          /* mov ax, [bp-8] */
  0x83, 0xC4, 0x02,          /* add  sp, 2 */
  0x5B,                      /* pop bx */
  0x59,                      /* pop cx */
  0x89, 0xEC,                /* mov  sp, bp */
  0x5D,                      /* pop  bp */
  0xC3                       /* ret near 0 */
};

static unsigned long __NEAR__ __FASTCALL__ __OPS_nop(volatile unsigned *time_val) =
{
  0x56,                      /* push si */
  0x1E,                      /* push ds */
  0x89, 0xC6,                /* mov  si, ax */
  0x8E, 0xDB,                /* mov  ds, bx */
  0x31, 0xC0,                /* xor  ax, ax */
  0x99,                      /* cwd */
                             /* @WaitTimer: */
  0x81, 0x3C, 0x00, 0x00,    /* cmp word [si], 0 */
  0x74, 0xFA,                /* jz @WaitTimer */
                             /* @L38: */
  0x81, 0x3C, 0x00, 0x00,    /* cmp word [si], 0 */
  0x75, 0x03,                /* jnz @L36 */
  0xE9, 0xF1, 0x03,          /* jmp @L37 */
                             /* @L36: */
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,
      0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90, 0x90,

  0x83, 0xC0, 0x01,          /* add  ax, 1 */
  0x83, 0xD2, 0x00,          /* adc  dx, 0 */
  0xE9, 0x06, 0xFC,          /* jmp  @L38 */
                             /* @L37: */
  0x1F,                      /* pop  ds */
  0x5E,                      /* pop  si */
  0xC3                       /* ret  near 0 */
};

static void __FAR__ __ret_far( void ) = { 0xCB };

static unsigned long __NEAR__ __FASTCALL__ __OPS_std(volatile unsigned *counter,char *arr8byte) =
{
   0x51,                     /* push    cx */
   0x53,                     /* push    bx */
   0x56,                     /* push    si */
   0x57,                     /* push    di */
   0x1E,                     /* push    ds */
   0x06,                     /* push    es */
   0x8E, 0xDB,               /* mov     ds,bx */
   0x89, 0xC6,               /* mov     si,ax */
   0x8E, 0xC2,               /* mov     es,dx */
   0x89, 0xCF,               /* mov     di,cx */
   0x31, 0xC0,               /* xor     ax,ax */
   0x99,                     /* cwd */
/*@WaitTimer:*/
   0x81, 0x3C, 0x00, 0x00,   /* cmp     word [si], 0 */
   0x74, 0xFA,               /* jz      @WaitTimer */
/* @L14: */
   0x81, 0x3C, 0x00, 0x00,   /* cmp     word [si], 0 */
   0x75, 0x03,               /* jnz     @L5 */
   0xE9, 0x90, 0x00,         /* jmp     @L6 */
/* @L5: */
   0x50,                     /* push    ax */
   0x52,                     /* push    dx */

   0xB8, 0x14, 0x00,         /* mov     ax,0014h */
   0xB9, 0x07, 0x00,         /* mov     cx,0007h */
   0xF7, 0xE1,               /* mul     cx */
   0xF7, 0xE9,               /* imul    cx */
   0xF7, 0xF1,               /* div     cx */
   0xF7, 0xF9,               /* idiv    cx */
   0x01, 0xC8,               /* add     ax,cx */
   0x83, 0xD0, 0x01,         /* adc     ax,0001h */
   0x29, 0xC8,               /* sub     ax,cx */
   0x83, 0xD8, 0x01,         /* sbb     ax,0001h */
   0x1E,                     /* push    ds */
   0x06,                     /* push    es */
   0x56,                     /* push    si */
   0x57,                     /* push    di */
   0x8C, 0xC0,               /* mov     ax,es */
   0x8E, 0xD8,               /* mov     ds,ax */
   0x89, 0xFE,               /* mov     si,di */
   0xA5,                     /* movsw */
   0xA7,                     /* cmpsw */
   0x5F,                     /* pop     di */
   0x5E,                     /* pop     si */
   0x07,                     /* pop     es */
   0x1F,                     /* pop     ds */
   0x9A, ((unsigned long)__ret_far),
   0x50,                     /* push    ax */
   0x52,                     /* push    dx */
   0x5A,                     /* pop     dx */
   0x58,                     /* pop     ax */
   0xB8, 0x14, 0x00,         /* mov     ax,0014h */
   0xB9, 0x07, 0x00,         /* mov     cx,0007h */
   0xF7, 0xE1,               /* mul     cx */
   0xF7, 0xE9,               /* imul    cx */
   0xF7, 0xF1,               /* div     cx */
   0xF7, 0xF9,               /* idiv    cx */
   0x01, 0xC8,               /* add     ax,cx */
   0x83, 0xD0, 0x01,         /* adc     ax,0001h */
   0x83, 0xD8, 0x01,         /* sbb     ax,0001h */
   0x29, 0xC8,               /* sub     ax,cx */
   0x1E,                     /* push    ds */
   0x06,                     /* push    es */
   0x56,                     /* push    si */
   0x57,                     /* push    di */
   0x8C, 0xC0,               /* mov     ax,es */
   0x8E, 0xD8,               /* mov     ds,ax */
   0x89, 0xFE,               /* mov     si,di */
   0xA5,                     /* movsw */
   0xA7,                     /* cmpsw */
   0x5F,                     /* pop     di */
   0x5E,                     /* pop     si */
   0x07,                     /* pop     es */
   0x1F,                     /* pop     ds */
   0x50,                     /* push    ax */
   0x52,                     /* push    dx */
   0x9A, ((unsigned long)__ret_far),
   0x5A,                     /* pop     dx */
   0x58,                     /* pop     ax */
   0xB8, 0x14, 0x00,         /* mov     ax,0014h */
   0xB9, 0x07, 0x00,         /* mov     cx,0007h */
   0xF7, 0xE1,               /* mul     cx */
   0xF7, 0xE9,               /* imul    cx */
   0xF7, 0xF1,               /* div     cx */
   0xF7, 0xF9,               /* idiv    cx */
   0x01, 0xC8,               /* add     ax,cx */
   0x83, 0xD0, 0x01,         /* adc     ax,0001h */
   0x29, 0xC8,               /* sub     ax,cx */
   0x83, 0xD8, 0x01,         /* sbb     ax,0001h */
   0xE8, 0x1E, 0x00,         /* call    @S9 */
   0xE9, 0x1C, 0x00,         /* jmp     @S10 */
/* @S16: */
   0xE8, 0x18, 0x00,         /* call    @S11 */
   0xE9, 0x12, 0x00,         /* jmp     @S12 */
/* @S15: */
   0x5A,                     /* pop     dx */
   0x58,                     /* pop     ax */
   0x83, 0xC0, 0x01,         /* add     ax, 1 */
   0x83, 0xD2, 0x00,         /* adc     dx, 0 */
   0xE9, 0x67, 0xFF,         /* jmp     @L14 */
/* @L6: */
   0x07,                     /* pop     es */
   0x1F,                     /* pop     ds */
   0x5F,                     /* pop     di */
   0x5E,                     /* pop     si */
   0x5B,                     /* pop     bx */
   0x59,                     /* pop     cx */
   0xC3,                     /* ret near 0 */
/* @S12: */
   0xE9, 0xEB, 0xFF,         /* jmp     @S15 */
/* @S11: */
   0xC3,                     /* ret near 0 */
/* S10: */
   0xE9, 0xE1, 0xFF,         /* jmp     @S16 */
   0x90, 0x90, 0x90
};

static unsigned long __NEAR__ __FASTCALL__ __FOPS_w_wait(volatile unsigned *counter,char *arr14bytes) =
{
   0x51,                     /* push    cx */
   0x56,                     /* push    si */
   0x57,                     /* push    di */
   0x06,                     /* push    es */
   0x1E,                     /* push    ds */
   0x8E, 0xDB,               /* mov     ds,bx */
   0x89, 0xC6,               /* mov     si,ax */
   0x8E, 0xC2,               /* mov     es,dx */
   0x89, 0xCF,               /* mov     di,cx */
   0x31, 0xC0,               /* xor     ax,ax */
   0x99,                     /* cwd */
 /* @WaitTimer: */
   0x81, 0x3C, 0x00, 0x00,   /* cmp     word [si], 0 */
   0x74, 0xFA,               /* jz      @WaitTimer */
/* @L26: */
   0x81, 0x3C, 0x00, 0x00,   /* cmp     word [si], 0 */
   0x75, 0x03,               /* jnz     @L24 */
   0xE9, 0x18, 0x01,         /* jmp     @L25 */
/* @L24: */
   0x9B, 0xDB, 0xE3,               /* finit */
   0x9B, 0x26, 0xDB, 0x6D, 0x04,   /* fld     tword [es: di+4] */
   0x9B, 0x26, 0xDB, 0x7D, 0x04,   /* fstp    tword [es: di+4] */
   0x9B, 0xDD, 0xD9,               /* fstp    st1 */
   0x9B, 0xD9, 0xEE,               /* fldz */
   0x9B, 0xD9, 0xE8,               /* fld1 */
   0x9B, 0xDE, 0xD9,               /* fcompp */
   0x9B, 0x26, 0xDD, 0x7D, 0x02,   /* fnstsw  word [es: di+2] */
   0x9B, 0x26, 0xD9, 0x3D,         /* fnstcw  [es: di] */
   0x9B, 0x26, 0xD9, 0x2D,         /* fldcw   [es: di] */
   0x9B, 0xD9, 0xEB,               /* fldpi */
   0x9B, 0xDD, 0xD9,               /* fstp    st1 */
   0x9B, 0xDD, 0xD2,               /* fst     st2 */
   0x9B, 0xDD, 0xD3,               /* fst     st3 */
   0x9B, 0xD9, 0xF0,               /* f2xm1 */
   0x9B, 0xD9, 0xE1,               /* fabs */
   0x9B, 0xD9, 0xE0,               /* fchs */
   0x9B, 0xD9, 0xF8,               /* fprem */
   0x9B, 0xD9, 0xF2,               /* fptan */
   0x9B, 0xD9, 0xF2,               /* fptan */
   0x9B, 0xD9, 0xFA,               /* fsqrt */
   0x9B, 0xD9, 0xFC,               /* frndint */
   0x9B, 0xDE, 0xC1,               /* faddp   st1,st0 */
   0x9B, 0xDD, 0xD9,               /* fstp    st1 */
   0x9B, 0xDE, 0xC9,               /* fmulp   st1,st0 */
   0x9B, 0xDD, 0xD9,               /* fstp    st1 */
   0x9B, 0xD9, 0xE8,               /* fld1 */
   0x9B, 0xDD, 0xD9,               /* fstp    st1 */
   0x9B, 0xD9, 0xF3,               /* fpatan */
   0x9B, 0xDD, 0xD9,               /* fstp    st1 */
   0x9B, 0xD9, 0xFD,               /* fscale */
   0x9B, 0xDD, 0xD9,               /* fstp    st1 */
   0x9B, 0xDE, 0xF1,               /* fdivrp  st1,st0 */
   0x9B, 0xDD, 0xD9,               /* fstp    st1 */
   0x9B, 0xDE, 0xE9,               /* fsubp   st1,st0 */
   0x9B, 0xDD, 0xD9,               /* fstp    st1 */
   0x9B, 0xD9, 0xF1,               /* fyl2x */
   0x9B, 0xDD, 0xD9,               /* fstp    st1 */
   0x9B, 0xD9, 0xF9,               /* fyl2xp1 */
   0x9B, 0xDD, 0xD9,               /* fstp    st1 */
   0x9B, 0x26, 0xDF, 0x65, 0x04,   /* fbld    tword [es: di+4] */
   0x9B, 0x26, 0xDF, 0x75, 0x04,   /* fbstp   tword [es: di+4] */
   0x9B, 0x26, 0xDF, 0x05,         /* fild    word [es: di] */
   0x9B, 0x26, 0xDF, 0x1D,         /* fistp   word [es: di] */
   0x9B, 0x26, 0xDB, 0x6D, 0x04,   /* fld     tword [es: di+4] */
   0x9B, 0x26, 0xDB, 0x7D, 0x04,   /* fstp    tword [es: di+4] */
   0x9B, 0xDD, 0xD9,               /* fstp    st1 */
   0x9B, 0xD9, 0xEE,               /* fldz */
   0x9B, 0xD9, 0xE8,               /* fld1 */
   0x9B, 0xDE, 0xD9,               /* fcompp */
   0x9B, 0x26, 0xDD, 0x7D, 0x02,   /* fnstsw   word [es: di+2] */
   0x9B, 0x26, 0xD9, 0x3D,         /* fnstcw   [es: di] */
   0x9B, 0x26, 0xD9, 0x2D,         /* fldcw    [es: di] */
   0x9B, 0xD9, 0xEB,               /* fldpi */
   0x9B, 0xDD, 0xD9,               /* fstp     st1 */
   0x9B, 0xDD, 0xD2,               /* fst      st2 */
   0x9B, 0xDD, 0xD3,               /* fst      st3 */
   0x9B, 0xD9, 0xF0,               /* f2xm1 */
   0x9B, 0xD9, 0xE1,               /* fabs */
   0x9B, 0xD9, 0xE0,               /* fchs */
   0x9B, 0xD9, 0xF8,               /* fprem */
   0x9B, 0xD9, 0xF2,               /* fptan */
   0x9B, 0xD9, 0xF2,               /* fptan */
   0x9B, 0xD9, 0xFA,               /* fsqrt */
   0x9B, 0xD9, 0xFC,               /* frndint */
   0x9B, 0xDE, 0xC1,               /* faddp    st1,st0 */
   0x9B, 0xDD, 0xD9,               /* fstp     st1 */
   0x9B, 0xDE, 0xC9,               /* fmulp    st1,st0 */
   0x9B, 0xDD, 0xD9,               /* fstp     st1 */
   0x9B, 0xD9, 0xE8,               /* fld1 */
   0x9B, 0xDD, 0xD9,               /* fstp     st1 */
   0x9B, 0xD9, 0xF3,               /* fpatan */
   0x9B, 0xDD, 0xD9,               /* fstp     st1 */
   0x9B, 0xD9, 0xFD,               /* fscale */
   0x9B, 0xDD, 0xD9,               /* fstp     st1 */
   0x9B, 0xDE, 0xF1,               /* fdivrp   st1,st0 */
   0x9B, 0xDD, 0xD9,               /* fstp     st1 */
   0x9B, 0xDE, 0xE9,               /* fsubp    st1,st0 */
   0x9B, 0xDD, 0xD9,               /* fstp     st1 */
   0x9B, 0xD9, 0xF1,               /* fyl2x */
   0x9B, 0xDD, 0xD9,               /* fstp     st1 */
   0x9B, 0xD9, 0xF9,               /* fyl2xp1 */
   0x9B, 0xDD, 0xD9,               /* fstp     st1 */
   0x83, 0xC0, 0x01,         /* add      ax,1 */
   0x83, 0xD2, 0x00,         /* adc      dx,0 */
   0xE9, 0xDF, 0xFE,         /* jmp      @L26 */
/*@L25: */
   0x1F,                     /* pop      ds */
   0x07,                     /* pop      es */
   0x5F,                     /* pop      di */
   0x5E,                     /* pop      si */
   0x59,                     /* pop      cx */
   0xC3                      /*ret near 0 */
};

static unsigned long __NEAR__ __FASTCALL__ __FOPS_nowait(volatile unsigned *counter,char *arr14bytes) =
{
   0x51,                     /* push    cx */
   0x56,                     /* push    si */
   0x57,                     /* push    di */
   0x06,                     /* push    es */
   0x1E,                     /* push    ds */
   0x8E, 0xDB,               /* mov     ds,bx */
   0x89, 0xC6,               /* mov     si,ax */
   0x8E, 0xC2,               /* mov     es,dx */
   0x89, 0xCF,               /* mov     di,cx */
   0x31, 0xC0,               /* xor     ax,ax */
   0x99,                     /* cwd */
 /* @WaitTimer: */
   0x81, 0x3C, 0x00, 0x00,   /* cmp     word [si], 0 */
   0x74, 0xFA,               /* jz      @WaitTimer */
/* @L26: */
   0x81, 0x3C, 0x00, 0x00,   /* cmp     word [si], 0 */
   0x75, 0x03,               /* jnz     @L24 */
   0xE9, 0xC5, 0x00,         /* jmp     @L25 */
/* @L24: */
   0xDB, 0xE3,               /* finit */
   0x26, 0xDB, 0x6D, 0x04,   /* fld     tword [es: di+4] */
   0x26, 0xDB, 0x7D, 0x04,   /* fstp    tword [es: di+4] */
   0xDD, 0xD9,               /* fstp    st1 */
   0xD9, 0xEE,               /* fldz */
   0xD9, 0xE8,               /* fld1 */
   0xDE, 0xD9,               /* fcompp */
   0x26, 0xDD, 0x7D, 0x02,   /* fnstsw  word [es: di+2] */
   0x26, 0xD9, 0x3D,         /* fnstcw  [es: di] */
   0x26, 0xD9, 0x2D,         /* fldcw   [es: di] */
   0xD9, 0xEB,               /* fldpi */
   0xDD, 0xD9,               /* fstp    st1 */
   0xDD, 0xD2,               /* fst     st2 */
   0xDD, 0xD3,               /* fst     st3 */
   0xD9, 0xF0,               /* f2xm1 */
   0xD9, 0xE1,               /* fabs */
   0xD9, 0xE0,               /* fchs */
   0xD9, 0xF8,               /* fprem */
   0xD9, 0xF2,               /* fptan */
   0xD9, 0xF2,               /* fptan */
   0xD9, 0xFA,               /* fsqrt */
   0xD9, 0xFC,               /* frndint */
   0xDE, 0xC1,               /* faddp   st1,st0 */
   0xDD, 0xD9,               /* fstp    st1 */
   0xDE, 0xC9,               /* fmulp   st1,st0 */
   0xDD, 0xD9,               /* fstp    st1 */
   0xD9, 0xE8,               /* fld1 */
   0xDD, 0xD9,               /* fstp    st1 */
   0xD9, 0xF3,               /* fpatan */
   0xDD, 0xD9,               /* fstp    st1 */
   0xD9, 0xFD,               /* fscale */
   0xDD, 0xD9,               /* fstp    st1 */
   0xDE, 0xF1,               /* fdivrp  st1,st0 */
   0xDD, 0xD9,               /* fstp    st1 */
   0xDE, 0xE9,               /* fsubp   st1,st0 */
   0xDD, 0xD9,               /* fstp    st1 */
   0xD9, 0xF1,               /* fyl2x */
   0xDD, 0xD9,               /* fstp    st1 */
   0xD9, 0xF9,               /* fyl2xp1 */
   0xDD, 0xD9,               /* fstp    st1 */
   0x26, 0xDF, 0x65, 0x04,   /* fbld    tword [es: di+4] */
   0x26, 0xDF, 0x75, 0x04,   /* fbstp   tword [es: di+4] */
   0x26, 0xDF, 0x05,         /* fild    word [es: di] */
   0x26, 0xDF, 0x1D,         /* fistp   word [es: di] */
   0x26, 0xDB, 0x6D, 0x04,   /* fld     tword [es: di+4] */
   0x26, 0xDB, 0x7D, 0x04,   /* fstp    tword [es: di+4] */
   0xDD, 0xD9,               /* fstp    st1 */
   0xD9, 0xEE,               /* fldz */
   0xD9, 0xE8,               /* fld1 */
   0xDE, 0xD9,               /* fcompp */
   0x26, 0xDD, 0x7D, 0x02,   /* fnstsw   word [es: di+2] */
   0x26, 0xD9, 0x3D,         /* fnstcw   [es: di] */
   0x26, 0xD9, 0x2D,         /* fldcw    [es: di] */
   0xD9, 0xEB,               /* fldpi */
   0xDD, 0xD9,               /* fstp     st1 */
   0xDD, 0xD2,               /* fst      st2 */
   0xDD, 0xD3,               /* fst      st3 */
   0xD9, 0xF0,               /* f2xm1 */
   0xD9, 0xE1,               /* fabs */
   0xD9, 0xE0,               /* fchs */
   0xD9, 0xF8,               /* fprem */
   0xD9, 0xF2,               /* fptan */
   0xD9, 0xF2,               /* fptan */
   0xD9, 0xFA,               /* fsqrt */
   0xD9, 0xFC,               /* frndint */
   0xDE, 0xC1,               /* faddp    st1,st0 */
   0xDD, 0xD9,               /* fstp     st1 */
   0xDE, 0xC9,               /* fmulp    st1,st0 */
   0xDD, 0xD9,               /* fstp     st1 */
   0xD9, 0xE8,               /* fld1 */
   0xDD, 0xD9,               /* fstp     st1 */
   0xD9, 0xF3,               /* fpatan */
   0xDD, 0xD9,               /* fstp     st1 */
   0xD9, 0xFD,               /* fscale */
   0xDD, 0xD9,               /* fstp     st1 */
   0xDE, 0xF1,               /* fdivrp   st1,st0 */
   0xDD, 0xD9,               /* fstp     st1 */
   0xDE, 0xE9,               /* fsubp    st1,st0 */
   0xDD, 0xD9,               /* fstp     st1 */
   0xD9, 0xF1,               /* fyl2x */
   0xDD, 0xD9,               /* fstp     st1 */
   0xD9, 0xF9,               /* fyl2xp1 */
   0xDD, 0xD9,               /* fstp     st1 */
   0x83, 0xC0, 0x01,         /* add      ax,1 */
   0x83, 0xD2, 0x00,         /* adc      dx,0 */
   0xE9, 0x32, 0xFF,         /* jmp      @L26 */
/*@L25: */
   0x1F,                     /* pop      ds */
   0x07,                     /* pop      es */
   0x5F,                     /* pop      di */
   0x5E,                     /* pop      si */
   0x59,                     /* pop      cx */
   0xC3                      /*ret near 0 */
};

static unsigned long __NEAR__ __FASTCALL__ __MOPS_std(volatile unsigned *counter,char *arr) =
{
   0x56,                     /* push     si */
   0x57,                     /* push     di */
   0x06,                     /* push     es */
   0x1E,                     /* push     ds */
   0x8E, 0xDB,               /* mov      ds,bx */
   0x89, 0xC6,               /* mov      si,ax */
   0x31, 0xC0,               /* xor      ax,ax */
   0x99,                     /* cwd */
/* @WaitTime: */
   0x81, 0x3C, 0x00, 0x00,   /* cmp      word [si], 0 */
   0x74, 0xFA,               /* jz       @WaitTime */
/* @L29: */
   0x81, 0x3C, 0x00, 0x00,   /* cmp      word [si], 0 */
   0x75, 0x03,               /* jnz      @L27 */
   0xE9, 0x0F, 0x01,         /* jmp      @L28 */
/* @L27: */
   0x0F, 0x77,               /* emmx */
   0x0F, 0x6E, 0xC0,         /* movd     mm0,eax */
   0x0F, 0x6B, 0xC5,         /* packssdw mm0,mm5 */
   0x0F, 0x63, 0xC4,         /* packsswb mm0,mm4 */
   0x0F, 0x67, 0xC7,         /* packuswb mm0,mm7 */
   0x0F, 0xFC, 0xDA,         /* paddb    mm3,mm2 */
   0x0F, 0xFE, 0xE9,         /* paddd    mm5,mm1 */
   0x0F, 0xE8, 0xE6,         /* psubsb   mm4,mm6 */
   0x0F, 0xD8, 0xDC,         /* psubusb  mm3,mm4 */
   0x0F, 0xDB, 0xCB,         /* pand     mm1,mm3 */
   0x0F, 0x76, 0xC0,         /* pcmpeqd  mm0,mm0 */
   0x0F, 0x64, 0xD2,         /* pcmpgtb  mm2,mm2 */
   0x0F, 0xF5, 0xFF,         /* pmaddwd  mm7,mm7 */
   0x0F, 0xD5, 0xF6,         /* pmullw   mm6,mm6 */
   0x0F, 0xEB, 0xD4,         /* por      mm2,mm4 */
   0x0F, 0xF3, 0xC6,         /* psllq    mm0,mm6 */
   0x0F, 0xE2, 0xCB,         /* psrad    mm1,mm3 */
   0x0F, 0xF8, 0xC9,         /* psubb    mm1,mm1 */
   0x0F, 0xE9, 0xD7,         /* psubsw   mm2,mm7 */
   0x0F, 0xD9, 0xD9,         /* psubusw  mm3,mm1 */
   0x0F, 0x6A, 0xC4,         /* punpckhdq mm0,mm4 */
   0x0F, 0x61, 0xD0,         /* punpcklwd mm2,mm0 */
   0x0F, 0xEF, 0xE2,         /* pxor     mm4,mm2 */
   0x0F, 0x6B, 0xC5,         /* packssdw mm0,mm5 */
   0x0F, 0x63, 0xC4,         /* packsswb mm0,mm4 */
   0x0F, 0x67, 0xC7,         /* packuswb mm0,mm7 */
   0x0F, 0xFC, 0xDA,         /* paddb    mm3,mm2 */
   0x0F, 0xFE, 0xE9,         /* paddd    mm5,mm1 */
   0x0F, 0xE8, 0xE6,         /* psubsb   mm4,mm6 */
   0x0F, 0xD8, 0xDC,         /* psubusb  mm3,mm4 */
   0x0F, 0xDB, 0xCB,         /* pand     mm1,mm3 */
   0x0F, 0x76, 0xC0,         /* pcmpeqd  mm0,mm0 */
   0x0F, 0x64, 0xD2,         /* pcmpgtb  mm2,mm2 */
   0x0F, 0xF5, 0xFF,         /* pmaddwd  mm7,mm7 */
   0x0F, 0xD5, 0xF6,         /* pmullw   mm6,mm6 */
   0x0F, 0xEB, 0xD4,         /* por      mm2,mm4 */
   0x0F, 0xF3, 0xC6,         /* psllq    mm0,mm6 */
   0x0F, 0xE2, 0xCB,         /* psrad    mm1,mm3 */
   0x0F, 0xF8, 0xC9,         /* psubb    mm1,mm1 */
   0x0F, 0xE9, 0xD7,         /* psubsw   mm2,mm7 */
   0x0F, 0xD9, 0xD9,         /* psubusw  mm3,mm1 */
   0x0F, 0x6A, 0xC4,         /* punpckhdq mm0,mm4 */
   0x0F, 0x61, 0xD0,         /* punpcklwd mm2,mm0 */
   0x0F, 0xEF, 0xE2,         /* pxor     mm4,mm2 */
   0x0F, 0x77,               /* emmx */
   0x0F, 0x6E, 0xC0,         /* movd     mm0,eax */
   0x0F, 0x6B, 0xC5,         /* packssdw mm0,mm5 */
   0x0F, 0x63, 0xC4,         /* packsswb mm0,mm4 */
   0x0F, 0x67, 0xC7,         /* packuswb mm0,mm7 */
   0x0F, 0xFC, 0xDA,         /* paddb    mm3,mm2 */
   0x0F, 0xFE, 0xE9,         /* paddd    mm5,mm1 */
   0x0F, 0xE8, 0xE6,         /* psubsb   mm4,mm6 */
   0x0F, 0xD8, 0xDC,         /* psubusb  mm3,mm4 */
   0x0F, 0xDB, 0xCB,         /* pand     mm1,mm3 */
   0x0F, 0x76, 0xC0,         /* pcmpeqd  mm0,mm0 */
   0x0F, 0x64, 0xD2,         /* pcmpgtb  mm2,mm2 */
   0x0F, 0xF5, 0xFF,         /* pmaddwd  mm7,mm7 */
   0x0F, 0xD5, 0xF6,         /* pmullw   mm6,mm6 */
   0x0F, 0xEB, 0xD4,         /* por      mm2,mm4 */
   0x0F, 0xF3, 0xC6,         /* psllq    mm0,mm6 */
   0x0F, 0xE2, 0xCB,         /* psrad    mm1,mm3 */
   0x0F, 0xF8, 0xC9,         /* psubb    mm1,mm1 */
   0x0F, 0xE9, 0xD7,         /* psubsw   mm2,mm7 */
   0x0F, 0xD9, 0xD9,         /* psubusw  mm3,mm1 */
   0x0F, 0x6A, 0xC4,         /* punpckhdq mm0,mm4 */
   0x0F, 0x61, 0xD0,         /* punpcklwd mm2,mm0 */
   0x0F, 0xEF, 0xE2,         /* pxor     mm4,mm2 */
   0x0F, 0x6B, 0xC5,         /* packssdw mm0,mm5 */
   0x0F, 0x63, 0xC4,         /* packsswb mm0,mm4 */
   0x0F, 0x67, 0xC7,         /* packuswb mm0,mm7 */
   0x0F, 0xFC, 0xDA,         /* paddb    mm3,mm2 */
   0x0F, 0xFE, 0xE9,         /* paddd    mm5,mm1 */
   0x0F, 0xE8, 0xE6,         /* psubsb   mm4,mm6 */
   0x0F, 0xD8, 0xDC,         /* psubusb  mm3,mm4 */
   0x0F, 0xDB, 0xCB,         /* pand     mm1,mm3 */
   0x0F, 0x76, 0xC0,         /* pcmpeqd  mm0,mm0 */
   0x0F, 0x64, 0xD2,         /* pcmpgtb  mm2,mm2 */
   0x0F, 0xF5, 0xFF,         /* pmaddwd  mm7,mm7 */
   0x0F, 0xD5, 0xF6,         /* pmullw   mm6,mm6 */
   0x0F, 0xEB, 0xD4,         /* por      mm2,mm4 */
   0x0F, 0xF3, 0xC6,         /* psllq    mm0,mm6 */
   0x0F, 0xE2, 0xCB,         /* psrad    mm1,mm3 */
   0x0F, 0xF8, 0xC9,         /* psubb    mm1,mm1 */
   0x0F, 0xE9, 0xD7,         /* psubsw   mm2,mm7 */
   0x0F, 0xD9, 0xD9,         /* psubusw  mm3,mm1 */
   0x0F, 0x6A, 0xC4,         /* punpckhdq mm0,mm4 */
   0x0F, 0x61, 0xD0,         /* punpcklwd mm2,mm0 */
   0x0F, 0xEF, 0xE2,         /* pxor     mm4,mm2 */
   0x83, 0xC0, 0x01,         /* add      ax,1 */
   0x83, 0xD2, 0x00,         /* adc      dx, 0 */
   0xE9, 0xE8, 0xFE,         /* jmp      @L29 */
/* @L28: */
   0x1F,                     /* pop      ds */
   0x07,                     /* pop      es */
   0x5F,                     /* pop      di */
   0x5E,                     /* pop      si */
   0xC3,                     /* ret near 0 */
};

static unsigned long __NEAR__ __FASTCALL__ __SSEOPS_std(volatile unsigned *counter,char *arr) =
{
   0x56,                     /* push     si */
   0x57,                     /* push     di */
   0x06,                     /* push     es */
   0x1E,                     /* push     ds */
   0x8E, 0xDB,               /* mov      ds,bx */
   0x89, 0xC6,               /* mov      si,ax */
   0x31, 0xC0,               /* xor      ax,ax */
   0x99,                     /* cwd */
/* @WaitTime: */
   0x81, 0x3C, 0x00, 0x00,   /* cmp      word [si], 0 */
   0x74, 0xFA,               /* jz       @WaitTime */
/* @L29: */
   0x81, 0x3C, 0x00, 0x00,   /* cmp      word [si], 0 */
   0x75, 0x03,               /* jnz      @L27 */
   0xE9, 0xDB, 0x00,         /* jmp      @L28 */
/* @L27: */
   0x0F, 0x77,               /* emmx */
   0x26, 0x0F, 0x28, 0x05,   /* movaps   xmm0,[es:di] */
   0x26, 0x0F, 0x16, 0x0D,   /* movhps   xmm1,[es:di] */
   0x26, 0x0F, 0x12, 0x15,   /* movlps   xmm2,[es:di] */
   0x26, 0x0F, 0x10, 0x1D,   /* movups   xmm3,[es:di] */
   0x0F, 0x58, 0xC1,         /* addps    xmm0,xmm1 */
   0xF3, 0x0F, 0x58, 0xC1,   /* addss    xmm0,xmm1 */
   0x0F, 0x2D, 0xC1,         /* cvtps2pi mm0,xmm1 */
   0x0F, 0x2C, 0xCA,         /* cvttps2pi mm1,xmm2 */
   0x0F, 0x5F, 0xC1,         /* maxps    xmm0,xmm1 */
   0xF3, 0x0F, 0x5F, 0xC1,   /* maxss    xmm0,xmm1 */
   0x0F, 0x5D, 0xC1,         /* minps    xmm0,xmm1 */
   0xF3, 0x0F, 0x5D, 0xC1,   /* minss    xmm0,xmm1 */
   0x0F, 0x59, 0xC1,         /* mulps    xmm0,xmm1 */
   0xF3, 0x0F, 0x59, 0xC1,   /* mulss    xmm0,xmm1 */
   0x0F, 0x54, 0xC1,         /* andps    xmm0,xmm1 */
   0x0F, 0x56, 0xC1,         /* orps     xmm0,xmm1 */
   0x0F, 0x57, 0xC9,         /* xorps    xmm1,xmm1 */
   0x0F, 0x5E, 0xC1,         /* divps    xmm0,xmm1 */
   0xF3, 0x0F, 0x5E, 0xC1,   /* divss    xmm0,xmm1 */
   0x0F, 0x53, 0xC1,         /* rcpps    xmm0,xmm1 */
   0xF3, 0x0F, 0x53, 0xC1,   /* rcpss    xmm0,xmm1 */
   0x0F, 0x52, 0xC1,         /* rsqrtps  xmm0,xmm1 */
   0xF3, 0x0F, 0x52, 0xC1,   /* rsqrtss  xmm0,xmm1 */
   0x0F, 0x51, 0xC1,         /* sqrtps   xmm0,xmm1 */
   0xF3, 0x0F, 0x51, 0xC1,   /* sqrtss   xmm0,xmm1 */
   0x0F, 0x5C, 0xC1,         /* subps    xmm0,xmm1 */
   0xF3, 0x0F, 0x5C, 0xC1,   /* subss    xmm0,xmm1 */
   0x0F, 0x2E, 0xC1,         /* ucomiss  xmm0,xmm1 */
   0x0F, 0x15, 0xC1,         /* unpckhps xmm0,xmm1 */
   0x0F, 0x14, 0xC1,         /* unpcklps xmm0,xmm1 */
   0x0F, 0x77,               /* emmx */
   0x26, 0x0F, 0x28, 0x05,   /* movaps   xmm0,[es:di] */
   0x26, 0x0F, 0x16, 0x0D,   /* movhps   xmm1,[es:di] */
   0x26, 0x0F, 0x12, 0x15,   /* movlps   xmm2,[es:di] */
   0x26, 0x0F, 0x10, 0x1D,   /* movups   xmm3,[es:di] */
   0x0F, 0x58, 0xC1,         /* addps    xmm0,xmm1 */
   0xF3, 0x0F, 0x58, 0xC1,   /* addss    xmm0,xmm1 */
   0x0F, 0x2D, 0xC1,         /* cvtps2pi mm0,xmm1 */
   0x0F, 0x2C, 0xCA,         /* cvttps2pi mm1,xmm2 */
   0x0F, 0x5F, 0xC1,         /* maxps    xmm0,xmm1 */
   0xF3, 0x0F, 0x5F, 0xC1,   /* maxss    xmm0,xmm1 */
   0x0F, 0x5D, 0xC1,         /* minps    xmm0,xmm1 */
   0xF3, 0x0F, 0x5D, 0xC1,   /* minss    xmm0,xmm1 */
   0x0F, 0x59, 0xC1,         /* mulps    xmm0,xmm1 */
   0xF3, 0x0F, 0x59, 0xC1,   /* mulss    xmm0,xmm1 */
   0x0F, 0x54, 0xC1,         /* andps    xmm0,xmm1 */
   0x0F, 0x56, 0xC1,         /* orps     xmm0,xmm1 */
   0x0F, 0x57, 0xC9,         /* xorps    xmm1,xmm1 */
   0x0F, 0x5E, 0xC1,         /* divps    xmm0,xmm1 */
   0xF3, 0x0F, 0x5E, 0xC1,   /* divss    xmm0,xmm1 */
   0x0F, 0x53, 0xC1,         /* rcpps    xmm0,xmm1 */
   0xF3, 0x0F, 0x53, 0xC1,   /* rcpss    xmm0,xmm1 */
   0x0F, 0x52, 0xC1,         /* rsqrtps  xmm0,xmm1 */
   0xF3, 0x0F, 0x52, 0xC1,   /* rsqrtss  xmm0,xmm1 */
   0x0F, 0x51, 0xC1,         /* sqrtps   xmm0,xmm1 */
   0xF3, 0x0F, 0x51, 0xC1,   /* sqrtss   xmm0,xmm1 */
   0x0F, 0x5C, 0xC1,         /* subps    xmm0,xmm1 */
   0xF3, 0x0F, 0x5C, 0xC1,   /* subss    xmm0,xmm1 */
   0x0F, 0x2E, 0xC1,         /* ucomiss  xmm0,xmm1 */
   0x0F, 0x15, 0xC1,         /* unpckhps xmm0,xmm1 */
   0x0F, 0x14, 0xC1,         /* unpcklps xmm0,xmm1 */

   0x83, 0xC0, 0x01,         /* add      ax,1 */
   0x83, 0xD2, 0x00,         /* adc      dx, 0 */
   0xE9, 0x1C, 0xFF,         /* jmp      @L29 */
/* @L28: */
   0x1F,                     /* pop      ds */
   0x07,                     /* pop      es */
   0x5F,                     /* pop      di */
   0x5E,                     /* pop      si */
   0xC3,                     /* ret near 0 */
};

#define __ASMPART_DEFINED 1
#include "biewlib/sysdep/ia16/cmn_ix86.c"
