var bm = require("../shared-modules/bookmarks");

const TIMEOUT = 5000;

const LOCAL_TEST_FOLDER = collector.addHttpResource('../test-files/');
const LOCAL_TEST_PAGE = LOCAL_TEST_FOLDER + 'test.html';
const LOCAL_TEST_FEED = LOCAL_TEST_FOLDER + 'feed.atom';

var setupModule = function(module) {
    module.controller = mozmill.getBrowserController();
    module.jum = {};
    module.desktopcouch = {};
    module.sync = {};
    Cu.import("resource://mozmill/modules/jum.js", module.jum);
    Cu.import("resource://bindwood/desktopcouch.jsm", module.desktopcouch);
    Cu.import("resource://bindwood/sync.jsm", module.sync);
    bm.clearBookmarks();
    module.couch = null;
    module.synchroniser = null;
};


var setupTest = function(test) {
    var done = false;
    desktopcouch.connect_desktopcouch("test_bookmarks", function(db) {
            couch = db;
            done = true;
        }, function (message) {});
    controller.waitFor(
        function() { return done; }, "Could not connect to CouchDB", TIMEOUT);
    jum.assertNotEquals(couch, null);

    try {
        couch.createDb();
    } catch (e) {
        if (e.error != 'file_exists')
            throw(e);
    }
};


var teardownTest = function(test) {
    if (synchroniser) {
        synchroniser.uninit();
        synchroniser = null;
    }
    bm.clearBookmarks();
    couch.deleteDb();
};


var test_sync_local = function() {
    var item_id = bm.bookmarksService.insertBookmark(
        bm.bookmarksService.toolbarFolder, bm.createURI(LOCAL_TEST_PAGE),
        bm.bookmarksService.DEFAULT_INDEX, "Bookmark title");

    // Perform synchronisation.
    synchroniser = new sync.Synchroniser(couch, "profile_name");
    synchroniser.init();
    synchroniser.sync();

    // The root and toolbar have been synchronised.
    var root_guid = synchroniser.guid_from_id(
        bm.bookmarksService.placesRoot);
    var toolbar_guid = synchroniser.guid_from_id(
        bm.bookmarksService.toolbarFolder)
    var doc = couch.open(root_guid);
    jum.assertNotNull(doc);
    var old_root_rev = doc._rev;

    doc = couch.open(toolbar_guid);
    jum.assertNotNull(doc);
    var old_toolbar_rev = doc._rev;

    // Our bookmark has also been synchronised.
    var item_guid = synchroniser.guid_from_id(item_id);
    doc = couch.open(item_guid);
    jum.assertNotNull(doc);
    jum.assertEquals(doc.title, "Bookmark title");

    // Making a local change to the bookmark.
    bm.bookmarksService.setItemTitle(item_id, "New title");
    synchroniser.sync();

    // The parent folders should be unchanged.
    doc = couch.open(root_guid);
    jum.assertEquals(doc._rev, old_root_rev);
    doc = couch.open(toolbar_guid);
    jum.assertEquals(doc._rev, old_toolbar_rev);

    // Our bookmark should have been updated.
    doc = couch.open(item_guid);
    jum.assertEquals(doc.title, "New title");
    var old_item_rev = doc._rev;

    // Performing another synchronisation should leave the item unchanged.
    synchroniser.sync();
    doc = couch.open(item_guid);
    jum.assertEquals(doc._rev, old_item_rev);
};

var test_sync_remote = function() {
    couch.save({
        _id: 'root_profile_name',
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/folder',
        record_type_version: 2,
        children: [
            'toolbar_profile_name',
            'menu_profile_name',
            'unfiled_profile_name'
        ]
    });

    var item_guid = '12345';
    var doc = {
        _id: item_guid,
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
        record_type_version: 2,
        parent_title: 'Bookmarks Toolbar',
        parent_guid: 'toolbar_profile_name',
        title: 'Bookmark title',
        uri: LOCAL_TEST_PAGE,
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    };
    couch.save(doc);
    var old_item_rev = doc._rev;

    // Perform synchronisation.
    synchroniser = new sync.Synchroniser(couch, "profile_name");
    synchroniser.init();
    synchroniser.sync();

    // Item has been created locally.
    var item_id = synchroniser.guid_to_id(item_guid);
    jum.assertEquals(
        bm.bookmarksService.getItemTitle(item_id), 'Bookmark title');

    // CouchDB record has not been modified.
    doc = couch.open(item_guid);
    jum.assertEquals(doc._rev, old_item_rev);

    // Modify the item and save it back.
    doc.title = 'New title';
    doc.application_annotations.Firefox.last_modified = (
        bm.bookmarksService.getItemLastModified(item_id) + 1);
    couch.save(doc);
    old_item_rev = doc._rev;

    // After synchronisation, the local bookmark has changed.
    synchroniser.sync();
    jum.assertEquals(
        bm.bookmarksService.getItemTitle(item_id), 'New title');

    // Again, the CouchDB record hasn't been modified by the sync.
    doc = couch.open(item_guid);
    jum.assertEquals(doc._rev, old_item_rev);
};

function get_bookmark_ids(profile) {
    var result = couch.view("bindwood/bookmarks", {key: profile});
    return [row.value for each (row in result.rows)];
}

var test_sync_remote_add_no_duplicate = function() {
    // Perform synchronisation.
    synchroniser = new sync.Synchroniser(couch, "profile_name");
    synchroniser.init();
    synchroniser.sync();

    var expected_bookmarks = [
        synchroniser.guid_from_id(bm.bookmarksService.placesRoot),
        synchroniser.guid_from_id(bm.bookmarksService.toolbarFolder),
        synchroniser.guid_from_id(bm.bookmarksService.bookmarksMenuFolder),
        synchroniser.guid_from_id(bm.bookmarksService.unfiledBookmarksFolder),
    ];
    var actual_bookmarks = get_bookmark_ids("profile_name");
    jum.assertEquals(expected_bookmarks.sort().join("\n"),
                     actual_bookmarks.sort().join("\n"))

    var item_guid = '12345';
    var doc = {
        _id: item_guid,
        record_type: 'http://www.freedesktop.org/wiki/Specifications/desktopcouch/bookmark',
        record_type_version: 2,
        parent_guid: 'toolbar_profile_name',
        title: 'Bookmark title',
        uri: LOCAL_TEST_PAGE,
        application_annotations: {
            Firefox: {
                profile: 'profile_name',
                last_modified: 1,
            }
        }
    };
    couch.save(doc);

    // After synchronisation, we expect to only find the one added
    // bookmark.
    synchroniser.sync();
    expected_bookmarks.push(item_guid);
    var actual_bookmarks = get_bookmark_ids("profile_name");
    jum.assertEquals(expected_bookmarks.sort().join("\n"),
                     actual_bookmarks.sort().join("\n"))
};
