var bm = require("../shared-modules/bookmarks");

const TIMEOUT = 5000;

const LOCAL_TEST_FOLDER = collector.addHttpResource('../test-files/');
const LOCAL_TEST_PAGE = LOCAL_TEST_FOLDER + 'test.html';
const LOCAL_TEST_FEED = LOCAL_TEST_FOLDER + 'feed.atom';

var setupModule = function(module) {
    module.controller = mozmill.getBrowserController();
    module.jum = {};
    module.desktopcouch = {};
    module.sync = {};
    Cu.import("resource://mozmill/modules/jum.js", module.jum);
    Cu.import("resource://bindwood/desktopcouch.jsm", module.desktopcouch);
    Cu.import("resource://bindwood/sync.jsm", module.sync);
    bm.clearBookmarks();
    module.couch = null;
    module.synchroniser = null;
};


var setupTest = function(test) {
    var done = false;
    desktopcouch.connect_desktopcouch("test_bookmarks", function(db) {
            couch = db;
            done = true;
        }, function (message) {});
    controller.waitFor(
        function() { return done; }, "Could not connect to CouchDB", TIMEOUT);
    jum.assertNotEquals(couch, null);

    try {
        couch.createDb();
    } catch (e) {
        if (e.error != 'file_exists')
            throw(e);
    }
    synchroniser = new sync.Synchroniser(couch, "profile_name");
};


var teardownTest = function(test) {
    bm.clearBookmarks();
    couch.deleteDb();
};


var test_export_bookmark = function() {
    var item_id = bm.bookmarksService.insertBookmark(
        bm.bookmarksService.toolbarFolder, bm.createURI(LOCAL_TEST_PAGE),
        bm.bookmarksService.DEFAULT_INDEX, "Bookmark title");

    var item_guid = synchroniser.guid_from_id(item_id);
    synchroniser.exportItem(item_guid);
    var doc = couch.open(item_guid);
    jum.assertEquals(doc.record_type, "http://www.freedesktop.org/wiki/" +
                     "Specifications/desktopcouch/bookmark");
    jum.assertEquals(doc.record_type_version, 2);
    jum.assertEquals(doc.application_annotations.Firefox.profile,
                     "profile_name");
    jum.assertEquals(doc.parent_guid, "toolbar_profile_name");
    jum.assertEquals(doc.uri, LOCAL_TEST_PAGE);
    jum.assertEquals(doc.title, "Bookmark title");

    // Make a change to the folder and re-export it.
    bm.bookmarksService.setItemTitle(item_id, "New title");
    synchroniser.exportItem(item_guid);
    doc = couch.open(item_guid);
    jum.assertEquals(doc.title, "New title");

    // A second sync with no changes does not update the revision ID.
    var old_revid = doc._rev;
    synchroniser.exportItem(item_guid);
    doc = couch.open(item_guid);
    jum.assertEquals(doc._rev, old_revid);
};

var test_export_folder = function() {
    var folder_id = bm.bookmarksService.createFolder(
        bm.bookmarksService.toolbarFolder, "Folder",
        bm.bookmarksService.DEFAULT_INDEX);
    var child1_id = bm.bookmarksService.insertBookmark(
        folder_id, bm.createURI(LOCAL_TEST_PAGE + "#one"),
        bm.bookmarksService.DEFAULT_INDEX, "Bookmark 1");
    var child2_id = bm.bookmarksService.insertBookmark(
        folder_id, bm.createURI(LOCAL_TEST_PAGE + "#two"),
        bm.bookmarksService.DEFAULT_INDEX, "Bookmark 2");

    var folder_guid = synchroniser.guid_from_id(folder_id);
    synchroniser.exportItem(folder_guid);
    var doc = couch.open(folder_guid);
    jum.assertEquals(doc.record_type, "http://www.freedesktop.org/wiki/" +
                     "Specifications/desktopcouch/folder");
    jum.assertEquals(doc.record_type_version, 2);
    jum.assertEquals(doc.application_annotations.Firefox.profile,
                     "profile_name");
    jum.assertEquals(doc.parent_guid, "toolbar_profile_name");
    jum.assertEquals(doc.title, "Folder");
    jum.assertEquals(doc.children.length, 2);
    jum.assertEquals(doc.children[0], synchroniser.guid_from_id(child1_id));
    jum.assertEquals(doc.children[1], synchroniser.guid_from_id(child2_id));

    // Make a change to the folder and re-export it.
    bm.bookmarksService.setItemTitle(folder_id, "New title");
    synchroniser.exportItem(folder_guid);
    doc = couch.open(folder_guid);
    jum.assertEquals(doc.title, "New title");

    // A second sync with no changes does not update the revision ID.
    var old_revid = doc._rev;
    synchroniser.exportItem(folder_guid);
    doc = couch.open(folder_guid);
    jum.assertEquals(doc._rev, old_revid);
};

var test_export_feed = function() {
    var item_id = bm.livemarkService.createLivemark(
        bm.bookmarksService.toolbarFolder, "Feed",
        null, bm.createURI(LOCAL_TEST_FEED),
        bm.bookmarksService.DEFAULT_INDEX);
    // Wait for the livemark to populate.
    controller.waitFor(function() {
        return bm.bookmarksService.getIdForItemAt(
            item_id, 0) >= 0;
    }, TIMEOUT);

    var item_guid = synchroniser.guid_from_id(item_id);
    synchroniser.exportItem(item_guid);
    var doc = couch.open(item_guid);
    jum.assertEquals(doc.record_type, "http://www.freedesktop.org/wiki/" +
                     "Specifications/desktopcouch/feed");
    jum.assertEquals(doc.record_type_version, 2);
    jum.assertEquals(doc.application_annotations.Firefox.profile,
                     "profile_name");
    jum.assertEquals(doc.parent_guid, "toolbar_profile_name");
    jum.assertEquals(doc.site_uri, "http://www.example.com/");
    jum.assertEquals(doc.feed_uri, LOCAL_TEST_FEED);
    jum.assertEquals(doc.title, "Feed");

    // Make a change to the folder and re-export it.
    bm.bookmarksService.setItemTitle(item_id, "New title");
    synchroniser.exportItem(item_guid);
    doc = couch.open(item_guid);
    jum.assertEquals(doc.title, "New title");

    // A second sync with no changes does not update the revision ID.
    var old_revid = doc._rev;
    synchroniser.exportItem(item_guid);
    doc = couch.open(item_guid);
    jum.assertEquals(doc._rev, old_revid);
};

var test_export_separator = function() {
    var item_id = bm.bookmarksService.insertSeparator(
        bm.bookmarksService.toolbarFolder, bm.createURI(LOCAL_TEST_PAGE));

    var item_guid = synchroniser.guid_from_id(item_id);
    synchroniser.exportItem(item_guid);
    var doc = couch.open(item_guid);
    jum.assertEquals(doc.record_type, "http://www.freedesktop.org/wiki/" +
                     "Specifications/desktopcouch/separator");
    jum.assertEquals(doc.record_type_version, 2);
    jum.assertEquals(doc.application_annotations.Firefox.profile,
                     "profile_name");
    jum.assertEquals(doc.parent_guid, "toolbar_profile_name");

    // A second sync with no changes does not update the revision ID.
    var old_revid = doc._rev;
    synchroniser.exportItem(item_guid);
    doc = couch.open(item_guid);
    jum.assertEquals(doc._rev, old_revid);
};

var test_export_deleted_item = function() {
    var item_id = bm.bookmarksService.insertBookmark(
        bm.bookmarksService.toolbarFolder, bm.createURI(LOCAL_TEST_PAGE),
        bm.bookmarksService.DEFAULT_INDEX, "Bookmark title");

    var item_guid = synchroniser.guid_from_id(item_id);
    synchroniser.exportItem(item_guid);

    bm.bookmarksService.removeItem(item_id);
    synchroniser.exportItem(item_guid);

    // The document has been deleted in CouchDB.
    var doc = couch.open(item_guid);
    jum.assertNull(doc);
};

var test_export_root = function() {
    var root_guid = synchroniser.guid_from_id(bm.bookmarksService.placesRoot);
    synchroniser.exportItem(root_guid);

    var doc = couch.open(root_guid);
    jum.assertEquals(doc.record_type, "http://www.freedesktop.org/wiki/" +
                     "Specifications/desktopcouch/folder");
    jum.assertEquals(doc.record_type_version, 2);
    jum.assertUndefined(doc.parent_guid);
    jum.assertEquals(doc.children.length, 3);
    jum.assertEquals(doc.children[0], synchroniser.guid_from_id(
        bm.bookmarksService.toolbarFolder));
    jum.assertEquals(doc.children[1], synchroniser.guid_from_id(
        bm.bookmarksService.bookmarksMenuFolder));
    jum.assertEquals(doc.children[2], synchroniser.guid_from_id(
        bm.bookmarksService.unfiledBookmarksFolder));
};

var test_push_changes = function() {
    var item_id = bm.bookmarksService.insertBookmark(
        bm.bookmarksService.toolbarFolder, bm.createURI(LOCAL_TEST_PAGE),
        bm.bookmarksService.DEFAULT_INDEX, "Bookmark title");

    // Trap calls to exportItem, to see what bookmarks are exported.
    var bookmarks = []
    synchroniser.exportItem = function(item_id) {
        bookmarks.push(item_id);
    }

    // Fake observer that returns a fixed set of changes.
    synchroniser.observer = {
        clear_changes: function() {
            var changes = {}
            changes[synchroniser.guid_from_id(
                bm.bookmarksService.toolbarFolder)] = true;
            changes[synchroniser.guid_from_id(item_id)] = true;
            return changes;
        }
    };

    // First call to pushItems() will export everything.
    synchroniser.pushChanges();
    bookmarks.sort();
    jum.assertEquals(bookmarks.length, 5);
    // This relies on the fact that the genrated IDs '{...}' sort
    // after the special ones.  If we change the generated IDs, we'll
    // need to change how this test works.
    jum.assertEquals(bookmarks[0], "menu_profile_name");
    jum.assertEquals(bookmarks[1], "root_profile_name");
    jum.assertEquals(bookmarks[2], "toolbar_profile_name");
    jum.assertEquals(bookmarks[3], "unfiled_profile_name");
    jum.assertEquals(bookmarks[4], synchroniser.guid_from_id(item_id));

    // Subsequent calls get the ID list from the bookmarks observer.
    bookmarks = [];
    synchroniser.pushChanges();
    bookmarks.sort()
    jum.assertEquals(bookmarks.length, 2);
    jum.assertEquals(bookmarks[0], "toolbar_profile_name");
    jum.assertEquals(bookmarks[1], synchroniser.guid_from_id(item_id));
};
