#!/usr/bin/env python
# -*- coding: utf-8 -*-

#  Bluemindo
#  bluemindo.py

#    Bluemindo: A really simple but powerful audio player in Python/PyGTK.
#    Copyright (C) 2007-2008  Erwan Briand

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation version 3 of the License.

#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

from os.path import isfile, exists, join

from common.functions import Functions
functions = Functions()

# Gettext internationalization
from gettext import (bindtextdomain, textdomain, bind_textdomain_codeset,
                     gettext as _)
from gtk.glade import (bindtextdomain as gtk_glade_bindtextdomain,
                       XML as glade_XML)

bindtextdomain('bluemindo')
gtk_glade_bindtextdomain('bluemindo', functions.localedir)

textdomain('bluemindo')
bind_textdomain_codeset('bluemindo', 'UTF-8')

# Optional DBUS support
try:
    import dbus
    dbus_imported = True
except ImportError:
    dbus_imported = False

if dbus_imported:
    from dbus.mainloop.glib import DBusGMainLoop
    my_native_main_loop = DBusGMainLoop(set_as_default=True)

from gtk import (main as gtk_main, window_set_default_icon_list, pygtk_version,
                 gtk_version, ICON_SIZE_LARGE_TOOLBAR, MenuItem,
                 FileChooserDialog, FILE_CHOOSER_ACTION_OPEN, STOCK_CANCEL,
                 RESPONSE_CANCEL, STOCK_OPEN, RESPONSE_OK, FileFilter)

# Raise if GTK is not up-to-date
if pygtk_version < (2, 10) or gtk_version < (2, 10):
    raise ImportError, 'Bluemindo needs GTK >= 2.10'

from atexit import register as atexit_register
from gobject import threads_init, idle_add
from gtk.gdk import pixbuf_new_from_file
from os import remove
from sys import argv, exit
from socket import socket, AF_UNIX, SOCK_DGRAM, error as socket_error
from thread import start_new_thread
import ConfigParser

if len(argv) > 1 and argv[1] in ('-h', '--help'):
    print _("Bluemindo  Copyright (C) 2007-2008  Erwan Briand\n"
    "This program comes with ABSOLUTELY NO WARRANTY.\n"
    "This is free software, and you are welcome to\n"
    "redistribute it under certain conditions.\n\n"
    "Usage: bluemindo[.py] [options]\n\n"
    "Available options:\n"
    "--current\t\t\t"
     "Show the current playing song artist, title and album\n"
    "--current-cover\t\t\t"
     "Show the path to the album cover of the current playing song\n"
    "--current-lyrics\t\t"
     "Show the lyrics for the current playing song\n\n"
    "--playpause, --play, --pause\t"
     "Play or pause a song\n"
    "--stop\t\t\t\t"
     "Stop a song\n"
    "--previous\t\t\t"
     "Jump to the previous song in playlist\n"
    "--next\t\t\t\t"
     "Jump to the next song in playlist\n\n"
    "--volume-more [STEP]\t\t"
     "Increase the volume, you can specify a step (0 > 100)\n"
    "--volume-less [STEP]\t\t"
     "Decrease the volume, you can specify a step (0 > 100)\n"
    "--volume=VOLUME\t\t\t"
     "Set the volume: 0 > 100\n\n"
    "--reload\t\t\t"
     "Reload the songs from your music folder\n"
    "--quit, --plunge\t\t"
     "Quit Bluemindo")
    exit(0)

from handlers import main as hmain
from common.config import ConfigLoader
from common.playlists import Playlists
from media.gstreamer import GStreamer
config = ConfigLoader()
gst = GStreamer()
playlists_management = Playlists()

from extensionsloader import ExtensionsLoader
extensions = ExtensionsLoader()

class Bluemindo:
    def __init__(self):
        # Set the name of the UNIX socket
        SOCKET_NAME = '/tmp/bluemindo'

        # If args
        if len(argv) > 1:
            if argv[1].startswith('--current'):
                # Get the current song
                current_playing = join(config.datadir, 'current-playing')
                if exists(current_playing):
                    file_ = open(current_playing)
                    csong = file_.read()
                    csong_ = csong.split(' (from: ')
                    calbum = csong_[1][:-1]

                    csong_ = csong_[0].split(' - ')
                    cartist = csong_[0]
                    ctitle = csong_[1]
                    file_.close()

                    # Send the current playing song album cover
                    if argv[1].endswith('-cover'):
                        file_ = join(config.datadir, 'modules', 'player',
                                functions.get_hash(calbum, cartist) + '.covers')
                        if isfile(file_):
                            print file_
                        else:
                            print 'Not found'
                    # Send the current playing song lyrics                
                    elif argv[1].endswith('-lyrics'):
                        file_ = join(config.datadir, 'modules', 'lyrics',
                                functions.get_hash(ctitle, cartist) + '.lyrics')
                        if isfile(file_):
                            lyric = open(file_)
                            print lyric.read()
                            lyric.close()
                        else:
                            print 'Not found'
                    # Send the current playing song artist and title
                    else:
                        print csong
            # Get the volume level
            elif argv[1] == '--volume':
                configfile = ConfigParser.ConfigParser()
                configfile.read(join(config.confdir, 'Bluemindo.cfg'))
                print int(float(configfile.get('Audio', 'volume')) * 100)
            # Others commands
            else:
                try:
                    # Create a client and connect to the UNIX socket in
                    # order to send the user's request
                    client = socket(AF_UNIX, SOCK_DGRAM)
                    client.connect(SOCKET_NAME)

                    # Send two args instead of just one
                    if len(argv) == 3:
                        client.send(argv[1] + ' ' + argv[2])
                    else:
                        client.send(argv[1])

                    client.close()
                except socket_error, error:
                    print 'Socket error, aborting.\n%s' % error

            # Close Bluemindo
            exit(0)

        # If the UNIX socket exists, remove it
        if exists(SOCKET_NAME):
            remove(SOCKET_NAME)

        # Create the UNIX socket server
        server = socket(AF_UNIX, SOCK_DGRAM)
        server.bind(SOCKET_NAME)
        atexit_register(server.close)

        # Receive args send on the socket
        def start_socket_server(server, extensions):
            while True:
                # Get the user's command
                datagram = server.recv(1024)
                if datagram:

                    # Playpause, play or pause
                    if datagram in ('--playpause', '--play', '--pause'):
                        idle_add(extensions.load_event, 'OnPlayPressed')

                    # Stop
                    elif datagram == '--stop':
                        idle_add(extensions.load_event, 'OnStopPressed')

                    # Next
                    elif datagram == '--next':
                        idle_add(extensions.load_event, 'OnNextPressed')

                    # Previous
                    elif datagram == '--previous':
                        idle_add(extensions.load_event, 'OnPreviousPressed')

                    # More volume
                    elif datagram.startswith('--volume-more'):
                        self.volume_more(datagram)

                    # Less volume
                    elif datagram.startswith('--volume-less'):
                        self.volume_less(datagram)

                    # Set the volume
                    elif datagram.startswith('--volume='):
                        self.volume_set(datagram.split('=')[1])

                    # Reload songs list
                    elif datagram == '--reload':
                        idle_add(extensions.load_event, 'OnMenuRefreshPressed',
                                                     widgets)

                    # Quit Bluemindo
                    elif datagram in ('--quit', '--plunge'):
                        idle_add(handler.on_menu_quit, 'socket')

                    # The command isn't handled by any action
                    else:
                        print 'Receive unknown event `%s`!' % datagram

        # Thread the above function
        thread = start_new_thread(start_socket_server, (server, extensions))

        # Handle keyboard if dbus is enabled
        if dbus_imported:
            try:
                bus = dbus.SessionBus()
                application = 'bluemindo'
                def on_dbus_key(app, key):
                    if app == application:
                        if key == 'Play':
                            extensions.load_event('OnPlayPressed')
                        elif key == 'PlayPause':
                            extensions.load_event('OnPlayPressed')
                        elif key == 'Next':
                            extensions.load_event('OnNextPressed')
                        elif key == 'Previous':
                            extensions.load_event('OnPreviousPressed')
                        elif key == 'Stop':
                            extensions.load_event('OnStopPressed')

                # Old API
                try:
                    obj = bus.get_object('org.gnome.SettingsDaemon',
                                         '/org/gnome/SettingsDaemon')
                    dbi = dbus.Interface(obj, 'org.gnome.SettingsDaemon')

                    dbi.GrabMediaPlayerKeys(application, 0)
                    dbi.connect_to_signal('MediaPlayerKeyPressed', on_dbus_key)

                # New API
                except dbus.exceptions.DBusException:
                    obj = bus.get_object('org.gnome.SettingsDaemon',
                                         '/org/gnome/SettingsDaemon/MediaKeys')
                    dbi = dbus.Interface(obj,
                                         'org.gnome.SettingsDaemon.MediaKeys')

                    dbi.GrabMediaPlayerKeys(application, 0)
                    dbi.connect_to_signal('MediaPlayerKeyPressed', on_dbus_key)

            except dbus.exceptions.DBusException, error:
                print 'DBus error!\n%s' % error

        # Start the main GTK thread
        threads_init()

        # GUI initialization
        widgets = glade_XML(functions.datadir + '/glade/mainwindow.glade',
                                                'window1', domain='bluemindo')

        # Use in callback
        self.widgets = widgets

        handler = hmain.MainHandlers(extensions, config, widgets, gst)
        window = widgets.get_widget('window1')
        functions.open_bluemindo(window)
        window.connect('delete_event', handler.on_window_close)

        # Add the pretty logo to all windows
        pixbuf = pixbuf_new_from_file(
                 functions.datadir + '/image/logo_head_small.png')
        window_set_default_icon_list(pixbuf)
        window.set_icon(pixbuf)
        window.set_title('Bluemindo')

        # This function opens the playlist
        def open_playlist(widgetmenu, playlist_name):
            extensions.load_event('OnPlaylistRequested', playlist_name)

        # This function imports a playlist file into `datadir`
        def import_playlist(widgetmenu):
            fcd_filter = FileFilter()
            fcd_filter.add_mime_type('audio/x-mpegurl')
            fcd_filter.set_name(_('Winamp-like playlists'))

            fcd_playlist = FileChooserDialog(_('Choose a playlist'), window,
                           FILE_CHOOSER_ACTION_OPEN,
                             (STOCK_CANCEL, RESPONSE_CANCEL,
                              STOCK_OPEN, RESPONSE_OK
                             ))
            fcd_playlist.add_filter(fcd_filter)
            playlist = fcd_playlist.run()

            if playlist == RESPONSE_OK:
                filename = fcd_playlist.get_filename()
                playlist_name = playlists_management.import_playlist(filename)

                extensions.load_event('OnPlaylistRequested', playlist_name)

            fcd_playlist.destroy()

        # Import a playlist
        widgets.get_widget('menu-playlist-import').connect(
                           'activate', import_playlist)

        # Saved Playlists
        playlists_list = playlists_management.get_playlists_list()
        playlists_menu = widgets.get_widget('submenu-playlist')

        for playlist_name in playlists_list:
            menu_item = MenuItem(label=playlist_name, use_underline=False)
            playlists_menu.append(menu_item)
            menu_item.show_all()
            menu_item.connect('activate', open_playlist, playlist_name)

        # Volume
        self.show_volume = False
        configfile = ConfigParser.ConfigParser()
        configfile_ = 'Bluemindo.cfg'
        configfile.read(join(config.confdir, configfile_))
        try:
            volume = float(configfile.get('Audio', 'volume'))
        except ConfigParser.NoSectionError:
            volume = 1.0

        self.volume = volume
        self.change_volume(volume)

        # This function changes views to `only player` or normal
        self.has_left_panel = False
        vbox = widgets.get_widget('vbox2')

        # Swith views
        def change_view(widget, view):
            if view == 'player':
                if vbox.get_property('visible'):
                    vbox.hide()
                    self.has_left_panel = True
                widgets.get_widget('vpaned1').hide()
                widgets.get_widget('toolbar4').hide()

                window = widgets.get_widget('window1')
                width = window.get_size()[0]
                window.resize(width, 120)

            elif view == 'all':
                if self.has_left_panel:
                    vbox.show()
                widgets.get_widget('vpaned1').show()
                widgets.get_widget('toolbar4').show()

                window = widgets.get_widget('window1')
                width = window.get_size()[0]
                window.resize(width, 240)

        radio_player = widgets.get_widget('view-player')
        radio_all = widgets.get_widget('view-all')

        # Activate menu items
        radio_player.set_group(radio_all)
        radio_all.set_active(True)
        radio_player.set_active(False)

        radio_player.connect('activate', change_view, 'player')
        radio_all.connect('activate', change_view, 'all')

        widgets.get_widget('tool-volume').connect('clicked',
                                                  self.show_volume_window)

        # Registering a few GTK handlers
        widgets.get_widget('tool-previous').connect('clicked',
                                                    handler.on_tool_previous)
        widgets.get_widget('tool-stop').connect('clicked', handler.on_tool_stop)
        widgets.get_widget('tool-play').connect('clicked', handler.on_tool_play)
        widgets.get_widget('tool-next').connect('clicked', handler.on_tool_next)

        widgets.get_widget('tool-edit').connect('clicked', handler.on_tool_edit)
        widgets.get_widget('tool-lyrics').connect('clicked',
                                                  handler.on_tool_lyrics)
        widgets.get_widget('tool-reloadlyrics').connect('clicked',
                                                handler.on_tool_reloadlyrics)
        widgets.get_widget('tool-savelyrics').connect('clicked',
                                              handler.on_tool_savelyrics)

        widgets.get_widget('menu-show-fullscreen').connect('activate',
                                                   handler.on_menu_fullscreen)

        widgets.get_widget('menu-refresh').connect('activate',
                                                   handler.on_menu_refresh)
        widgets.get_widget('menu-quit').connect('activate',
                                                handler.on_menu_quit)
        widgets.get_widget('menu-prefs').connect('activate',
                                                 handler.on_menu_prefs)
        widgets.get_widget('menu-about').connect('activate',
                                                 handler.on_menu_about)

        extensions.load_event('OnBluemindoStarted', widgets)
        gtk_main()

    # This function handles the user's command for increase the volume
    def volume_more(self, datagram):
        is_step = datagram.split('--volume-more ')
        if len(is_step) == 2:
            step = float(is_step[1]) / 100.
        else:
            step = 0.1

        volume = self.volume + step
        self.change_volume(volume)

    # This function handles the user's command for decrease the volume
    def volume_less(self, datagram):
        is_step = datagram.split('--volume-less ')
        if len(is_step) == 2:
            step = float(is_step[1]) / 100.
        else:
            step = 0.1

        volume = self.volume - step
        self.change_volume(volume)

    # This function handles the user's command for change the volume
    def volume_set(self, volume):
        self.change_volume((float(volume) / 100.))

    # This function shows a popup window with a volume scale
    def show_volume_window(self, widget):
        if self.show_volume:
            # Close the window
            self.show_volume.destroy()
            self.show_volume = False
        else:
            # Create the popup window and place it near the cursor
            volume_widg = glade_XML(
                          functions.datadir + '/glade/volumecontrol.glade')
            main_window = self.widgets.get_widget('window1')
            volume_window = volume_widg.get_widget('window1')
            self.show_volume = volume_window

            x0, y0 = main_window.get_position()
            x1, y1 = main_window.get_pointer()
            volume_window.move((x0 + x1), (y0 + y1 + 30))

            volume_widg.get_widget('vscale1').set_value(self.volume * 100.0)

            def more_volume(widget):
                # Add a level to the vertical scale
                volume = self.volume + 0.1
                volume_widg.get_widget('vscale1').set_value(volume * 100.0)
                self.change_volume(volume)

            def less_volume(widget):
                # Delete a level to the vertical scale
                volume = self.volume - 0.1
                volume_widg.get_widget('vscale1').set_value(volume * 100.0)
                self.change_volume(volume)

            def adapt_volume(widget, gtkscroll, value):
                # Adapt the level with the vertical scale
                volume = (value / 100.0)
                self.change_volume(volume)

            # GTK handlers
            volume_widg.get_widget('button-more').connect(
                                   'clicked', more_volume)
            volume_widg.get_widget('button-less').connect(
                                   'clicked', less_volume)
            volume_widg.get_widget('vscale1').connect(
                                   'change-value', adapt_volume)

    # This function changes the volume
    def change_volume(self, volume):
        # Verify the volume
        if volume < 0.0:
            volume = 0.0
        elif volume > 1.0:
            volume = 1.0

        self.volume = volume

        # Send the new volume to Gstreamer
        player = gst.getplayer()
        player.set_property('volume', volume)

        # Change the icon
        if volume > 0.66:
            img = 'stock_volume-max'
        elif volume > 0.33:
            img = 'stock_volume-med'
        elif volume > 0:
            img = 'stock_volume-min'
        else:
            img = 'stock_volume-mute'

        self.widgets.get_widget('tool-volume-img').set_from_icon_name(img,
                                ICON_SIZE_LARGE_TOOLBAR)

        # Save the new volume level
        configfile = ConfigParser.ConfigParser()
        _file_ = join(config.confdir, 'Bluemindo.cfg')

        if exists(_file_):
            configfile.read(_file_)

            if configfile.has_section('Window'):
                configfile.set('Window', 'width', configfile.get('Window',
                                                                 'width'))
                configfile.set('Window', 'height', configfile.get('Window',
                                                                  'height'))
                configfile.set('Window', 'x', configfile.get('Window', 'x'))
                configfile.set('Window', 'y', configfile.get('Window', 'y'))

            configfile.set('Audio', 'volume', volume)
        else:
            configfile.add_section('Audio')
            configfile.set('Audio', 'volume', volume)

        configfile.write(open(_file_, 'w'))

def main():
    bluemindo = Bluemindo()
   
if __name__ == "__main__":
    main()
