# -*- coding: utf-8 -*-

#  Bluemindo
#  gajim.py

#    Bluemindo: A really simple but powerful audio player in Python/PyGTK.
#    Copyright (C) 2008  Ľubomír Remák
#    Copyright (C) 2008  Erwan Briand

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation version 3 of the License.

#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

from os import makedirs
from os.path import join, isdir, isfile, expanduser
from sys import exit
from gettext import gettext as _
from string import replace as str_replace
from gtk import (WIN_POS_CENTER_ALWAYS, MessageDialog, DIALOG_MODAL,
                 MESSAGE_INFO, BUTTONS_OK)

from gtk.glade import XML as glade_XML
import ConfigParser

from common.config import ConfigLoader
from common.functions import Functions

import dbus
import dbus.service
import dbus.glib

OBJ_PATH = '/org/gajim/dbus/RemoteObject'
INTERFACE = 'org.gajim.dbus.RemoteInterface'
SERVICE = 'org.gajim.dbus'
BASENAME = 'gajim-remote'

class Gajim:
    def __init__(self, plugin):
        self.plugin = {'name': 'Gajim',
                       'version': 0.1,
                       'logo': 'plugins/gajim/gajim_small.png',
                       'configurable': True,
                       'authors': 'Ľubomír Remák <lubomirr88@gmail.com>',
                       'license': 'GNU General Public License 3',
                       'description': _('Change Gajim status with the '
                                        'current playing song.')}

        plugin.connect('OnModuleConfiguration', self.loadconfig)
        plugin.connect('OnModuleConfigurationSave', self.saveconfig)
        plugin.connect('OnPlayNewSong', self.newsong)
        plugin.connect('OnStopPressed', self.stopsong)

        config = ConfigLoader()
        self.plugin_conf = join(config.confdir, 'plugins', 'gajim')
        if not isdir(self.plugin_conf):
            makedirs(self.plugin_conf)

        self.configfile = ConfigParser.ConfigParser()
        self.configfile_ = self.plugin['name'] + '.cfg'
        self.default_active = 0
        self.default_status = 0
        self.default_statusmsg = '♫ {artist} - {title} ♫'

        if not isfile(join(self.plugin_conf, self.configfile_)):
            self.write_default_config()

        self.functions = Functions()

        self.sbus = dbus.SessionBus()
        obj = self.sbus.get_object(SERVICE, OBJ_PATH)
        interface = dbus.Interface(obj, INTERFACE)
        self.method1 = interface.__getattr__('change_status')
        self.method2 = interface.__getattr__('get_status')

    def newsong(self, song):
        self.configfile.read(join(self.plugin_conf, self.configfile_))
        active = int(self.configfile.get(self.plugin['name'], 'active'))
        status = int(self.configfile.get(self.plugin['name'], 'status'))
        statusmsg = self.configfile.get(self.plugin['name'], 'statusmsg')

        if active == 1:
            track = song[0]
            title = song[1]
            artist = song[2]
            album = song[3]
            genre = song[4]
            length = song[5]

            def parse(str_):
                str_ = str_replace(str_, '{title}', title)
                str_ = str_replace(str_, '{artist}', artist)
                str_ = str_replace(str_, '{album}', album)
                str_ = str_replace(str_, '{track}', str(track))
                str_ = str_replace(str_, '{length}', length)
                str_ = str_replace(str_, '{genre}', genre)

                return str_

            if status == 1:
                gajim_status = 'online'
            elif status == 2:
                gajim_status = 'chat'
            elif status == 3:
                gajim_status = 'away'
            elif status == 4:
                gajim_status = 'xa'
            elif status == 5:
                gajim_status = 'dnd'
            else:
                gajim_status = self.getstatus()

            gajim_statusmsg = parse(statusmsg)
            self.changestatus(gajim_status, gajim_statusmsg);

    def stopsong(self):
        self.configfile.read(join(self.plugin_conf, self.configfile_))
        active = int(self.configfile.get(self.plugin['name'], 'active'))
        status = int(self.configfile.get(self.plugin['name'], 'status'))

        if active == 1:
            if status == 1:
                gajim_status = 'online'
            elif status == 2:
                gajim_status = 'chat'
            elif status == 3:
                gajim_status = 'away'
            elif status == 4:
                gajim_status = 'xa'
            elif status == 5:
                gajim_status = 'dnd'
            else:
                gajim_status = self.getstatus()

            gajim_statusmsg = ''
            self.changestatus(gajim_status, gajim_statusmsg);

    def changestatus(self, status, statusmsg):
        self.method1(dbus.String(status.decode('UTF-8')),
                     dbus.String(statusmsg.decode('UTF-8')), dbus.String(''))

    def getstatus(self):
        return self.method2(dbus.String(''))

    def saveconfig(self, extension):
        if extension == self.plugin['name']:
            try:
                check = self.conf_widgets.get_widget('checkbutton1')
                combo = self.conf_widgets.get_widget('combobox1')
                entry = self.conf_widgets.get_widget('entry1')

                check_val = int(check.get_active())
                combo_val = combo.get_active()
                entry_text = entry.get_text()

                widgets_retrieved = True
            except:
                widgets_retrieved = False

            if widgets_retrieved:
                try:
                    self.configfile.add_section(self.plugin['name'])
                except:
                    pass

                self.configfile.set(self.plugin['name'], 'active', check_val)
                self.configfile.set(self.plugin['name'], 'status', combo_val)
                self.configfile.set(self.plugin['name'], 'statusmsg', entry_text)

                self.configfile.write(open(join(self.plugin_conf,
                                                self.configfile_), 'w'))

    def loadconfig(self, (module, confglade)):
        if module == self.plugin['name']:
            self.conf_widgets = glade_XML('plugins/gajim/configuration.glade', 
                                          'vbox1', domain='bluemindo')

            hbox = confglade.get_widget('hbox2')

            try:
                kids = hbox.get_children()
                hbox.remove(kids[2])
            except:
                pass

            hbox.add(self.conf_widgets.get_widget('vbox1'))

            try:
                self.configfile.read(join(self.plugin_conf, self.configfile_))
                check_val = self.configfile.get(self.plugin['name'], 'active')
                combo_val = self.configfile.get(self.plugin['name'], 'status')
                entrytxt = self.configfile.get(self.plugin['name'], 'statusmsg')

                file_exists = True
            except:
                file_exists = False

            check = self.conf_widgets.get_widget('checkbutton1')
            combo = self.conf_widgets.get_widget('combobox1')
            entry = self.conf_widgets.get_widget('entry1')
            label = self.conf_widgets.get_widget('label4')

            if file_exists:
                check.set_active(int(check_val))
                combo.set_active(int(combo_val))
                entry.set_text(entrytxt)
                label.set_markup(
                _('Available tags:\n <b><tt>{title}</tt></b> - '
                  'Insert the title\n <b><tt>{artist}</tt></b> - '
                  'Insert the artist\n <b><tt>{album}</tt></b> - '
                  'Insert the album\n <b><tt>{track}</tt></b> - '
                  'Insert the track number\n <b><tt>{length}</tt></b> - '
                  'Insert the length\n <b><tt>{genre}</tt></b> - '
                  'Insert the genre'))

    def write_default_config(self):
        self.configfile.add_section(self.plugin['name'])
        self.configfile.set(self.plugin['name'], 'active', self.default_active)
        self.configfile.set(self.plugin['name'], 'status', self.default_status)
        self.configfile.set(self.plugin['name'],
                                        'statusmsg', self.default_statusmsg)

        self.configfile.write(open(join(self.plugin_conf,
                                        self.configfile_), 'w'))
